#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ccvt.h"
#include "VideoDevice.h"

#ifdef HAVE_PWCIOCTL_H
#include "pwc-ioctl.h"
#endif

/**
  \class CVideoDevice
  \brief A device wrapper for Video4Linux devices
  
  This class wraps itself around a Video4Linux device; at the moment it
  is primarely oriented at webcams, but grabber cards will be supported
  as well. This device can be 'opened' multiple times, so more than one
  class can use this device (of course they all get the same picture).
  
  The class can return video images in RGB, YUV or both formats; it also
  has an optional buffer scheme for the images; this can prevent the
  otherwise deep copy that is needed when, for example, image diffs
  need to be calculated.

  The class will use mmap() when it is supported by the device driver. Also
  some device support select(); if not, a QTimer is used. Use the Framerate
  functions to set the framerate or timer.
*/

/**
  \fn CVideoDevice::CVideoDevice(const QString &node_name)
  \param node_name The /dev device name of the video device
  
  The constructor will do some basic checks to see if this is a valid device.
  If yes, \ref IsValid() will return TRUE, otherwise FALSE.
 */  

CVideoDevice::CVideoDevice(const QString &node_name)
{
   Opened = 0;
   Buffers = 0;
   FrameRate = 25;
   PalRGB = 0;
   PalYUV = 0;
   vid_buffer = NULL;
   NodeName = node_name;
   pImageSocket = NULL;
   pImageTimer = NULL;
   validated = FALSE;
   CurrentInput = -1;

   Inputs.setAutoDelete(TRUE);

   CamFD = ::open(NodeName, O_RDONLY);
   Init(TRUE);
   if (CamFD < 0) {
     if (errno == EBUSY)
       validated = TRUE;
   }
   else {
     validated = TRUE;
     close(CamFD);
   }
   CamFD = -1;
}

/**
  \fn CVideoDevice::~CVideoDevice()
  
  Destroys the object; also closes the associated file descriptor
 */
CVideoDevice::~CVideoDevice()
{
   if (Opened > 1)
     qWarning("Warning: CVideoDevice `%s' was destroyed when it was in use more than once.", (const char *)NodeName);
   CleanUp();
   if (CamFD >= 0)
     close(CamFD);

   ResetImagesRGB();
   ResetImagesYUV();
   delete RGB;
   delete Y;
   delete U;
   delete V;
}

// private


/** 
  \brief Gather data from video device, like name, size, etc. Also initilizes buffers
  \param First Set to TRUE when called from constructor (init fase, don't allocate memory)
  
  This function must be called after every first open.
*/
void CVideoDevice::Init(bool First)
{
   char name[33];
   struct video_capability vcap;
   struct video_window vwin;
   int i;

   if (First) {
     RGB = NULL;
     Y = U = V = NULL;
     memset(&VMBuf, 0, sizeof(VMBuf));
     max_w = 0;
     max_h = 0;
     min_w = 0;
     min_h = 0;
   }

   if (CamFD < 0 || Opened > 1)
     return;
   
   if (ioctl(CamFD, VIDIOCGCAP, &vcap) < 0)
     return;
   max_w = vcap.maxwidth;
   max_h = vcap.maxheight;
   min_w = vcap.minwidth;
   min_h = vcap.minheight;
   strncpy(name, vcap.name, 32);
   name[32] = '\0';
   IntfName = name;

   if (ioctl(CamFD, VIDIOCGPICT, &VPic) < 0)
     return;
   Palette = VPic.palette; // To start with

   HasFramerate = FALSE;
   memset(&vwin, 0, sizeof(struct video_window));
   if (ioctl(CamFD, VIDIOCGWIN, &vwin) == 0) {
#if defined(PWC_FPS_SHIFT)
     if ((vwin.flags & PWC_FPS_FRMASK) >> PWC_FPS_SHIFT) {
       HasFramerate = TRUE;
//qDebug("VideoDevice supports framerate setting [%x].", vwin.flags);
     }
#endif
   }
   CurBuffer = 0;

   /* Query inputs */
   Inputs.clear();
   if (vcap.channels > 0) {
     Inputs.resize(vcap.channels);
     for (i = 0; i < vcap.channels; i++)
        Inputs.insert(i, new CVideoDeviceInput(this, i));
   }

   GetSize();

   /* See if we can use mmap(); */
   vid_buffer_size = image_buffer_size = image_w * image_h * 4;
   VMBuf.size = 0;
   VMBuf.frames = 0;
   if (ioctl(CamFD, VIDIOCGMBUF, &VMBuf) == 0) {
      vid_buffer_size = VMBuf.size;
      if (VMBuf.frames < Buffers) {
        qWarning("CVideoDevice::Init(): there are more buffers requested than MBUF can provide.\nFalling back to non-mmap()ed mode.\n");
        VMBuf.size = 0;
      }
   }
   
   /* Don't malloc memory at first probe */
   if (First)
     return;

   if (VMBuf.size > 0) {
      vid_buffer = (uchar *)mmap(NULL, vid_buffer_size, PROT_READ, MAP_SHARED, CamFD, 0);
      if (vid_buffer == (uchar *)-1) {
        qWarning("CVideoDevice::Init(): mmap() failed (%d). Falling back to non-mmap()ed mode.\n", errno);
        VMBuf.size = 0;
      }
   }
   if (VMBuf.size == 0) { // No mmap: use a single buffer for read() and hope the device queues for us.
     vid_buffer_size = image_buffer_size;
     vid_buffer = new uchar[vid_buffer_size];
   }

   RGB = new QImage[Buffers];
   Y   = new QImage[Buffers];
   U   = new QImage[Buffers];
   V   = new QImage[Buffers];
}

/** 
  \brief Cleans up mmap()ed stuff and buffers

  This function must be called after the last Close()
*/
void CVideoDevice::CleanUp()
{
   if (VMBuf.size > 0) {
     MSync(); // Just to make sure
     munmap(vid_buffer, vid_buffer_size);
     VMBuf.size = 0;
   }
   else {
     if (vid_buffer != NULL)
       delete vid_buffer;
   }
   vid_buffer = NULL;
}

/**
  \fn bool CVideoDevice::TryPalette(int pal, int depth)
  \brief Tries to set a VIDEO_PALETTE_* palette.
  \param pal One of the VIDEO_PALETTE_* palettes.
  \param depth visual depth (?) [Okay, CPIA driver, have it your way *GRRR*]
  \return \b TRUE on success, \b FALSE on failure
 */
bool CVideoDevice::TryPalette(int pal, int depth)
{
   VPic.palette = pal;
   VPic.depth = depth;
   if (ioctl(CamFD, VIDIOCSPICT, &VPic) < 0)
     return FALSE;
   /* Sigh. It was to be expected. The OV511 and IBMCam don't pay attention to the palette field */
   if (ioctl(CamFD, VIDIOCGPICT, &VPic) < 0)
     return FALSE;
   if (VPic.palette == pal) {
     Palette = pal;
     return TRUE;
   }
   return FALSE;
}

void CVideoDevice::SetPalette()
{
   /* Determine most preferable palette... With more and more 
      color conversion going into if the apps, we must try quite a few
      palettes before may hit one that we like.
      In case we want both YUV and RGB output, we prefer the YUV output.
    */
   Palette = 0;
   if (CamFD < 0)
     return;

   if (PalYUV) {
     TryPalette(VIDEO_PALETTE_YUV420P, 16) ||
     TryPalette(VIDEO_PALETTE_YUV420, 16)  ||
     TryPalette(VIDEO_PALETTE_YUYV, 16)    ||
     TryPalette(VIDEO_PALETTE_RGB24, 24)   ||
     TryPalette(VIDEO_PALETTE_RGB32, 32)   ||
     TryPalette(VIDEO_PALETTE_GREY, 8);
   }
   else if (PalRGB) {
     TryPalette(VIDEO_PALETTE_RGB32, 32)   ||
     TryPalette(VIDEO_PALETTE_RGB24, 24)   ||
     TryPalette(VIDEO_PALETTE_YUYV, 16)    ||
     TryPalette(VIDEO_PALETTE_YUV420, 16)  ||
     TryPalette(VIDEO_PALETTE_YUV420P, 16) ||
     TryPalette(VIDEO_PALETTE_GREY, 8);
   }
   qDebug("CVideoDevice::SetPalette picked palette %d", Palette);
}   

/** 
  \brief Do ioctl(MCAPTURE), start grabbing frame in buffer
  \param buf The mmap buffer (will be set modulo total buffers)
  
  This function will start the capture of an image into buffer \b buf
  with the current width & height
 */
int CVideoDevice::MCapture(int buf)
{
   struct video_mmap vm;
   if (VMBuf.size == 0)
     return 0; // ignore call

   vm.frame = buf % Buffers;
   vm.format = Palette;
   vm.width = image_w;
   vm.height = image_h;
#ifdef TRACE_VIDEODEV_READ
   printf("CVideoDevice::MCapture(): buffer %d, format %d, size (%dx%d)\n", CurBuffer, Palette, image_w, image_h);
#endif  
   if (ioctl(CamFD, VIDIOCMCAPTURE, &vm) < 0)
     return -errno;
   return 0;
}

/**
  \brief Do ioctl(SYNC), releasing buffer
*/
int CVideoDevice::MSync()
{
   if (VMBuf.size == 0)
     return 0; // ignore call

#ifdef TRACE_VIDEODEV_READ
   printf("CVideoDevice::MSync(): buffer %d\n", CurBuffer);
#endif  
   if (ioctl(CamFD, VIDIOCSYNC, &CurBuffer) < 0)
     return -errno;
   return 0;
}

void CVideoDevice::CreateImagesRGB()
{
   for (int i = 0; i < Buffers; i++)
      RGB[i].create(image_w, image_h, 32);
}

void CVideoDevice::ResetImagesRGB()
{
   if (RGB == NULL)
     return;
   for (int i = 0; i < Buffers; i++)
      RGB[i].reset();
}

void CVideoDevice::CreateImagesYUV()
{
   int i, c;

   for (i = 0; i < Buffers; i++) {
      Y[i].create(image_w,      image_h,      8, 256);
      U[i].create(image_w >> 1, image_h >> 1, 8, 256);
      V[i].create(image_w >> 1, image_h >> 1, 8, 256);
      for (c = 0; c < 256; c++) {
         // Default is gray palette
         Y[i].setColor(c, qRgb(c, c, c));
         U[i].setColor(c, qRgb(c, c, c));
         V[i].setColor(c, qRgb(c, c, c));
      }
   }
}

void CVideoDevice::ResetImagesYUV()
{
   if (Y == NULL || U == NULL || V == NULL)
     return;
   for (int i = 0; i < Buffers; i++) {
      Y[i].reset();
      U[i].reset();
      V[i].reset();
   }
}


// private slots

/**
  \brief stub for automated loading
*/
void CVideoDevice::LoadImage()
{
   int e;

   e = ReadImage();
   if (e < 0) {
     printf("Error loading image: %d\n", e);
     if (pImageTimer)
       pImageTimer->stop();
     /*
     if (pImageSocket)
       pImageSocket->setEnabled(FALSE);
     */
     emit Error(e);
   }
} 

// protected

// protected slots

// public

bool CVideoDevice::IsValid()
{
   return validated;
}

/**
  \brief Open the device, even multiple times
  \param buf Number of image buffers
  \return 0 upon success, otherwise \b -errno .
  
  This function increments the usage counter of the device; the first time
  this function is called the device is really opened and initialized; all
  subsequent calls will return 0. You will need as many calls to Close() as
  you placed to Open() to get the device really closed.
  
  Open() will also start the automatic loading of image. In case the device
  supports select() a QSocketNotifier is installed, otherwise a timer
  is used.
  
  For double buffering purposes, use a \b buf value > 1. The \b buf parameter
  can only be set at the first Open(); subsequent calls will have no
  effect on the number of buffers.
*/
int CVideoDevice::Open(int buf)
{
   if (Opened++) {
#ifdef TRACE_VIDEODEV_OPEN   
     qDebug("Opening CVideoDevice again (count = %d).", Opened);
#endif     
     return 0;
   }
     
   if (CamFD >= 0) {
     qWarning("Warning: VideoDevice already opened ?!?!");
     return 0;
   }
   CamFD = ::open(NodeName, O_RDONLY);
   if (CamFD < 0) {
     Opened = 0;
     return -errno;
   }

   Buffers = buf;
   if (Buffers < 1) Buffers = 1;
   if (Buffers > 8) Buffers = 8;
   Init();

   /* Determine if we can use select() on this cam. At the moment, 
      only the Philips cam has select().
      The BTTV cards and USB CPiA, IBMCAM & OV511 do not, however.
    */
   UseSelect = FALSE;
   if (IntfName.find("Philips") == 0)
     UseSelect = TRUE;

   if (UseSelect) {
     /* Tie into the Qt framework. Neat, huh? */
     pImageSocket = new QSocketNotifier(CamFD, QSocketNotifier::Read, this);
     connect(pImageSocket, SIGNAL(activated(int)), this, SLOT(LoadImage()));
   }
   else {
      /* For devices without select() we use a timer. */
     pImageTimer = new QTimer(this);
     connect(pImageTimer, SIGNAL(timeout()), this, SLOT(LoadImage()));
     pImageTimer->start(1000 / FrameRate); // default is 10 fps
   }
   
#ifdef TRACE_VIDEODEV_OPEN   
   qDebug("CVideoDevice::Open(%s, %d): using %s%s. Initial size (%dx%d)",
       (const char *)NodeName, buf,
       UseSelect ? "select()" : "timer",
       VMBuf.size > 0 ? " and mmap()" : "",
       image_w, image_h);
#endif
       
   if (MCapture(CurBuffer) < 0) // Start capture for mmap()
     perror("Starting mcapture: ");

   return 0;
}

/**
  \fn void CVideoDevice::Close()
  \brief Closes the device provisionally
  
  This function decrements the usage counter of the VideoDevice. If the
  counter reaches 0, the device is really closed. See also \ref Open().
 */
void CVideoDevice::Close()
{
   if (Opened) {
     if (Opened == 1) {
#ifdef TRACE_VIDEODEV_OPEN
       printf("CVideoDevice::Close(): last close.\n");
#endif       
       delete pImageTimer;
       pImageTimer = NULL;
       delete pImageSocket;
       pImageSocket = NULL;
       CleanUp();
       close(CamFD);
       CamFD = -1;
     }
     Opened--;
   }
}


/** 
  \fn int CVideoDevice::GetDescriptor() const
  \return file descriptor.
  
  This functions returns the file descriptor for this device. If the device
  is not opened, -1 is returned.
 */
int CVideoDevice::GetDescriptor() const
{
   return CamFD;
}



/** 
  \fn void CVideoDevice::EnableRGB(bool isOn)
  \brief Enable/disable retrieval of RGB image(s)
  
  This tells the object if RGB images are desired. This will help in
  selecting the proper PALETTE for the device. Both RGB and YUV images
  may be enabled. See also \ref EnableYUV.
  
  Multiple calls to EnableRGB(TRUE) will require the same amount of calls
  to EnableRGB(FALSE) to turn RGB retrieval completely off.
 */
void CVideoDevice::EnableRGB(bool isOn)
{
   if (isOn)
     PalRGB++;
   else
     PalRGB--;
   if (PalRGB < 0)
     qWarning("Warning: VideoDevice PalRGB is negative?\n");
   if (PalRGB == 0)
     ResetImagesRGB();
   if (isOn && PalRGB == 1)
     CreateImagesRGB();
   SetPalette();
}

/** 
  \fn void CVideoDevice::EnableYUV(bool isOn)
  \brief Enable/disable retrieval of YUV image(s)
  
  This tells the object if YUV (planar) images are desired. This will help in
  selecting the proper PALETTE for the device. Both YUV and RGB images
  may be enabled. See also \ref EnableRGB.

  Multiple calls to EnableYUV(TRUE) will require the same amount of calls
  to EnableYUV(FALSE) to turn YUV retrieval completely off.
 */
void CVideoDevice::EnableYUV(bool isOn)
{
   if (isOn)
     PalYUV++;
   else
     PalYUV--;
   if (PalYUV < 0)
     printf("Warning: VideoDevice PalYUV is negative?\n");
   if (PalYUV == 0)
     ResetImagesYUV();
   if (isOn && PalYUV == 1)
     CreateImagesYUV();
   SetPalette();
}

/** 
  \brief Returns device inode name
  
  This function returns the name of the device inode, like /dev/video0,
  /dev/video1, etc.
*/
QString CVideoDevice::GetNodeName() const
{
   return NodeName;
}

/**
  \brief Returns internal name of device.
  
  This function returns the name of the device through the V4L interface.
*/
QString CVideoDevice::GetIntfName() const
{
   return IntfName;
}
   

/**
  \fn QSize CVideoDevice::GetMinSize() const
  \brief Return the minimum image size this device supports.
  \return an Object of type QSize
  
  With this function the minium image size in pixels is retrieved. 
*/
QSize CVideoDevice::GetMinSize() const
{
   return QSize(min_w, min_h);
}

/**
  \fn QSize CVideoDevice::GetSize()
  \return An object of type QSize.
  \brief Return current size from the driver.

  Returns the current image size as reported by the device. Returns a 
  size of (0, 0) when the device is closed or an error occured.
 */  
QSize CVideoDevice::GetSize()
{
   struct video_window vwin;

   image_w = 0;
   image_h = 0;
   if (CamFD >= 0 && ioctl(CamFD, VIDIOCGWIN, &vwin) == 0) {
     image_w = vwin.width;
     image_h = vwin.height;
   }

//qDebug("CVideoDevice::GetSize() returns %dx%d", image_w, image_h);
   image_buffer_size = image_w * image_h * 4;
   return QSize(image_w, image_h);
}

/**
  \fn QSize CVideoDevice::GetMaxSize() const
  \brief Return the maximum image size this device supports.
  \return an Object of type QSize
  
  With this function the maximum image size in pixels is retrieved. See
  also \ref GetMinSize and \ref SetSize
  
  Not all sizes between the minimum and maximum size may be allowed by the
  device; however, there is currently no way to retrieve a list of possible
  sizes. It's safest to stick to CIF (352x288) and SIF (320x240) formats
  and subsizes hereof, and VGA. Also it's wise to use to multiples of
  8 in both directions. \ref SetSize will return FALSE when the size
  was rejected by the driver.
*/
QSize CVideoDevice::GetMaxSize() const
{
   return QSize(max_w, max_h);
}

/**
  \fn bool CVideoDevice::SetSize(int width, int height)
  \brief Set a new image size.
  \return FALSE when the call failed, TRUE when the new size was accepted by the device.

  This function will attempt to set a new image size; not all sizes between
  the minimum and maximum size may be allowed by the device; however, there
  is currently no way to retrieve a list of possible sizes. It is safest to
  stick to CIF (352x288) and SIF (320x240) formats and subsizes hereof, and
  VGA. Also it's wise to use to multiples of 8 in both directions. SetSize
  will return FALSE when the size was rejected by the driver.
 */  
bool CVideoDevice::SetSize(int width, int height)
{
   struct video_window vwin;

printf("CVideoDevice::SetSize(%d, %d)\n", width, height);
   if (CamFD < 0 || width > max_w || height > max_h)
     return FALSE;
   if (ioctl(CamFD, VIDIOCGWIN, &vwin) < 0) {
     perror("GWIN: ");
     return FALSE;
   }
   MSync(); // Drop current frame if we're mmap()ing

   vwin.width = width;
   vwin.height = height;
   vwin.clipcount = 0;
   if (ioctl(CamFD, VIDIOCSWIN, &vwin) < 0) {
     perror("SWIN");
     return FALSE;
   }
   /* Read back; it might be the hardware decides a bit differently 
      (for example, multiples of 8)
    */
   GetSize();
   if (image_w == 0 && image_h == 0) /* woops */
     return FALSE;

   MCapture(CurBuffer); // (opt) new frame 

   // Reset images
   if (PalRGB)
     CreateImagesRGB();
   if (PalYUV)
     CreateImagesYUV();
   emit Resized(QSize(image_w, image_h));
   return TRUE;
}

/**
  \override
*/
bool CVideoDevice::SetSize(const QSize &new_size)
{
   return SetSize(new_size.width(), new_size.height());
}


/**
  \brief Returns the current framerate.
  \return The framerate in frames per second.
  
  This applies to some webcams that allow setting of a framerate. In 
  case of a device that does not support select() we use the framerate
  to set the timer. By default the framerate is set to 25 (assuming a 
  videograbber card at PAL).
  
  Returns -1 in case of error.
 */
int CVideoDevice::GetFramerate() const
{
   struct video_window vwin;

   if (CamFD < 0)
     return -1;
   if (HasFramerate) {
     if (ioctl(CamFD, VIDIOCGWIN, &vwin) < 0)
       return -1;
#if defined(PWC_FPS_SHIFT)       
     return (vwin.flags & PWC_FPS_FRMASK) >> PWC_FPS_SHIFT;
#endif     
   }
   return FrameRate;
}

/**
  \brief Try to set a framerate
  \param fps The framerate, in frames per second.
  \param snap Use 'snapshot' mode
  \return TRUE if the framerate was accepted, FALSE otherwise
  
  Some webcams allow their framerate to be set; this functions tries to do
  just that; in general, the camera will use the framerate closest to what it
  supports. In case a device does not support framerates or only a fixed
  framerate (grabber cards!) we use the framerate to set the timer.
  
  Some cams can have a snapshot mode; the \b snap parameter enables this mode.
 */
bool CVideoDevice::SetFramerate(int fps, bool snap)
{
   struct video_window vwin;

   if (CamFD < 0)
     return FALSE;
   if (ioctl(CamFD, VIDIOCGWIN, &vwin) < 0)
     return FALSE;
   if (HasFramerate) { 
     MSync(); // Drop current frame
     FrameRate = fps;
     if (FrameRate <= 0)
       FrameRate = 0;
     if (FrameRate > 63)
       FrameRate = 63;
#if defined(PWC_FPS_SHIFT)
     vwin.flags = (vwin.flags & ~PWC_FPS_MASK) | (FrameRate << PWC_FPS_SHIFT);
     if (snap)
       vwin.flags |= PWC_FPS_SNAPSHOT;
printf("Setting framerate -> 0x%x\n", vwin.flags);
#endif     
     if (ioctl(CamFD, VIDIOCSWIN, &vwin) < 0)
       return FALSE;
     MCapture(CurBuffer); // Try to grab new frame
   }
   else {
     FrameRate = fps;
     if (FrameRate <= 0) 
       FrameRate = 1;
     if (FrameRate > 60)
       FrameRate = 60;
     if (pImageTimer)
       pImageTimer->start(1000 / FrameRate);
   }
   return TRUE;
}

/**
  \brief Return available buffers
  \return 0 if no mmap() support exists,
  \condition The device must be opened

  In case the device supports mmap(), this returns the number of buffers
  that are available and mapped.
*/
int CVideoDevice::GetMBuffers() const
{
   if (VMBuf.size == 0)
     return 0;
   return VMBuf.frames;
}  

/**
  \brief Return number of input channels (sources)

  See \ref CVideoDeviceInput. A device should report at least one
  channel, but buggy device drivers may choose to return 0.
*/
int CVideoDevice::GetInputs() const
{
   return Inputs.count();
}

/**
  \brief Return current input
  \return input number, or -1 if current input is unknown
  
  This will return the current input, if known. Unfortunately, there is 
  no way to query the video device for the selected channel so until
  \ref SelectInput() is called this function returns -1.
*/
int CVideoDevice::GetCurrentInput() const
{
   return CurrentInput;
}


/**
  \brief Returns an input channel (source) object
  \param number The desired input channel
  \return An object of type \ref CVideoDeviceInput, or NULL if \b number is out of range
  
*/
CVideoDeviceInput *CVideoDevice::GetInput(int number) const
{
   return Inputs.at((uint) number);
}

/**
  \brief Select a new input channel.
  
  This function will program the card/chip to use the selected input channel.
  Return TRUE if the call succeeded, otherwise FALSE.
*/
bool CVideoDevice::SelectInput(int number)
{
   struct video_channel arg;
   bool ret;

   ret = FALSE;
   if (CamFD >= 0 && (number >= 0 && number < (int)Inputs.count())) {
     arg.channel = number;
     if (ioctl(CamFD, VIDIOCGCHAN, &arg) == 0) {
       if (ioctl(CamFD, VIDIOCSCHAN, &arg) == 0) {
         CurrentInput = number;
         emit ChangedInput(number);
         ret = TRUE;
       }
#ifdef TRACE_VIDEODEV_IOCTL  
       else
         perror("SCHAN");
#endif   
     }
#ifdef TRACE_VIDEODEV_IOCTL  
     else
       perror("GCHAN");
#endif   
   }
   return ret;
}



/**
  \brief Return brightness setting of device.
  \return unsigned integer in the range 0-65535. 65535 may indicate setting is not available.
 */
int CVideoDevice::GetBrightness() const
{
   return VPic.brightness;
}

/**
  \fn bool CVideoDevice::SetBrightness(int val)
  \brief Set brightness in device
  \param val An integer in the range 0-65535.
  \return FALSE if the device is closed or the call failed.
  
  The value returned by GetBrightness may be slightly different from 
  what is set with SetBrightness.
*/
bool CVideoDevice::SetBrightness(int val)
{
   if (CamFD < 0)
     return FALSE;

   VPic.brightness = val & 0xffff;
   if (ioctl(CamFD, VIDIOCSPICT, &VPic) < 0)
     return FALSE;
   return TRUE;
}

/**
  \fn int CVideoDevice::GetContrast() const
  \brief Return contrast setting of device.
  \return unsigned integer in the range 0-65535. 65535 may indicate setting is not available.
 */
int CVideoDevice::GetContrast() const
{
   return VPic.contrast;
}

/**
  \fn bool CVideoDevice::SetContrast(int val)
  \brief Set contrast in device.
  \param val An integer in the range 0-65535.
  \return FALSE if the device is closed or the call failed.
  
  The value returned by GetContrast may be slightly different from 
  what is set with SetContrast.
*/
bool CVideoDevice::SetContrast(int val)
{
   if (CamFD < 0)
     return FALSE;

   VPic.contrast = val & 0xffff;
   if (ioctl(CamFD, VIDIOCSPICT, &VPic) < 0)
     return FALSE;
   return TRUE;
}

/**
  \fn int CVideoDevice::GetHue() const
  \brief Return hue (color shift) setting of device.
  \return unsigned integer in the range 0-65535. 65535 may indicate setting is not available.
  
  Hue is a way to correct for colour deviations. 
  It is something different than \ref GetColour.
 */
int CVideoDevice::GetHue() const
{
   return VPic.hue;
}

/**
  \fn bool CVideoDevice::SetHue(int val)
  \brief Set hue in device
  \param val An integer in the range 0-65535.
  \return FALSE if the device is closed or the call failed.
  
  Hue is a way to correct for colour deviations.
  The value returned by GetHue may be slightly different from 
  what is set with SetHue.
*/
bool CVideoDevice::SetHue(int val)
{
   if (CamFD < 0)
     return FALSE;

   VPic.hue = val & 0xffff;
   if (ioctl(CamFD, VIDIOCSPICT, &VPic) < 0)
     return FALSE;
   return TRUE;
}

/**
  \fn int CVideoDevice::GetColour() const
  \brief Return colour saturation setting of device.
  \return unsigned integer in the range 0-65535. 65535 may indicate setting is not available.
  
  A colour saturation of 0 means no colour at all, so the returned
  images are grayscale.
 */
int CVideoDevice::GetColour() const
{
   return VPic.colour;
}

/**
  \fn bool CVideoDevice::SetColour(int val)
  \brief Set colour saturation in device.
  \param val An integer in the range 0-65535.
  \return FALSE if the device is closed or the call failed.
  
  Colour saturation sets how bright colours should appear. A 
  saturation of 0 yields grayscale images.

  The value returned by GetColour may be slightly different from 
  what is set with SetColour.
*/
bool CVideoDevice::SetColour(int val)
{
   if (CamFD < 0)
     return FALSE;

   VPic.colour = val & 0xffff;
   if (ioctl(CamFD, VIDIOCSPICT, &VPic) < 0)
     return FALSE;
   return TRUE;
}

/**
  \fn int CVideoDevice::GetWhiteness() const
  \brief Return gamma setting of device.
  \return unsigned integer in the range 0-65535. 65535 may indicate setting is not available.
  
  Sometimes used as a brightness contrast, but more generally this returns
  the gamma correction the device applies to the image.
 */
int CVideoDevice::GetWhiteness() const
{
   return VPic.whiteness;
}

/**
  \fn bool CVideoDevice::SetWhiteness(int val)
  \brief Set gamma value in device
  \param val An integer in the range 0-65535.
  \return FALSE if the device is closed or the call failed.

  Whiteness is sometimes used as brightness, but usually this sets the
  gamma correction the device will apply to the image.
  
  The value returned by GetWhiteness may be slightly different from 
  what is set with SetWhiteness.
*/
bool CVideoDevice::SetWhiteness(int val)
{
   if (CamFD < 0)
     return FALSE;

   VPic.whiteness = val & 0xffff;
   if (ioctl(CamFD, VIDIOCSPICT, &VPic) < 0)
     return FALSE;
   return TRUE;
}


/**
  \fn int CVideoDevice::ReadImage()
  \brief Read image into Buffers.
  \return 0 on success, \b -errno otherwise.
  
  This function reads the raw data from the device and transforms it into
  RGB and/or YUV images, doing all necessary conversions.
 */
int CVideoDevice::ReadImage()
{
   uchar *dst, *src, *dy, *du, *dv;
   QRgb *drgb;
   int i, n;

   if (CamFD < 0)
     return -ENODEV;
   if (vid_buffer == NULL)
     return -ENOMEM;
     
   if (VMBuf.size > 0) { // mmap()ed.
     if (MSync() < 0)
       return -errno;
     src = vid_buffer + VMBuf.offsets[CurBuffer];
     if (Buffers > 1) // we have space, start capture immediately in next buffer. Otherwise, see end of function
       MCapture(CurBuffer + 1); 
   }
   else {
     if (read(CamFD, vid_buffer, image_buffer_size) < 0)
       return -errno;
     src = vid_buffer; 
   }
     
   dst = RGB[CurBuffer].bits();
   dy = Y[CurBuffer].bits();
   du = U[CurBuffer].bits();
   dv = V[CurBuffer].bits();
   n = image_w * image_h;
   
   /* Okay, we have our (raw) data. Let's see what needs to be done... */
   switch (Palette) {
     case VIDEO_PALETTE_RGB32:
       /* Quick-n-dirty palette */
       if (PalRGB)
         memcpy(dst, src, n * 4);
       if (PalYUV) {
         // convert to YUV planes
       }
       break;

     case VIDEO_PALETTE_RGB24:
       /* "Broken" BGR palette */
       if (PalRGB) {
         drgb = (QRgb *)dst;
         for (i = 0; i < n; i++) {
            *drgb++ = qRgb(src[2], src[1], src[0]);
            src += 3;
         }
       }
       if (PalYUV)
         ccvt_bgr24_420p(image_w, image_h, vid_buffer, dy, du, dv);
       break;

     case VIDEO_PALETTE_YUV420P:
       // Planar format... easy :)
       if (PalYUV) {
         if (dy != NULL)
           memcpy(dy, src, n);
         src += n;
         n /= 4; /* UV boxes are only a quarter size */
         if (du != NULL)
           memcpy(du, src, n);
         src += n;
         if (dv != NULL)
           memcpy(dv, src, n);
       }
       else {
         dy = src;
         du = src + n;
         dv = du + (n / 4);
       }
      
       if (PalRGB) 
         ccvt_420p_bgr32(image_w, image_h, dy, du, dv, dst);
       break;

     case VIDEO_PALETTE_YUV420:
       // Non planar format... Hmm...
       if (PalYUV)
         ccvt_420i_420p(image_w, image_h, src, dy, du, dv);
       if (PalRGB)
         ccvt_420i_bgr32(image_w, image_h, src, dst);
       break;
       
     case VIDEO_PALETTE_YUYV:
       // Hmm. CPiA cam has this nativly. Anyway, it's just another format. The last one, as far as I'm concerned ;)
       // No assembly yet.
       if (PalYUV) {
         int l, j;

         for (; n > 0; n--) {
            *dy = *src;
            dy++;
            src += 2;
         }
         src = vid_buffer + 1;
         for (l = 0; l < image_h; l += 2) {
            for (j = 0; j < image_w; j += 2) {
               *du = *src;
               du++;
               src += 2;
               *dv = *src;
               dv++;
               src += 2;
            }
            src += image_w * 2;
         }
       }
         
       if (PalRGB) {
         /* we need to work this one out... */
         src = vid_buffer;
         ccvt_yuyv_bgr32(image_w, image_h, src, dst);
       }
       break;
   }

   emit Notify();

   /* Go to next buffer (including capture). In case of 1 buffer CurBuffer
      will get stuck at 0, so we do this after we processed the buffer
      content.
    */
   CurBuffer = (CurBuffer + 1) % Buffers;
   if (Buffers == 1) {
     if (MCapture(CurBuffer) < 0)
       return -errno;
   }

   return 0;
}

/**
  \fn QImage *CVideoDevice::GetRGB(int offset) const
  \param offset Offset in images buffer.
  \brief Get an RGB image.
  
  Retrieve pointer to an RGB image; note that this is a pointer, not a
  (shallow) copy. The QImage is a 32 bit deep image.
  
  When buffering is active any of the previous images in the buffer can be
  retrieved. The 'offset' parameter indicates the negative offset in the
  (circular) list of images. When offset = 0 (the default) the latest image
  is returned; when offset = 1 is the previous image, offset = 2 the image
  before that, etc. up to the number of Buffers - 1.

  If <b>offset</b> is outside the range a NULL pointer will be returned.
 */
QImage *CVideoDevice::GetRGB(int offset) const
{
   if (offset < 0 || offset >= Buffers)
     return NULL;
   return &RGB[(Buffers + CurBuffer - offset) % Buffers];
}

/**
  \fn QImage *CVideoDevice::GetY(int offset) const
  \param offset Offset in circular buffer.
  \brief Get Y (luminance) component of image.
  
  Retrieve pointer to a Y image; note that this is a pointer, not a
  (shallow) copy. The QImage is an 8 bit deep image with grayscale palette.
  
  See \ref GetRGB about double buffering.
 */
QImage *CVideoDevice::GetY(int offset) const
{
   if (offset < 0 || offset >= Buffers)
     return NULL;
   return &Y[(Buffers + CurBuffer - offset) % Buffers];
}

/**
  \fn QImage *CVideoDevice::GetU(int offset) const
  \param offset Offset in circular buffer.
  \brief Get U (chrominance) component of image.
  
  Retrieve pointer to a U image; note that this is a pointer, not a
  (shallow) copy. The QImage is an 8 bit deep image with grayscale palette.
  
  See \ref GetRGB about double buffering.
 */
QImage *CVideoDevice::GetU(int offset) const
{
   if (offset < 0 || offset >= Buffers)
     return NULL;
   return &U[(Buffers + CurBuffer - offset) % Buffers];
}

/**
  \fn QImage *CVideoDevice::GetV(int offset) const
  \param offset Offset in circular buffer.
  \brief Get V (chrominance) component of image.
  
  Retrieve pointer to a V image; note that this is a pointer, not a
  (shallow) copy. The QImage is an 8 bit deep image with grayscale palette.
  
  See \ref GetRGB about double buffering.
 */
QImage *CVideoDevice::GetV(int offset) const
{
   if (offset < 0 || offset >= Buffers)
     return NULL;
   return &V[(Buffers + CurBuffer - offset) % Buffers];
}
