/*
 * KON2 - Kanji ON Console -
 * Copyright (C) 1992-1996 Takashi MANABE (manabe@papilio.tutics.tut.ac.jp)
 *
 * CCE - Console Chinese Environment -
 * Copyright (C) 1998-1999 Rui He (herui@cs.duke.edu)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */

/*
  This code is based on framebuffer code.
  */

#if defined(linux)

#include	<stdio.h>
#include	<fcntl.h>
#include	<termios.h>
#include	<string.h>
#include	<unistd.h>
#include	<linux/types.h>
#include	<sys/mman.h>
#include	<stdlib.h>
#include	<linux/fb.h>

#include	<getcap.h>
#include	<defs.h>
#include	<errors.h>
#include	<vc.h>
#include	<vt.h>
#include	<vga.h>
#include	<font.h>

static union 
{
    u_short cfb16[16];
    u_long  cfb24[16];
    u_long  cfb32[16];
} fb_cmap;

u_short savedRed[16], savedGreen[16],savedBlue[16];

/***********************************************************************
 *                        Palette Operations                           *
 ***********************************************************************/

//#define MAX_PELS        16

static void VgaFBSetPels(u_short *red, u_short *green, u_short *blue)
{
   int fd;
   struct fb_cmap cmap;

   fd = open("/dev/fb",O_RDWR);

   cmap.start = 0;
   cmap.len = MAX_PELS;
   cmap.red = red;
   cmap.green = green;
   cmap.blue = blue;

   if (ioctl(fd, FBIOPUTCMAP, &cmap) == -1)
     fatal("FBIOPUTCMAP error!\n");
   close(fd);
}

static void VgaFBGetPels(u_short *red, u_short *green, u_short *blue)
{
    int fd;
    struct fb_cmap cmap;

    fd = open("/dev/fb", O_RDWR);

    cmap.red = red;
    cmap.green = green;
    cmap.blue = blue;
    cmap.start = 0;
    cmap.len = MAX_PELS;

    ioctl(fd, FBIOGETCMAP, &cmap);

    close(fd);
}

/***************************************************************************
 *                                VgaFB routines                             *
 ***************************************************************************/

void VgaFBSetCursorAddress(CursorInfo *ci, u_int x, u_int y)
{
    ci->addr = y * dispInfo.tlineByte + cursorTop
		* dispInfo.glineByte + x * dispInfo.bpp;

}

/* p is index, 0-4095  */
void VgaFBSetAddress(u_int p)
{
    writeAddr = (p % dispInfo.txmax) * dispInfo.bpp  + 
           (p / dispInfo.txmax) * dispInfo.tlineByte;
} 

void VgaFBSetInputAddress(u_int p)
{
   writeAddr = (dispInfo.gydim - dispInfo.input) * dispInfo.glineByte + 
       (p % dispInfo.tx_avail) * dispInfo.bpp + 
        6 * dispInfo.glineByte ;  
}

/*************************************************************************
 *              Read the Palette/Register Values from cce.cfg            *
 *************************************************************************/

void VgaFBInitColorMap(void)
{
    int i;
    u_int red, green, blue;
    int fd;
  //  struct fb_fix_screeninfo fix;
    struct fb_var_screeninfo var;

    fd = open("/dev/fb", O_RDWR);

  //  ioctl(fd, FBIOGET_FSCREENINFO, &fix);
    ioctl(fd, FBIOGET_VSCREENINFO, &var);

    for(i = 0; i < 16; i++)
    {
        red = red16[i];
        green = green16[i];
        blue = blue16[i];

        switch(var.bits_per_pixel)
        { 
        case 15:  /* 1:5:5:5 */
            fb_cmap.cfb16[i] =  ((red   & 0xf800) >> 1) |
                                ((green & 0xf800) >> 6) |
                                ((blue  & 0xf800) >> 11); 

            break;
        case 16:  /* 0:5:6:5 */
            fb_cmap.cfb16[i] =  ((red   & 0xf800) ) |
                                ((green & 0xfc00) >> 5) |
                                ((blue  & 0xf800) >> 11);
            break;
        case 24:
            fb_cmap.cfb24[i] = ((red  & 0xff00) << 8) | 
                               (green & 0xff00) | 
                               (blue >> 8);
            break;
        case 32:
            fb_cmap.cfb32[i] = ((red  & 0xff00) << 8) |
                               (green & 0xff00) |
                               (blue >> 8);
            break;            
        }
    }

   close(fd);
}

void VgaFBInitRegs(void)
{
    int fd;
    int fontwidth = 8, fontheight = 16;

    struct fb_fix_screeninfo fix;
    struct fb_var_screeninfo var;

    fd = open("/dev/fb", O_RDWR);

    ioctl(fd, FBIOGET_FSCREENINFO, &fix);
    ioctl(fd, FBIOGET_VSCREENINFO, &var);

    dispInfo.gxdim = var.xres;
    dispInfo.gydim = var.yres;

    dispInfo.input = 30;

    dispInfo.tx_avail = dispInfo.gxdim / fontwidth;
    if (dispInfo.linegap < 0 || dispInfo.linegap > 10)
         dispInfo.linegap = 2;
    dispInfo.glineChar = fontheight + dispInfo.linegap;
    dispInfo.ty_avail = (dispInfo.gydim - dispInfo.input) / dispInfo.glineChar;

    if (dispInfo.txmax <= 0 || dispInfo.txmax > dispInfo.tx_avail)
           dispInfo.txmax = dispInfo.tx_avail;
    if (dispInfo.tymax <= 0 || dispInfo.tymax > dispInfo.ty_avail)
           dispInfo.tymax = dispInfo.ty_avail;

    dispInfo.bpp = var.bits_per_pixel;
    if (dispInfo.bpp == 15) dispInfo.bpp = 16; 
     /* 2/4/8/15/16/24/32 */

    dispInfo.glineByte = fix.line_length;
    dispInfo.gsize = dispInfo.glineByte * dispInfo.gydim;
    dispInfo.tlineByte = dispInfo.glineChar * dispInfo.glineByte;

    dispInfo.tsize = dispInfo.tlineByte * dispInfo.tymax;

    close(fd);
}

/* VGA initialize & uninitialize */

int VgaFBAttach(void)
{
    int	devFb;

    if ((devFb = open("/dev/fb", O_RDWR) ) < 0) 
    {
	Perror("/dev/fb");
	return FAILURE;
    }

    gramMem = (u_char *)mmap((__ptr_t)0, dispInfo.gsize,
          PROT_READ|PROT_WRITE, MAP_SHARED, devFb, 0);

    close(devFb);

    if ((long)gramMem < 0) 
    {
	Perror("mmap");
	return FAILURE;
    }
    
    return SUCCESS;
}

void VgaFBDetach(void)
{
    munmap(gramMem, dispInfo.gsize);
}

/***********************************************************************
 *                           Configure                                 *
 ***********************************************************************/

void VgaFBInit(void)
{
    VgaFBInitRegs();
    VgaFBInitColorMap();

    VgaFBAttach();
}

void VgaFBScreenSaver(bool blank)
{
}
 
/****************************************************************************
 *                 4 Bits ( 16 Color) Frame Buffer Operation                *
 *     Part of the code from /usr/src/linux/drivers/video/fbcon-cfb4.c      *
 ****************************************************************************/

static u_short nibbletab_cfb4[] = 
{
    0x0000,0xf000,0x0f00,0xff00,
    0x00f0,0xf0f0,0x0ff0,0xfff0,
    0x000f,0xf00f,0x0f0f,0xff0f,
    0x00ff,0xf0ff,0x0fff,0xffff
};

void fbcon_cfb4_clear(int sy, int height, u_char bc)
{
    u_char *dest, *p, bgx;
    int i;

    dest = gramMem + sy * dispInfo.glineByte;

    bgx = bc & 0x0F;
    bgx |= bgx << 4;
    while( height-- > 0)
    {
       p = dest;
       for(i = 0; i < dispInfo.gxdim/2; i++)
       {
          *p++ = bgx;
       }
       dest += dispInfo.glineByte;
    }
}

void fbcon_cfb4_putc(u_char *cdat, u_char fc, u_char bc, int fontwidth)
{
        u_char *dest, bits;
        int rows, fontheight = 16;
        u_int eorx,fgx,bgx;

        dest = gramMem + writeAddr;

        fgx = fc;
        bgx = bc;
        fgx |= (fgx << 4);
        fgx |= (fgx << 8);
        bgx |= (bgx << 4);
        bgx |= (bgx << 8);
        eorx = fgx ^ bgx;

        for (rows = fontheight ; rows-- ; dest += dispInfo.glineByte) 
        {
             bits = *cdat++;
             ((u_short *)dest)[0]= (nibbletab_cfb4[bits >> 4] & eorx) ^ bgx;
             if (fontwidth < 8) continue; // fontwidth = 4

             ((u_short *)dest)[1] = (nibbletab_cfb4[bits & 0xf] & eorx) ^ bgx;
             if (fontwidth < 12) continue; // fontwidth = 8
 
             bits = *cdat++;
             ((u_short *)dest)[2]= (nibbletab_cfb4[bits >> 4] & eorx) ^ bgx;
             if (fontwidth < 16) continue; // fontwidth = 12

             ((u_short *)dest)[3] = (nibbletab_cfb4[bits & 0xf] & eorx) ^ bgx;
             if (fontwidth < 20) continue; // fontwidth = 16

             bits = *cdat++;
             ((u_short *)dest)[4]= (nibbletab_cfb4[bits >> 4] & eorx) ^ bgx;
             if (fontwidth < 24) continue; // fontwidth = 20

             ((u_short *)dest)[5] = (nibbletab_cfb4[bits & 0xf] & eorx) ^ bgx;
             // fontwidth = 24

        }
}


/****************************************************************************
 *                 8 Bits (256 Color) Frame Buffer Operation                *
 *     Part of the code from /usr/src/linux/drivers/video/fbcon-cfb8.c      *
 ****************************************************************************/

static u_int nibbletab_cfb8[] = 
{
    0x00000000,0xff000000,0x00ff0000,0xffff0000,
    0x0000ff00,0xff00ff00,0x00ffff00,0xffffff00,
    0x000000ff,0xff0000ff,0x00ff00ff,0xffff00ff,
    0x0000ffff,0xff00ffff,0x00ffffff,0xffffffff
};

void fbcon_cfb8_clear(int sy, int height, u_char bc)
{
    u_char *dest,*p;
    int i;

    dest = gramMem + sy * dispInfo.glineByte;

    while( height-- > 0)
    {
       p = dest;
       for(i = 0; i < dispInfo.gxdim; i++)
       {
          *p++ = bc;
       }
       dest += dispInfo.glineByte;
    }
}

void fbcon_cfb8_putc(u_char *cdat, u_char fc, u_char bc, int fontwidth)
{
    u_char *dest, bits;
    int fontheight = 16,rows;
    u_int eorx,fgx,bgx;

    dest = gramMem + writeAddr;

    fgx = fc;
    bgx = bc;
    fgx |= (fgx << 8);
    fgx |= (fgx << 16);
    bgx |= (bgx << 8);
    bgx |= (bgx << 16);
    eorx = fgx ^ bgx;

    for (rows = fontheight ; rows-- ; dest += dispInfo.glineByte)
    {
         bits = *cdat++;
         ((u_int *)dest)[0]= (nibbletab_cfb8[bits >> 4] & eorx) ^ bgx;
         if (fontwidth < 8) continue; // fontwidth = 4

         ((u_int *)dest)[1]= (nibbletab_cfb8[bits & 0xf] & eorx) ^ bgx;
         if (fontwidth < 12) continue; // fontwidth = 8
         
         bits = *cdat++;
         ((u_int *)dest)[2]= (nibbletab_cfb8[bits >> 4] & eorx) ^ bgx;
         if (fontwidth < 16) continue; // fontwidth = 12

         ((u_int *)dest)[3]= (nibbletab_cfb8[bits & 0xf] & eorx) ^ bgx;
         if (fontwidth < 20) continue; // fontwidth = 16

         bits = *cdat++;
         ((u_int *)dest)[4]= (nibbletab_cfb8[bits >> 4] & eorx) ^ bgx;
         if (fontwidth < 24) continue; // fontwidth = 20

         ((u_int *)dest)[5]= (nibbletab_cfb8[bits & 0xf] & eorx) ^ bgx;
          // fontwidth = 24
    }
}

/****************************************************************************
 *                16 Bits (64K Color) Frame Buffer Operation                *
 *     Part of the code from /usr/src/linux/drivers/video/fbcon-cfb16.c     *
 ****************************************************************************/

static u_int nibbletab_cfb16[] = 
{
    0x00000000, 0xffff0000, 0x0000ffff, 0xffffffff
};

void fbcon_cfb16_clear(int sy, int height, u_char bc)
{
    u_char *dest;
    u_short *p;
    int i;

    dest = gramMem + sy * dispInfo.glineByte;

    while( height-- > 0)
    {
       p = (u_short*)dest;
       for(i = 0; i < dispInfo.gxdim; i++)
       {
          *p++ = fb_cmap.cfb16[bc];
          *p++ = fb_cmap.cfb16[bc];
       }
       dest += dispInfo.glineByte;
    }
}

void fbcon_cfb16_putc(u_char *cdat, u_char fc, u_char bc,int fontwidth)
{
    u_char *dest, bits;
    int rows;
    u_int eorx, fgx, bgx;
    int fontheight = 16;

    dest = gramMem + writeAddr;

    fgx = fb_cmap.cfb16[fc];
    bgx = fb_cmap.cfb16[bc];

    fgx |= (fgx << 16);
    bgx |= (bgx << 16);
    eorx = fgx ^ bgx;

    for (rows = fontheight; rows--; dest += dispInfo.glineByte) 
    {
         bits = *cdat++;
         ((u_int *)dest)[0] = (nibbletab_cfb16[bits >> 6] & eorx) ^ bgx;
         ((u_int *)dest)[1] = (nibbletab_cfb16[bits >> 4 & 3] & eorx) ^ bgx;
         if (fontwidth < 8)  continue;  // fontwidth = 4

         ((u_int *)dest)[2] = (nibbletab_cfb16[bits >> 2 & 3] & eorx) ^ bgx;
         ((u_int *)dest)[3] = (nibbletab_cfb16[bits & 3] & eorx) ^ bgx;
         if (fontwidth < 12) continue;  // fontwidth = 8
 
         bits = *cdat++;
         ((u_int *)dest)[4] = (nibbletab_cfb16[bits >> 6] & eorx) ^ bgx;
         ((u_int *)dest)[5] = (nibbletab_cfb16[bits >> 4 & 3] & eorx) ^ bgx;
         if (fontwidth < 16) continue;  // fontwidth = 12

         ((u_int *)dest)[6] = (nibbletab_cfb16[bits >> 2 & 3] & eorx) ^ bgx;
         ((u_int *)dest)[7] = (nibbletab_cfb16[bits & 3] & eorx) ^ bgx;
         if (fontwidth < 20) continue;  // fontwidth = 16

         bits = *cdat++;
         ((u_int *)dest)[8] = (nibbletab_cfb16[bits >> 6] & eorx) ^ bgx;
         ((u_int *)dest)[9] = (nibbletab_cfb16[bits >> 4 & 3] & eorx) ^ bgx;
         if (fontwidth < 16) continue;  // fontwidth = 20

         ((u_int *)dest)[10] = (nibbletab_cfb16[bits >> 2 & 3] & eorx) ^ bgx;
         ((u_int *)dest)[11] = (nibbletab_cfb16[bits & 3] & eorx) ^ bgx;
                  // fontwidth = 24
    }
}


/****************************************************************************
 *                24 Bits (16M Color) Frame Buffer Operation                *
 *     Part of the code from /usr/src/linux/drivers/video/fbcon-cfb24.c     *
 ****************************************************************************/

static inline void store4pixels(u_int d1, u_int d2, u_int d3, u_int d4, u_int *dest)
{
    *dest++ = d1         | (d2 << 24);
    *dest++ = (d2 >> 8)  | (d3 << 16);
    *dest++ = (d3 >> 16) | (d4 << 8);
}

 void fbcon_cfb24_clear(int sy, int height, u_char bc)
{
    u_char *dest,*p, red, green, blue;
    int i;

    dest = gramMem + sy * dispInfo.glineByte;

    red = red16[bc] >> 8;
    green = green16[bc] >> 8;
    blue = blue16[bc] >> 8;

    while( height-- > 0)
    {
       p = dest;
       for(i = 0; i < dispInfo.gxdim; i++)
       {
          *p++ = blue;
          *p++ = green;
          *p++ = red;
       }  
       dest += dispInfo.glineByte;
    }
}

void fbcon_cfb24_putc(u_char *cdat, u_char fc, u_char bc, int fontwidth)
{
    u_char *dest, bits;
    int rows;
    u_int eorx, fgx, bgx, d1, d2, d3, d4;
    int fontheight = 16;

   if (cdat == NULL) return;

    dest = gramMem + writeAddr;

    fgx = fb_cmap.cfb24[fc];
    bgx = fb_cmap.cfb24[bc];
    eorx = fgx ^ bgx;

    for (rows = fontheight; rows--; dest += dispInfo.glineByte) 
    {
        bits = *cdat++;
        d1 = (-(bits >> 7) & eorx) ^ bgx;
        d2 = (-(bits >> 6 & 1) & eorx) ^ bgx;
        d3 = (-(bits >> 5 & 1) & eorx) ^ bgx;
        d4 = (-(bits >> 4 & 1) & eorx) ^ bgx;
        store4pixels(d1, d2, d3, d4, (u_int *)dest);
        if (fontwidth < 8) continue;  // fontwidth = 4

        d1 = (-(bits >> 3 & 1) & eorx) ^ bgx;
        d2 = (-(bits >> 2 & 1) & eorx) ^ bgx;
        d3 = (-(bits >> 1 & 1) & eorx) ^ bgx;
        d4 = (-(bits & 1) & eorx) ^ bgx;
        store4pixels(d1, d2, d3, d4, (u_int *)(dest+12));
        if (fontwidth < 12) continue; // fontwidth = 8

        bits = *cdat++;
        d1 = (-(bits >> 7) & eorx) ^ bgx;
        d2 = (-(bits >> 6 & 1) & eorx) ^ bgx;
        d3 = (-(bits >> 5 & 1) & eorx) ^ bgx;
        d4 = (-(bits >> 4 & 1) & eorx) ^ bgx;
        store4pixels(d1, d2, d3, d4, (u_int *)(dest+24));
        if (fontwidth < 16) continue;  // fontwidth = 12

        d1 = (-(bits >> 3 & 1) & eorx) ^ bgx;
        d2 = (-(bits >> 2 & 1) & eorx) ^ bgx;
        d3 = (-(bits >> 1 & 1) & eorx) ^ bgx;
        d4 = (-(bits & 1) & eorx) ^ bgx;
        store4pixels(d1, d2, d3, d4, (u_int *)(dest+36));
        if (fontwidth < 20) continue;  // fontwidth = 16

        bits = *cdat++;
        d1 = (-(bits >> 7) & eorx) ^ bgx;
        d2 = (-(bits >> 6 & 1) & eorx) ^ bgx;
        d3 = (-(bits >> 5 & 1) & eorx) ^ bgx;
        d4 = (-(bits >> 4 & 1) & eorx) ^ bgx;
        store4pixels(d1, d2, d3, d4, (u_int *)(dest+48));
        if (fontwidth < 24) continue;  // fontwidth = 20

        d1 = (-(bits >> 3 & 1) & eorx) ^ bgx;
        d2 = (-(bits >> 2 & 1) & eorx) ^ bgx;
        d3 = (-(bits >> 1 & 1) & eorx) ^ bgx;
        d4 = (-(bits & 1) & eorx) ^ bgx;
        store4pixels(d1, d2, d3, d4, (u_int *)(dest+60));
           // fontwidth = 24
    }
}

/* XOR all 1s, for 2/4/8 bit, they are using colormaps, can't do it in this way
  */

void VgaFBCursor(CursorInfo *ci)
{
    char       *gram;
    u_char      x;
    int         i;
    int bottom = ( cursorBtm + 1 <= dispInfo.glineChar ?
        cursorBtm + 1 : dispInfo.glineChar );

    gram = gramMem + ci->addr;

    x = cursorTop;
    if (kanjiCursor && ci->kanji)
    {
        for (;x < bottom;x ++, gram += dispInfo.glineByte)
        {
            for( i = 0; i < dispInfo.bpp*2; i++)
               if (dispInfo.bpp <= 8)  // palette-based
                   gram[i] ^= 0x07;
               else gram[i] ^= 0xFF;
        }
    }
    else
    {
        for (;x < bottom;x ++, gram += dispInfo.glineByte)
        {
            for( i = 0; i < dispInfo.bpp; i++)
               if (dispInfo.bpp <= 8)
                    gram[i] ^= 0x07;
               else gram[i] ^= 0xFF;
        }
    }
}


void VgaFBClearAll(int color)
{
   switch(dispInfo.bpp)
    {
    case 4:
        fbcon_cfb4_clear(0, dispInfo.gydim- dispInfo.input, color);
        break;
    case 8:
        fbcon_cfb8_clear(0, dispInfo.gydim-dispInfo.input, color);
        break;
    case 16:
        fbcon_cfb16_clear(0, dispInfo.gydim-dispInfo.input, color);
        break;
    case 24:
        fbcon_cfb24_clear(0, dispInfo.gydim-dispInfo.input, color);
        break;
    case 32:
    //    fbcon_cfb24_clear(0, dispInfo.gydim-dispInfo.input, color);
        break;
    }
}

void VgaFBClearInput(int color)
{
   switch(dispInfo.bpp)
    {
    case 4:
        fbcon_cfb4_clear(dispInfo.gydim-dispInfo.input, dispInfo.input, color);
        break;
    case 8:
        fbcon_cfb8_clear(dispInfo.gydim-dispInfo.input, dispInfo.input, color);
        break;
    case 16:
        fbcon_cfb16_clear(dispInfo.gydim-dispInfo.input, dispInfo.input, color);
        break;
    case 24:
        fbcon_cfb24_clear(dispInfo.gydim-dispInfo.input, dispInfo.input, color);
        break;
    case 32:
    //    fbcon_cfb24_clear(dispInfo.gydim-dispInfo.input, dispInfo.input, color);
        break;
    }
}


/***********************************************************************
 *                Single/Double Character Output                       *
 ***********************************************************************/

/* After set the writing address(writeAddr), call this to output the
  value gramMem is the absolute starting address */

void VgaFBWput(u_char *code, u_char fc, u_char bc)
{
    switch(dispInfo.bpp)
    {
    case 4:
        fbcon_cfb4_putc(code, fc, bc, 16);
        break;
    case 8:
        fbcon_cfb8_putc(code, fc, bc, 16);
        break;
    case 16:
        fbcon_cfb16_putc(code, fc, bc, 16);
        break;
    case 24:
        fbcon_cfb24_putc(code, fc, bc, 16);
        break;
    case 32:
    //    fbcon_cfb24_putc(code, fc,  bc, 16);
        break;
    }
}

void VgaFBSput(u_char *code, u_char fc, u_char bc)
{
    switch(dispInfo.bpp)
    {
    case 4:
        fbcon_cfb4_putc(code, fc, bc, 8);
        break;
    case 8:
        fbcon_cfb8_putc(code, fc, bc, 8);
        break;
    case 16:
        fbcon_cfb16_putc(code, fc, bc, 8);
        break;
    case 24:
        fbcon_cfb24_putc(code, fc, bc, 8);
        break;
    case 32:
    //    fbcon_cfb24_putc(code, fc,  bc, 8);
        break;
    }
}

// only one line, input area
void VgaFBInputSput(int x,u_char ch,int fg,int bg)
{

   sbFReg = &fSRegs[0];
   VgaFBSetInputAddress(x);

   VgaFBSput(sbFReg->bitmap + ((int)ch << 4), 7, 1); //fg, bg);
}

void VgaFBInputWput(int x,u_char ch1,u_char ch2,int fg,int bg)
{
    u_int fnt;

    dbFReg = &fDRegs[0];  // hard-coded here, GB here
    VgaFBSetInputAddress(x);
    fnt = dbFReg->addr(ch2, ch1);
    if (fnt < dbFReg->size)
       VgaFBWput(dbFReg->bitmap + fnt, 7, 1); //fg, bg);
}

void VgaFBTextMode(void)
{
    VgaFBSetPels(savedRed, savedGreen, savedBlue);
}

void VgaFBGraphMode(void)
{
     VgaFBSetPels(red16, green16, blue16);

     VgaFBClearAll(0);

     VgaFBClearInput(1);

   // RefreshInputArea();
}

void VgaFBStart(void)
{
    VgaFBGetPels(savedRed, savedGreen, savedBlue);

 //   VgaFBClearInput(1);
}


VideoInfo VgaFBInfo =
{
//    FALSE,
    VgaFBInit,
    VgaFBStart,
    VgaFBTextMode,
    VgaFBGraphMode,
    VgaFBWput,
    VgaFBInputWput,
    VgaFBSput,
    VgaFBInputSput,
    VgaFBSetCursorAddress,
    VgaFBSetAddress,
    VgaFBSetInputAddress,
    VgaFBCursor,
    VgaFBClearAll,
    VgaFBClearInput,
    VgaFBScreenSaver,
    VgaFBDetach
};

#endif
