/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h> /* floor() */

#include <libintl.h>
#include <locale.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include <ncursesw/ncurses.h>
#include <ncursesw/menu.h>

#include "cdw_ui.h"
#include "options.h"
#include "gettext.h"
#include "color.h"
#include "log.h"
#include "config.h"
#include "main.h"
#include "utils.h"
#include "cdw_tmp_files.h"
#include "cdw_widgets.h"


#define MENU_ITEMS 10 /* amount of items in main cdw menu on the left */

/* main menu on the left side of main cdw window */
struct curses_menu {
	char add_file[22];
	char delete_file[22];
	char create_image[22];
	char copy_data[22]; /* copy content of cd to image file on your hard drive */
	char copy_audio[22];
	char write_data[22];
	/* char write_audio[22]; not in this release */
	/* char clone_disc[22]; not in this release */
	char blank[22];
	char options[22];
	char quit[22];
	char nul[1];
} buttons;


extern struct conf config;



/* table of pointers to paths */
char **filetable = NULL;

fileitem *filei;

extern int num;
int maxfiles;
double size;
int menupos;
ITEM **items = NULL;
ITEM **cditems = NULL;

char *TITLE = NULL;
MENU *menu = NULL;
MENU *cdmenu = NULL;

WINDOW *mainwin = NULL; /* main application window */
WINDOW *addwin = NULL;
WINDOW *menuwin_sub = NULL;
WINDOW *select_sub = NULL;
WINDOW *addwinlist = NULL;
WINDOW *menuwin = NULL;

#define SELECT_SUB_BEGIN_X 27
#define SELECT_SUB_BEGIN_Y 3
#define SELECT_SUB_NLINES (LINES - 6)
#define SELECT_SUB_NCOLS (COLS - 28)

void selected_files_info_area(void);
void show_disk_usage(double files_size);

void title_clean(void)
{
	if (TITLE != NULL) {
		free(TITLE);
		TITLE = NULL;
	}

	return;
}



void fill_info(void)
{
	wrefresh(menuwin_sub);
	wattrset(menuwin_sub, A_BOLD | COLOR_PAIR(8));
	if (strcmp(config.fromimage, "1") == 0) {
		/* 2TRANS: string displayed in status area, indicating that
		   user has selected in options writing to cd from image
		   file as default action for "Write Data CD" operation */
		mvwprintw(menuwin_sub, LINES - 11, 2, _("Write from image"));
	} else {
		/* 2TRANS: string displayed in status area, indicating that
		   user has selected in options writing to cd files selected
		   from hard disc as default action for "Write Data CD"
		   operation */
		mvwprintw(menuwin_sub, LINES - 11, 2, _("Write direct    "));

	}
	/* 2TRANS: string displayed in status area, indicating that
	user has selected in options writing to cd with speeed %s */
	mvwprintw(menuwin_sub, LINES - 10, 2, _("Speed: %sX"), config.speed);

	if (strcmp(config.multi, "1") == 0) {
		/* 2TRANS: string displayed in status area, indicating that
		   user has selected in options writing data in multisession
		   mode */
		mvwprintw(menuwin_sub, LINES - 9, 2, _("Multi-session "));
	} else {
		/* 2TRANS: string displayed in status area, indicating that
		   user has selected in options writing data in single
		   session mode */
		mvwprintw(menuwin_sub, LINES - 9, 2, _("Single session"));
	}
	wattrset(menuwin_sub, A_NORMAL | COLOR_PAIR(1));

	wrefresh(menuwin_sub);
}




/*
 * Simple file selector window
 *
 * File/directory selection window with one list of files. Allows browsing
 * tree of directories using keyboard.
 *
 * Uses global variables addwinlist and filei.
 *
 * \param char *dir
 * \param ind width - width ow list window
 * \param int height - height of list window
 *
 */
void load_dir(char *dir, int width, int height)
{
	int i, n;
	struct dirent **eps;

	werase(addwinlist);
	n = scandir(dir, &eps, (void *) 0, alphasort);
	filei = (fileitem *) malloc(n * sizeof(fileitem));

	if (n > 0) {
		fill_filelist(dir, &filei, eps, n);
	}

	/* FIXME - maybe rest of this function should be put into one of branches of if? */

	/* number of items on directory listing;
	 * 2 = current dir ('.') + parent dir ('..') */
	maxfiles = n - 2;
	for (i = 0; i < maxfiles + 1; i++) {
		if (filei[i].type == 0) {
			wattrset(addwinlist, A_BOLD | COLOR_PAIR(2));
			mvwprintw(addwinlist, i, 0, "/%.*s", getmaxx(addwinlist) - 7, filei[i].filename);
			mvwprintw(addwinlist, i, getmaxx(addwinlist) - 5,
				  /* 2TRANS: this is label indicating that labeled
				     item is a directory; please keep as short as possible */
				  _("DIR"));
		}
		if (filei[i].type == 1) {
			wattrset(addwinlist, COLOR_PAIR(2));
			mvwprintw(addwinlist, i, 0, "%.*s", getmaxx(addwinlist) - 7, filei[i].filename);
			mvwprintw(addwinlist, i, getmaxx(addwinlist) - 5, "%4d%s", (filei[i].size / 1024 / 1024 > 1 ? filei[i].size / 1024 / 1024 : filei[i].size / 1024),
				  (filei[i].size / 1024 / 1024 > 1 ?
						  /* 2TRANS: suffix for megabytes,
						     please keep as short as possible */
						  _("M") :
						  /* 2TRANS: suffix for kilobytes,
						     please keep as short as possible */
						  _("K")));
		}

	}

	for (i = 0; i < n; i++){
		free(eps[i]);
		eps[i] = NULL;
	}
	free(eps);
	eps = NULL;

	/* 2TRANS: this is title of dialog window in which user selects files
	   from hard disc, 'adds' them to set of files to be written to CD
	   or to iso file */
	nice_box(addwin, _("Add files"), dir);

	wrefresh(addwin);
}




/*
 * Display one item (and its attributes) from directory listing list
 *
 * Display in given row one item from directory listing list, set its
 * font/background properly, display additional attributes in the same row.
 *
 * Set color of direcrory listing item based on its type (dir or regular file).
 *
 * \param int pos - position on list (position in window)
 * \param fileitem file_p - file item with file properties (name, type, size)
 * \param int isreverse - if = 0 then black background and light font,
 *                         otherwise reverse attributes applied
 */
void highlight(int pos, fileitem file_p, int isreverse)
{
	if (file_p.type == 0) {
		if (isreverse == 0) {
			wattrset(addwinlist, A_REVERSE | COLOR_PAIR(2));
		} else {
			wattrset(addwinlist, A_BOLD | COLOR_PAIR(2));
		}
		mvwprintw(addwinlist, pos, 0, "/%.*s", getmaxx(addwinlist) - 7, file_p.filename);
	}
	if (file_p.type == 1) {
		if (isreverse == 0) {
			wattrset(addwinlist, A_REVERSE | COLOR_PAIR(2));
		} else {
			wattrset(addwinlist, COLOR_PAIR(2));
		}
		mvwprintw(addwinlist, pos, 0, "%.*s", getmaxx(addwinlist) - 7, file_p.filename);
	}
	wrefresh(addwinlist);
}




/**
 * Show window that allows user to select files and directories
 *
 * Show window that allows user to select files and directories
 * that will be put into CD disc.
 *
 * \param int width
 * \param int height
 * \param int y
 * \param int x
 *
 */
void put_addwin(int width, int height, int y, int x)
{
	char path[2000];
	struct stat stbuf;
	char name[1000];
	char *pos;
	int c;
	int rv;
	int screenpos = 0; /* cursor position on addwinlist */
	char *linkfile = malloc(2000);

	menupos = 0; /* position of cursor on whole list of files, so it may be larger than y size of window */
	addwin = newwin(width, height, y, x);
	keypad(addwin, TRUE);
	wbkgd(addwin, A_NORMAL | COLOR_PAIR(2));
	wattrset(addwin, A_NORMAL | COLOR_PAIR(2));
	addwinlist = derwin(addwin, width - 2, height - 2, 1, 1);

	/* 2TRANS: this is title of dialog window in which user selects files
	   from hard disc, 'adds' them to set of files to be written to CD
	   or to iso file */
	nice_box(addwin, _("Add files"), ""); /* second string would be erased by 'current dir path' string */

	sprintf(path, "%s", getenv("HOME"));
	load_dir(path, width, height);
	highlight(0, filei[menupos], 0);
	while ((c = wgetch(addwin)) != 27) { /* escape char */
		switch (c) {
		/* First item */
		case KEY_HOME:
			if (menupos > 0) {
				menupos = 0;
				screenpos = 0;
				werase(addwinlist);
				scroll_addlist(addwinlist, menupos, filei, maxfiles);
				highlight(screenpos, filei[menupos], 0);
				/* wrefresh(addwinlist); */
			}
			break;
		/* Last item */
		case KEY_END:
			if (menupos < maxfiles) { /* maxfiles is number of files/dirs in current dir */
				menupos = maxfiles - (maxfiles % getmaxy(addwinlist));
				werase(addwinlist);
				scroll_addlist(addwinlist, menupos, filei, maxfiles);
				menupos = maxfiles;
				screenpos = menupos % getmaxy(addwinlist);
				highlight(screenpos, filei[menupos], 0);
				/* refresh(addwinlist); */
			}
			break;
		/* Next item */
		case KEY_DOWN:
			if (menupos < maxfiles) {
				menupos++;
				if ((menupos % (getmaxy(addwinlist)) != 0) || (menupos == 0)) {
					if (menupos < getmaxy(addwinlist))
						screenpos = menupos;
					else
						screenpos = menupos % getmaxy(addwinlist);
					highlight(screenpos - 1, filei[menupos - 1], 1);
				} else {
					werase(addwinlist);
					scroll_addlist(addwinlist, menupos, filei, maxfiles);
					screenpos = menupos % getmaxy(addwinlist);
				}
				highlight(screenpos, filei[menupos], 0);
				wrefresh(addwinlist);
			}
			break;
		/* Prev item */
		case KEY_UP:
			if (menupos > 0) {
				menupos--;
				if ((menupos % (getmaxy(addwinlist)) != getmaxy(addwinlist) - 1) || (menupos == 0)) {
					if (menupos < getmaxy(addwinlist))
						screenpos = menupos;
					else
						screenpos = menupos % getmaxy(addwinlist);
					highlight(screenpos + 1, filei[menupos + 1], 1);
				} else {
					werase(addwinlist);
					scroll_addlist(addwinlist, menupos, filei, maxfiles);
					screenpos = getmaxy(addwinlist) - 1;
				}
				highlight(screenpos, filei[menupos], 0);
				wrefresh(addwinlist);
			}
			break;
		case KEY_PPAGE: /* Previous Page, Page Up */
			if (menupos > 0) {
				if ( (menupos == 0) || (menupos < getmaxy(addwinlist)) ) { /* top of dir listing, do not scroll */
					highlight(screenpos, filei[menupos], 1);
					menupos = 0; /* we are on top of dir listing, so there is only one place to move cursor and file index */
					screenpos = 0;
				} else {
					menupos -= getmaxy(addwinlist); /* move file index exactly one screensize up */
					/* screenpos = screenpos; // cursor does not change its position */
					werase(addwinlist);
					scroll_addlist(addwinlist, menupos, filei, maxfiles);
				}
				highlight(screenpos, filei[menupos], 0);
				wrefresh(addwinlist);
			}
			break;
		case KEY_NPAGE: /* Next Page, Page Down */
			if ((maxfiles - menupos) > getmaxy(addwinlist) - 1) {
				/* (remainder of) dir listing is long enough
				   to move it by whole page */
				/* cursoir does not change it's relative
				   position, because if we scroll list by
				   whole page, the cursor will still be on
				   valid position */
				highlight(screenpos, filei[menupos], 1);
				menupos += getmaxy(addwinlist);
				/* screenpos = screenpos; */
				werase(addwinlist);
				scroll_addlist(addwinlist, menupos, filei, maxfiles);
				highlight(screenpos, filei[menupos], 0);
				wrefresh(addwinlist);
			} else {
				/* move cursor to last item on the list,
				   display remainder of the list; cursor
				   will have to change it's position */
				screenpos = maxfiles % getmaxy(addwinlist); /* relative position of cursor */
				menupos = maxfiles;  /* non-relative number of highlighted item */
				werase(addwinlist);
				scroll_addlist(addwinlist, menupos, filei, maxfiles);
				highlight(screenpos, filei[menupos], 0);
				wrefresh(addwinlist);
			}
			break;
		/* Chdir */
		case 10:
			memset(name, '\0', sizeof(name));
			strcpy(name, path);
			strcat(name, "/");
			strcat(name, filei[menupos].filename);
			if (stat(name, &stbuf) == -1) {
				/* 2TRANS: this is title of dialog window */
				dialogbox(_("Error"),
					  /* 2TRANS: this is message in dialog window:
					     user highlighted a file and pressed enter,
					     but the file does not exit */
					  _("File does not exists or cannot change directory."), DIALOG_OK);
				break;
			}
			if ((stbuf.st_mode & S_IFMT) == S_IFDIR) {
				strcpy(linkfile, path);
				if (strcmp(filei[menupos].filename, "..") == 0) {
					if ((pos = (char *) strrchr((char *) path, '/')) != NULL)
						memset(pos, '\0', 1);
					if (strlen(path) <= 0)
						strcat(path, "/");
				} else {
					if (path[strlen(path) - 1] != '/')
						strcat(path, "/");
					strcat(path, filei[menupos].filename);
				}
				if (access(path, R_OK | X_OK) == -1) {
					strcpy(path, linkfile);
				} else {
					free(filei);
					filei = NULL;
					menupos = 0;
					load_dir(path, width, height);
					highlight(0, filei[menupos], 0);
				}
			}
			break;
		/* Add item to list of files/directories that
		 * are going to be put into CD disk */
		case ' ':
			/* copy full path of selected file to linkfile */
			strcpy(linkfile, path); /* path = full current dir path */
			strcat(linkfile, "/");
			strcat(linkfile, filei[menupos].filename); /* name of selected file/dir */

			/* add this full path to files list */
			rv = filelist_add(linkfile);

			/* 2TRANS: this is title of dialog window in which user
			   selects files from hard disc, 'adds' them to set of
			   files to be written to CD or to iso file */
			nice_box(addwin, _("Add files"), path); /* FIXME: why this re-appears in this file? why call this function here? */

			select_window(TRUE); /* FIXME: why call this function here? */
			break;
		}
		wrefresh(addwinlist);
	}
	free(filei);
	free(linkfile);
	filei = NULL;
	filelist_to_file();
	delwin(addwin);

	return;
}




void mainform_init()
{
	/* main NCURSES window */
	mainwin = newwin(LINES, COLS, 0, 0);
	keypad(mainwin, TRUE);
	wbkgd(mainwin, A_NORMAL | COLOR_PAIR(1));
	box(mainwin, 0, 0);

	/* title bar at the top */
	mvwaddch(mainwin, 2, 0, ACS_LTEE);
	mvwhline(mainwin, 2, 1, ACS_HLINE, COLS - 2);
	mvwaddch(mainwin, 2, COLS - 1, ACS_RTEE);
	wattrset(mainwin, A_BOLD | COLOR_PAIR(3));
	mvwprintw(mainwin, 1, (COLS / 2) - (strlen(TITLE) / 2), TITLE);
	wattrset(mainwin, A_NORMAL | COLOR_PAIR(1));

	/* left-hand menu area */
	{
		/* 2TRANS: this is label of button in main menu: Add files
		   from file system to list of files that will be written
		   to CD or to image files*/
		strcpy(buttons.add_file,     _(" Add files           "));
		/* 2TRANS: this is label of button in main menu: Remove
		   item(s) from list of files that will be written to CD
		   or to image files*/
		strcpy(buttons.delete_file,  _(" Delete files        "));
		/* 2TRANS: this is label of button in main menu: write
		   selected files to iso image */
		strcpy(buttons.create_image, _(" Create image        "));
		/* 2TRANS: this is label of button in main menu: copy
		   content of your data cd to hard drive */
		strcpy(buttons.copy_data,    _(" Copy data CD        "));
		/* 2TRANS: this is label of button in main menu: copy
		   content of your audio cd to hard drive */
		strcpy(buttons.copy_audio,   _(" Copy audio CD       "));
		/* 2TRANS: this is label of button in main menu: write
		   selected file or preselected iso image file to your
		   data cd */
		strcpy(buttons.write_data,   _(" Write data CD       "));
		/* strcpy(buttons.write_audio, _(" Write audio CD      ")); not in this release */
		/* strcpy(buttons.clone_disc, _(" Clone data CD       ")); not in this release */
		/* 2TRANS: this is label of button in main menu:
		   erase content of CD-RW */
		strcpy(buttons.blank,        _(" Blank CD-RW         "));
		/* 2TRANS: this is label of button in main menu:
		   change preferences of cdw */
		strcpy(buttons.options,      _(" Preferences         "));
		/* 2TRANS: this is label of button in main menu: exit cdw */
		strcpy(buttons.quit,         _(" Quit                "));

		items = (ITEM **) calloc(MENU_ITEMS, sizeof(ITEM *));

		items[0] = new_item(buttons.add_file, "");
		items[1] = new_item(buttons.delete_file, "");
		items[2] = new_item(buttons.create_image, "");
		items[3] = new_item(buttons.copy_data, "");
		items[4] = new_item(buttons.copy_audio, "");
		items[5] = new_item(buttons.write_data, "");
		/* items[5] = new_item(buttons.wau, ""); */
		/* items[6] = new_item(buttons.cdc, ""); */
		items[6] = new_item(buttons.blank, "");
		items[7] = new_item(buttons.options, "");
		items[8] = new_item(buttons.quit, "");
		items[9] = new_item((char *) 0, "");
		/* ***************** */

		/* just a reminder:
		WINDOW *derwin(WINDOW *orig, int nlines, int ncols,
		               int begin_y, int begin_x); */
		menu = new_menu((ITEM **) items);
		set_menu_win(menu, mainwin);
		set_menu_sub(menu, derwin(mainwin, LINES - 8, 26, 5, 3));
		set_menu_fore(menu, COLOR_PAIR(4) | A_REVERSE);
		set_menu_back(menu, COLOR_PAIR(4));
		set_menu_mark(menu, "");
	}



	/* 'Selected files' part of screen */
	{
		select_sub = derwin(mainwin, SELECT_SUB_NLINES, SELECT_SUB_NCOLS,
				    SELECT_SUB_BEGIN_Y, SELECT_SUB_BEGIN_X);

		keypad(select_sub, TRUE);
		/* 2TRANS: this is title of biggest area in cdw UI that
		   shows files selected for writing to cd or image file */
		nice_box(select_sub, _("Selected files"), "");
		wattrset(select_sub, A_NORMAL | COLOR_PAIR(1));
	}

	selected_files_info_area();


	/* Left hand cdw menu */
	menuwin_sub = derwin(mainwin, LINES - 6, 26, 3, 1);
	/* 2TRANS: this is title of left-hand menu area of cdw UI */
	nice_box(menuwin_sub, _("Menu"), "");

	mvwaddch(menuwin_sub, LINES - 12, 0, ACS_LTEE);
	mvwhline(menuwin_sub, LINES - 12, 1, ACS_HLINE, 24);
	mvwaddch(menuwin_sub, LINES - 12, 25, ACS_RTEE);
	mvwprintw(menuwin_sub, LINES - 12, 2, "( %s )",
		  /* 2TRANS: this is title of area displaying
		     'status' - values of most important settings */
		  _("Info"));
	fill_info();


	wattrset(mainwin, A_BOLD | COLOR_PAIR(2));
	post_menu(menu);
	wrefresh(mainwin);

	return;
}



/**
 * Deallocate all resources relatd to main app window
 *
 * \returns 0
 */
int mainform_clean(void)
{
	cdmenu_clean();
	menu_clean();
	delwin(mainwin);
	mainwin = NULL;

	return 0;
}




/**
 * Delete file(s) from list of selected files
 *
 * Delete file(s) from list of selected files. This function will move
 * cursor to list of currently selected files and let user remove (using
 * Delete button) remove item(s) from this list.
 */
void delete_files(void)
{
	int c;
	char filetodelete[200] = "\0";

	set_menu_fore(cdmenu, COLOR_PAIR(1) | A_REVERSE);

	while (((c = wgetch(mainwin)) != 27) && (num > 0)) {
		switch (c) {
		case KEY_HOME:
			menu_driver(cdmenu, REQ_FIRST_ITEM);
			break;
		case KEY_END:
			menu_driver(cdmenu, REQ_LAST_ITEM);
			break;
		case KEY_DOWN:
			menu_driver(cdmenu, REQ_DOWN_ITEM);
			break;
		case KEY_UP:
			menu_driver(cdmenu, REQ_UP_ITEM);
			break;
		case KEY_DC:
			/* FIXME - merge those two lines */
			strcpy(filetodelete, item_name(current_item(cdmenu)));
			filelist_remove(filetodelete);

			select_window(TRUE);
			set_menu_fore(cdmenu, COLOR_PAIR(1) | A_REVERSE);
			fflush(stdin);
			break;
		}
		// wrefresh(derwin(mainwin, LINES - 8, COLS - 31, 4, 28));
		wrefresh(select_sub);
	}
	set_menu_fore(cdmenu, COLOR_PAIR(1));
	filelist_to_file();
	return;
}



int move_radio_selection(WINDOW * window, int radiobutton_number)
{
	mvwaddch(window, 4, 3, ' ');
	mvwaddch(window, 5, 3, ' ');
	mvwaddch(window, 6, 3, ' ');
	mvwaddch(window, 7, 3, ' ');
	if (radiobutton_number == 0) {
		mvwaddch(window, 4, 3, '*');
	} else if (radiobutton_number == 1) {
		mvwaddch(window, 5, 3, '*');
	} else if (radiobutton_number == 2) {
		mvwaddch(window, 6, 3, '*');
	} else if (radiobutton_number == 3){
		mvwaddch(window, 7, 3, '*');
	}

	return radiobutton_number;
}




/*
 * Show window allowing user to select disk size
 */
void size_selector(void)
{
	WINDOW *volwin, *edit;
	int ch = 0;
	int rv = 0; /* generic return value variable */
	char cdsize[20];

	int nlines = 10, ncols = 40;
	int begin_y = (LINES - nlines) / 2, begin_x = (COLS - ncols) / 2;

	volwin = newwin(nlines, ncols, begin_y, begin_x);

	wbkgd(volwin, COLOR_PAIR(2));
	werase(volwin);
	keypad(volwin, TRUE);

	/* 2TRANS: this is title of dialog window allowing user to
	   select size of disc currently in drive */
	nice_box(volwin, _("CD size"),
		 /* 2TRANS: this is a tooltip at the bottom of size selector
		    window. User can select size pressing corresponding
		    number key. Keep no longer than 34 chars. */
		 _("Select using number key"));

	/* 2TRANS: this is label in dialog window in which user can select
	   size of disc currently in drive */
	mvwprintw(volwin, 2, 3, _("Select CD size"));
	/* 2TRANS: this is radio button label: 650MB CD */
	mvwprintw(volwin, 4, 5, _("1 - 74 Min (650 MB)"));
	/* 2TRANS: this is radio button label: 700MB CD */
	mvwprintw(volwin, 5, 5, _("2 - 80 Min (700 MB)"));
	/* 2TRANS: this is radio button label: DVD */
	mvwprintw(volwin, 6, 5, _("3 - DVD - (4700 MB)"));
	/* 2TRANS: this is radio button label, after this string an input
	   field is displayed and user can enter value */
	mvwprintw(volwin, 7, 5, _("4 -"));

	/* small, one-line text input area */
	edit = derwin(volwin, 1, 8, 7, 9);
	wbkgd(edit, COLOR_PAIR(6));
	werase(edit);

	/* initial value displayed in text input area */
	sprintf(cdsize, "%d", config.user_cdsize);
	mvwprintw(edit, 0, 0, cdsize);

	switch (config.cdsize) {
		case 657:
			move_radio_selection(volwin, 0);
			break;
		case 702:
			move_radio_selection(volwin, 1);
			break;
		case 4700:
			move_radio_selection(volwin, 2);
			break;
		default:
			move_radio_selection(volwin, 3);
			break;
	}

	wrefresh(edit);
	wmove(volwin, 9, 29);
	while ((ch != CDW_ESCAPE) && (ch != CDW_ENTER)) {
		ch = wgetch(volwin);
		switch (ch) {
			case '1':
				move_radio_selection(volwin, 0);
				config.cdsize = 657;
				break;
			case '2':
				move_radio_selection(volwin, 1);
				config.cdsize = 702;
				break;
			case '3':
				move_radio_selection(volwin, 2);
				config.cdsize = 4700;
				break;
			case '4':
				move_radio_selection(volwin, 3);
				wrefresh(volwin);
				sprintf(cdsize, "%d", config.user_cdsize);
				rv = inbox(edit, cdsize, false);

				if (rv == CDW_ESCAPE) {
					;
				} else { /* enter pressed, use entered value */
					config.cdsize = atoi(cdsize);
					config.user_cdsize = atoi(cdsize);
				}
				break;
		}

		wrefresh(volwin);
	}


	write_conf();

	delwin(volwin);
	show_disk_usage(size); /* FIXME: using global variable */
	main_win_refresh_part(nlines, ncols, begin_y, begin_x);

	return;
}




/**
 * Display CD usage summary on bottom of cdw window
 *
 * Display amount of space used by selected files, amount of
 * space that would stay free on disk, and percent of CD disk usage.
 * CD\DVD size is taken from config panel/file.
 *
 * \param double files_size - size of selected files
 */
void show_disk_usage(double files_size)
{
	int i;

	/* clean row in which usage bar and information about space usage is displayed */
	for (i = 1; i < 40; i++) {
		mvwaddch(mainwin, LINES - 5, 30 + i, ' ');
		mvwaddch(mainwin, LINES - 6, 30 + i, ' ');
	}

	/* left and right border of usage bar*/
	mvwaddch(mainwin, LINES - 5, 30, '[');
	mvwaddch(mainwin, LINES - 5, 60, ']');

	if (files_size <= config.cdsize) {
		float percent = (files_size / ((double) config.cdsize / 100));
		for (i = 1; i < (int) ((percent * 30) / 100); i++) {
			mvwaddch(mainwin, LINES - 5, 30 + i, ACS_BLOCK);
		}

		mvwprintw(mainwin, LINES - 6, 30,
			  /* 2TRANS: this is information about usage of CD
			     space: '%2.1f%%' is information about used space,
			     displayed as percentage of total disc size,
			     '%.0f' is how many megabytes will be unused
			     (free) out of '%d' - total value of megabytes */
			  _("Used: %2.1f%%, free: %.0f/%d MB"), percent, config.cdsize - files_size, config.cdsize);
	} else { /* files size is larger than available space */
		 /* use red font color to alarm user */
		wattrset(mainwin, A_BOLD | COLOR_PAIR(3));
		for (i = 1; i < 30; i++) {
			mvwaddch(mainwin, LINES - 5, 30 + i, ACS_BLOCK);
		}

		mvwprintw(mainwin, LINES - 6, 30,
			  /* 2TRANS: this is information about usage of CD
			     space: size of files selected ('%2.1f') and
			     space available on CD ('%d'). */
			  _("Files size: %2.1f MB, disc size %d MB"), files_size, config.cdsize);
		wattrset(mainwin, A_NORMAL | COLOR_PAIR(1));
	}

	return;
}




/**
 * Refresh content of main app window
 *
 * Refresh content of main app window - the one containing menu
 * on the left and selected files/dirs list on the right + some
 * statistics at the bottom. This may need reading (possibly)
 * changed content of files-to-cd listing.
 *
 * \param bool real - do you want to refresh content of main
 *        window by re-reading it from listing on hard-disk
 *        (may be time-consuming)?
 *
 * \returns 0
 */
int select_window(bool real)
{
	/* Do time-consuming refresh of file list?
	 * It is time consuming, because this is not ordinary curses
	 * wrefresh(), this code has to allocate some memory and move data from
	 * one place to another and print this data to screen. */
	if (real) {

		/* clean up previous selected files list and all associated data structures */
		cdmenu_clean();

		/* fill table with selected files paths, return number of paths */
		num = filelist_items(&filetable);

		/* debug code

		if (num > 0) {
			fprintf(stderr, "first path is");
			fprintf(stderr, "\"%s\"\n", *(filetable + num - 1));
		}
		*/

		cditems = (ITEM **) calloc(num + 1, sizeof(ITEM *));

		if (num > 0) {
			int i;
			int pos = 0;
			/* this file list is in reality a ncurses menu, so
			 * we will use menu **ITEM data structure (cditems) here */
			for (i = 0; i < num; ++i) {
				/* FIXME find a way to save path with file type
				 * like it was done in original implementation of cdw */
				/*
				if (cdfiles[i]->d_type == DT_DIR) {
					cditems[pos++] = new_item(cdfiles[i]->d_name, " DIR");
				} else if (cdfiles[i]->d_type != DT_DIR) {
					cditems[pos++] = new_item(cdfiles[i]->d_name, "");
				}
				*/

				/* FIXME - here, instad of "    ", should be file/DIR info */
				cditems[pos++] = new_item( *(filetable + i), "    " );
			}
		}

		/* Let's continue constructing ncurses menu cdmenu using
		 * ncurses data structure **ITEM cditems. Note that this menu
		 * will be constructed even if there are no files */

		/* last element must be NULL - ncurses requirement */
		cditems[num] = (ITEM *) NULL;

		cdmenu = new_menu((ITEM **) cditems);
		set_menu_win(cdmenu, mainwin);
		set_menu_sub(cdmenu, derwin(mainwin, LINES - 13, COLS - 31, 4, 28));
		set_menu_format(cdmenu, LINES - 13, 1);
		set_menu_fore(cdmenu, COLOR_PAIR(1));
		set_menu_back(cdmenu, COLOR_PAIR(1));
		post_menu(cdmenu);

	}

	/* redraws frames and it's content at the bottom of window (below
	selected fiels list) */
	selected_files_info_area();
	/* size = 0; */ /* FIXME: why this was set to zero? */

	wrefresh(mainwin);

	return 0;
}







/**
 * Small wrapper function - shift content of 'Add files' list in window
 *
 * \param WINDOW *addwinlist - list to scroll
 * \param int menupos - cursor position on whole list of files (index of highlighted file); it has value calculated after key has been hit
 * \param fileitem *filei - current directory listing
 * \param int maxfiles - number of items in current directory listing
 */
void scroll_addlist(WINDOW *addwinlist, int menupos, fileitem *filei, int maxfiles)
{
	int startpos = 0; // index of first file that will be displayed in scrolled window
	int i; /* rows index of addwinlist window */

	/* start displaying items starting from item with index equal (menupos - "cursor position");
	 * cursor position is (menupos % getmaxy(addwinlist)) - this is cursor pos. in window */
	startpos = menupos - (menupos % getmaxy(addwinlist));


	/* increment window row number and dir listing index at the same time */
	for (i = 0, startpos;
	     startpos <= maxfiles && i < getmaxy(addwinlist);
	     i++, startpos++) {
		/* display startpos-th element in i-th row of window */
		highlight(i, filei[startpos], 1);

		/* TODO: check if this piece of code does not repeat itself in scroll_addlist() */
		/* file name is displayed, so we could just go to next iteration,
		 * but let's display some additional file info (in the same row) */
		if (filei[startpos].type == 0) { /* let user know that this is a dir */
			wattrset(addwinlist, A_BOLD | COLOR_PAIR(2));
			mvwprintw(addwinlist, i, getmaxx(addwinlist) - 5,
				  /* 2TRANS: this is label indicating that
				     labeled item is a directory; please
				     keep as short as possible */
				  _("DIR"));
		} else if (filei[startpos].type == 1) { /* show file size (human readable) */
			wattrset(addwinlist, COLOR_PAIR(2));
			mvwprintw(addwinlist, i, getmaxx(addwinlist) - 5, "%4d%s",
					(filei[startpos].size / 1024 / 1024 > 1 ?
						filei[startpos].size / 1024 / 1024 :
						filei[startpos].size / 1024), (filei[startpos].size / 1024 / 1024 > 1 ?
							/* 2TRANS: suffix for megabytes,
							   please keep as short as possible */
						  	_("M") :
							/* 2TRANS: suffix for kilobytes,
						           please keep as short as possible */
							_("K")));

		}
	}
	return;
}








int cdw_title_init(void)
{
	int title_len = strlen(PACKAGE) + 2 + strlen(VERSION) + 9;
	TITLE = (char *) malloc(title_len);
	sprintf(TITLE,
		/* 2TRANS: fancy application title in main cdw window:
		   first %s is app name, second %s is app version */
		_("::: %s-%s :::"), PACKAGE, VERSION);

	return 0;
}




/**
 * Run common ncurses setup code - simple wrapper
 *
 * \returns 0 on success, -1 if console has too few lines or columns
 */
int cdw_curses_init(void)
{
	initscr();
	start_color();
	cbreak();
	noecho();
	keypad(stdscr, TRUE);

	/* turn cursor off - now cursor will be
	turned on only if really needed */
	curs_set(0);

	init_curses_colors();

	// lines = LINES - 2;
	if ((LINES < 24) || (COLS < 79)) {
		return -1;
	}

	return 0;

}




/*
 * Remove menu containing paths selected for recording to CD
 *
 * Remove menu in central part of cdw window. This function is
 * called either when user adds/removes files to list (every time
 * new list (in fact: ncurses menu) is allocated, so previous
 * version must be cleaned up) or at the end of the program.
 */
void cdmenu_clean(void)
{
	int i;
	if (cdmenu != NULL) {
		unpost_menu(cdmenu);
		free_menu(cdmenu);
		cdmenu = NULL;

		for (i = 0; i < num; i++) {
			free_item(cditems[i]);
			cditems[i] = NULL;
		}

		if (cditems != NULL ){
			free(cditems);
			cditems = NULL;
		}
		if (filetable != NULL) {
			free(filetable);
			filetable = NULL;
		}
	}

	return;
}




/*
 * Remove menu containing cdw commands
 *
 * Remove menu in left part of cdw window. This function is
 * called at the end of the program.
 */
void menu_clean(void)
{
	int i;
	if (menu != NULL) {
		unpost_menu(menu);
		free_menu(menu);
		menu = NULL;

		for (i = 0; i < MENU_ITEMS; ++i){
			free_item(items[i]);
			items[i] = NULL;
		}

		free(items);
		items = NULL;

		/* we don't free cdw menu labels since they are static */
	}
	return;
}




/*
 * Check for existence of bootimage
 *
 * This functio nwas created because this code appeared twice in main.c.
 * Maybe it should be replaced by generic file_exists(char *file).
 *
 * \returns 0 on failure, 1 on success
 */
int check_for_boot_image(void)
{
	char bootimage[255];
	FILE *isofile;

	int boot_exists;

	sprintf(bootimage, "%s/.cdw/%s", getenv("HOME"), config.bootimg);
	if ((isofile = fopen(bootimage, "r")) == NULL) {
		/* 2TRANS: this is title of dialog window */
		boot_exists = dialogbox(_("Boot image error"),
					/* 2TRANS: this is message in dialog window */
					_("Boot image doesn't exists!"), DIALOG_OK);
		boot_exists = 0; /* FIXME - why we are assigning 0 here? */
		return 0;
	} else {
		fclose(isofile);
		return 1;
	}
}




/*
 * Fill list with file names, types and sizes of files in given dir
 *
 * Fill custom data structure: list of items, each containing information
 * about file: name, size and type. So in fact - extract some information
 * about current dir that is stored in struct dirent **eps.
 *
 * \param char *dir
 * \param fileitem **filei - allocated list of file information items, big enough to store information about all files in dir
 * \param struct dirent **eps
 * \param int n - number of files in given dir
 *
 * \returns 0 on success, -1 on failure
 */
int fill_filelist(char *dir, fileitem **filei, struct dirent **eps, int n)
{
	struct stat stbuf;
	int pos = 0;
	bool failure = false;
	int rv = 0;

	/* Here we have two for loops - first checks if eps item
	 * is a DIR - so we will have directories on top of list.
	 * Then second looks for files. Common counter (pos) ensures that we will
	 * check all items of eps. */
	/* FIXME - I'm sure that filei can be handled more elegantly */
	int i;
	for (i = 0; i < n; ++i) {
		char *fullpath = concat(dir, "/", eps[i]->d_name, NULL);
		rv = is_valid_path(fullpath, -1, CDW_DIR, &stbuf);
		if (rv == -1) {
			free(fullpath);
			failure = true;
			continue; /* don't break this loop - this may be a file */
		}
		free(fullpath);

		if (((stbuf.st_mode & S_IFMT) == S_IFDIR) && (strcmp(eps[i]->d_name, ".") != 0)) {
			strcpy((*filei)[pos].filename, eps[i]->d_name);
			(*filei)[pos].size = 0;
			(*filei)[pos].type = 0;
			pos++;
		}
	}
	for (i = 0; i < n; ++i) {
		char *fullpath = concat(dir, "/", eps[i]->d_name, NULL);
		rv = is_valid_path(fullpath, -1, CDW_FILE, &stbuf);
		if (rv == -1) {
			free(fullpath);
			failure = true;
			continue; /* don't break this loop - this may be a dir */
		}
		free(fullpath);

		if ((stbuf.st_mode & S_IFMT) != S_IFDIR) {
			strcpy((*filei)[pos].filename, eps[i]->d_name);
			(*filei)[pos].size = stbuf.st_size;
			(*filei)[pos].type = 1;
			pos ++;
		}
	}
	if (failure) {
		return -1;
	} else {
		return 0;
	}

}



/**
 * Widget allowing user to select blanking method
 *
 * Widget allowing user to select blanking method: "fast" or "all".
 * User can select one of them using UP/DOWN arrow keys or numerical
 * values assigned to labels "fast" and "all", visible in the
 * widget.
 *
 * \returns CDW_ENTER if user accepted blanking method using ENTER key,
 *          CDW_ESCAPE if user pressed 'q', 'Q' or ESCAPE
 */
int blank_method_selector(void)
{
	/* dialog window */
	WINDOW *blankwin = newwin(10, 30, (LINES - 10) / 2, (COLS - 30) / 2);
	wbkgd(blankwin, COLOR_PAIR(2));
	werase(blankwin);
	keypad(blankwin, true); /* enable arrow keys and other keys for this window */
	/* 2TRANS: this is title of dialog window, the window allows
	   user to select blanking method */
	nice_box(blankwin, _("Blanking method"), "");


	/* label and kind-of radio buttons in dialog windows */
	/* 2TRANS: this is label in dialog window, the window allows
	   user to select blanking method */
	mvwprintw(blankwin, 2, 3, _("Select CD blanking method"));
	/* 2TRANS: this is radio button label: 'erase fast' method,
	   only table of content is erased */
	mvwprintw(blankwin, 4, 5, _("1 - fast"));
	/* 2TRANS: this is radio button label:  'arase all' method,
	   whole content is erased */
	mvwprintw(blankwin, 5, 5, _("2 - all"));


	/* both radiobuttons are blank, preselect one of them based on
	   current configuration */
	if ( !strcmp(config.blank, "fast") ) {
		mvwaddch(blankwin, 4, 3, '*');
	} else if ( !strcmp(config.blank, "all") ) {
		mvwaddch(blankwin, 5, 3, '*');
	} else { /* default will be "fast" */
		mvwaddch(blankwin, 4, 3, '*');
	}


	char tmp_blank[5];
	strcpy(tmp_blank, config.blank); /* work on copy of config value */

	int ch = 'k'; /* just a safe, non-meaningful initial value */
	while (( ch != CDW_ESCAPE) && ( ch != CDW_ENTER) && (ch != 'q') && (ch != 'Q')) {
		ch = wgetch(blankwin);
		if ( ch == '1' ) {
			strcpy(tmp_blank, "fast");
		} else if ( ch == '2' ) {
			strcpy(tmp_blank, "all");
		} else if ( ch == KEY_UP || ch == KEY_DOWN) {
			if ( !strcmp(tmp_blank, "fast") ) {
				/* "fast" is currently selected, but user
				    moves selection to "all" */
				strcpy(tmp_blank, "all");
			} else {
				/* "all" is currently selected, but user
				   moves selection to "fast" */
				strcpy(tmp_blank, "fast");
			}
		} else {
			; /* FIXME - should this be left empty? */
		}

		/* user changed selection (using numerical keys or arrow
		   keys - this is now irrelevant), it was written to
		   config.blank, now update radio button accordingly */
		if ( !strcmp(tmp_blank, "fast") ) {
			mvwaddch(blankwin, 4, 3, '*');
			mvwaddch(blankwin, 5, 3, ' ');
		} else if ( !strcmp(tmp_blank, "all") ) {
			mvwaddch(blankwin, 4, 3, ' ');
			mvwaddch(blankwin, 5, 3, '*');
		} else { /* some error occured, don't change anything */
			;
		}
		wrefresh(blankwin);
	}


	delwin(blankwin);
	main_win_refresh_part(10, 30, (LINES - 10) / 2, (COLS - 30) / 2);

	if ((ch == CDW_ESCAPE) || (ch == 'q') || (ch == 'Q')) {
		return CDW_ESCAPE;
	} else {
		/* save current setting to real config variable */
		strcpy(config.blank, tmp_blank);
		write_conf();
		return CDW_ENTER;
	}
}




/**
 * Ask user for volume label
 *
 * Check if user has to be asked for volume label. Show input dialog if
 * this is the case. Save volume label in config variable.
 *
 * \returns CDW_OK if user entred volume label (or accepted default value)  and pressed enter;
 *          CDW_OK if user should not be asked for volume label; (default/stored value should be used)
 *          CDW_CANCEL if user pressed escape when asked for volume label
 */
cdw_rv_t conditional_volume_label_dialog(void)
{
	int rv = 0;

	if (strcmp(config.showvol, "1") == 0) { /* ask for disk label */
		/* 2TRANS: this is title of dialog window - user has to
		   enter CD label */
		rv = input_box(_("Volume ID"),
			       /* 2TRANS: this is message in dialog window:
			          there is input field below */
			       _("Enter volume label:"), config.volumeid, 0);
		return rv; /* CDW_OK or CDW_CANCEL - user decision */
	} else {
		return CDW_OK; /* use defaut value or value stored in options file */
	}
}




/**
 * Check if output image file can be created with given permissions or
 * already exists. Ask user in case of doubts.
 *
 * Check if output image file (with path set in config.tempdir) exists.
 * If it does then user have to decide if existing file should be
 * overwritten or not, or user can enter new path. If the file does not exist
 *
 * Function asks for new path when file with given path cannot be (for some
 * reason) created. Check "man 2 open" page for list of handled errors
 * (all open() errors are handled).
 *
 * If the path to image
 *
 * \param filedes -
 *
 * \return true if file with path set in config.tempdir can be written,
 *         false if user cancels writing image or tries too many (6) times
 *         without success.
 */
bool open_target_image_file(int *filedes)
{
	int attempt = 0;
	int total_attempts = 6; /* only 6 tries */
	for (attempt = 0; attempt < total_attempts; attempt++) {
		/* O_CREAT - if the file does not exist it will be created
		 * O_EXCL  - when used with O_CREAT, if the file already exists it is an error and the open will fail */
		*filedes = open(config.tempdir, O_CREAT | O_EXCL, S_IWRITE);
		if (*filedes > 0) { /* the file previousy did not existed, path is ok */
			return true;
		} else { /* *filedes == -1, errno is set */
			int e = errno;
			int rv = 0;

			if (attempt == (total_attempts - 1)) {
				break; /* because even if user agrees to enter new path and provides it, loop condition turns false before new path can be tested */
			}

			if (e == EEXIST) { /* pathname już istnieje, a użyto O_CREAT i O_EXCL. */
				/* 2TRANS: this is title of dialog window */
				rv = dialogbox(_("Error"),
					       /* 2TRANS: this is message dialog window, user can press Yes, No or Cancel button */
					       _("Image file already exists. Overwrite?"), DIALOG_YES_NO_CANCEL);
				if (rv == BUTTON_YES) { /* yes, overwrite existing image */
					/* can we write to file? does it have correct permissions? */
					int rv2 = access(config.tempdir, W_OK);
					if (rv2 == -1) {
						if (errno == EACCES) {
							/* 2TRANS: this is title of dialog window */
							dialogbox(_("Error"),
									/* 2TRANS: this is message in dialog window, user can press OK button */
									_("Image file exists, but I cannot write to it."), DIALOG_OK);
							continue; /* continue top level for() */
						} else { /* some other error */
							/* 2TRANS: this is title of dialog window */
							dialogbox(_("Error"),
									/* 2TRANS: this is message in dialog window, user can press OK button */
									_("The path is stilll invalid. Please check path."), DIALOG_OK);
							continue; /* continue top level for() */
						}
					} else {
						return true;
					}
				} else if (rv == BUTTON_NO) { /* don't overwrite, user will netr new path */
					/* 2TRANS: this is title of dialog window */
					input_box(_("Path to iso file"),
							/* 2TRANS: this is message in dialog window, below it is displayed an input area */
							_("Enter path to new iso file"), config.tempdir, 0);
					// wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
					continue; /* user entered new path - check (again) if the file exists */
				} else {
					return false;
				}
			} else { /* this is more complicated than simple "file exists - overwrite?" */
				switch (e) {
					case EISDIR: // pathname odnosi się do katalogu, a żądany był dostęp z prawem zapisu (tzn. ustwine było O_WRONLY lub O_RDWR).
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press OK or Cancel button */
								_("The path points to directory. Enter new path?"), DIALOG_OK_CANCEL);
						break;
					case EACCES: // Żądany dostęp do pliku nie jest dozwolony, jeden z katalogów w pathname nie ma praw przeszukiwania (wykonywania), lub plik nie istnieje, a katalog nadrzędny nie ma praw zapisu.
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press OK or Cancel button */
								_("Wrong access rights to parent directory. Enter new path?"), DIALOG_OK_CANCEL);
						break;
					case ENAMETOOLONG: // pathname było zbyt długie.
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press OK or Cancel */
								_("Path is too long. Enter new path?"), DIALOG_OK_CANCEL);
						break;

					case ENOENT: // O_CREAT nie było ustawione, a plik o zadanej nazwie nie istnieje.  Lub, składnik pathname, który powinien być katalogiem nie istnieje  lub jest wiszącym dowiązaniem symbolicznym.
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press OK or Cancel */
								_("Part of directory path does not exist. Enter new path?"), DIALOG_OK_CANCEL);
						break;

					case ENOTDIR: //Składnik użyty w pathname jako katalog w rzeczywistości nie jest katalogiem lub podano O_DIRECTORY, a pathname nie było katalogiem.
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press OK or Cancel */
								_("Path is invalid. Enter new path?"), DIALOG_OK_CANCEL);
						break;
					case ENXIO: // Podano  O_NONBLOCK  |  O_WRONLY,  plik  o zadanej nazwie stanowi FIFO i nie jest ono otwarte dla żadnego procesu do odczytu. Lub plik jest plikiem urządzenia specjalnego, a odpowiadające mu urządzenie nie istnieje.
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press OK or Cancel */
								_("Path is invalid. Enter new path?"), DIALOG_OK_CANCEL);
						break;
					case ENODEV: // pathname odnosi się do pliku urządzenia specjalnego, a odpowiadające mu urządzenie nie istnieje. (Jest to błąd w jądrze  Linuksa  -  ENXIO
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press OK or Cancel */
								_("Path is invalid. Enter new path?"), DIALOG_OK_CANCEL);
						break;
					case EROFS: //  pathname odnosi się do pliku na systemie plików tylko dla odczytu, a żądano otwarcia w trybie zapisu
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press Yes, No or Cancel */
								_("Path to file in read-only file system. Enter new path?"), DIALOG_YES_NO_CANCEL);
						break;
					case ETXTBSY: // pathname odnosi się do wykonywalnego obrazu, który obecnie jest wykonywany, a zażądano dostępu dla zapisu.
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press OK or Cancel */
								_("Path is invalid. Enter new path?"), DIALOG_OK_CANCEL);
						break;
					case EFAULT: // pathname wskazuje poza dostępną dla użytkownika przestrzeń adresową.
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press OK or Cancel */
								_("Path is invalid. Enter new path?"), DIALOG_OK_CANCEL);
						break;
					case ELOOP: //  Podczas  rozwiązywania  pathname  napotkano  zbyt  wiele dowiązań symbolicznych lub podano O_NOFOLLOW, a pathname jest dowiązaniem symbolicznym.
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press Yes, No or Cancel */
								_("Path uses too many symbolic links. Enter new path?"), DIALOG_YES_NO_CANCEL);
						break;
					case ENOSPC: // Gdy pathname miało być utworzone, okazało się, że na urządzeniu na którym miało się znajdować brak miejsca na nowy plik.
						/* 2TRANS: this is title of dialog window */
						rv = dialogbox(_("Error"),
								/* 2TRANS: this is message in dialog window, user can press OK or Cancel */
								_("There is no space on this device for the file. Enter new path?"), DIALOG_OK_CANCEL);
						break;
					case ENOMEM: // Brak dostępnej pamięci jądra.
						{
							int i = 0;
							/* fprintf(stderr, "ENOMEM\n"); */
							for (i = 0; i < 3; i++) {
								/* 2TRANS: this is title of dialog window */
								int rv2 = dialogbox(_("System error"),
										/* 2TRANS: this is message in dialog window, user can press OK or Cancel */
										_("No system memory. Retry with the same path?"), DIALOG_OK_CANCEL);
								if (rv2 == BUTTON_OK) {
									sleep(2);
								} else {
									return false;
								}
							}
							/* 2TRANS: this is title of dialog window */
							dialogbox(_("System error"),
									/* 2TRANS: this is message in dialog window, user can press OK */
									_("No system memory. Operation abandoned."), DIALOG_OK);
						}
						break;
					case EMFILE: // Proces ma już otwartą maksymalną liczbę plików
					case ENFILE: // Osiągnięto ograniczenie dla łącznej liczby otwartych plików w systemie.
						{
							int i = 0;
							/* fprintf(stderr, "EMFILE\n"); */
							for (i = 0; i < 3; i++) {
								/* 2TRANS: this is title of dialog window */
								int rv2 = dialogbox(_("System error"),
										/* 2TRANS: this is message in dialog window, user can press OK or Cancel */
										_("Too many files opened. Retry with the same path?"), DIALOG_OK_CANCEL);
								if (rv2 == BUTTON_OK) {
									sleep(2);
								} else {
									return false;
								}
							}
							/* 2TRANS: this is title of dialog window */
							dialogbox(_("System error"),
									/* 2TRANS: this is message in dialog window, user can press OK */
									_("Too many files opened. Operation abandoned."), DIALOG_OK);
						}
						break;
				} /* switch (errno) */

				if (rv == BUTTON_OK) { /* user wants to enter new path */
					/* 2TRANS: this is title of dialog window */
					input_box(_("Path to iso file"),
							/* 2TRANS: this is label in dialog window, there is input field below */
							_("Enter path to new iso file"), config.tempdir, 0);
					continue; /* user entered new path - check (again) if the file exists */
				} else { /* (rv == BUTTON_NO):  don't want new file, use existing */
					return false;
				}
			}
		} /* if (*filedes == 0) */
	} /* for() */

	/* 2TRANS: this is title of dialog window */
	dialogbox(_("Message"),
		  /* 2TRANS: this is message in dialog window, user can press OK */
		  _("Operation of image file selection failed after several attempts. Please check your filesystem."), DIALOG_OK);

	return false;
}




void selected_files_info_area(void)
{
	double files_size = 0;
	long long int ds = 0;

	int i;

	/* line just below 'selected files' list */
	mvwaddch(select_sub, LINES - 12, 0, ACS_LTEE);
	mvwhline(select_sub, LINES - 12, 1, ACS_HLINE, COLS - 30);
	mvwaddch(select_sub, LINES - 12, COLS - 29, ACS_RTEE);

	mvwaddch(select_sub, LINES - 10, 0, ACS_LTEE);
	mvwhline(select_sub, LINES - 10, 1, ACS_HLINE, COLS - 30);
	mvwaddch(select_sub, LINES - 10, COLS - 29, ACS_RTEE);

	wattrset(mainwin, A_BOLD | COLOR_PAIR(8));

	/* print total size of all items in selected files/dirs list and number of items on this list */
	for (i = 30; i < COLS - 3; i++)
		mvwaddch(mainwin, LINES - 8, i, ' ');

	if (num > 0) {
		ds = dirsize(filetable, num);
		files_size = (double) ((ds / 1024) / 1024);
	}
	size = files_size; /* FIXME - size is global variable - find a way to remove it */
	wattrset(mainwin, A_BOLD | COLOR_PAIR(8));
	mvwprintw(mainwin, LINES - 8, 30,
		  /* 2TRANS: this string displays total size of dirs/files
		  selected ('%.0f') and number of items selected ('%d') -
		  number of items on 'Selected files' list. I'm using word
		  'items' because it can be files or directories, so they
		  are _items_ on list of selected files or dirs. */
		  _("Size: %.0f MB in %d items"), files_size, num);

	show_disk_usage(files_size);
	wattrset(mainwin, A_NORMAL | COLOR_PAIR(1));
}




void main_win_refresh(bool real)
{
	if (mainwin != NULL) {
		wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
		select_window(real);
	}

	return;
}


void main_win_refresh_part(int nlines, int ncols, int begin_y, int begin_x)
{
	wrefresh(derwin(mainwin, nlines, ncols, begin_y, begin_x));

	return;
}





/* ******************************* */
/* ****** unused code below ****** */
/* ******************************* */




#if 0

			int rv = dialogbox(_("Please confirm"), _("Image file exist!\nDo you wish to overwrite it?"), DIALOG_YES_NO_CANCEL);

			if (rv == BUTTON_NO) { /* do not overwrite, ask for new path to iso file */
				input_box(_("Path to iso file"), _("Enter path to new iso file"), config.tempdir, 0);
				// wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				continue; /* user entered new path - check (again) if the file exists */
			} else if (rv == BUTTON_YES) { /* yes, overwrite existing iso image */
				file_set = true;
				return true;
			} else { /* BUTTON_CANCEL - don't create image */
				return false;
			}
		} else { /* file doesn't exist yet, it is safe to create iso file with such path/name */
			/* FIXME: access(2) sets errno in case of error - check it's value */
			int rv = 0;
			switch (errno) {
				case EACCES:
					fprintf(stderr, "EACCESS\n");
					rv = dialogbox(_("Message"), _("Wrong permissions for image file or dir. Enter new path?"), DIALOG_OK_CANCEL);
					break;
				case ELOOP:
					fprintf(stderr, "ELOOP\n");
					dialogbox(_("Error"), _("Wrong permissions for image file or dir. Enter new path?"), DIALOG_OK_CANCEL);
					break;
				case ENAMETOOLONG:
					fprintf(stderr, "ENAMETOOLONG\n");
					break;
				case ENOENT:
					fprintf(stderr, "ENOENT\n");
					break;
				case ENOTDIR:
					fprintf(stderr, "ENOTDIR\n");
					break;
				case EROFS:
					fprintf(stderr, "EROFS\n");
					break;
				case EFAULT:
					fprintf(stderr, "EFAULT\n");
					break;
				case EINVAL:
					fprintf(stderr, "EINVAL\n");
					break;
				case EIO:
					fprintf(stderr, "EIO\n");
					break;
				case ENOMEM:
					fprintf(stderr, "ENOMEM\n");
					break;
				}

		}
}








void processwin_finished(char *status_string, bool wait)
{
	wrefresh(processwin);
	return;
}




#endif



