/*
 * ImagePanel.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  ImagePanel.hpp

  The image panel for the Fl_Tabs in the main window. 
*/
#ifndef ImagePanel_hpp
#define ImagePanel_hpp

#include <FL/Fl_Choice.H>
#include <FL/Fl_Float_Input.H>
#include <FL/Fl_Int_Input.H>
#include "Fl_Table_Row.H"

#include "Br.hpp"               // BracketingCore::Event
#include "BrGUI.hpp"            // global `BrGUI' (later local!)

#include "YS/ys_dbg.hpp"        // CTOR(), DTOR()


/**================================================================
 *
 * ImageTable...
 *
 * Diese Tabelle praesentiert Daten aus dem globalen Container Br.imgVec.
 * Im Konstruktionsmoment konstruiert sie sich mit soviel Zeilen wie 
 * Bilder in Br.imgVec (gewoehnlich 0). (Nach)Laden von Bildern erfordert
 * dann Zeilenzahl anzupassen!
 * 
 *=================================================================*/
class ImageTable : public Fl_Table
{
    Fl_Float_Input* input_;     // for input of exposure times
    int             row_edit;
    const static int col_edit = 4;  // column of exposure time 

protected:
    void draw_cell(TableContext context,    // table cell drawing
                   int R=0, int C=0, int X=0, int Y=0, int W=0, int H=0);

public:
    ImageTable(int X, int Y, int W, int H, const char *la=0); 
    ~ImageTable() { DTOR(label()) }
    
    void        cb_event ();
    static void cb_event_(Fl_Widget*, ImageTable* I)
      { I -> cb_event(); }

private:      
    void        cb_input ();
    static void cb_input_(Fl_Widget*, ImageTable* I)
      { I -> cb_input(); }  
};


/**================================================================
 *
 * ImagePanel  --  the base panel in the Fl_Tabs in the main menu
 *
 *=================================================================*/
class ImagePanel : public Fl_Group
{
    typedef struct {const char* text; float value;} MenuMapItem_stops;
    
    static MenuMapItem_stops  menuMap_stops[];
    static Fl_Menu_Item       menu_choice_stops[];
    static const char*        tooltip_stops;
    static const char*        tooltip_activate;

    // widgets...
    ImageTable*      table_;
    Fl_Choice*       choice_stops_;   
    Fl_Float_Input*  input_smoothing_;
    Fl_Int_Input*    input_n_grid_points_;
    Fl_Button        *button_init_camera_, 
                     *button_compute_CCD_, 
                     *button_compute_HDR_, 
                     *button_make_HDR_, 
                     *button_follow_;  
    
    double val_stops_;        // stops value

public:
    ImagePanel (int X, int Y, int W, int H, const char* la=0);
    ~ImagePanel();
    
    ImageTable* table()        const { return table_; }
    Fl_Choice*  choice_stops() const { return choice_stops_; }
    
private:        
    // Fl_Callbacks...      
    void        cb_init_camera (Fl_Button*)    { BrGUI.init_camera(); }
    static void cb_init_camera_(Fl_Button* o, ImagePanel* I)
      { I -> cb_init_camera(o); }  

    void        cb_compute_CCD (Fl_Button*)    { BrGUI.make_CCD_curve(); }
    static void cb_compute_CCD_(Fl_Button* w, ImagePanel* I) 
      { I -> cb_compute_CCD(w); }

    void        cb_compute_HDR (Fl_Button*)    { BrGUI.complete_HDR(); }
    static void cb_compute_HDR_(Fl_Button* w, ImagePanel* I) 
      { I -> cb_compute_HDR(w); }
    
    void        cb_make_HDR    (Fl_Button*)    { BrGUI.make_HDR(); }
    static void cb_make_HDR_(Fl_Button* w, ImagePanel* I) 
      { I -> cb_make_HDR(w); }
    
    void        cb_FollowCurveWindow ()        { BrGUI.do_FollowCurveWindow(); }
    static void cb_FollowCurveWindow_(Fl_Widget*, ImagePanel* I)
      { I -> cb_FollowCurveWindow(); }  

    void        cb_stops_choice (Fl_Menu_*);
    static void cb_stops_choice_(Fl_Menu_* w, ImagePanel* I) 
      { I -> cb_stops_choice(w); }
    
    void        cb_grid_points (Fl_Int_Input*);
    static void cb_grid_points_(Fl_Int_Input* w, ImagePanel* I) 
      { I -> cb_grid_points(w); }
  
    void        cb_smoothing (Fl_Float_Input*);
    static void cb_smoothing_(Fl_Float_Input* w, ImagePanel* I) 
      { I -> cb_smoothing(w); }
  
    // other...
    void set_smoothing (double coeff);
    void set_n_grid_points (int n);
  
    // Distributor callbacks...
    void        refpic_changed (int pic);
    static void refpic_changed_(int* pic, ImagePanel* I)
      { I -> refpic_changed (*pic); }

    void        event_msg (BracketingCore::Event);
    static void event_msg_(BracketingCore::Event* e, ImagePanel* I)
      { I -> event_msg (*e); } 
};

#endif  // ImagePanel_hpp
                    
// END OF FILE
