/*
*
* Template Numerical Toolkit (TNT)
*
* Mathematical and Computational Sciences Division
* National Institute of Technology,
* Gaithersburg, MD USA
*
*
* This software was developed at the National Institute of Standards and
* Technology (NIST) by employees of the Federal Government in the course
* of their official duties. Pursuant to title 17 Section 105 of the
* United States Code, this software is not subject to copyright protection
* and is in the public domain. NIST assumes no responsibility whatsoever for
* its use by other parties, and makes no guarantees, expressed or implied,
* about its quality, reliability, or any other characteristic.
*
*/
/*
  Adds und some changes by Hartmut Sbosny, marked by "HS".

  Changes:
  - In `<<'-Operator (std::cout) formating (slightly) improved.

  Adds:
  - dot product
  - Euclidian norm
*/


#ifndef TNT_ARRAY1D_UTILS_H
#define TNT_ARRAY1D_UTILS_H

#include <cstdlib>
#include <cassert>

#include "tnt_array1d.hpp"
#include "tnt_math_utils.hpp"    // dot()

namespace TNT
{


template <class T>
std::ostream& operator<<(std::ostream &s, const Array1D<T> &A)
{
    int N=A.dim1();

#ifdef TNT_DEBUG_1
    s << "addr: " << (void *) &A[0] << '\n';
#endif
    s << "Array1D [ " << N << " ]:\n";
    for (int j=0; j<N; j++)
    {
        s << " [" << j << "]\t" << A[j] << '\n';
    }

    return s;
}

template <class T>
std::istream& operator>>(std::istream &s, Array1D<T> &A)
{
    int N;
    s >> N;

    Array1D<T> B(N);
    for (int i=0; i<N; i++)
        s >> B[i];
    A = B;
    return s;
}



template <class T>
Array1D<T> operator+(const Array1D<T> &A, const Array1D<T> &B)
{
    int n = A.dim1();

    if (B.dim1() != n )
        return Array1D<T>();

    else
    {
        Array1D<T> C(n);

        for (int i=0; i<n; i++)
        {
            C[i] = A[i] + B[i];
        }
        return C;
    }
}



template <class T>
Array1D<T> operator-(const Array1D<T> &A, const Array1D<T> &B)
{
    int n = A.dim1();

    if (B.dim1() != n )
        return Array1D<T>();

    else
    {
        Array1D<T> C(n);

        for (int i=0; i<n; i++)
        {
            C[i] = A[i] - B[i];
        }
        return C;
    }
}


template <class T>
Array1D<T> operator*(const Array1D<T> &A, const Array1D<T> &B)
{
    int n = A.dim1();

    if (B.dim1() != n )
        return Array1D<T>();

    else
    {
        Array1D<T> C(n);

        for (int i=0; i<n; i++)
        {
            C[i] = A[i] * B[i];
        }
        return C;
    }
}


template <class T>
Array1D<T> operator/(const Array1D<T> &A, const Array1D<T> &B)
{
    int n = A.dim1();

    if (B.dim1() != n )
        return Array1D<T>();

    else
    {
        Array1D<T> C(n);

        for (int i=0; i<n; i++)
        {
            C[i] = A[i] / B[i];
        }
        return C;
    }
}




template <class T>
Array1D<T>&  operator+=(Array1D<T> &A, const Array1D<T> &B)
{
    int n = A.dim1();

    if (B.dim1() == n)
    {
        for (int i=0; i<n; i++)
        {
                A[i] += B[i];
        }
    }
    return A;
}


template <class T>
Array1D<T>&  operator-=(Array1D<T> &A, const Array1D<T> &B)
{
    int n = A.dim1();

    if (B.dim1() == n)
    {
        for (int i=0; i<n; i++)
        {
            A[i] -= B[i];
        }
    }
    return A;
}


template <class T>
Array1D<T>&  operator*=(Array1D<T> &A, const Array1D<T> &B)
{
    int n = A.dim1();

    if (B.dim1() == n)
    {
        for (int i=0; i<n; i++)
        {
            A[i] *= B[i];
        }
    }
    return A;
}


template <class T>
Array1D<T>&  operator/=(Array1D<T> &A, const Array1D<T> &B)
{
    int n = A.dim1();

    if (B.dim1() == n)
    {
        for (int i=0; i<n; i++)
        {
            A[i] /= B[i];
        }
    }
    return A;
}




// =======
// My Adds:
// =======

/**
    Scalar product (dot-product, inner product) of two Array1D's.

    - Assume contigious storing of data.
    - Primitve version without any attention to overflow/underflow or
      cascade summation.
*/
template <class T>
T dot (const Array1D<T> & A, const Array1D<T> & B)
{
    assert (A.dim() == B.dim());

    return dot (A.dim(), &A[0], &B[0]);        // chrashes for Null-Arrays

#if 0
    const T* a = &A[0];                        // chrashes for Null-Arrays
    const T* b = &B[0];
    const T* end = a + A.dim();
    T sum = 0;
    while (a < end)
        sum +=    *a++ * *b++;

    return sum;
#endif
}


template <class T>
T norm2 (const Array1D<T> & A, const Array1D<T> & B)
{
    assert (A.dim() == B.dim());

    return sqrt(dot(A,B));
}



} // namespace TNT

#endif
