/* DesPack.c	By Ken Stephenson, Univ. of Tenn. 1996

	cc -o DP DesPack.c cp_remote_library.c -g -lm 
 
Argument "-cmd" puts this program in command line mode, otherwise it
expects to be talking to a parent routine such as CirclePack.

Program for development of routines for handling dessins. Interface
pieces for running this remotely from CirclePack may be found in
"remote_shell.c" with CirclePack distribution. But this experimental
procedure is designed for both direct use and remote use via CirclePack.

DESSINS (d'enfants):

The packings considered are associated with certain triangulations of
surfaces; always an even number of triangles, half to be mapped to
upper half-plane, half to lower, and special vertices, called d_vertices,
partitioned into those mapping to 0, 1, and infty, respectively.

Always try to get the d_vertices to have the lowest vertex numbers.
These structures can be built out of "hex" triangles; three d_verts,
three edge barycenter verts, and one mid-face vert. An orientation is
attached to tell order of 0,1,infty; need this and knowledge of only
ONE d_vertex to get whole structure. 

Once a simple structure is developed, it can be refined in CirclePack
via the "hex_refine" command. Fortunately, this leaves all the
original vertices with their same numbers, relative locations, and
meanings. Thus, although there may be only a few dessin faces, an
actual packing may have thousands of faces and circles.

Some conventions: New data is stored as follows in packing files
or separate files. It should be in this order:

----------------

(1). First get the number of dessin faces and for each face, orientation,
1 for upper half plane, -1 for lower; then the three
neighboring faces: the first shares 0-pt and 1-pt, next, 1-pt and infty-pt,
and last, infty-pt and 0-pt. When orientation is -1, these neighbors are 
in clockwise (negative) order around the geometric face.).

	DESSIN_FACECOUNT: 28

	1	1	2  12  6
	2	-1	1  13  3
	...
	...
	28 	-1	7  5  27


(2). Next get the vertices for each face, order 0-vert, 1-vert, inf-vert.
If this is not provided, the program will create it using "bd" command 
(for "build dessin").

	DESSIN_FACE_VERTS: 

	1	1 3 7
	2	1 4 8
	...
	...
	28	5 9 10


(3). Finally, get "redchain" of faces; that is, list of faces which defines
a fundamental domain in multiply connected cases.

	DESSIN_FACE_RED_LIST:

	2 4 10 9 ...... 2
	(done)

-----------------

When a packing has been computed for a given dessin structure, the
actual "dessin d'enfant" (the 'drawing' itself) is just the union of
the edges from 0 to 1 in each dessin face, which can be found from
the dessin structural data.


========================================================================
======================================================================== */

#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>
#include "cp_types.h"
#include "cp_proto.h"

typedef struct
{
	int orient;	/* half-plane, +1 upper, -1 lower */
	int verts[3];	/* vertices, always listed 0, 1, infty */
	int new_vert[3];/* utility storage for resetting vert no's */
	int nghb[3];	/* neighboring faces across [0,1], ... */
	int mark;	/* useful flag */
	int midvert;	/* vert at face center */
	int midices[3]; /* flower index in verts[j] towards face center */
	int rwb_flag;	/* for marking red chain */
} des_face;

typedef struct
{
	int facecount;		/* number of d-faces */
	des_face *d_faces;	/* pointer to face data */
} Dessin_data;


/* ============== prototypes ===============*/
int which_face(int v,int w,int u);
int go_around(struct p_data *p,int v,int w,int n,int lrflag);
int process_edge_chain(struct p_data *p,struct Vertlist *v_list,
    int sidelength,struct Edgelist **edgelist,
    struct Vertlist **facelist,int want);
int process_face(struct p_data *p,int v,int w,int u,int ind,
    int sidelength,struct Edgelist **edgelist,
    struct Vertlist **facelist,int want);
int process_edge(struct p_data *p,int v,int w,int ind,int sidelength,
    struct Edgelist **edgelist,struct Vertlist **facelist,int want,
    int *outdex); 
int find_sidelength(struct p_data *p,int v);
int place_cross_edge_face_no(struct p_data *p,int f,int v,int sidelength,
    int *nghbf,int *midcirc);
int which_face(int v,int w,int u);
struct Vertlist * ck_in_list(struct Vertlist *redlist,int v,int w);
int ck_edge(struct p_data *p,int v,int w,int ind,int sidelength);
int check_list(struct Vertlist *list,int v);
int next_face(struct p_data *p,int v,int f,int lrflag);
int nghb_face(struct p_data *p,int v,int w);
int send_edge_fv(FILE *fp,struct p_data *p,int f,int v,int q,
    int pflag);
int send_edge(struct p_data *p,int v,int w,int ind,int q,int pflag);
int set_send_face(FILE *fp,struct p_data *p,int f,int q,int setflag,
    int cflag,int col,int show,int pflag);
int genus_bdry(Dessin_data *d_data,int *g,int *e,int *n,int *geom);
int number_verts(Dessin_data *d_data);
int ck_smooth(Dessin_data *d_data);
int make_basic_pack(struct p_data *basepack);
int bd_call(struct p_data *p);
int build_dessin_packing(struct p_data *p,struct p_data *q,
    struct p_data *r,Dessin_data *d_data);
int add_vertex(Dessin_data *d_data,int *f1,int *f2,int *nf1,int *nf2);
int add_free_edge(Dessin_data *d_data,int *f1,int *f2,int *nf1,int *nf2);
int add_new_vert(Dessin_data *d_data,int *f1,int *f2,int *nf1,
    int *nf2,int *nf3,int *nf4);
int add_loop(Dessin_data *d_data,int *f1,int *f2,int *nf1,int *nf2,
    int *nf3,int *nf4);
int add_bridge_edge(Dessin_data *d_data,int *f1,int *f2,int *nf1,
    int *nf2,int *nf3,int *nf4);
int mark_red_verts(struct p_data *p,Dessin_data *d_data,
    struct Vertlist *redlist,struct Vertlist **redverts);
int send_red_chain(FILE *fp,struct p_data *p,Dessin_data *d_data,
    struct Vertlist *redlist,struct Vertlist **redverts,int pnum);
int read_dessin_data(struct p_data *p,Dessin_data *d_data);
int write_dessin_data(FILE *fp,Dessin_data *d_data,int flag);
int ck_for_red_edge(int f);
int start_data_file();
int exec_data_file();
int carbon_copy(struct p_data *p,struct p_data *q);
int bdry_flags(struct p_data *p);
int pnv(struct p_data *p,Dessin_data *d_data,int f,int nf);
int vsp(int *oldnew,int n,int f,int nf);
int bugwrite(struct p_data *p);
int get_parent_data(char *datastr);
int get_next_cmd();
int msg();
int emsg();
int swap_faces(Dessin_data *d_data,int f1,int f2);

/* user specified variables set before compiling */

struct p_data *PackData;
struct p_data *packdata;

char emsgbuf[BUFSIZE+10],*msgbuf,next[BUFSIZE],buf[BUFSIZE];
   /* Note: msgbuf, emsgbuf, and msg(), emsg() defined here
      are necessary for library routines */

char nextcmd[BUFSIZE],datafilename[256];	
	/* utility buffers for strings. */
double toler=TOLER,okerr=OKERR;
int cmdmode,sidelength,datafilecount=0,Pid;

FILE *fp;
FILE *datafp;
struct Vertlist *dessin_redlist=NULL,*des_redverts=NULL;
struct Vertlist *ck_in_list();
int **face_org;	/* temp ptrs to list of faces by vertex */

Dessin_data *dessin; 

int main(int argc,char *argv[])
{
	int i,pnum=-1,w,v,u,k;
	int flag,pflag,nflag,nnflag,sflag;
	int nv,nw,nnv,nnw,midcirc,jj,f,ind,j;
	struct Edgelist *edgelist=NULL;
	struct Vertlist *facelist=NULL,*ftrace;
	struct Vertlist *v_list=NULL,*vtrace,*vtmp;
	char *cmdptr,filename[256];

	msgbuf=emsgbuf+10; 
	strcpy(emsgbuf,"**ERROR** "); /* set up standard buffers
					 which routines expect for
					 messages/errors */
	
/* put command line info in cmdstr; check for cmd/remote mode. */

	cmdmode=0; /* default, get commands from parent program
		      like CirclePack */
	for (i=1;i<=argc;i++) 
		if (strcmp(argv[i-1],"-cmd")==0) {cmdmode=1;break;}
		/* direct from user (-cmd) or through parent program */
	face_org=NULL;
	Pid = (int)getpid();

/* ---------- initialization --------------- */

	dessin=(Dessin_data *)calloc(1,sizeof(Dessin_data));
	PackData=(struct p_data *)
		calloc(3,sizeof(struct p_data));
	alloc_pack_space(&PackData[0],500,0);
	packdata=&PackData[0]; /* convenient pointer */
	make_basic_pack(&PackData[1]); /* ?? defaults to basic pieces now */
	make_basic_pack(&PackData[2]);
	reverse_orient(&PackData[2]);

/* ---------- hello world ------------------ */
	sprintf(msgbuf,"Remote Program DesPack: compute/return dessin-related data on packings");
	msg();
	sprintf(msgbuf,"    Type ?? for help with commands. ");
	msg();

/* --------- cycle through commands --------- */

	while (get_next_cmd()) /* either from user or parent program */
	 {
		cmdptr=nextcmd;
		if (grab_next(&cmdptr,next)) 

switch(next[0]) {

case 'a':
 {
/* add new bridge edge (crossing cell) */
   if (strncmp(next,"ab",2)==0 && sscanf(cmdptr,"%d %d",&v,&w)==2) 
    {
	if (v>0 && v<=dessin->facecount
		&& w>0 && w<=dessin->facecount )
	 {
	   if (!add_bridge_edge(dessin,&v,&w,&nv,&nw,&nnv,&nnw))
	    sprintf(msgbuf,"add_bridge_edge failed");
	   else
	    {
		sprintf(msgbuf,
		   "added bridge: new faces %d %d %d %d next to %d and %d.",
		   nv,nw,nnw,nnv,v,w);
		bd_call(packdata);
		mark_red_verts(packdata,
		   dessin,dessin_redlist,&des_redverts);
	    }
	   msg();
	 }
    }
/* add new edge (sticking out) */
   else if (strncmp(next,"ae",2)==0 && sscanf(cmdptr,"%d %d",&v,&w)==2) 
    {
	if (v>0 && v<=dessin->facecount
		&& w>0 && w<=dessin->facecount )
	 {
	   if (!add_free_edge(dessin,&v,&w,&nv,&nw)
	    || !add_vertex(dessin,&nv,&nw,&nnv,&nnw))
		sprintf(msgbuf,"add_free_edge failed");
	   else
	    {
		sprintf(msgbuf,
		   "added edge: new faces %d %d %d %d between %d and %d.",
		   nw,nv,nnv,nnw,v,w);
		bd_call(packdata);
		mark_red_verts(packdata,
		   dessin,dessin_redlist,&des_redverts);
	    }
	   msg();
	 }
    }
/* add vert to end of free edge */
   else if (strncmp(next,"av",2)==0 && sscanf(cmdptr,"%d %d",&v,&w)==2) 
    {
	if (v>0 && v<=dessin->facecount
		&& w>0 && w<=dessin->facecount )
	 {
	    if (!add_vertex(dessin,&v,&w,&nv,&nw))
	     sprintf(msgbuf,
	 	"add_vertex failed: f1 f2 must share edge and 1-pt.");
	   else
	    {
	        sprintf(msgbuf,
		   "added vert: new faces %d %d between %d and %d.",
		   nv,nw,v,w);
		bd_call(packdata);
		mark_red_verts(packdata,
		   dessin,dessin_redlist,&des_redverts);
	    }
	    msg();
	 }
    }
/* add loop to 0-pt */
   else if (strncmp(next,"al",2)==0 && sscanf(cmdptr,"%d %d",&v,&w)==2) 
    {
	if (v>0 && v<=dessin->facecount
		&& w>0 && w<=dessin->facecount )
	 {
	   if (!add_loop(dessin,&v,&w,&nv,&nw,&nnv,&nnw))
		   sprintf(msgbuf,
		    "add_loop failed: f1 f2 share edge and 0-pt");
	   else
	    {
		sprintf(msgbuf,
		   "added loop: new face fan: %d %d %d %d between %d and %d.",
		   nv,nnv,nnw,nw,v,w);
		bd_call(packdata);
		mark_red_verts(packdata,
		   dessin,dessin_redlist,&des_redverts);
	    }
	   msg();
	 }
    }
/* add 0-pt in middle of edge */
   else if (strncmp(next,"am",2)==0 && sscanf(cmdptr,"%d %d",&v,&w)==2) 
    {
	if (v>0 && v<=dessin->facecount
		&& w>0 && w<=dessin->facecount )
	 {
	   if (!add_new_vert(dessin,&v,&w,&nv,&nw,&nnv,&nnw))
	    sprintf(msgbuf,
	      "add_new_vert failed: f1 f2 share edge getting new vert.");
	   else
	    {
		sprintf(msgbuf,
	           "added new vert: new faces %d %d %d %d between %d and %d.",
		   nv,nw,nnv,nnw,v,w);
		bd_call(packdata);
		mark_red_verts(packdata,
		   dessin,dessin_redlist,&des_redverts);
	    }
	   msg();
	 }
    }
/* add free edge */
   else if (strncmp(next,"af",2)==0 && sscanf(cmdptr,"%d %d",&v,&w)==2) 
    {
	if (v>0 && v<=dessin->facecount
		&& w>0 && w<=dessin->facecount )
	 {
	   if (!add_free_edge(dessin,&v,&w,&nv,&nw))
	     sprintf(msgbuf,
		"add_free_edge failed: f1 f2 must share 0-pt.");
	   else
	    {
		sprintf(msgbuf,
		   "added free edge: new faces %d %d between %d and %d.",
		   nv,nw,v,w);
		bd_call(packdata);
		mark_red_verts(packdata,
		   dessin,dessin_redlist,&des_redverts);
	    }
	   msg();
	 }
    }


/* adjoin dessin data to file */
   else if (strncmp(next,"ad",2)==0 && grab_next(&cmdptr,next)) 
    {
	if ((fp=fopen(next,"a"))==NULL
	   || !write_dessin_data(fp,dessin,1)) 
	 {
	   sprintf(msgbuf,"DP: write of dessin failed.");
	   emsg();
	 }
	if (fp) fclose(fp);
    }
   break;
 }



case 'b': 
 {
   if (!strncmp(next,"bd",2)) 
	/* build dessin packing (from loaded dessin structure).
	   Don't need this if packing is read in from elsewhere. */
    {
	bd_call(packdata);
    }
   if (strncmp(nextcmd,"bye",3)==0) 
    {
	msg("DP: exited successfully.\n");
	exit(0);
    }
   break;
 }



case 'c':
 {
   if (!strncmp(next,"ck",2)) /* check dessin */
    {
	ck_smooth(dessin);
    }
   if (!strcmp(next,"c") || !strcmp(next,"c-p")) 
		/* send draw/print instructions to CirclePack
		for chain (eg, redchain) of dessin edges from 
		given list of verts. */
    {
	if (v_list) vert_free(&v_list);
	vtmp=v_list;
	while (grab_next(&cmdptr,next)
	   && (v=atoi(next))>0 && v<=packdata->nodecount)
	 {
		vtrace=(struct Vertlist *)malloc(sizeof(struct Vertlist));
		vtrace->next=NULL;
		vtrace->v=v;
		if (!vtmp) v_list=vtrace;
		else vtmp->next=vtrace;
	 }
	start_data_file();
	if (v_list && v_list->next 
	   && process_edge_chain(packdata,v_list,sidelength,
		&edgelist,&facelist,1) )
	 {
		if (!strcmp(next,"c-p")) sprintf(msgbuf,"post -p%d -F ",pnum);
		else sprintf(msgbuf,"disp -p%d -F ",pnum);
		ftrace=facelist;
		while (ftrace)
		 {
			sprintf(buf,"%d ",ftrace->v);
			strcat(msgbuf,buf);
			ftrace=ftrace->next;
		 }
		fprintf(fp,"%s\n",msgbuf);
	 }
	else
	 {
	   sprintf(msgbuf,
	    "given vertex string, starting %d, %d, .., seems to be in error.",
		v_list->v,v_list->next->v);
	   emsg();
	 }
	exec_data_file();
	vert_free(&facelist); 
	break;
    }
   else if (strcmp(next,"cr")==0) /* convert/send redchain of pack faces. */
    {
    	start_data_file();
	send_red_chain(datafp,packdata,dessin,dessin_redlist,
		&des_redverts,pnum);
	exec_data_file();
	break;
    }
   break;
 }



case 'd':
 {
   if (!strcmp(next,"de") || !strcmp(next,"de-p")) 
		/* draw/print d-edges given endpoints. */
    {
	flag=1;
	pflag=!strcmp(next,"de-p");
	while (flag && grab_next(&cmdptr,next)
	   && (v=atoi(next))>0 && v<=packdata->nodecount
	   && grab_next(&cmdptr,next)
	   && (w=atoi(next))>0 && w<=packdata->nodecount) 
	 {
		if (grab_next(&cmdptr,next) && (ind=atoi(next))>=0
			&& ind<=packdata->packK_ptr[v].num);
		else ind=-1;		
		flag=send_edge(packdata,v,w,ind,pnum,pflag);
	 }
    }
   else if (!strcmp(next,"dt") || !strcmp(next,"dts") 
	|| !strcmp(next,"dt-p") || !strcmp(next,"dts-p")) 
		/* draw/print d-faces from vert-triples, lay out
		via bdry chains */
    {
	pflag=(!strcmp(next,"dt-p") || !strcmp(next,"dts-p"));
	sflag=(!strcmp(next,"dts") || !strcmp(next,"dts-p"));
	flag=1;
	start_data_file();
	while (flag && grab_next(&cmdptr,next)
	   && (v=atoi(next))>0 && v<=packdata->nodecount
	   && grab_next(&cmdptr,next)
	   && (w=atoi(next))>0 && w<=packdata->nodecount
	   && grab_next(&cmdptr,next)
	   && (u=atoi(next))>0 && u<=packdata->nodecount )
	 {
		if (!(f=which_face(v,w,u)))
		 {
		   sprintf(msgbuf,
			"incorrect dessin face data p%d: <%d,%d,%d>.",
			 pnum,v,w,u);
		   emsg();
		   flag=0;
		 }
		else set_send_face(datafp,packdata,
			f,pnum,1,sflag,0,1,pflag);
	 }
	exec_data_file();
	break;
    }
   else if (!strcmp(next,"df") || !strcmp(next,"dfn") 
	|| !strcmp(next,"dfnn")
	|| !strcmp(next,"dfs") || !strcmp(next,"df-p") 
	|| !strcmp(next,"dfs-p") || !strcmp(next,"dfn-p") 
	|| !strcmp(next,"dfnn-p") )
		/* draw/print faces (by number) (opt: with number, shaded) */
   {
	pflag=(!strcmp(next,"df-p") || !strcmp(next,"dfn-p") 
		|| !strcmp(next,"dfnn-p"));
	if (next[2]=='s') sflag=1; /* shade faces? */
	else sflag=0;
	if (next[2]=='n') nflag=1; /* put in face number? */
	else nflag=0;
	if (next[3]=='n') nnflag=1; /* draw nghb face no. across red edge? */
	else nnflag=0;
	flag=1;
	start_data_file();
	while (flag && grab_next(&cmdptr,next)
	   && (f=atoi(next))>0 && f<=dessin->facecount)
	 {
		if (!(flag=set_send_face(datafp,packdata,
			f,pnum,1,sflag,1,1,pflag)) )
		 {
		   sprintf(msgbuf,"there is no dessin face %d, p%d",f,pnum);
		   emsg();
		   break;
		 }
		if (nflag) /* want number; convention: goes at center of 
		   dessin face center circle, generally numbered 		
		   (dessin->facecount + f) .*/
		 {
		   if (pflag) 
			sprintf(msgbuf,"post -p%d -nl %d %d",
				pnum,dessin->facecount+f,f);
		   else
		   	sprintf(msgbuf,"disp -p%d -nl %d %d",
				pnum,dessin->facecount+f,f);
		   fprintf(datafp,"%s\n",msgbuf);
		   if (nnflag && (k=ck_for_red_edge(f))>=0) 
			/* get number across red edge */
		    {
			place_cross_edge_face_no(packdata,
				f,k,sidelength,&jj,&midcirc);
			if (pflag)
				sprintf(msgbuf,"post -p%d -nl %d %d",
				   pnum,midcirc,jj);
		   	else
			   	sprintf(msgbuf,"disp -p%d -nl %d %d",
				   pnum,midcirc,jj);
			fprintf(datafp,"%s\n",msgbuf); /* send nghb face no. */
		    }
		 }
	 }
	exec_data_file();
	break;
    }
   else if (!strcmp(next,"dbw") || !strcmp(next,"dbw-p")) 
	/* draw (or print) black/white dessin faces */
    {
	if (dessin->facecount<1)
	 {
		sprintf(msgbuf,"DP: No dessin loaded for p%d",pnum);
		emsg();
		break;
	 }
	pflag=(!strcmp(next,"dbw-p")); /* send for printing */
	start_data_file();
	for (f=1;f<=dessin->facecount;f++)
	 {
			/* first, layout in order and draw outer edge 
			   shaded if lower-hemisphere face */
		set_send_face(datafp,packdata,f,pnum,1,
			(dessin->d_faces[f].orient<0),
			(dessin->d_faces[f].orient<0),1,pflag);
	 }
	exec_data_file();
	break;
    }
   else if (!strcmp(next,"dd") || !strcmp(next,"dd-p")) 
		/* draw/print whole dessin (ie. all 0-1 edges) */
    {
	if (dessin->facecount<1)
	 {
		sprintf(msgbuf,"DP: No dessin loaded for p%d.",pnum);
		emsg();
		break;
	 }
	pflag=(!strcmp(next,"dd-p")); /* send for printing */
		/* first, thru red faces for all copies of same 01 edge. */
	vtrace=dessin_redlist;
	start_data_file();
	while (vtrace)
	 {
		f=vtrace->v;
		if (dessin->d_faces[f].orient>0) v=dessin->d_faces[f].verts[0];
		else v=dessin->d_faces[f].verts[1];
		set_send_face(datafp,packdata,
			f,pnum,1,0,0,0,0); /* place the face */
		send_edge_fv(datafp,packdata,f,v,pnum,pflag);
		vtrace=vtrace->next;
	 }
		/* go get the rest (get some redundancy with red faces) */	
	for (f=1;f<=dessin->facecount;f++)
	   if (dessin->d_faces[f].orient>0) /* to avoid some of duplication. */
	    {
		set_send_face(datafp,packdata,
			f,pnum,1,0,0,0,0); /* place the face */
		send_edge_fv(datafp,packdata,f,
			dessin->d_faces[f].verts[0],pnum,pflag);
	    }
	exec_data_file();
	break;
    }
   else if (!strcmp(next,"di") || !strcmp(next,"d0") 
	|| !strcmp(next,"d1")
	|| !strcmp(next,"di-p") || !strcmp(next,"d0-p") 
	|| !strcmp(next,"d1-p")
	|| !strcmp(next,"dia") || !strcmp(next,"d0a") 
	|| !strcmp(next,"d1a")
	|| !strcmp(next,"dia-p") || !strcmp(next,"d0a-p") 
	|| !strcmp(next,"d1a-p")
)
		/* draw shaded circles at 0,1,infty verts of faces. */
    {
	k=strlen(next);
	if (next[k-1]=='p') pflag=1;
	else pflag=0;
	start_data_file();
	if (next[2]=='a') 
	 for (f=1;f<=dessin->facecount;f++)
	  {	
		j=0;
		if (next[1]=='1') j=1;
		else if (next[1]=='i') j=2;
		v=dessin->d_faces[f].verts[j];
	 	if (check_list(des_redverts,v)) 	
		   set_send_face(datafp,packdata,
			f,pnum,1,0,0,0,0); /* place the face */
		if (pflag) 
			sprintf(msgbuf,"post -p%d -cf %d",pnum,v);
		else
	   		sprintf(msgbuf,"disp -p%d -cf %d",pnum,v);
		fprintf(fp,"%s\n",msgbuf);
	  }
	else 
	 while (grab_next(&cmdptr,next)
	   && (f=atoi(next))>0 && f<=dessin->facecount)
	  {
		j=0;
		if (next[1]=='1') j=1;
		else if (next[1]=='i') j=2;
		v=dessin->d_faces[f].verts[j];
	 	if (check_list(des_redverts,v)) 	
		   set_send_face(datafp,packdata,
			f,pnum,1,0,0,0,0); /* place the face */
		if (pflag) 
			sprintf(msgbuf,"post -p%d -cf %d",pnum,v);
		else
	   		sprintf(msgbuf,"disp -p%d -cf %d",pnum,v);
		fprintf(fp,"%s\n",msgbuf);
	  } /* end of while */
	exec_data_file();
    }
   break;
 }









case 'l':
 {
   if (strncmp(next,"lf",2)==0 || strncmp(next,"lfn",3)==0
	|| strncmp(next,"lf-p",4)==0 || strncmp(next,"lfn-p",5)==0)	
		/* layout/number draw/print all faces */
    {
 	if (dessin->facecount<1)
	 {
		sprintf(msgbuf,"DP: No dessin loaded for p%d.",pnum);
		emsg();
		break;
	 }
	pflag=(!strcmp(next,"lf-p") || !strcmp(next,"lfn-p")); /* print? */
	nflag=((!strcmp(next,"lfn") || !strcmp(next,"lfn-p"))); 
		/* want numbers? */
/* first, thru red faces for all copies of same 01 edge. */
	for (f=1;f<=dessin->facecount;f++) dessin->d_faces[f].mark=0;
	vtrace=dessin_redlist;
	start_data_file();
	while (vtrace)
	 {
		f=vtrace->v;
		dessin->d_faces[f].mark=1;
		if ((flag=set_send_face(datafp,packdata,
			f,pnum,1,0,1,1,pflag)) && nflag)
		 {
		   if (pflag) 
			sprintf(msgbuf,"post -p%d -nl %d %d\n",
				pnum,dessin->facecount+f,f);
		   else
		   	sprintf(msgbuf,"disp -p%d -nl %d %d\n",
				pnum,dessin->facecount+f,f);
		   fprintf(datafp,"%s\n",msgbuf); /* send face number */
		   if ((k=ck_for_red_edge(f))>=0) 
			/* position number across red edge */
		    {
			place_cross_edge_face_no(packdata,
				f,k,sidelength,&jj,&midcirc);
			if (pflag)
				sprintf(msgbuf,"post -p%d -nl %d %d\n",
				   pnum,midcirc,jj);
		   	else
			   	sprintf(msgbuf,"disp -p%d -nl %d %d\n",
				   pnum,midcirc,jj);
			fprintf(datafp,"%s\n",msgbuf); /* send nghb face no. */
		     }
		  }
		 vtrace=vtrace->next;
	 } /* end of while */
/* go get the rest */	
	for (f=1;f<=dessin->facecount;f++)
	   if (!dessin->d_faces[f].mark) 
	    {
		set_send_face(datafp,packdata,
			f,pnum,1,0,1,1,pflag); /* place the face */
		if (nflag)
		 {
		   if (pflag) 
			sprintf(msgbuf,"post -p%d -nl %d %d\n",
				pnum,dessin->facecount+f,f);
		   else
		   	sprintf(msgbuf,"disp -p%d -nl %d %d\n",
				pnum,dessin->facecount+f,f);
		   fprintf(datafp,"%s\n",msgbuf);
		 }
	    }
	exec_data_file();
	break;
    }
 }








case 'r':
 {
   if (strncmp(next,"rp",2)==0 && grab_next(&cmdptr,next)) 
	/* read packing */
    {
	flag=0;
	if (cmdmode==0) /* required to get packing from parent. */
	 {
		if ((pnum=atoi(next))<0 || pnum>4
			|| !r_pack_read(packdata,pnum,0017) 
			|| !readpack(stdin,packdata)) 
		 {
			sprintf(msgbuf,"DP: read of pack p%d failed.",pnum);
			emsg();
		 }
		else {flag=1;packdata->status=1;}
	 }
	else
	 {
		if ((fp=fopen(next,"r"))==NULL
			|| !readpack(fp,packdata)) 
		 {
			sprintf(msgbuf,"DP: read of %s failed.",next);
			emsg();
		 }
		else {flag=1;packdata->status=1;}
		if (fp) fclose(fp);
	 }
	if (flag)
	 {
		if (packdata->faces!=NULL) 
		 {free(packdata->faces);packdata->faces=NULL;}
		complex_count(packdata,1);
		catalog_faces(packdata);
		if ((sidelength=find_sidelength(packdata,1))==0)
		 {
			sprintf(msgbuf,
			  "DP: Caution, pack may not be 'dessin' type.");
			emsg();
		 }
		else
		 {
			sprintf(msgbuf,
			  "DP: read from p%d appears okay.",pnum);
			msg();
		 }
	 }
    }
   else if (!strcmp(next,"rd")) /* read dessin data 
			(hopefully, for this packing) */
    {
	if (!grab_next(&cmdptr,filename) 
	   || !(fp=fopen(filename,"r")) )
	 {
		sprintf(msgbuf,"DP: Cannot open file %s.",filename);
		emsg();
	 }
	else if (!read_dessin_data(packdata,dessin) || !bd_call(packdata))
	 {
		sprintf(msgbuf,
		   "DP: Reading/processing of dessin data failed.");
		emsg();
	 }
	else
	 {
		sprintf(msgbuf,
		  "DP: read of dessin appears okay.");
		msg();
	 }
	if (fp) fclose(fp);
	break;
    }
   break;
}



case 's':  
 {
   if (!strncmp(next,"swap",2) && sscanf(cmdptr,"%d %d",&v,&w)==2) 
	/* swap face no.s */
    {
	if (!swap_faces(dessin,v,w)) 
	 {
		sprintf(msgbuf,"face swap didn't work.");
		emsg();
	 }
    }
   break;
 }




case 'w':
 {
   if (strncmp(next,"wp",2)==0 && grab_next(&cmdptr,next)) 
	/* write packing and send redchain, if available */
    {
	if (cmdmode==0)
	 {
		pnum=atoi(next);
		if (!r_pack_write(packdata,pnum,1)) /* write comb data only */
		 {
			sprintf(msgbuf,"DP: write to p%d failed.",pnum);
			emsg();
		 }
		else 
		 {
		 	start_data_file();
		 	send_red_chain(datafp,packdata,dessin,dessin_redlist,
			   &des_redverts,pnum);
			exec_data_file();
		 }
	 }
	else
	 {
		if ((fp=fopen(next,"w"))==NULL
			|| !writepack(fp,packdata,1,0)) 
		 {
			sprintf(msgbuf,"DP: write to %s failed.",
				next);
			emsg();
		 }
		else 
		 {
		 	start_data_file();
		 	send_red_chain(datafp,packdata,dessin,dessin_redlist,
			   &des_redverts,pnum);
			exec_data_file();
		 }
		if (fp) fclose(fp);
	 }
    }
   else if (strncmp(next,"wd",2)==0 && grab_next(&cmdptr,next)) 
		/* write dessin */
    {
	if ((fp=fopen(next,"w"))==NULL
		|| !write_dessin_data(fp,dessin,1)) 
	 {
	   sprintf(msgbuf,"DP: write of dessin data to %s failed.",
		next);
	   emsg();
	 }
	if (fp) fclose(fp);
    }
   break;
 }
case '?': /* send summary of commands */
 {
sprintf(msgbuf,"read pack:   rp filename  (or from pack n: rp n)");
msg();
sprintf(msgbuf,"read dessin:   rd filename");
msg();
sprintf(msgbuf,"write pack:   wp filename  (or to pack n: wp n)");
msg();
sprintf(msgbuf,"write dessin:   wd filename");
msg();
sprintf(msgbuf,"append dessin data to file:   ad filename");
msg();
sprintf(msgbuf,"send pack redchain (from dessin redchain): cr");
msg();

sprintf(msgbuf,"draw d_faces (by vert-triple): dt <(v w u) ..> (shaded: dts)");
msg();
sprintf(msgbuf,"draw d_faces (by no.): df <f..>  (with no: dfn, shaded: dfs)");
msg();
sprintf(msgbuf,"draw dessin itself (0-1 edges):   dd ");
msg();
sprintf(msgbuf,"draw shaded lower hemisphere faces:  dbw (dbw-p)");
msg(); 
sprintf(msgbuf,"draw d_edges:   de <(v w) ..>");
msg();
sprintf(msgbuf,"draw verts (0,1, or i): d0 <f..>; d1 <f..>; di <f..> dia (all)");
msg();
sprintf(msgbuf,"draw d_edges along vert chain:    c <v..>");
msg();
sprintf(msgbuf,"layout faces:  lf   (numbered: lfn)");
msg();
sprintf(msgbuf,"     (To print, add '-p' to end of command: e.g., dd-p.");
msg();

sprintf(msgbuf,"build dessin packing (from dessin struct): bd");
msg();
sprintf(msgbuf,"check dessin: ck");
msg();
sprintf(msgbuf,"swap face numbers:   swap f1 f2");
msg();
sprintf(msgbuf,"add new edge:   ae f1 f2");
msg();
sprintf(msgbuf,"add loop:   al f1 f2");
msg();
sprintf(msgbuf,"add vert to 0-1 edge:   am f1 f2");
msg();
sprintf(msgbuf,"add bridge:   ab f1 f2");
msg();
sprintf(msgbuf,"add free edge:   af f1 f2");
msg();
sprintf(msgbuf,"add end to free edge:   av f1 f2");
msg();
sprintf(msgbuf,"quit: bye");
msg();
 	break;
 }



case ' ': {break;}
/* unrecognized command */
		 } /* end of switch */
	 } /* end of while */

/* -------- alternately, watch for message from CirclePack --- */

	if (scanf("%s",buf) && strcmp(buf,"die")==0)
	 {
		msg("DP: died, as directed.\n");
	 }
	exit(7);

} /* end of main */

int get_next_cmd()
/* fill nextcmd from parent (or user) */
{
	nextcmd[0]='\0';
	if (cmdmode==0) /* from parent */
	 {
		sprintf(msgbuf,"Send next command:");
		msg();
		while (!strlen(nextcmd))
			scanf(" %[^\n]BUFSIZE",nextcmd); /* wait for input */  
		return 1;
	 }
	printf("Enter next command (or ?):");
	if (fgets(nextcmd,BUFSIZE,stdin))
		  return 1;
	else return 0;
} /* get_next_cmd */

int get_parent_data(char *datastr)
/* read incoming data requested from parent */
{
	datastr[0]='\0';
	if (cmdmode==0) /* from parent */
		scanf(" %[^\n]BUFSIZE",datastr);
	else /* from user */
		fgets(datastr,BUFSIZE,stdin);
	return 1;
} /* get_parent_data */


/* ---------------------

new routines for manipulating complexes, particularly with dessins in mind

------------------------- */


int find_sidelength(struct p_data *p,int v)
/* assume dessin-type packing, v = dessin vert.
Find length of dessin-face edges. Look for shortest 6..4..6
degree vert pattern (there could be accidental longer ones). 
Just check 0, 1 flower directions. Return 0 on
failure (means not a dessin-type packing). */
{
	int next,last,sidelength,length,ahead;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
/* check index 0 direction */
	last=v;
	next=pK_ptr[v].flower[0];
	length=1;
	while ( pK_ptr[next].num==6 
		&& (ahead=go_around(p,last,next,3,-1)) )
	 {
		length++;
		last=next;
		next=ahead;
	 }
	if (pK_ptr[next].num!=4) return 0; /* something wrong */
	sidelength = 2*length;
/* compare index 1 direction */
	last=v;
	next=pK_ptr[v].flower[1];
	length=1;
	while ( pK_ptr[next].num==6 
		&& (ahead=go_around(p,last,next,3,-1)) )
	 {
		length++;
		last=next;
		next=ahead;
	 }
	if (2*length < sidelength) return (2*length);
	else return sidelength;
} /* find_sidelength */

int process_edge(struct p_data *p,int v,int w,int ind,int sidelength,
    struct Edgelist **edgelist,struct Vertlist **facelist,int want,
    int *outdex) 
/* Go along dessin-type tri side from v in dir ind. If end not w, 
return 0. Else get data; outdex is index of last edge wrt w.
want: 1=faces, 2=edges, 3=both, 0=neither.
Lists are new, will be set NULL on error or if not requested. */
{
	int k,last,next,ahead,count=1,flag=0,length;
	struct Edgelist *etrace;
	struct Vertlist *trace;
	struct K_data *pK_ptr;

	if (!face_org) /* shouldn't happen */
	  {
	    *edgelist=NULL;
	    *facelist=NULL;
	    return 0;
	  }
	pK_ptr=p->packK_ptr;
	*edgelist=NULL;
	*facelist=NULL;
	last=v;
	next=pK_ptr[v].flower[ind];
	etrace=(struct Edgelist *)malloc(sizeof(struct Edgelist));
	etrace->next=NULL;
	etrace->v=last;
	etrace->w=next;
	*edgelist=etrace;
	trace=(struct Vertlist *)malloc(sizeof(struct Vertlist));
	trace->next=NULL;
	trace->v=nghb_face(p,last,next);
	*facelist=trace;
	length=1;
	while ( pK_ptr[next].num==6 
		&& (ahead=go_around(p,last,next,3,-1)) )
	 {
		length++;
		last=next;
		next=ahead;
			/* record edge */
		etrace=etrace->next=
		   (struct Edgelist *)malloc(sizeof(struct Edgelist));
		etrace->next=NULL;
		etrace->v=last;
		etrace->w=next;
			/* record two faces */
		for (k=1;k<=2;k++)
		 {
		   trace->next=
			(struct Vertlist *)malloc(sizeof(struct Vertlist));
		   trace->next->next=NULL;
		   trace->next->v=next_face(p,last,trace->v,-1);
		   trace=trace->next;
		 }
	 }
	if (sidelength==2*length && pK_ptr[next].num==4
		&& (ahead=go_around(p,last,next,2,-1)) )
	 {
		flag=1;
		last=next;
		next=ahead;
			/* record edge */
		etrace=etrace->next=
		   (struct Edgelist *)malloc(sizeof(struct Edgelist));
		etrace->next=NULL;
		etrace->v=last;
		etrace->w=next;
			/* record one face */
		trace->next=
		   (struct Vertlist *)malloc(sizeof(struct Vertlist));
		trace->next->next=NULL;
		trace->next->v=next_face(p,last,trace->v,-1);
		trace=trace->next;
		while ( count<length && pK_ptr[next].num==6
		   && (ahead=go_around(p,last,next,3,-1)) )
		 {
		   count++;
		   last=next;
		   next=ahead;
			/* record edge */
		   etrace=etrace->next=
		      (struct Edgelist *)malloc(sizeof(struct Edgelist));
		   etrace->next=NULL;
		   etrace->v=last;
		   etrace->w=next;
			/* record two faces */
		   for (k=1;k<=2;k++)
		    {
			trace->next=
			 (struct Vertlist *)malloc(sizeof(struct Vertlist));
			trace->next->next=NULL;
			trace->next->v=next_face(p,last,trace->v,-1);
			trace=trace->next;
		    }
		 }
	 }
	if (!flag || next!=w) /* not an edge */
	 {
		edge_free(edgelist);
		vert_free(facelist);
		return 0;
	 }
	if (want==1 || want==0) edge_free(edgelist);
	if (want==2 || want==0) vert_free(facelist); 
	*outdex=nghb(p,w,last);
	return 1;
} /* process_edge */

int process_face(struct p_data *p,int v,int w,int u,int ind,
    int sidelength,struct Edgelist **edgelist,
    struct Vertlist **facelist,int want) 
/* If <v,w,u> is dessin-type face, create desired lists.
(if ind>=0, try that index from v first.)
Return 0 on error, lists NULL. want: 1=faces, 2=edges,
3=both, 0=neither. */
{
	int j,J,outdex,flag=-1,dir;
	struct K_data *pK_ptr;
	struct Edgelist *etrace,*etmp;
	struct Vertlist *trace,*tmp;

	pK_ptr=p->packK_ptr;
	*edgelist=NULL;
	*facelist=NULL;
	pK_ptr=p->packK_ptr;
	for (j=0;j<pK_ptr[v].num+pK_ptr[v].bdry_flag && flag==-1;j++)
		/* don't know direction from v */
	 {
		if (ind>=0) J=(j+ind)%pK_ptr[v].num;
			/* ind might give hint of direction to go */
		else J=j;
		if (ck_edge(p,v,w,J,sidelength)
			&& process_edge(p,v,w,J,sidelength,
			&etrace,&trace,want,&outdex) )
		 {
			*edgelist=etmp=etrace;
			while(etmp!=NULL && etmp->next!=NULL) 
				etmp=etmp->next;
			*facelist=tmp=trace;
			while(tmp!=NULL && tmp->next!=NULL) 
				tmp=tmp->next;
			flag=1;
		 }
		if (flag==1)
		 {
			dir=(outdex+pK_ptr[w].num-2)%pK_ptr[w].num;
			if (ck_edge(p,w,u,dir,sidelength)
			   && process_edge(p,w,u,dir,sidelength,
				&etrace,&trace,want,&outdex) )
			 {
				if (etmp!=NULL && etrace!=NULL)
					etmp->next=etrace;
				while(etmp!=NULL && etmp->next!=NULL) 
					etmp=etmp->next;
				if (tmp!=NULL && trace!=NULL)
					tmp->next=trace;
				while(tmp!=NULL && tmp->next!=NULL)
					tmp=tmp->next;
				flag=2;
			 }
			if (flag==2)
			 {
				dir=(outdex+pK_ptr[u].num-2)%pK_ptr[u].num;
				if (ck_edge(p,u,v,dir,sidelength)
				   && process_edge(p,u,v,dir,sidelength,
					&etrace,&trace,want,&outdex) )
				 {
				      if (etmp!=NULL && etrace!=NULL)
					etmp->next=etrace;
				      while(etmp!=NULL && etmp->next!=NULL) 
					etmp=etmp->next;
				      if (tmp!=NULL && trace!=NULL)
					tmp->next=trace;
				      while(tmp!=NULL && tmp->next!=NULL)
					tmp=tmp->next;
				      if (*facelist!=NULL && tmp!=NULL)
					 /* close up facelist */
					{
					 tmp=tmp->next=(struct Vertlist *)
					   malloc(sizeof(struct Vertlist));
					 tmp->v=(*facelist)->v;
					 tmp->next=NULL;
					}  
				      flag=3;
				 }
			 }
		 }
		if (flag!=3) /* failure */
		 {
			edge_free(edgelist);
			vert_free(facelist);
			flag=-1;
		 }
	 } /* end of loop through neighbors */
	if (flag!=3) return 0;
	return 1;
} /* process_face */	
	
int process_edge_chain(struct p_data *p,struct Vertlist *v_list,
    int sidelength,struct Edgelist **edgelist,
    struct Vertlist **facelist,int want) 
/* From list of vertices of dessin-type packing, build facelist 
(on left-hand side of edges) and/or edgelist. 
want: 1=faces,2=edges,3=both,0=neither. 
Return 0 on error and set lists to NULL; don't change v_list. */
{
	int j,v,w,ind,outdex,k,n,num,flag=1;
	int f,first_vert,last_outdex=-1,first_ind;
	struct Edgelist *etrace,*etmp;
	struct Vertlist *trace,*tmp,*vtmp;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
	*edgelist=etmp=NULL;
	*facelist=tmp=NULL;
	if (!(vtmp=v_list)) return 0;
	else first_vert=vtmp->v; /* keep for possible later closing */
	while (flag && vtmp && vtmp->next) /* while successful */
	 {
		flag=0;
		v=vtmp->v;
		w=vtmp->next->v;
		if (v==w) flag=1; /* skip redundancy */
		else
		 {
			/* find first good index */ 
		   for (j=0;j<pK_ptr[v].num && !flag;j++)
		    {
		     ind=(last_outdex+1+j)%pK_ptr[v].num;
		     if (ck_edge(p,v,w,ind,sidelength)
			 && process_edge(p,v,w,ind,sidelength,
			    &etrace,&trace,want,&outdex) )
		      {
			if (vtmp==v_list) /* just starting */
				first_ind=ind;
			if (etrace) 
			 {
			   if (etmp) etmp->next=etrace;
			   else *edgelist=etmp=etrace; /* start edgelist*/
			   while (etmp && etmp->next)
				etmp=etmp->next;
			 }
 			if (trace && tmp && last_outdex>=0) 
				/* need connecting faces*/
			 {
				num=pK_ptr[v].num;
				k=(last_outdex)%num;
				n=(ind+num+1)%num;
				if (k<n) k=k+num;
				while ((k=(k-1)) > n)
				 {
				   if ((f=nghb_face(p,
					pK_ptr[v].flower[k%num],v)))
				    {
				      tmp=tmp->next=
					(struct Vertlist *)
				      malloc(sizeof(struct Vertlist));
				     tmp->next=NULL;
				     tmp->v=f;
				    }
				 }
				tmp->next=trace;
				while (tmp && tmp->next)
					tmp=tmp->next;
			 }


			if (trace && !tmp) /* start facelist */
			 {
			   *facelist=tmp=trace;
			   while (tmp && tmp->next)
				tmp=tmp->next;
			 }
			last_outdex=outdex;
			flag=1;
		      }
		    } /* end of j loop */
		 } /* end of else */
		vtmp=vtmp->next;
	 } /* end of while */
	if (flag && vtmp && vtmp->v==first_vert 
	   && tmp && last_outdex>=0)
		/* v_list is closed, patch last face to first */
	 {
		num=pK_ptr[first_vert].num;
		k=(last_outdex)%num;
		n=(first_ind+num+1)%num;
		if (k<n) k=k+num;
		while ((k=(k-1)) >= n) /* include repeat of first face */
		 {
		   if ((f=nghb_face(p,
			pK_ptr[first_vert].flower[k%num],first_vert)))
		    {
		     tmp=tmp->next=
			(struct Vertlist *)
			 malloc(sizeof(struct Vertlist));
		     tmp->next=NULL;
		     tmp->v=f;
		    }
		 }
	 }
	if (!flag) /* problems, chain breaking up */
	 {
		edge_free(edgelist);
		vert_free(facelist);
		return 0;
	 }
	return 1;
} /* process_edge_chain */

int place_cross_edge_face_no(struct p_data *p,int f,int v,int sidelength,
    int *nghbf,int *midcirc)
/* given face f, vert v, find face index of face next to f in 
pos. edge from v. Identify middle circle. 0 on error. */
{
	int last,next,ahead,dir,ind,length;
	des_face *dfptr;
	struct K_data *pK_ptr;

	if (f<1 || f>dessin->facecount) return 0;
	dfptr=dessin->d_faces+f;
	if (v==dfptr->verts[0]) ind=0;
	else if (v==dfptr->verts[1]) ind=1;
	else if (v==dfptr->verts[2]) ind=2;
	else return 0; 
	pK_ptr=p->packK_ptr;	
	if (dfptr->orient>0) *nghbf=dfptr->nghb[ind];
	else *nghbf=dfptr->nghb[(ind+2)%3];
	dir=dfptr->midices[ind];
	if (dir==0) dir=pK_ptr[v].num-1;
	else dir=dir-1;
		/* find middle of this edge */
	last=v;
	next=pK_ptr[v].flower[dir];
	length=1;
	while ( pK_ptr[next].num==6 && ((2*length) < sidelength)
		&& (ahead=go_around(p,last,next,3,-1)) )
	 {
		length++;
		last=next;
		next=ahead;
	 }
	if ((2*length)<sidelength) return 0; /* some error in edge */
		/* get latest data from parent */
	*midcirc=next;
	return 1;
} /* place_cross_edge_face_no */
			
int go_around(struct p_data *p,int v,int w,int n,int lrflag)
/* From v to w, pass around n faces to left/right depending 
on lrflag=-/+. return vert or 0 on error. ?? Assume complex w/o bdry */
{
	int ind,m;

	if ((ind=nghb(p,w,v)) < 0 ) return 0;
	if (lrflag<=0) /* go left around w */
	 {
	   if (p->packK_ptr[w].bdry_flag && n>ind) return 0;
	   m=(p->packK_ptr[w].num+ind-n) % p->packK_ptr[w].num;
	   return (p->packK_ptr[w].flower[m]);
	 }
	else
	 {
	   if (p->packK_ptr[w].bdry_flag && (ind+n)>p->packK_ptr[w].num)
		return 0;
	   m=(p->packK_ptr[w].num+ind-n) % p->packK_ptr[w].num;
	   return (p->packK_ptr[w].flower[m]);
	 }
} /* go_around */

int which_face(int v,int w,int u)
/* return number of dessin face with v,w,u. return 0 if 
it doesn't exist */
{
	int f,j,spot;
	des_face *dfptr;

	dfptr=dessin->d_faces;
	for (f=1;f<=dessin->facecount;f++)
	 {
		spot=0;
		for (j=0;j<3;j++)
			if (dfptr[f].verts[j]==v) spot++;
		if (spot)
		 {
			spot=0;
			for (j=0;j<3;j++)
				if (dfptr[f].verts[j]==w) spot++;
			if (spot)
			 {
				spot=0;
				for (j=0;j<3;j++)
					if (dfptr[f].verts[j]==w) spot++;
				if (spot) return f;
			 }
		 }
	 }
	return 0;
} /* which_face */

struct Vertlist * ck_in_list(struct Vertlist *redlist,int v,int w)
/* check list for v,w pair, return pointer to v */
{
	struct Vertlist *ftrace;

	if ((ftrace=redlist)==NULL) return NULL;
	while(ftrace && ftrace->next)
	 {
		if (ftrace->v==v && ftrace->next->v==w)
		return ftrace;
		ftrace=ftrace->next;
	 }
	return NULL;
} /* ck_in_list */

int ck_edge(struct p_data *p,int v,int w,int ind,int sidelength)
/* 1 if w is along edge index ind from v. */
{
	int count=0,flag=0,last,next,ahead,length=1;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
	if (ind<0 || v<1 || v>p->nodecount 
		|| w<1 || w>p->nodecount
		|| ind>pK_ptr[v].num) return 0;
	last=v;
	next=pK_ptr[v].flower[ind];
	while ( pK_ptr[next].num==6 
		&& (ahead=go_around(p,last,next,3,-1)) )
	 {
		length++;
		last=next;
		next=ahead;
	 }
	if (sidelength==2*length && pK_ptr[next].num==4
		&& (ahead=go_around(p,last,next,2,-1)) )
	 {
		flag=1;
		last=next;
		next=ahead;
		while ( count<length && pK_ptr[next].num==6
		   && (ahead=go_around(p,last,next,3,-1)) )
		 {
		   count++;
		   last=next;
		   next=ahead;
		 }
	 }
	if (!flag || next!=w) return 0;
	return 1;
} /* ck_edge */

int check_list(struct Vertlist *list,int v)
/* is v in list? */
{
	struct Vertlist *trace;

	if (!(trace=list)) return 0;
	while (trace)	
	 {
		if (trace->v==v) return 1;
		trace=trace->next;
	 }
	return 0;
} /* check_list */

int next_face(struct p_data *p,int v,int f,int lrflag)
/* return face at v next to f, lrflag=+/- means left/right. */
{
	int n,num;

	n=1;
	num=p->packK_ptr[v].num;
	while ( n<=num && (*(face_org[v]+n)!=f) ) n++;
	if (n>num 
	   || (n==num && lrflag>0 && p->packK_ptr[v].bdry_flag)
	   || (n==1 && lrflag<0 && p->packK_ptr[v].bdry_flag) ) 
		return 0;
	if (n==num && lrflag>0) return *(face_org[v]+1);
	if (n==1 && lrflag<0) return *(face_org[v]+num);
	if (lrflag>0) return *(face_org[v]+(n+1));
	else return *(face_org[v]+(n-1));
} /* next_face */

int nghb_face(struct p_data *p,int v,int w)
/* return face with ordered edge <v,w> */
{
	int ind;

	if ((ind=nghb(p,v,w))<0 
	   || (ind==p->packK_ptr[v].num && p->packK_ptr[v].bdry_flag) )
		return 0;
	return *(face_org[v]+(ind+1));
} /* nghb_face */

int send_edge_fv(FILE *fp,struct p_data *p,int f,int v,int q,
    int pflag)
/* send ctr-clw edge from v in face f, print if pflag. Assume 
vert centers already in place. return 0 on error. */
{
     int outdex,flag=0,shift,dir,a,b;
     struct Vertlist *facelist;
     struct Edgelist *edgelist,*etrace;
     des_face *dfptr;

     if (f<1 || f>dessin->facecount) return 0;
     dfptr=dessin->d_faces+f;
     if (v==dfptr->verts[0]) shift=0;
     else if (v==dfptr->verts[1]) shift=1;
     else if (v==dfptr->verts[2]) shift=2;
     else return 0;
	a=dfptr->verts[shift];
	if (dfptr->orient >0 )	b=dfptr->verts[(shift+1)%3];
	else b=dfptr->verts[(shift+3-1)%3];
	dir=dfptr->midices[shift];
	if (dir==0) dir=p->packK_ptr[a].num-1;
	else dir=dir-1;
     if (process_edge(p,a,b,dir,sidelength,
             &edgelist,&facelist,2,&outdex))
      {
          if (pflag) sprintf(msgbuf,"post -p%d -e ",q);
          else sprintf(msgbuf,"disp -p%d -e ",q);
          etrace=edgelist;
          while (etrace!=NULL)
           {
               sprintf(buf,"%d %d ",etrace->v,etrace->w);
               strcat(msgbuf,buf);
               etrace=etrace->next;
           }
          fprintf(fp,"%s\n",msgbuf);
          flag=1;
      }
     edge_free(&edgelist);
     return flag;
} /* send_edge_fv */     

int send_edge(struct p_data *p,int v,int w,int ind,int q,int pflag)
/* send edge display data to CirclePack. May be more than one edge 
between v, w. Use index ind if >= 0. pflag for print. */
{
	int k,outdex,flag=0;
	struct Vertlist *facelist;
	struct Edgelist *edgelist,*etrace;

	for (k=0;k<(p->packK_ptr[v].num+
		p->packK_ptr[v].bdry_flag);k++)
	 {
	   if ( ((ind>=0 && k==ind) || (ind<0))
		&& process_edge(p,v,w,k,sidelength,
		   &edgelist,&facelist,2,&outdex))
	    {
		if (pflag) sprintf(msgbuf,"post -p%d -e ",q);
		else sprintf(msgbuf,"disp -p%d -e ",q);
		etrace=edgelist;
		while (etrace!=NULL)
		 {
			sprintf(buf,"%d %d ",etrace->v,etrace->w);
			strcat(msgbuf,buf);
			etrace=etrace->next;
		 }
		fprintf(fp,"%s\n",msgbuf);
		sprintf(msgbuf,"Edge length p%d = %d; index from %d = %d.",
		   q,sidelength,v,k);
		msg();
		flag=1;
	    }
	   edge_free(&edgelist);
	 }
	return flag;
} /* send_edge */

int set_send_face(FILE *fp,struct p_data *p,int f,int q,int setflag,
    int cflag,int col,int show,int pflag) 
	/* tell CirclePack to layout and display a dessin face f. 
setflag=0 means don't tell CirclePack to recompute centers. 
cflag=1 for shaded. If show=0, just compute centers, don't 
draw. pflag means "post" rather than "disp" */
{
	int v,w,u,dum,ind,dir,shift=0;
	struct Vertlist *facelist,*ftrace;
	struct Edgelist *edgelist,*etrace;
	des_face *dfptr;

	dfptr=dessin->d_faces+f;
	if (p->genus==0 && (p->euler==1 || p->euler==2))
		setflag=0; 		/* simply connected */
	if (setflag && dessin_redlist) /* try to start with white vert */
	 {
		while ( shift<3 
		   && check_list(des_redverts,dfptr->verts[shift]) )
			shift++;
		if (shift==3) shift=0; /* didn't find white */
	 }		
	v=dfptr->verts[shift];
	w=dfptr->verts[(shift+1)%3];
	u=dfptr->verts[(shift+2)%3];
	ind=dfptr->midices[shift];
	if (dfptr->orient<0) /* switch first two if neg. orient */
	    {dum=w;w=u;u=dum;} 
	if (ind==0) dir=p->packK_ptr[v].num-1;
	else dir=ind-1;
	if (process_face(p,v,w,u,dir,sidelength,
		&edgelist,&facelist,3) )
	 {
		if (setflag)
		 {
			sprintf(msgbuf,"fix -p%d -r ",q);
			ftrace=facelist;
			while (ftrace)
			 {
			   sprintf(buf,"%d ",ftrace->v);
			   strcat(msgbuf,buf);
			   ftrace=ftrace->next;
			 }
			fprintf(fp,"%s\n",msgbuf);
		 }
		if (show && cflag) /* shade the face */
		 {
			if (col) /* only do fg and bg now */
			 {
			  if (pflag) sprintf(msgbuf,"post -p%d -sffg ",q);
			  else sprintf(msgbuf,"disp -p%d -sffg ",q);
			 }
			else
			 {
			  if (pflag) sprintf(msgbuf,"post -p%d -sfbg ",q);
			  else sprintf(msgbuf,"disp -p%d -sfbg ",q);
			 }
			etrace=edgelist;
			while (etrace)
			 {
			   sprintf(buf,"%d %d ",etrace->v,etrace->w);
			   strcat(msgbuf,buf);
			   etrace=etrace->next;
			 }
			fprintf(fp,"%s\n",msgbuf);
		 }
		else if (show && !cflag) /* just draw bdry */
		 {
			if (pflag) sprintf(msgbuf,"post -p%d -s ",q);
			else sprintf(msgbuf,"disp -p%d -s ",q);
			etrace=edgelist;
			while (etrace)
			 {
			   sprintf(buf,"%d %d ",etrace->v,etrace->w);
			   strcat(msgbuf,buf);
			   etrace=etrace->next;
			 }
			fprintf(fp,"%s\n",msgbuf);
		 }
		edge_free(&edgelist);
		vert_free(&facelist);
		return 1;
	 }
	return 0;
} /* set_send_face */

/* ------------------- new way to keep face neighbor relations ----- */

int genus_bdry(Dessin_data *d_data,int *g,int *e,int *n,int *geom) 
     /* compute genus, euler char, and no. bdrys,
orientation of dessin. o=1 means oriented, o=0, unoriented. geom for the
curvature, default eucl. Return 0 for any failure, including nonoriented.
This is cursory check; e.g., doesn't check connected, etc. Assume compact. */
{
	int fcount,ecount,vcount,bcount,k,i,f,hit,v;
	int *verts;
	des_face *fptr;

	fcount=d_data->facecount;
	fptr=d_data->d_faces;
	verts=(int *)malloc(3*(fcount+1)*sizeof(int));
	for (f=1;f<=fcount;f++) verts[f]=0;

   /* count no. verts */

	verts[1]=fptr[1].verts[0];
	vcount=1;
	for (f=1;f<=fcount;f++) for (i=0;i<3;i++)
	 {
		v=fptr[f].verts[i];
		hit=0;
		for (k=1;k<=vcount;k++) if (v==verts[k]) hit=1;
		if (!hit)
		 {vcount++;verts[vcount]=v; }
	 }

   /* count edges: ?? need to assume compact and consistent for now ?? */

	ecount=(fcount*3)/2;
	if (2*ecount!=(fcount*3)) 
	 {
		free(verts);
		return 0; /* inconsistency */
	 }


   /* count bdry components: ?? need to assume there are none for now ?? */

	bcount=0;

   /* now to compute the results */

	free(verts);
	*n=bcount;
	*e=fcount-ecount+vcount;
	*g=(2-*e-*n)/2;
	if ((2-*e-*n)!=2*(*g)) return 0; /* genus must be integer */
	if (*g==0) *geom=1;		/* sphere */
	else if (*g==1) *geom=0;	/* eucl */
	else if (*g>1) *geom=-1; 	/* hyperbolic */
	return 1;
} /* genus_bdry */

int swap_faces(Dessin_data *d_data,int f1,int f2) 
/* interchange face numbers */
{
  int f,facecount,k;
  des_face hold;

  if (f1<1 || f2<1 || f1==f2 || f1>(facecount=d_data->facecount)
      || f2>facecount) return 0;
  hold=d_data->d_faces[f1];
  d_data->d_faces[f1]=d_data->d_faces[f2];
  d_data->d_faces[f2]=hold;
  for (f=1;f<=facecount;f++) for (k=0;k<3;k++)
    {
      if (d_data->d_faces[f].nghb[k]==f1) d_data->d_faces[f].nghb[k]=f2;
      else if (d_data->d_faces[f].nghb[k]==f2) d_data->d_faces[f].nghb[k]=f1;
    }
  return (number_verts(d_data));
} /* swap_faces */
	

int number_verts(Dessin_data *d_data)
/* (re)numbers the vertices of dessin */
{
	int f,vcount,flag,count,kk,next,safty,maxv,old,k;
	int *oldnew;
	des_face *fptr;

	fptr=d_data->d_faces;
	count=d_data->facecount;
	for (f=1;f<=count;f++)
		fptr[f].mark=fptr[f].verts[0]
		   =fptr[f].verts[1]=fptr[f].verts[2]=0;
/* keep track of new vert numbers */
	vcount=1;
/* work on 0-pts, then 1-pts, then infty-pts */
	for (kk=0;kk<3;kk++)
	 {
	   flag=1;
	   while (flag)	/* go as long as you find needed new number */
	    {
		flag=0;
		f=1;
		while (fptr[f].verts[kk] && f<count) f++;
		if (fptr[f].verts[kk]) break;
		fptr[f].verts[kk]=vcount;
		if (fptr[f].orient==1) next=fptr[f].nghb[kk];
		else next=fptr[f].nghb[(kk+2) % 3];
		safty=1;
		while (next!=f && safty<10000) /* fix up all faces around this vert */
		 {
			fptr[next].verts[kk]=vcount;
			if (fptr[next].orient==1) next=fptr[next].nghb[kk];
			else next=fptr[next].nghb[(kk+2) % 3];
			safty++;
		 }
		if (safty>=10000) return 0;
		vcount++;
		flag++;
	    }
	 }
/* set largest deg infty-pt to largest vertex number */
	vcount--;
	oldnew=(int *)malloc((vcount+1)*sizeof(int));
	for (k=1;k<=vcount;k++) oldnew[k]=0;
	for (f=1;f<=count;f++) oldnew[fptr[f].verts[2]]++;
	old=fptr[count].verts[2];
	maxv=oldnew[old];
	for (k=1;k<=vcount;k++) 
		if (oldnew[k]>maxv) 
		 {
			maxv=oldnew[k];
			old=k;
		 }
	free(oldnew);
	for (f=1;f<=count;f++)
	 {
		if (fptr[f].verts[2]==old) fptr[f].verts[2]=vcount;
		else if (fptr[f].verts[2]==vcount) fptr[f].verts[2]=old;
	 }
	return 1;
} /* number_verts */

int ck_smooth(Dessin_data *d_data)
/* check consistency of dessin, print info about dessin.
Return 0 if okay, error number otherwise. */
{
	int i,f,k,count[3],facecount,nf,vcount;
	int *vertid;
	des_face *fptr;

	if (d_data==NULL || (facecount=d_data->facecount)<1) return 1;
	fptr=d_data->d_faces;
	count[0]=count[1]=count[2]=0;
/* check that faces neighbors consistent, vertices consistent */
	for (f=1;f<=facecount;f++) for (k=0;k<3;k++)
	 {
		nf=fptr[f].nghb[k];
		if (fptr[nf].nghb[k]!=f && fptr[nf].nghb[k]!=0) return 2;
			/* discrepancy */
			/* should we ever want to allow bdry, 
				indicate with 0 index for that neighbor */
		if (fptr[nf].nghb && fptr[nf].verts[k]!=fptr[f].verts[k]) 
			return 3;
			/* vert indices don't agree */
	 }
/* count vertices */
	vertid=(int *)malloc(3*(facecount+1)*sizeof(int));
	for (i=1;i<=3*facecount;i++) vertid[i]=0;
	for (f=1;f<=facecount;f++) for (k=0;k<3;k++) 
	 {
		if (!vertid[fptr[f].verts[k]]) count[k]++;
		vertid[fptr[f].verts[k]]++;
	 }
	vcount=0;
	for (i=1;i<=3*facecount;i++) if (vertid[i]) vcount++;
	sprintf(msgbuf,"dessin has %d 0-verts, %d 1-verts, %d infty-verts and %d faces.",
		count[0],count[1],count[2],facecount);
	msg();
	free(vertid);
	return 0;
} /* ck_smooth */

int make_basic_pack(struct p_data *basepack)
/* puts basic hex "triangle" in packing p */
{
	int k;
	struct K_data *pK;

	alloc_pack_space(basepack,100,0);
	basepack->nodecount=7;
	basepack->intnode=basepack->status=1;
	basepack->hes=-1;
	basepack->alpha=7;
	basepack->beta=basepack->gamma=1;
	basepack->faces=NULL;
	basepack->overlap_status=0;
	pK=basepack->packK_ptr;
	pK[1].num=pK[2].num=pK[3].num=pK[3].num=pK[4].num
		=pK[5].num=pK[6].num=2;
	pK[7].num=6;
	for (k=1;k<=7;k++) 
	   pK[k].flower=(int *)calloc((size_t)(pK[k].num+1),sizeof(int));
	pK[1].flower[0]=4;pK[1].flower[1]=7;pK[1].flower[2]=6;
	pK[2].flower[0]=5;pK[2].flower[1]=7;pK[2].flower[2]=4;
	pK[3].flower[0]=6;pK[3].flower[1]=7;pK[3].flower[2]=5;
	pK[4].flower[0]=2;pK[4].flower[1]=7;pK[4].flower[2]=1;
	pK[5].flower[0]=3;pK[5].flower[1]=7;pK[5].flower[2]=2;
	pK[6].flower[0]=1;pK[6].flower[1]=7;pK[6].flower[2]=3;
	pK[7].flower[0]=1;pK[7].flower[1]=4;pK[7].flower[2]=2;
	  pK[7].flower[3]=5;pK[7].flower[4]=3;pK[7].flower[5]=6;
	  pK[7].flower[6]=1;
	pK[1].bdry_flag=pK[2].bdry_flag=pK[3].bdry_flag
		=pK[4].bdry_flag=pK[5].bdry_flag=pK[6].bdry_flag=1;
	pK[7].bdry_flag=0;
	for (k=1;k<=7;k++) basepack->packR_ptr[k].rad=.1;
	basepack->status=1;
	complex_count(basepack,0);
	return 1;
} /* make_basic_pack */
	  
int bd_call(struct p_data *p)
/* call to build the packing using available dessin data.
return 0 on problem. */
{
	int i,genus,euler,bdrycount,alp,derror;

	if (dessin==NULL) return 0;
	make_basic_pack(p); /* ?? currently defaults to basic pieces*/
	catalog_faces(p);
	if ( (derror=build_dessin_packing(p,
	   &PackData[1],&PackData[2],dessin))!=0
	   || !genus_bdry(dessin,&genus,&euler,&bdrycount,&(p->hes)) )
	 {
		sprintf(msgbuf,"bd returned with error code %d.",derror);
		emsg();
		p->nodecount=0;
		p->status=0;
		if (p->faces!=NULL)
		 {free(p->faces);p->faces=NULL;}
	 }
	else
	 {
		if (p->faces!=NULL) 
		 {free(p->faces);p->faces=NULL;}
		complex_count(p,1);
		catalog_faces(p);
		if ((sidelength=find_sidelength(p,1))==0)
		 {
			sprintf(msgbuf,
			  "DP: Caution, pack may not be 'dessin' type.");
			emsg();
		 }
		sprintf(msgbuf,"dessin built: genus %d, %d bdry curves.",
			genus,bdrycount);
		msg();
		if ( genus!=0 
			&& (!dessin_redlist || !mark_red_verts(p,
			dessin,dessin_redlist,&des_redverts)) )
		 {
			vert_free(&des_redverts);
			vert_free(&dessin_redlist);
			sprintf(msgbuf,
			   "DP: note, dessin just built has no redchain.");
			msg();
		 }
	 }
		/* set aims, initial radii */
	for (i=1;i<=p->nodecount;i++)
	 {
		p->packR_ptr[i].aim=2*M_PI;
		p->packR_ptr[i].rad=.1;
	 }

		/* for normalization, 0_pt for alpha, infty_pt for beta,
			1_pt for gamma */
	alp=dessin->d_faces[1].verts[0];
	p->beta=dessin->d_faces[dessin->facecount].verts[2];
	p->gamma=dessin->d_faces[dessin->facecount].verts[1];
	i=2;
	while (check_list(des_redverts,alp) && i<=dessin->facecount) 
		/* try for non-red 0-vert for alpha. */
	 {
 		alp=dessin->d_faces[i].verts[0];	
		i++;
	 }
	if (i>dessin->facecount) /* arn't any? use mid-circle of face 1. */
		alp=dessin->facecount+1;
	p->alpha=alp;
	return 1;	
} /* bd_call */

int build_dessin_packing(struct p_data *p,struct p_data *q,
    struct p_data *r,Dessin_data *d_data)
     /* given dessin structure (assume correct), a packing is built 
with faces from packings q and r. Assume q and r simply connected, 
there are verts 1,2,3 positively about bdry of q, neg about bdry 
of r. Typically q=basepack, r=mirror image. Return 0 on success. */
{
	int i,j,k,kk,v,w,start,pastelength,f,nf;
	int mark,premark,length,facecount,dflag;
	struct p_data *pack_ptr=NULL;
	des_face *fptr=NULL;
	int edge_len[3],indx[2];
	int *oldnew=NULL;

	if (p==NULL || d_data==NULL) return 1;
	fptr=d_data->d_faces;
	facecount=d_data->facecount;

	if (q->nodecount>=7 && r->nodecount>=7) kk=7; 
		/* want sensible midverts */
	else kk= (q->nodecount>=r->nodecount) ? r->nodecount : q->nodecount;
	for (f=1;f<=facecount;f++) 
	 {
		fptr[f].mark=0;
			/* mark will indicate pasted status of each face:
			sum with 1 ==> 0,1 verts pasted, 2==> 1,2 pasted,
			and 4==> 2,0 pasted */ 
		fptr[f].midvert=kk;
	 }
		
/* first, empty p, copy q into p */ 

	carbon_copy(p,q);

/* get edge lengths for q and r */

	for (k=0;k<3;k++)
	 {
		length=1;
		v=k+1;w=q->packK_ptr[v].flower[0];
		while (w!=((v % 3)+1) && length<q->nodecount)
		 {
			length++;
			w=q->packK_ptr[w].flower[0];
		 }
		if (length==q->nodecount) return 4;
		edge_len[k]=length;
		length=1;
		v=k+1;w=r->packK_ptr[v].flower[r->packK_ptr[v].num];
		while (w!=((v % 3)+1) && length<r->nodecount)
		 {
			length++;
			w=r->packK_ptr[w].flower[r->packK_ptr[w].num];
		 }
		if (length==r->nodecount) return 5;
		if (length!=edge_len[k]) return 6; /* match for pasting? */
	 }

/* register current labels for d_faces; update to keep new number of verts. */
	for (f=1;f<=facecount;f++)
		for (i=0;i<3;i++) fptr[f].new_vert[i]=i+1;

/* if dessin is sphere */
	if (facecount==2)
	 {
		if (!adjoin(p,r,1,1,-1,&oldnew))
			return 3;
		complex_count(p,0);
		catalog_faces(p);
		return 0;
	 }
	
/* now adjoin faces systematically */
	dflag=1;
	while (dflag) /* cycle thru faces while pastings continue */
	 {
	   dflag=0;
	   for (f=1;f<=facecount;f++)
	   while ((mark=fptr[f].mark)<7) /* still not done with face f */
	    {
			/* first, which is next edge to paste? what face? */
		if (mark==6 || mark==4 || mark==2 || (mark==0 && f==1))
			/* e.g., need paste verts[0],verts[1] */
		 {
			indx[0]=0;
			indx[1]=1;
			nf=fptr[f].nghb[0];
			premark=1;
		 }
		else if (mark==1 || mark==5)
		 {
			indx[0]=1;
			indx[1]=2;
			nf=fptr[f].nghb[1];
			premark=2;
		 }
		else if (mark==3)
		 {
			indx[0]=2;
			indx[1]=0;
			nf=fptr[f].nghb[2];
			premark=4;
		 }
		else break; /* f not contiguous to any attached face */
 
		pastelength=edge_len[indx[0]];
	/* check for extra pasting between f and nf */
		if (fptr[f].nghb[indx[1]]==nf) /* yes, next edge also */
		 {
			pastelength += edge_len[indx[1]];
			indx[1]=(indx[1]+1) % 3;
			if (premark==1) premark=3;
			else if (premark==2) premark=6;
			else if (premark==4) premark=5;
		 }
		else if (fptr[f].nghb[(indx[1]+1)%3]==nf) 
			/* yes, prev. edge also*/
		 {
			pastelength += edge_len[(indx[1]+1)%3];
			indx[0]=(indx[0]+2)%3;
			if (premark==1) premark=5;
			else if (premark==2) premark=3;
			else if (premark==4) premark=6;
		 }
	/* pasting depends on orient and on whether nf is already part of p */
		if (fptr[nf].mark==0) /* nf is virgin */
		 {
			if (fptr[f].orient==1) pack_ptr=r;
			else pack_ptr=q;
		 }
		else pack_ptr=p;
	/* recall!! paste clockwise on p */
		if (fptr[f].orient==1) start=indx[1];
		else start=indx[0];

/* ------------------- debugging -------------------------------- */
/* 
FILE *fbug1;
fbug1=fopen("bugname1","w");
fprintf(fbug1,"f=%d <<----------------------<< paste on <<--------------------<< nf=%d",
f,nf);
fprintf(fbug1,"\nf mark=%d;    nf mark=%d;     premark=%d;     nodecount=%d.\n",
fptr[f].mark,fptr[nf].mark,premark,p->nodecount);

fprintf(fbug1," verts: face f=%d:",f);
for (i=0;i<3;i++) fprintf(fbug1," %d ",fptr[f].new_vert[i]);
fprintf(fbug1," (");
for (i=0;i<3;i++) fprintf(fbug1,"%d",p->packK_ptr[fptr[f].new_vert[i]].bdry_flag);
fprintf(fbug1,") ");
fprintf(fbug1,"  face nf=%d:",nf);
for (i=0;i<3;i++) fprintf(fbug1," %d",fptr[nf].new_vert[i]);
fprintf(fbug1," (");
for (i=0;i<3;i++) fprintf(fbug1,"%d",p->packK_ptr[fptr[nf].new_vert[i]].bdry_flag);
fprintf(fbug1,")");

fprintf(fbug1,"\n Boundary count starting at %d is %d.",
fptr[f].new_vert[start],bdry_comp_count(p,fptr[f].new_vert[start]));
fprintf(fbug1,"\n\n");

fprintf(fbug1,"paste verts: ");
if (fptr[f].orient==1)
 {
	st=fptr[f].new_vert[start];
	for (i=0;i<=pastelength;i++) 
	 {
		fprintf(fbug1,"%d ",st);
		st=p->packK_ptr[st].flower[p->packK_ptr[st].num];
	 }
	fprintf(fbug1,"  against: ");
	st=fptr[nf].new_vert[start];
	for (i=0;i<=pastelength;i++) 
	 {
		fprintf(fbug1,"%d ",st);
		st=pack_ptr->packK_ptr[st].flower[0];
	 }
	fprintf(fbug1,"\n");
 }
else
 {
	st=fptr[nf].new_vert[start];
	for (i=0;i<=pastelength;i++) 
	 {
		fprintf(fbug1,"%d ",st);
		st=pack_ptr->packK_ptr[st].flower[pack_ptr->packK_ptr[st].num];
	 }
	fprintf(fbug1,"  against: ");
	st=fptr[f].new_vert[start];
	for (i=0;i<=pastelength;i++) 
	 {
		fprintf(fbug1,"%d ",st);
		st=p->packK_ptr[st].flower[0];
	 }
	fprintf(fbug1,"\n");
 }

fprintf(fbug1," Current faces of f=27: %d, %d, %d",fptr[27].new_vert[0],
	fptr[27].new_vert[1],fptr[27].new_vert[2]);
fprintf(fbug1, "  bdry_flags: (%d%d%d)\n",
p->packK_ptr[fptr[27].new_vert[0]].bdry_flag,
p->packK_ptr[fptr[27].new_vert[1]].bdry_flag,
p->packK_ptr[fptr[27].new_vert[2]].bdry_flag);

node=p->nodecount; */  /* pnv(p,d_data,f,nf) or bugwrite(p) */

/* --------------------- end debugging ------------------------------- */

		if (!adjoin(p,pack_ptr,fptr[f].new_vert[start],
		   fptr[nf].new_vert[start], pastelength,&oldnew))
			{free(oldnew); return 6;}

		complex_count(p,0);

	/* register changes in mark */
		dflag++;
		fptr[f].mark += premark;
		fptr[nf].mark += premark;
	/* register new vertex info: only for nf if p!=pack_ptr */
		if (p!=pack_ptr) /* only modify nf */
		 {
			fptr[nf].midvert=oldnew[fptr[nf].midvert];
			for (kk=0;kk<3;kk++) 
			   fptr[nf].new_vert[kk]=oldnew[fptr[nf].new_vert[kk]];
		 }
		else 
		 {
			for (i=1;i<=facecount;i++) 
			if (fptr[i].mark)
			{
				fptr[i].midvert=oldnew[fptr[i].midvert];
				for (kk=0;kk<3;kk++)
				   fptr[i].new_vert[kk]=
					oldnew[fptr[i].new_vert[kk]];
			 }
		 }
		free(oldnew);oldnew=NULL;

/* ------------------------- debugging ----------------------- */
/*
fprintf(fbug1,"Afterward:f mark=%d;  nf mark=%d;    nodecount=%d.\n",
fptr[f].mark,fptr[nf].mark,p->nodecount);

fprintf(fbug1," verts: face f=%d:",f);
for (i=0;i<3;i++) fprintf(fbug1," %d ",fptr[f].new_vert[i]);
fprintf(fbug1," (");
for (i=0;i<3;i++) fprintf(fbug1,"%d",p->packK_ptr[fptr[f].new_vert[i]].bdry_flag);
fprintf(fbug1,") ");
fprintf(fbug1,"  face nf=%d:",nf);
for (i=0;i<3;i++) fprintf(fbug1," %d",fptr[nf].new_vert[i]);
fprintf(fbug1," (");
for (i=0;i<3;i++) fprintf(fbug1,"%d",p->packK_ptr[fptr[nf].new_vert[i]].bdry_flag);
fprintf(fbug1,")");
fprintf(fbug1,"\n\n\n");
*/
/* ------------------------------------- end debugging ------------ */


	      } /* end of inner for and while loops */
	 } /* end of outer while loop */

/* Now must establish the desired d_vert no's of p from original dessin data. */
	for (v=1;v<=p->nodecount;v++) p->packK_ptr[v].util_flag=0;
	for (f=1;f<=facecount;f++)
	 {
		for (i=0;i<3;i++) /* desired d_verts */
		   p->packK_ptr[fptr[f].new_vert[i]].util_flag=fptr[f].verts[i];
		if (p->packK_ptr[fptr[f].midvert].util_flag==0 
			&& (facecount+f)<=p->nodecount) /* desired midverts */
		   p->packK_ptr[fptr[f].midvert].util_flag= facecount+f;
	 }
	do
	 {
		kk=0;
		for (v=1;v<=p->nodecount;v++)
		 {
		   if (p->packK_ptr[v].flower[0]==
			p->packK_ptr[v].flower[p->packK_ptr[v].num])
			p->packK_ptr[v].bdry_flag=0;
		   if ((j=p->packK_ptr[v].util_flag)!=0 && j!=v) 
			/* want to swap */
		    {
			swap_nodes(p,v,j);
			kk++;
		    }
		 }
	 } while (kk>0);
	for (f=1;f<=facecount;f++) for (j=0;j<3;j++)
	 {
		fptr[f].midvert=facecount+f;
		fptr[f].midices[j]=nghb(p,fptr[f].verts[j],fptr[f].midvert);
	 }
	sidelength=find_sidelength(p,1);
	free(oldnew);
	p->status=1;
	catalog_faces(p);
	return 0;
} /* build_dessin_packing */

int add_vertex(Dessin_data *d_data,int *f1,int *f2,int *nf1,int *nf2)
/* add a vert to free edge between faces f1, f2.
Faces in counterclockwise order, f1,nf1,nf2,f2. */
{
	int i, maxvert=0,facecount,new,f;
	des_face *fptr1,*fptr2,*nfptr1,*nfptr2,*tptr,*fptr;


	facecount=d_data->facecount;
	fptr=(des_face *)malloc((facecount+3)*sizeof(des_face));
	for (f=1;f<=facecount;f++)
		fptr[f]=d_data->d_faces[f];
	d_data->d_faces=fptr;
	fptr1=(fptr)+(*f1);
	fptr2=(fptr)+(*f2);
	nfptr1=fptr+facecount+1;
	nfptr2=fptr+facecount+2;
	if (fptr1->orient!=1) /* interchange */
	 {i=*f1;*f1=*f2;*f2=i;tptr=fptr1;fptr1=fptr2;fptr2=tptr;}
	if (fptr1->verts[2]!=fptr2->verts[2]
		|| fptr2->nghb[0]!=*f1
		|| fptr2->nghb[1]!=*f1)
		return 0; 

/* okay, go ahead */
	d_data->facecount += 2; 
/* find largest vert number */
	for (f=1;f<=facecount;f++) for (i=0;i<3;i++) 
		if (d_data->d_faces[f].verts[i]>maxvert) 
			maxvert=d_data->d_faces[f].verts[i];
	new=maxvert+1;
/* data for new faces */
	*nf1=facecount+1;
	*nf2=facecount+2;
	nfptr1->orient=-1;
	nfptr2->orient=1;
	nfptr1->verts[0]=nfptr2->verts[0]=new;
	nfptr1->verts[1]=nfptr2->verts[1]=fptr1->verts[1];
	nfptr1->verts[2]=nfptr2->verts[2]=fptr1->verts[2];
	nfptr1->nghb[0]=nfptr1->nghb[2]=*nf2;
	nfptr2->nghb[0]=nfptr2->nghb[2]=*nf1;
	nfptr1->nghb[1]=*f1;
	nfptr2->nghb[1]=*f2;
/* fix old faces */
	fptr1->nghb[1]=*nf1;
	fptr2->nghb[1]=*nf2;
	if (!number_verts(d_data) || ck_smooth(d_data)) return 0;
	return 1;
} /* add_vertex */

int add_free_edge(Dessin_data *d_data,int *f1,int *f2,int *nf1,int *nf2)
/* add a free edge between faces f1, f2.
Faces in counterclockwise order, f1,nf1,nf2,f2. */
{
	int i, maxvert=0,facecount,new,f;
	des_face *fptr1,*fptr2,*nfptr1,*nfptr2,*tptr,*fptr;
	struct Vertlist *rptr,*hold;

	facecount=d_data->facecount;
	fptr=(des_face *)malloc((facecount+3)*sizeof(des_face));
	for (f=1;f<=facecount;f++)
		fptr[f]=d_data->d_faces[f];
	d_data->d_faces=fptr;
	fptr1=(fptr)+(*f1);
	fptr2=(fptr)+(*f2);
	nfptr1=fptr+facecount+1;
	nfptr2=fptr+facecount+2;
	if (fptr1->orient!=1) /* interchange */
	    {i=*f1;*f1=*f2;*f2=i;tptr=fptr1;fptr1=fptr2;fptr2=tptr;}
	if (fptr1->verts[0]!=fptr2->verts[0]
		|| fptr1->nghb[2]!=*f2)
		return 0; 

/* okay, go ahead */
	d_data->facecount += 2; 
/* find largest vert number */
	for (f=1;f<=facecount;f++) for (i=0;i<3;i++) 
		if (d_data->d_faces[f].verts[i]>maxvert) 
			maxvert=d_data->d_faces[f].verts[i];
	new=maxvert+1;
/* data for new faces */
	*nf1=facecount+1;
	*nf2=facecount+2;
	nfptr1->orient=-1;
	nfptr2->orient=1;
	nfptr1->verts[0]=nfptr2->verts[0]=fptr1->verts[0];
	nfptr1->verts[1]=nfptr2->verts[1]=new;
	nfptr1->verts[2]=nfptr2->verts[2]=fptr1->verts[2];
	nfptr1->nghb[0]=nfptr1->nghb[1]=*nf2;
	nfptr2->nghb[0]=nfptr2->nghb[1]=*nf1;
	nfptr1->nghb[2]=*f1;
	nfptr2->nghb[2]=*f2;
/* fix old faces */
	fptr1->nghb[2]=*nf1;
	fptr2->nghb[2]=*nf2;
/* modify redchain */
	if ( dessin_redlist && (rptr=ck_in_list(dessin_redlist,*f2,*f1)) )
	 {
		hold=rptr->next;
		rptr->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next->next=hold;
		rptr->next->v=*nf2;
		rptr->next->next->v=*nf1;
	 }
	if (!number_verts(d_data) || ck_smooth(d_data)) return 0;
	return 1;
} /* add_free_edge */


int add_new_vert(Dessin_data *d_data,int *f1,int *f2,int *nf1,
    int *nf2,int *nf3,int *nf4)
     /* add new 0-pt on edge between faces f1 and f2 (which must 
share 0-pt and 1-pt). 4 new faces in counterclockwise order, 
f2,n2,n4,n3,n1,f1 . */
{
	int i, maxvert=0,facecount,new1,new2,alpha,beta,f,f3,f4;
	des_face *fptr1,*fptr2,*fptr3,*fptr4,*fptr;
	des_face *nfptr1,*nfptr2,*nfptr3,*nfptr4,*tptr;
	struct Vertlist *rptr,*hold;

	f3=d_data->d_faces[*f1].nghb[1];
	f4=d_data->d_faces[*f2].nghb[1];
	facecount=d_data->facecount;
	fptr=(des_face *)malloc((facecount+5)*sizeof(des_face));
	for (f=1;f<=facecount;f++)
		fptr[f]=d_data->d_faces[f];
	d_data->d_faces=fptr;
	fptr1=(fptr)+(*f1);
	fptr2=(fptr)+(*f2);
	nfptr1=fptr+facecount+1;
	nfptr2=fptr+facecount+2;
	nfptr3=fptr+facecount+3;
	nfptr4=fptr+facecount+4;

	if (fptr1->orient!=1) /* interchange */
	    {i=*f1;*f1=*f2;*f2=i;tptr=fptr1;fptr1=fptr2;fptr2=tptr;}
	if (fptr1->verts[0]!=fptr2->verts[0]
		|| fptr1->nghb[0]!=*f2)
		return 0; 

/* okay, go ahead */
	d_data->facecount += 4; 
/* find largest vert number */
	for (f=1;f<=facecount;f++) for (i=0;i<3;i++) 
		if (d_data->d_faces[f].verts[i]>maxvert) 
			maxvert=d_data->d_faces[f].verts[i];
	new1=maxvert+1;
	new2=maxvert+2;
	alpha=fptr1->nghb[1];
	beta=fptr2->nghb[1];
	fptr3=(fptr)+alpha;
	fptr4=(fptr)+beta;
/* data for new faces */
	*nf1=facecount+1;
	*nf2=facecount+2;
	*nf3=facecount+3;
	*nf4=facecount+4;
	nfptr1->orient=-1;
	nfptr2->orient=1;
	nfptr3->orient=1;
	nfptr4->orient=-1;
	nfptr1->verts[0]=nfptr2->verts[0]=
		nfptr3->verts[0]=nfptr4->verts[0]=new2;
	nfptr1->verts[1]=nfptr2->verts[1]=new1;
	nfptr3->verts[1]=nfptr4->verts[1]=fptr1->verts[1];
	nfptr1->verts[2]=nfptr3->verts[2]=fptr1->verts[2];
	nfptr2->verts[2]=nfptr4->verts[2]=fptr2->verts[2];
	
	nfptr1->nghb[0]=nfptr4->nghb[2]=*nf2;
	nfptr2->nghb[0]=nfptr3->nghb[2]=*nf1;
	nfptr1->nghb[2]=nfptr4->nghb[0]=*nf3;
	nfptr2->nghb[2]=nfptr3->nghb[0]=*nf4;
	nfptr1->nghb[1]=*f1;
	nfptr2->nghb[1]=*f2;
	nfptr3->nghb[1]=alpha;
	nfptr4->nghb[1]=beta;
/* fix old faces */
	fptr1->verts[1]=fptr2->verts[1]=new1;
	fptr1->nghb[1]=*nf1;
	fptr2->nghb[1]=*nf2;
	fptr3->nghb[1]=*nf3;
	fptr4->nghb[1]=*nf4;
/* modify redchain */
	if ( dessin_redlist && (rptr=ck_in_list(dessin_redlist,*f1,f3)) )
	 {
		hold=rptr->next;
		rptr->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next->next=hold;
		rptr->next->v=*nf1;
		rptr->next->next->v=*nf3;
	 }
	if ( dessin_redlist && (rptr=ck_in_list(dessin_redlist,f4,*f2)) )
	 {
		hold=rptr->next;
		rptr->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next->next=hold;
		rptr->next->v=*nf4;
		rptr->next->next->v=*nf2;
	 }
	if (!number_verts(d_data) || ck_smooth(d_data)) return 0;
	return 1;
} /* add_new_vert */

int add_loop(Dessin_data *d_data,int *f1,int *f2,int *nf1,int *nf2,
    int *nf3,int *nf4)
/* add loop to 0-pt, positive face f1, neg, f2 which must share 0-pt. 
4 new faces in counterclockwise order about the base 0-pt. 
f1,nf1,nf3,nf4,nf2,f2. */
{
	int i, maxvert=0,facecount,new1,new2,f;
	des_face *fptr1,*fptr2,*fptr;
	des_face *nfptr1,*nfptr2,*nfptr3,*nfptr4,*tptr;
	struct Vertlist *rptr,*hold;

	facecount=d_data->facecount;
	fptr=(des_face *)malloc((facecount+5)*sizeof(des_face));
	for (f=1;f<=facecount;f++)
		fptr[f]=d_data->d_faces[f];
	d_data->d_faces=fptr;
	fptr1=(fptr)+(*f1);
	fptr2=(fptr)+(*f2);
	nfptr1=fptr+facecount+1;
	nfptr2=fptr+facecount+2;
	nfptr3=fptr+facecount+3;
	nfptr4=fptr+facecount+4;

	if (fptr1->orient!=1) /* interchange */
	    {i=*f1;*f1=*f2;*f2=i;tptr=fptr1;fptr1=fptr2;fptr2=tptr;}
	if (fptr1->verts[0]!=fptr2->verts[0]
		|| fptr1->nghb[2]!=*f2)
		return 0; 

/* okay, go ahead */
	d_data->facecount += 4; 
/* find largest vert number */
	for (f=1;f<=facecount;f++) for (i=0;i<3;i++) 
		if (d_data->d_faces[f].verts[i]>maxvert) 
			maxvert=d_data->d_faces[f].verts[i];
	new1=maxvert+1;
	new2=maxvert+2;
/* data for new faces */
	*nf1=facecount+1;
	*nf2=facecount+2;
	*nf3=facecount+3;
	*nf4=facecount+4;
	nfptr1->orient=-1;
	nfptr2->orient=1;
	nfptr3->orient=1;
	nfptr4->orient=-1;
	nfptr1->verts[0]=nfptr2->verts[0]=
		nfptr3->verts[0]=nfptr4->verts[0]=fptr1->verts[0];
	nfptr1->verts[1]=nfptr2->verts[1]=
		nfptr3->verts[1]=nfptr4->verts[1]=new1;
	nfptr1->verts[2]=nfptr2->verts[2]=fptr1->verts[2];
	nfptr3->verts[2]=nfptr4->verts[2]=new2;
	nfptr1->verts[2]=nfptr3->verts[2]=new2;
	
	nfptr1->nghb[0]=nfptr4->nghb[2]=nfptr4->nghb[1]=*nf3;
	nfptr2->nghb[0]=nfptr3->nghb[2]=nfptr3->nghb[1]=*nf4;
	nfptr1->nghb[1]=nfptr4->nghb[0]=*nf2;
	nfptr2->nghb[1]=nfptr3->nghb[0]=*nf1;
	nfptr1->nghb[2]=*f1;
	nfptr2->nghb[2]=*f2;
/* fix old faces */
	fptr1->nghb[2]=*nf1;
	fptr2->nghb[2]=*nf2;
	if (!number_verts(d_data) || ck_smooth(d_data)) return 0;
/* adjust redchain */
	if ( dessin_redlist && (rptr=ck_in_list(dessin_redlist,*f2,*f1)) )
	 {
		hold=rptr->next;
		rptr->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next->next->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next->next->next->next=hold;
		rptr->next->v=*nf2;
		rptr->next->next->v=*nf4;
		rptr->next->next->next->v=*nf3;
		rptr->next->next->next->next->v =*nf1;
	 }
	return 1;
} /* add_loop */

int add_bridge_edge(Dessin_data *d_data,int *f1,int *f2,int *nf1,
    int *nf2,int *nf3,int *nf4)
/* add a new edge cutting a cell. Faces f1, f2 must share common 
infty-pt, but not 0-infty edge. New faces in counterclockwise order 
around new 1-vert: nf1,nf2,nf4,nf3 */
{
	int i, maxvert=0,facecount,new1,new2,f,f3,f4;
	des_face *fptr1,*fptr2,*fptr3,*fptr4,*fptr;
	des_face *nfptr1,*nfptr2,*nfptr3,*nfptr4,*tptr;
	struct Vertlist *rptr,*hold;

	f3=d_data->d_faces[*f1].nghb[2];
	f4=d_data->d_faces[*f2].nghb[2];
	facecount=d_data->facecount;
	fptr=(des_face *)malloc((facecount+5)*sizeof(des_face));
	for (f=1;f<=facecount;f++)
		fptr[f]=d_data->d_faces[f];
	d_data->d_faces=fptr;
	fptr1=(fptr)+(*f1);
	fptr2=(fptr)+(*f2);
	fptr3=(fptr)+fptr1->nghb[2];
	fptr4=(fptr)+fptr2->nghb[2];
	nfptr1=fptr+facecount+1;
	nfptr2=fptr+facecount+2;
	nfptr3=fptr+facecount+3;
	nfptr4=fptr+facecount+4;
	if (fptr1->orient!=1) /* interchange */
	    {i=*f1;*f1=*f2;*f2=i;tptr=fptr1;fptr1=fptr2;fptr2=tptr;}
	if (fptr1->verts[2]!=fptr2->verts[2]
		|| fptr1->nghb[2]==*f2)
		return 0; 

/* okay, go ahead */
	d_data->facecount += 4; 
/* find largest vert number */
	for (f=1;f<=facecount;f++) for (i=0;i<3;i++) 
		if (d_data->d_faces[f].verts[i]>maxvert) 
			maxvert=d_data->d_faces[f].verts[i];
	new1=maxvert+1;
	new2=maxvert+2;
/* data for new faces */
	*nf1=facecount+1;
	*nf2=facecount+2;
	*nf3=facecount+3;
	*nf4=facecount+4;
	nfptr1->orient=nfptr4->orient=-1;
	nfptr2->orient=nfptr3->orient=1;
	nfptr1->verts[0]=nfptr3->verts[0]=fptr1->verts[0];
	nfptr1->verts[1]=nfptr2->verts[1]=
		nfptr3->verts[1]=nfptr4->verts[1]=new1;
	nfptr1->verts[2]=nfptr2->verts[2]=fptr1->verts[2];
	nfptr3->verts[2]=nfptr4->verts[2]=new2;
	nfptr1->nghb[0]=nfptr4->nghb[1]=*nf3;
	nfptr1->nghb[1]=nfptr4->nghb[0]=*nf2;
	nfptr1->nghb[2]=*f1;
	nfptr2->nghb[0]=nfptr3->nghb[1]=*nf4;
	nfptr2->nghb[1]=nfptr3->nghb[0]=*nf1;
	nfptr2->nghb[2]=*f2;
	nfptr3->nghb[2]=fptr1->nghb[2];
	nfptr4->nghb[2]=fptr2->nghb[2];
/* fix old faces */
	fptr1->nghb[2]=*nf1;
	fptr2->nghb[2]=*nf2;
	fptr3->verts[2]=fptr4->verts[2]=new2;
	fptr3->nghb[2]=*nf3;
	fptr4->nghb[2]=*nf4;
	if (!number_verts(d_data) || ck_smooth(d_data)) return 0;
/* modify redchain */
	if ( dessin_redlist && (rptr=ck_in_list(dessin_redlist,f3,*f1)) )
	 {
		hold=rptr->next;
		rptr->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next->next=hold;
		rptr->next->v=*nf3;
		rptr->next->next->v=*nf1;
	 }
	if ( dessin_redlist && (rptr=ck_in_list(dessin_redlist,*f2,f4)) )
	 {
		hold=rptr->next;
		rptr->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		rptr->next->next->next=hold;
		rptr->next->v=*nf2;
		rptr->next->next->v=*nf4;
	 }
	return 1;
} /* add_bridge_edge */

int mark_red_verts(struct p_data *p,Dessin_data *d_data,
    struct Vertlist *redlist,struct Vertlist **redverts)
/* make list of red dessin verts. These will be verts on 'right' 
side of red face chain. Not a lot of checking done here; easily 
might screw up. 0 on error. */
{
	int face,lastface,nextface,flag,v,w,flip,j;
	int n1,n2,f0,f1,f2;
	struct Vertlist *ftrace=NULL,*ftmp,*vtrace=NULL,*vtmp,*hold;
	des_face *dfptr;

	if ( !redlist || !(redlist->next) ||!(redlist->next->next)
	   || !p || !p->status ) 
		return 0; /* not enough of a list */
	if (*redverts) vert_free(redverts);
	face=redlist->v;
	ftmp=redlist;
	while (ftmp && ftmp->next->next) ftmp=ftmp->next;
	lastface=ftmp->v; /* one just before end of list */
	if (ftmp->next->v!=face) return 0; /* not closed */
	ftrace=redlist;
	while (ftrace->next && (nextface=ftrace->next->v)==face)
	 {
		hold=ftrace->next;
		ftrace->next=ftrace->next->next;
		free(hold);
	 }
	if (nextface==face) return 0; /* didn't find distinct next face */
	ftrace=redlist;
/* go around the chain, create list of vert numbers */
	while (ftrace->next) 
	 {
		v=w=0;
		dfptr=d_data->d_faces+face;
		flip=dfptr->orient;
		flag=0;
		if ((face-nextface)*(face-lastface)*
			(lastface-nextface)) /* distinct */
		 {
		   for (j=0;j<3 && !flag;j++)
			if ((f0=dfptr->nghb[j])==lastface
			   && (f1=dfptr->nghb[n1=(j+3+flip)%3])!=lastface)
		 	 {
				n2=(j+3+flip+flip)%3;
				f2=dfptr->nghb[n2];
				if (f1==nextface)
				 {
					flag=1;
					if (flip>0) v=dfptr->verts[n1];
					else v=dfptr->verts[j];
					w=0;
				 }
				else if (f2==nextface)
				 {
					flag=1;
					if (flip>0)
					 {
					   v=dfptr->verts[n1];
					   w=dfptr->verts[n2];
					 }
					else
					 {
					   v=dfptr->verts[j];
					   w=dfptr->verts[n1];
					 }
				 }
				else /* error */
				 {
					vert_free(redverts);
					return 0; 
				 }
			 }
		 }
		else /* nextface and lastface must be same */
		 {
		   for (j=0;j<3 && !flag;j++)
			if ((f0=dfptr->nghb[j])==lastface
			   && (f2=dfptr->nghb[n2=(j+3+flip+flip)%3])==lastface
			   && (f1=dfptr->nghb[n1=(j+3+flip)%3])==nextface)
			 {
				flag=1;
				v=dfptr->nghb[j];
				w=dfptr->nghb[n1];
			 }
		 }
		if (!flag) /* error */
		 {
			vert_free(redverts);
			return 0; 
		 }
		if (flag && v!=0 && (!vtrace || v!=vtrace->v))
		 {
			vtmp=(struct Vertlist *)
				malloc(sizeof(struct Vertlist));
			vtmp->v=v;
			vtmp->next=NULL;
			if (!(*redverts)) *redverts=vtrace=vtmp;
			else vtrace=vtrace->next=vtmp;
	   	 }
		if (flag && w!=0 && (!vtrace || w!=vtrace->v))
		 {
			vtmp=(struct Vertlist *)
				malloc(sizeof(struct Vertlist));
			vtmp->v=w;
			vtmp->next=NULL;
			if (!(*redverts)) *redverts=vtrace=vtmp;
			else vtrace=vtrace->next=vtmp;
	   	 }
	   	ftrace=ftrace->next;
	   	lastface=face;
	   	face=ftrace->v;
		nextface=0;
	   	while (ftrace->next && (nextface=ftrace->next->v)==face)
	   	 {
			hold=ftrace->next;
			ftrace->next=ftrace->next->next;
			free(hold);
	   	 }
		if (nextface==face)  /* didn't find distinct next face */	
		 {
			vert_free(redverts);
			return 0;
		 }
	 } /* end of while */
	if (vtrace && vtrace->v!=(*redverts)->v) 
		/* must close up vertlist */
	 {
		vtmp=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		vtmp->next=NULL;
		vtmp->v=(*redverts)->v;
	 }
	return 1;
} /* mark_red_verts */

int send_red_chain(FILE *fp,struct p_data *p,Dessin_data *d_data,
    struct Vertlist *redlist,struct Vertlist **redverts,int pnum)
/* convert red chain given for this dessin (from file of dessin data) 
into red chain of actual pack faces, send to CirclePack. */
{
	struct Edgelist *edgelist;
	struct Vertlist *facelist,*ftrace;

	if (p->genus==0 && (p->euler==1 || p->euler==2))
		return 0; /* simply connected */
	if (d_data->facecount<=0 || !redverts)
	 {
		sprintf(msgbuf,"DP: there is no redchain data.");
		emsg();
		return 0;
	 }
	if (process_edge_chain(p,*redverts,sidelength,
		&edgelist,&facelist,1))
	 {
		sprintf(msgbuf,"fix -p%d -f ",pnum);
		ftrace=facelist;
		while (ftrace)
		 {
			sprintf(buf,"%d ",ftrace->v);
			strcat(msgbuf,buf);
			ftrace=ftrace->next;
		 }
		fprintf(fp,"%s\n",msgbuf);
	 }
	else
	 {
	   sprintf(msgbuf,
	   "DP: Dessin redchain failed to give packing redchain for p%d.",
		pnum);
	   emsg();
	 }
	vert_free(&facelist); 
	return 1;
} /* send_red_chain */

int read_dessin_data(struct p_data *p,Dessin_data *d_data)
/* read in dessin data. return 0 on error. */
{
	int total=0,eof=0,i,j,v,ok=0,dum;
	struct Vertlist *ftrace=NULL,*ftmp;

  /* first, read dessin face info */
	do
	 {
		eof=fscanf(fp,"%s",next);
	 }
	while (eof!=EOF && strcmp(next,"DESSIN_FACECOUNT:")!=0
	   && strcmp(next,"END")!=0);
	if (eof==EOF || strcmp(next,"END")==0 
	   || !fscanf(fp,"%d",&total) )
		return 0;
	if (d_data->facecount) /* throw out old */
	 {
		free(d_data->d_faces);
		d_data->d_faces=NULL;
		d_data->facecount=0;
		vert_free(&dessin_redlist);
		vert_free(&des_redverts);
	 }
	d_data->facecount=total;
	d_data->d_faces=
	   (des_face *)malloc((d_data->facecount+1)*sizeof(des_face));
	for (i=1;i<=d_data->facecount;i++)
	if (fscanf(fp,"%d %d %d %d %d",&dum,&(d_data->d_faces[i].orient),
	   &(d_data->d_faces[i].nghb[0]),&(d_data->d_faces[i].nghb[1]),
	   &(d_data->d_faces[i].nghb[2]))!=5)
	 {
		sprintf(msgbuf,"DP: Problem with d_data.");
		emsg();
		free(d_data->d_faces);
		d_data->d_faces=NULL;
		d_data->facecount=0;
		return 0;
	 }
  /* next, look for vert nos.; otherwise, create preferred numbering. */
	do
	 {
		eof=fscanf(fp,"%s",next);
	 }
	while (eof!=EOF && strcmp(next,"DESSIN_FACE_VERTS:")!=0
	   && strcmp(next,"DESSIN_RED_FACE_LIST:")!=0
	   && strcmp(next,"END")!=0);
	if (eof==EOF || strncmp(next,"END",3)==0 
	   || strcmp(next,"DESSIN_RED_FACE_LIST:")==0)
	 {
		number_verts(d_data);
		ok=ck_smooth(d_data); /* consistent ? */
	 }
	if (strcmp(next,"DESSIN_FACE_VERTS:")==0)
	   for (i=1;i<=d_data->facecount;i++)
	    if (fscanf(fp,"%d %d %d %d",
		&dum,&(d_data->d_faces[i].verts[0]),
		&(d_data->d_faces[i].verts[1]),
		&(d_data->d_faces[i].verts[2]))!=4)
	    {
		sprintf(msgbuf,"DP: File has faulty vertex data.");
		emsg();
		free(d_data->d_faces);
		d_data->d_faces=NULL;
		d_data->facecount=0;
		return 0;
	    }
  /* finally, see if there is redchain info. */
	if (strcmp(next,"DESSIN_RED_FACE_LIST:")) /* keep looking */
	 {
	   do
	    {
		eof=fscanf(fp,"%s",next);
	    }
	   while (eof!=EOF && strcmp(next,"DESSIN_RED_FACE_LIST:")!=0
		&& strcmp(next,"END")!=0);
	 }
	if (!strcmp(next,"DESSIN_RED_FACE_LIST:")) /* file gives redchain */
	 {
		for (j=1;j<=d_data->facecount;j++)
			d_data->d_faces[j].rwb_flag=0;
		if (dessin_redlist) 
		 {vert_free(&dessin_redlist);vert_free(&des_redverts);}
	/* read in the list */
		while (fscanf(fp,"%d",&v) && v>0 && v<=d_data->facecount)
		 {
			d_data->d_faces[v].rwb_flag=1; /* red face */
			ftmp=(struct Vertlist *)
			   malloc(sizeof(struct Vertlist));
			ftmp->next=NULL;
			ftmp->v=v;
			if (!dessin_redlist) dessin_redlist=ftrace=ftmp;
			else ftrace=ftrace->next=ftmp;
		 }
		if (dessin_redlist && ftrace->v!=dessin_redlist->v) 
			/* must close up */
		 {
			ftrace->next=(struct Vertlist *)
			   malloc(sizeof(struct Vertlist));
			ftrace->next->next=NULL;
			ftrace->next->v=dessin_redlist->v;
		 }
	/* mark the red vertices (not faces) */	
		if (!mark_red_verts(p,
			dessin,dessin_redlist,&des_redverts))
			vert_free(&des_redverts);
		
	 }
	else 
	 {
		sprintf(msgbuf,
		   "DP: dessin data gives no red chain info.");
		msg();
	 }
	return 1;
} /* read_dessin_data */

int write_dessin_data(FILE *fp,Dessin_data *d_data,int flag)
/* append dessin data to file; assume fp ready. */
{	
	int count,i,icount=0;
	des_face *ptr;
	struct Vertlist *ftrace;

	if (d_data==NULL) return 0;
	count=d_data->facecount;
	if (d_data==NULL || count<=0) return 0;
	fprintf(fp,"\nDESSIN_FACECOUNT: %d\n",count);
	for (i=1;i<=count;i++) 
	 {
		ptr=d_data->d_faces+i;
		fprintf(fp,"   %d\t  %d\t %d %d %d\n",
			i,ptr->orient,
			ptr->nghb[0],ptr->nghb[1],ptr->nghb[2]);
	 }
	if (flag) /* write vertex data for information */
	 {
		fprintf(fp,"\n DESSIN_FACE_VERTS:\n");
		for (i=1;i<=count;i++) 
		 {
			ptr=d_data->d_faces+i;
			fprintf(fp,"   %d\t  %d %d %d\n",
			i,ptr->verts[0],ptr->verts[1],ptr->verts[2]);
		 }
	 }
	if (dessin_redlist) /* record with dessin data */
	 {
		fprintf(fp,"\nDESSIN_RED_FACE_LIST:\n");
		ftrace=dessin_redlist;
		while (ftrace)
		 {
			fprintf(fp,"%d  ",ftrace->v);
			icount++;
			if (icount>9)
			 {
				icount=0;
				fprintf(fp,"\n");
			 }
			ftrace=ftrace->next;
		 }
		fprintf(fp,"\n(done)\n");
	 }
	fprintf(fp,"\nEND");
	return 1;
} /* write_dessin_data */

int ck_for_red_edge(int f)
/* check d-face to see if it has a red edge; if not, return -1, 
else return index of vert from which edge goes in positive direction. */
{
	int j,k,n;

	if (f<1 || f>dessin->facecount) return -1;
	for (j=0;j<3;j++) /* red edge? */
	 {
		k=dessin->d_faces[f].verts[j];
		n=dessin->d_faces[f].
		   verts[(j+3+dessin->d_faces[f].orient)%3];
		if (check_list(des_redverts,k) 
		   && check_list(des_redverts,n) )
			return k;
	 }
	return -1;
} /* ck_for_red_edge */

int start_data_file()
/* open file to put data in for CirclePack */
{

	datafilecount++;
	if (datafp) fclose(datafp); /* shouldn't be open */
	sprintf(datafilename,"/tmp/DPx%dx%d.cpc",Pid,
		datafilecount);
	if ((datafp=fopen(datafilename,"w"))==NULL)
	 {
	 	sprintf(msgbuf,"DP: couldn't open file %s.",
	 		datafilename);
	 	emsg();
	 	return 0;
	 }
	return 1;
} /* start_data_file */

int exec_data_file()
/* close file; tell parent to execute cmds in that file */
{
  int count=0;
	
  if (datafp) fclose(datafp); /* close file */ 
  while (!fopen(datafilename,"r") && count++ < 1000)
    if (count >= 1000) fprintf(stderr,"delay in closing datafile");
  if (cmdmode)
    sprintf(msgbuf,"output in file %s",datafilename);
  else sprintf(msgbuf,"fexec %s",datafilename);
  r_cmsg(msgbuf); 
  return 1;
} /* exec_data_file */

int carbon_copy(struct p_data *p,struct p_data *q)
/* cheap carbon copy: p <--- q. */
{
  int i,j;
  struct K_data *pK_ptr;
  struct R_data *pR_ptr;

  alloc_pack_space(p,q->nodecount+1,0); /* dump old data in p */
  pK_ptr=p->packK_ptr;
  pR_ptr=p->packR_ptr;
  *p=*q;
  p->faces=NULL; /* reset some data */
  p->overlap_status=0;
  p->packK_ptr=pK_ptr;
  p->packR_ptr=pR_ptr;
  for (i=1;i<=q->nodecount;i++)
    {
      pK_ptr[i]=q->packK_ptr[i];
      pR_ptr[i]=q->packR_ptr[i];
      pK_ptr[i].flower=(int *)
	malloc((pK_ptr[i].num+1)*sizeof(int));
      for (j=0;j<=pK_ptr[i].num;j++)
	pK_ptr[i].flower[j]=q->packK_ptr[i].flower[j];
    }
  complex_count(p,0);
  catalog_faces(p);
  return 1;
} /* carbon_copy */

int bdry_flags(struct p_data *p)
/* set bdry flags */
{
  int v;

  for (v=1;v<=p->nodecount;v++)
    {
      if (p->packK_ptr[v].flower[0]==
	  p->packK_ptr[v].flower[p->packK_ptr[v].num])
	p->packK_ptr[v].bdry_flag=0;
      else p->packK_ptr[v].bdry_flag=1;
    }
  return 1;
} /* bdry_flags */

/* ----------------------------- debugging --------------------- */

int pnv(struct p_data *p,Dessin_data *d_data,int f,int nf) 
{
  FILE *ffp;
  int i;
  des_face *fptr;
  
  fptr=d_data->d_faces;
  ffp=fopen("aa_vert","w");
  fprintf(ffp,"face %d <<-------------dessin face data before "\
	  "pasting -------- face %d\n",f,nf);
  for (i=1;i<=d_data->facecount;i++)
    if (fptr[i].mark)
      fprintf(ffp," face %d, \t(%d, %d, %d); \t(%d%d%d); \tmark = %d.\n",
	      i,fptr[i].new_vert[0],fptr[i].new_vert[1],fptr[i].new_vert[2],
	      p->packK_ptr[fptr[i].new_vert[0]].bdry_flag,
	      p->packK_ptr[fptr[i].new_vert[1]].bdry_flag,
	      p->packK_ptr[fptr[i].new_vert[2]].bdry_flag,
	      fptr[i].mark);
  if (ffp) fclose(ffp);
  return 1;
} /* pnv */

int vsp(int *oldnew,int n,int f,int nf)
{
  FILE *ffp;
  int i,j,k;
	
  ffp=fopen("aa_swap","w");
  fprintf(ffp,"face %d << ------------- vertex swaps after "
	  "pasting --------- face %d\n",f,nf);
  for (i=1;i<=n;i=i+5)
    {
      fprintf(ffp,"\n");
      if (i<=n) for (j=0;j<5;j++)
	{
	  k=i+j;
	  if (k<=n && k!=oldnew[k]) 
	    fprintf(ffp,"%d -> %d; ",k,oldnew[k]);
	}
    }
  if (ffp) fclose(ffp);
  return 1;
} /* vsp */

int bugwrite(struct p_data *p)
{
  FILE *ffp;

  ffp=fopen("aa_pack","w");
  fprintf(ffp,"\n\n");
  writepack(ffp,p,0,0);
  if (ffp) fclose(ffp);
  return 1;
} /* bugwrite */
	

/* have to provide buffers emsgbuf (and msgbuf, part of it) and
the routines emsg() and msg() because library routines will need
them */
int msg()
{
  if (cmdmode) fprintf(stderr,"%s\n",msgbuf);
  else fprintf(stdout,"%s\n",msgbuf);
  return 1;
} /* msg */

int emsg()
{
  if (cmdmode) fprintf(stderr,"%s\n",emsgbuf);
  else fprintf(stdout,"%s\n",emsgbuf);
  return 1;
} /* emsg */
