;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: html-generation.lisp,v 1.33 2001/11/12 20:07:08 jesse Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.

(in-package :imho)

;;
;; Functions for programmatically generating SGML markup
;;

;; with-tag - A macro that wraps some code inside an SGML tag.
;;
;; example:
;;
;; (with-tag (:tag "P")
;;   (with-tag (:tag "I")
;;     (format t "It was a dark and stormy night.")))
;;
;; expands to code that writes:
;;
;; <P><I>It was a dark and stormy night.</I></P>
;;
;; the body is optional, and if you want to omit the close tag, say:
;;
;; (with-tag (:tag "IMG" :noclose t :attr '(("SRC" . "http://localhost/gif.jpg"))))
;;
;; which produces
;;
;; <IMG SRC="http://localhost/gif.jpg">

(defvar *html-escaped*
  '((#\& . "&amp;")
    (#\= . "&3D")
    (#\< . "&lt;")
    (#\> . "&gt;")))

(defun html-escape (str)
  "return HTML escaped string (eg, '<' replaced by '&lt;')"
  (dolist (pair *html-escaped*)
    (setq str (string-replace (car pair) (cdr pair) str)))
  str)

(defmacro with-tag ((&key tag (stream '*response-stream*)
			  (attr nil) (noclose nil)) &body body)
  `(locally
    (let ((stream ,stream))
      (format stream "<~A" ,tag)
      (dolist (pair ,attr)
        (if (null (cdr pair))
            (progn
              (write-char #\  stream)
              (write-string (car pair) stream))
            (format stream " ~A=\"~A\"" (car pair) (cdr pair))))
      (format stream ">")
      ,@body
      ,(if (not noclose)
	  `(format stream "</~A>" ,tag)))))

(defmacro with-tag2 ((tag &key (stream '*response-stream*)
			  (attr nil) (noclose nil)) &body body)
  `(locally
    (let ((stream ,stream))
      (format stream "<~A" ,tag)
      (dolist (pair ,attr)
        (if (null (cdr pair))
            (progn
              (write-char #\  stream)
              (write-string (car pair) stream))
            (format stream " ~A=\"~A\"" (car pair) (cdr pair))))
      (format stream ">")
      ,@body
      ,(if (not noclose)
	  `(format stream "</~A>" ,tag)))))

(defvar *add-element-comments* t)

(defmacro without-element-comments (&body body)
  `(progv '(*add-element-comments*)
    (list nil)
    ,@body))

(defmacro with-reference ((reference target stream) &body rest)
  (if target 
      `(with-tag (:stream ,stream
                  :tag "A"
 :attr `(("TARGET" . ,,target)
         ("HREF" . ,,reference)))
        ,@rest)
      `(with-tag (:stream ,stream
                  :tag "A"
                  :attr `(("HREF" . ,,reference)))
        ,@rest)))

;; Now, rather than
;; 
;; (with-tag (:stream stream :tag "A" :attr `(("HREF" . (refer-wm frob widget))))
;;   (:princ "Frob the Widget"))
;; 
;; you say
;; 
;; (with-action (stream frob widget)
;;   (:princ "Frob the Widget"))

(defmacro with-action ((stream element method &rest args) &body body)
  "Wrap the output of BODY in an HREF which will invoke METHOD on
element, passing along ARGS.  Write this all to STREAM"
  (let (target)
    (when (consp method)
      (setq target (cadr method))
      (setq method (car method)))
    `(with-reference ((build-element-url ,element
                       :reference (refer-wm ,method ,@args)) ,target ,stream)
      (html-stream
       stream
       ,@body))))

