;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: warp.lisp,v 1.21 2002/03/29 22:55:40 craig Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.

(in-package :imho)

;; The VERSION of this implementation.
(defconstant +warp-version-string+ "0.5")

;; The RID associated with the connection controller handler (0x00000).
(defconstant +rid-connection+ #x00000)

;; The RID indicating that the connection must be closed (0x0ffff).
(defconstant +rid-disconnect+ #x0ffff)

;; The RID minimum value (0x00001).
(defconstant +rid-min+ #x00001)

;; The RID maximum value (0x0fffe).
(defconstant +rid-max+ #x0fffe)

(defconstant +typ-coninit-hst+ #x00000)
(defconstant +typ-coninit-hid+ #x00001)
(defconstant +typ-coninit-app+ #x00002)
(defconstant +typ-coninit-aid+ #x00003)
(defconstant +typ-coninit-req+ #x00004)
(defconstant +typ-coninit-rid+ #x00005)
(defconstant +typ-coninit-err+ #x0000F)
(defconstant +typ-reqinit-met+ #x00010)
(defconstant +typ-reqinit-uri+ #x00011)
(defconstant +typ-reqinit-arg+ #x00012)
(defconstant +typ-reqinit-pro+ #x00013)
(defconstant +typ-reqinit-hdr+ #x00014)
(defconstant +typ-reqinit-var+ #x00015)
(defconstant +typ-reqinit-run+ #x0001D)
(defconstant +typ-reqinit-err+ #x0001E)
(defconstant +typ-reqinit-ack+ #x0001F)
(defconstant +typ-request-sta+ #x00020)
(defconstant +typ-request-hdr+ #x00021)
(defconstant +typ-request-cmt+ #x00022)
(defconstant +typ-request-dat+ #x00023)
(defconstant +typ-request-err+ #x0002E)
(defconstant +typ-request-ack+ #x0002F)

(defun gethostid ()
  1234)

(defun getappid ()
  4321)


(defvar *requests* nil)

(defvar *claimed-rids* nil)

(defun claim-rid (&aux (rid-idx 0))
  ;; avoid using RIDs that are still apparently 'claimed' by a request
  (loop
   (or (member rid-idx *claimed-rids* :test #'=)
       (return (push rid-idx *claimed-rids*)))
   (if (= rid-idx +rid-max+)
       (setf rid-idx +rid-min+)
       (incf rid-idx)))
  rid-idx)

(defun unclaim-rid (rid)
  ;; for #'terminate-request
  (setf *claimed-rids*
        (delete rid *claimed-rids* :test #'=))
  t)

(define-condition warp-shutdown ()
  )

(defun warp-server (stream)
  (and (debugging :warp)
         (note-stat :cgi-servers :increment))
  (let ((*requests* (make-hash-table)))
    (unwind-protect
         (handler-case
             (loop (exec-warp-packet (read-warp-packet stream) stream))
           (warp-shutdown ()))
      (progn
        (clrhash *requests*)
              (and (debugging :warp)
                       (note-stat :cgi-servers :decrement))))))

(defun read-warp-packet (stream)
  (list (read-int16 stream)
        (read-int16 stream)
        (read-length-bytes stream)))

;;
;; maximum length of warp packets is 65535 bytes
;;

(defvar *warp-maxpacket* 65530)

(defun write-warp-packet (rid type payload stream)
  (etypecase payload
    (string
     (let ((len (length payload)))
       (dformat :warp "WARP: String response, ~d octets" len)
       (when (debugging :warp-outfile)
         (with-open-file (stream "/tmp/warp.out.string"
                                 :element-type 'unsigned-byte
                                 :direction :output :if-exists :overwrite
                                 :if-does-not-exist :create)
           (cgi-write-length-string payload stream)))
       (if (<= len *warp-maxpacket*)
           (progn
             (dformat :warp "Warp: Writing ~d bytes" len)
             (write-int16 rid stream)
             (write-int16 type stream)
             (cgi-write-length-string payload stream))
           (dotimes (i (ceiling len *warp-maxpacket*))
             (let* ((start (* i *warp-maxpacket*))
                    (stop  (min len (* (1+ i) *warp-maxpacket*)))
                    (pay (subseq payload start stop)))
               (dformat :warp "Warp: Writing ~d bytes" (length pay))
               (write-int16 rid stream)
               (write-int16 type stream)
               (cgi-write-length-string pay stream))))))
    (cons
     (write-warp-packet rid type (cdr payload) stream))
    (vector
     (let ((len (length payload)))
       (dformat :warp "WARP: Vector response, ~d octets" len)
       (when (debugging :warp-outfile)
         (with-open-file (stream (format nil "/tmp/warp.out.vector.~d" (random 1000))
                                 :element-type 'unsigned-byte
                                 :direction :output :if-exists :supersede
                                 :if-does-not-exist :create)
           (write-length-bytes payload stream)
           (finish-output stream)))
       (if (<= len *warp-maxpacket*)
           (progn
             (dformat :warp "Warp: Writing ~d bytes~%" len)
             (write-int16 rid stream)
             (write-int16 type stream)
             (write-length-bytes payload stream))
           (dotimes (i (ceiling len *warp-maxpacket*))
             (let* ((start (* i *warp-maxpacket*))
                    (stop  (min len (* (1+ i) *warp-maxpacket*)))
                    (pay (subseq payload start stop)))
               (dformat :warp "Warp: Writing ~d bytes" (length pay))
               (write-int16 rid stream)
               (write-int16 type stream)
               (write-length-bytes pay stream)
               (finish-output stream)))))))
  (finish-output stream))

(defun initiate-request ()
  "Initiate a warp request, returning a new request id"
  (let ((rid (claim-rid)))
    (setf (gethash rid *requests*)
          (make-request :rid rid :response-callback #'warp-send-response))
    (and (debugging :warp)
         (note-stat :active-warp-requests :increment))
    rid))

(defun terminate-request (rid)
  "Terminate the warp request identified by RID, returning t"
  (destroy-request (gethash rid *requests*))
  (remhash rid *requests*)
  (unclaim-rid rid)
  (and (debugging :warp)
       (note-stat :active-warp-requests :decrement))
  t)

(defun make-keyword (string)
  (intern (string-upcase string) :keyword))

(defun exec-warp-request (rid typ request stream)
  (let ((active-request (gethash rid *requests*)))
    (unless active-request
      (error "Request ~d is not in progress." rid))
    (ecase typ
      (#.+typ-reqinit-met+              ; method
       (setf (request-http-method active-request) (read-string request)))
      (#.+typ-reqinit-uri+              ; URI
       (parse-uri active-request (read-string request)))
      (#.+typ-reqinit-pro+              ; protocol
       (setf (request-protocol active-request) (read-string request)))
      (#.+typ-reqinit-arg+              ; protocol
       (setf (request-args active-request) (read-string request)))
      (#.+typ-reqinit-var+              ; content
       (if (request-client-content active-request)
           (setf (request-client-content active-request)
                 (concatenate 'vector
                              (request-client-content active-request)
                              (read-length-bytes request)))
           (setf (request-client-content active-request)
                 (read-length-bytes request))))
      (#.+typ-reqinit-hdr+              ; headers
       (let ((header (make-keyword (read-string request))))
         (case header
           (:content-type
            (push (cons header (parse-content-type-header (read-string request)))
                  (request-headers-in active-request)))
           (t
            (push (cons header (read-string request))
                  (request-headers-in active-request))))))
      (#.+typ-reqinit-run+              ; execute request
       (unwind-protect
            (progn
              (write-warp-packet rid +typ-reqinit-ack+ #() stream)
              (setf (request-stream active-request) stream)
              (parse-content active-request)
              (handle-request active-request)
              (let ((response-stream (make-byte-array-output-stream)))
                (cgi-write-length-string "End of request" response-stream)
                (write-warp-packet rid +typ-request-ack+ (get-output-stream-data response-stream) stream)))
         (terminate-request rid))))))

(defun exec-warp-packet (packet stream)
  (destructuring-bind (rid typ buf)
      packet
    (let ((request (make-byte-array-input-stream buf)))
      (case rid
        (#.+rid-connection+
         (ecase typ
           (#.+typ-coninit-hst+
            (read-string request)       ; name
            (read-int16 request)        ; port
            (let ((reply-packet (make-byte-array-output-stream)))
              (write-int16 (gethostid) reply-packet)
              (write-warp-packet 0 +typ-coninit-hid+ (get-output-stream-data reply-packet) stream)))
           (#.+typ-coninit-app+
            (read-int16 request)        ; HID
            (read-string request)       ; name
            (read-string request)       ; path
            (let ((reply-packet (make-byte-array-output-stream)))
              (write-int16 (getappid) reply-packet)
              (write-warp-packet 0 +typ-coninit-aid+ (get-output-stream-data reply-packet) stream)))
           (#.+typ-coninit-req+
            (read-int16 request)
            (read-int16 request)
            (let ((reply-packet (make-byte-array-output-stream)))
              (write-int16 (initiate-request) reply-packet)
              (write-warp-packet 0 +typ-coninit-rid+ (get-output-stream-data reply-packet) stream)))))
        (#.+rid-disconnect+
         (error 'warp-shutdown))
        (t
         (exec-warp-request rid typ request stream)))))
  (values))

(defun warp-send-response (request)
  (let ((stream (request-stream request))
        (rid (request-rid request)))
    (let ((reply-packet (make-byte-array-output-stream)))
      (cgi-write-length-string (request-protocol request) reply-packet)
      (write-int16 200 reply-packet)
      (cgi-write-length-string "" reply-packet)
      (write-warp-packet rid +typ-request-sta+ (get-output-stream-data reply-packet) stream))
    
    (let ((reply-packet (make-byte-array-output-stream)))
      (cgi-write-length-string "Pragma" reply-packet)
      (cgi-write-length-string "No-Cache" reply-packet)
      (write-warp-packet rid +typ-request-hdr+ (get-output-stream-data reply-packet) stream))
    
    (when-bind (response-type (request-response-type request))
      (let ((reply-packet (make-byte-array-output-stream)))
        (cgi-write-length-string "Content-Type" reply-packet)
        (cgi-write-length-string response-type reply-packet)
        (write-warp-packet rid +typ-request-hdr+ (get-output-stream-data reply-packet) stream)))
  
    (when-bind (response-length (request-response-length request))
      (let ((reply-packet (make-byte-array-output-stream)))
        (cgi-write-length-string "Content-Length" reply-packet)
        (cgi-write-length-string (princ-to-string response-length) reply-packet)
        (write-warp-packet rid +typ-request-hdr+ (get-output-stream-data reply-packet) stream)))
    
    
    (dolist (header (request-headers-out request))
      (let ((reply-packet (make-byte-array-output-stream)))
        (cgi-write-length-string (car header) reply-packet)
        (cgi-write-length-string (cdr header) reply-packet)
        (write-warp-packet rid +typ-request-hdr+ (get-output-stream-data reply-packet) stream)))
    
    (write-warp-packet rid +typ-request-cmt+ #() stream)
    (write-warp-packet rid +typ-request-dat+ (request-response-body request) stream)))
