;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: wcof -*-
;;; $Id: modules.lisp,v 1.5 2002/03/29 04:23:43 craig Exp $
;;;
;;; Copyright (c) 2000, 2001 onShore Development, Inc.
;;;
;;; ODCL Module System
;;;
;;; Purpose: Provide support for defining, loading and configuring
;;; "features" or code sets.  It's at this point just a set of
;;; defgenerics/methods for a simple define/load/configure API which
;;; speciailize on the name of a normal defsystem system.  Hopefully
;;; it can be integrated into defsystem or otherwise driven by it.

(in-package :odcl)

(defvar *odcl-module-config-path* "odcl:config;*.lisp"
  "logical host name which when given as the second argument to merge-pathname, with the first argument being the string specifying the module, well result in a pathname identifying the configuration file for that module.")

(defvar *odcl-modules* (make-hash-table)
  "Hash table keyed on system names which maps to MODULE-STATUS objects.")

;;; Module lifecycle:
;;; Register -> Load -> Configure -> Init -> Activate ->? Deactivate

(defclass module-status ()
  ((loaded
    :initarg loaded
    :initform nil)
   (configured
    :initform nil)
   (initialized
    :initform nil)
   (activated
    :initform nil)
   (config-file
    :initform nil
    :initarg config-file)
   (original-name
    :initform nil
    :initarg :original-name))
  (:documentation "Status of a module, not for public consumption"))

(defun %module-name (module-name)
  (etypecase module-name
    (string
     (intern module-name :keyword))
    (symbol
     module-name)))

(defmethod module-known? (module-name)
  "Is the module known to the system?"
  (gethash (%module-name module-name) *odcl-modules*))

(defmethod module-loaded? (module-name)
  "Has the code for the module been loaded?"
  (when-bind (mod (module-known? module-name))
    (slot-value mod 'loaded)))

(defmethod module-configured? (module-name)
  "Has the module been configured?"
  (when-bind (mod (module-known? module-name))
    (slot-value mod 'loaded)))

(defmethod module-initialized? (module-name)
  "Has the module been runtime initialized?"
  (when-bind (mod (module-known? module-name))
             (slot-value mod 'loaded)))

(defmethod module-activated? (module-name)
  "Has the module been runtime initialized?"
  (when-bind (mod (module-known? module-name))
             (slot-value mod 'activated)))

(defmethod module/register (module-name
                            &rest args
                            &key
                            config-file
                            loaded)
  (declare (ignore args))
  (let ((status (make-instance
                 'module-status
                 :original-name module-name
                 :loaded loaded
                 :config-file config-file)))
    (setf (gethash (%module-name module-name)
                   *odcl-modules*) status)))

(defmethod module/load (module-name
                        &rest args
                        &key already-loaded)
  (let ((status (module-known? module-name)))
    (unless status
      (apply #'module/register module-name args))
    (unless already-loaded
      (mk:oos (slot-value status 'original-name) :load))
    (setf (slot-value status 'loaded) t)))

(defmethod module/default-config-file (module-name)
  (when-bind (status (module-known? module-name))
    (unless (equal (slot-value status 'config-file) :none)
      (ignore-errors
        (merge-pathnames
         module-name *odcl-module-config-path*)))))

(defun %normalize-module-config-file (module-name file)
  (or (ignore-errors
        (merge-pathnames file *odcl-module-config-path*))
      (unless (equal file :none) file)
      (module/default-config-file module-name)))

(defmethod module/generate-configuration (module-name
                                          &key
                                          config-file)
  (declare (ignore module-name config-file))
  t)

(defmethod module/configure (module-name
			     &rest args
                             &key
                             config-file
                             (generate-configuration t))
  (unless (module-loaded? module-name)
    (apply #'module/load module-name args))
  (when-bind (conf (%normalize-module-config-file module-name config-file))
    (if generate-configuration
        (restart-case
            (load conf :verbose nil)
          (configure-wco ()
            :report "Set up your configurtion now."
            (module/generate-configuration module-name)))
        (load conf :verbose nil)))
  (setf (slot-value (module-known? module-name) 'configured) t))

(defmethod module/initialize (module-name
			      &rest args
                              &key
                              (cold-boot t)
                              &allow-other-keys)
  (declare (ignore cold-boot))
  (unless (module-configured? module-name)
    (apply #'module/configure module-name args))
  (when (apply #'module/init
               (%module-name module-name)
               args)
    (setf (slot-value (module-known? module-name) 'initialized) t)))

(defmethod module/init (module-name
                        &rest args)
  (declare (ignore args module-name))
  t)

(defmethod module/activate (module-name
                            &rest args)
  (unless (module-initialized? module-name)
    (apply #'module/initialize module-name args))
  (setf (slot-value (module-known? module-name) 'activated) t))

(defmethod module/deactivate (module-name
                              &rest args)
  (declare (ignore args))
  (when-bind (status (module-known? module-name))
    (setf (slot-value status 'activated) nil)))

(defmethod module/uninit (module-name
                        &rest args)
  (declare (ignore args module-name))
  t)

(defmethod module/uninitialize (module-name
                                &rest args)
  (when (module-initialized? module-name)
    (apply #'module/uninit module-name args)
    (setf (slot-value (module-known? module-name) 'initialized) nil)))

(pushnew :odcl-module-support *features*)
