;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: regression.lisp,v 1.24 2003/03/24 21:51:31 adam Exp $
;;;
;;; Copyright (c) 2000 - 2003 onShore Development, Inc.

;;; Regression testing for WebCheckout
;;;
;;; Tests are keyed by class, and bugid. To run a test for a single bug:
;;;
;;; (regress [class] [bugid])
;;;
;;; To run all tests for a given class:
;;;
;;; (regress [class])
;;;
;;; To run all tests:
;;;
;;; (regress)
;;;
;;; Tests ought to be iterable, ought not to affect eachother when run
;;; serially, and generally assume that the database has been freshly
;;; :cold-boot'ed.

(in-package :odcl)

(defvar *regression-funcs* (make-hash-table :test #'equal)
  "mapping of class + bugid to test function")

(defun clear-regression-class (class &aux to-clear)
  (maphash (lambda (k v)
             (declare (ignore v))
             (when (equal class (car k))
               (push k to-clear)))
           *regression-funcs*)
  (dolist (clear to-clear)
    (remhash clear *regression-funcs*)))

(defmacro defregression ((class testid &rest bugids) &body body)
  "define a function FUNCTION-NAME with body BODY that tests all bugs
in BUGID-LIST"
  (declare (ignore bugids))
  (let ((doc (if (and (listp body)
			(stringp (car body)))
               (car body)
	       (warn "No docstring provided to regression test ~A ~A" class testid)))
        (test-fun (symconcat *package* "%regression-" class "-" (ensure-string testid))))
    `(progn
      (setf (gethash (list ,class ,testid) *regression-funcs*)
       (cons ,doc ',test-fun))
      (defun ,test-fun ()
        ,@body))))

(defvar *regression-stats* nil)

(defun regress (&optional class bugid verbose)
  "entry point for regression testing, called with no parameters to
run all tests"
  (cond (bugid
         (if-bind (function (gethash (list class bugid) *regression-funcs*))
             (funcall (cdr function))
             (cmsg "test ~s ~s not found" class bugid)))
        (class
         (let* ((all-keys (hashkeys *regression-funcs*))
                (class-keys (remove-if-not (lambda (k) (eq (car k) class)) all-keys)))
           (if (null class-keys)
               (progn
                 (cmsg "No tests found for class ~s" class)
                 (cmsg "Valid classes are: ~{~s~^, ~}" (remove-duplicates (mapcar #'car all-keys))))
               (progn
                 (cmsg "~%;; Running ~d tests for class ~s~%;;" (length class-keys) class)
                 (setq class-keys (sort class-keys (lambda (x y)
                                                     (let ((x (cadr x))
                                                           (y (cadr y)))
                                                       (cond ((numberp x)
                                                              (or (not (numberp y))
                                                                  (< x y)))
                                                             ((numberp y)
                                                              (not (or (not (numberp x))
                                                                       (< x y))))
                                                             (t
                                                              (string<= (string x) (string y))))))))
                 (dolist (test-key class-keys)
                   (%regress-with-handler test-key :verbose verbose))))))
        (t
         (cmsg "Starting regression run")
         (let ((*regression-stats* (list (cons :running t)
                                         (cons :test-count 0)
                                         (cons :failure-count 0))))
           (mapc #'regress
                 (remove-duplicates (mapcar #'car (hashkeys *regression-funcs*))))
           (cmsg "Regression run complete, ~d tests run, ~d failures"
                 (get-alist :test-count *regression-stats*)
                 (get-alist :failure-count *regression-stats*))
           (if (< 0 (get-alist :failure-count *regression-stats*))
               (values nil *regression-stats*)
               (values t *regression-stats*))))))

(defun %regress-with-handler (bugid &key verbose &aux testerr)
  (when *regression-stats*
    (let ((count (get-alist :test-count *regression-stats*)))
      (update-alist :test-count (1+ count) *regression-stats*)))
  (let ((function (gethash bugid *regression-funcs*))
        (time (get-utime))
        (message (make-string-output-stream)))
    (format message "~&;; ~16a ~6a -" (car bugid) (cadr bugid))
    (force-output t)
    (handler-case
        (let ((*console-spam* verbose))
          (funcall (cdr function))
          (format message "              "))
      (error (e)
        (when *regression-stats*
          (let ((failures (get-alist :failure-count *regression-stats*)))
            (update-alist :failure-count (1+ failures) *regression-stats*)))
        (format message " ***FAILURE*** ")
	;(terpri message)
	;(terpri message)
	;(format message "~2T~A" e)
        (setf testerr e)))
    (format message " ~6,2F - ~a~%"
            (float (/ (ceiling (/ (- (get-utime) time) 100) 100) 100))
            (or (car function) ""))
    (when (or testerr verbose)
      (write-string (get-output-stream-string message) ))
    (when testerr
      (cmsg-c :regression "~A" testerr))))

;; helper

(defmacro expect-error (error err-form)
  "Throw an error if ERR-FORM doesn't generate ERROR"
  (let ((pass (gensym)))
    `(let ((,pass t))
      (handler-case
          (progn
            ,err-form
            (setq ,pass nil))
        (,error (e)
          (cmsg "Caught expected error: ~s" e)))
      (unless (eql ,pass t)
        (error "Logic error: The error ~s should have been thrown by form ~s" ',error ',err-form)))))
