/*
	$Id: font.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		Simple font support.
*/

//! component="Display 2D"

#ifndef header_font
#define header_font

class CL_Surface;
class CL_Target;
class CL_Font_Description;
class CL_ResourceManager;

class CL_Font
//: ClanLib's font class.
// <p>This class allows you to easilly draw text onto the backbuffer or to a
// renderer target.</p>
//
// <p>The font can be created in two ways. Either it is loaded from a
// resource, or it is created using the font description interface. The font
// and its description have the same relationship/functionality as the <a
// class="CL_Surface">surface</a> and its <a
// class="CL_SurfaceProvider">provider</a>.</p>
//
// <p>Please read the CL_Font_Description page if you want an description on
// how to create your a font.</p>
{
protected:
	CL_Surface *chars[256];
	int space_len;
	int subtract_width;
	int height;

public:
	enum alignments
	{
		ALIGN_LEFT,
		ALIGN_CENTER,
		ALIGN_RIGHT
	};

	static CL_Font *load(
		const char *res_id,
		CL_ResourceManager *resource_manager);
	// Loads the font from resources.
	//!param: res_id - The resource ID of the font.
	//!param: resource_manager - The resource manager used to load the
	//!param:                    resources from.

	CL_Font(CL_Font_Description *font_desc);
	// Loads the font from a font description.
	//!param: font_desc - The font description used to construct the font.

	virtual ~CL_Font();

	int get_height() { return height; }
	//: Returns the height of the font.
	//!retval: Width of the font.

	int get_text_width(const char *text);
	//: Return the width of 'text' if drawn with this font.
	//!param: text - Text string the width should be calculated from.
	//!retval: Width of the text (in pixels).
	
	void print_left(int x, int y, const char *text);
	//: Prints 'text' left aligned at coordinates ('x','y').
	//!param: x - X coordinate (in pixels) the text is drawn at.
	//!param: y - Y coordiante (in pixels) the text is drawn at.
	//!param: text - The text to be drawn with this font.

	void print_left(int x, int y, float scale_x, float scale_y, const char *text);
	//: Prints 'text' left aligned at coordinates ('x','y').
	//!param: x - X coordinate (in pixels) the text is drawn at.
	//!param: y - Y coordiante (in pixels) the text is drawn at.
	//!param: scale_x - Scale font width scale_x times.
	//!param: scale_y - Scale font height scale_y times.
	//!param: text - The text to be drawn with this font.
	
	void print_center(int x, int y, const char *text);
	//: Prints 'text' center aligned at coordinates ('x','y').
	//!param: x - X coordinate (in pixels) the text is centered at.
	//!param: y - Y coordiante (in pixels) the text is centered at.
	//!param: text - The text to be drawn with this font.

	void print_right(int x, int y, const char *text);
	//: Prints 'text' right aligned at coordinates ('x','y').
	//!param: x - X coordinate (in pixels) the text is ends at.
	//!param: y - Y coordiante (in pixels) the text is drawn at.
	//!param: text - The text to be drawn with this font.
	
	void put_target(int x, int y, const char *text, CL_Target *target, int alignment=ALIGN_LEFT);
	//: Prints 'text' aligned with alignment at coordinates ('x','y') to the target surface.
	//: If CL_Font::ALIGN_LEFT is specified the text is draw starting from x, y.
	//: If CL_Font::ALIGN_CENTER is specified the text is centered at x, y.
	//: If CL_Font::ALIGN_RIGHT is specified the text ends at x, y;
	//!param: x - X coordinate (in pixels)
	//!param: y - Y coordiante (in pixels)
	//!param: text - The text to be drawn with this font.
	//!param: target - the target surface to blit to.
	//!param: alignment - the alignment of the text, CL_Font::ALIGN_LEFT, CL_Font::ALIGN_CENTER, CL_Font::ALIGN_RIGHT
};

#endif

