
//! component="Resources"

#ifndef header_resourcetype_boolean
#define header_resourcetype_boolean

#include "resource_manager.h"
#include "resourcetype.h"
#include "resource.h"
#include "../System/clanstring.h"
#include "../System/error.h"
#include "../IOData/inputsource.h"
#include "../IOData/inputsource_provider.h"

class CL_Res_Boolean : public CL_ResourceType
//: The Boolean resource is used to read boolean values from a resource file.
// CL_Res_Boolean, CL_Res_Integer and CL_Res_String helps you with seperating
// the game code from the constants used in the game.
// <br>
// Using these resource types, you can easily change the text used in the game.
// Just by changing the resource file, you can make you game speak Danish (or Dutch), or
// any other language. You can change the constants like weapon damage, movement
// speed, and so on.
// <br>
// These resources can also be used to apply theme support into your application.
// By letting each theme become a datafile, you can give the player full control
// of the visual parts of the game - and without using much work on it yourself.
//!res_type: boolean
//!res_desc: The boolean resource is used to read a boolean from a resourcefile
{
public:
	static bool load(std::string res_id, CL_ResourceManager *manager, bool _default);
	static bool load(std::string res_id, CL_ResourceManager *manager );

	CL_Res_Boolean();

private:
friend CL_ResourceManager;
	virtual CL_Resource *create_from_location(
		std::string name,
		std::string location,
		CL_ResourceOptions *options,
		CL_ResourceManager *parent);

	virtual CL_Resource *create_from_serialization(
		std::string name,
		CL_ResourceManager *parent);
};

class CL_BooleanResource : public CL_Resource
{
public:
	CL_BooleanResource(
		std::string name,
		bool value)
	: CL_Resource("boolean", name)
	{
		this->value = value;
		load_count = 0;
	}

	CL_BooleanResource(
		std::string name,
		std::string location,
		CL_ResourceOptions *options,
		CL_ResourceManager *parent)
	: CL_Resource("boolean", name)
	{
		if (location == "true") value = true;
		else if (location == "false") value = false;
		else
		{
			CL_String err;
			err << "Boolean resource has invalid value: " << location;
			throw CL_Error(err.get_string());
		}
		load_count = 0;
	}

	CL_BooleanResource(
		std::string name,
		CL_ResourceManager *parent)
	: CL_Resource("boolean", name)
	{
		CL_InputSource *input =
			parent->get_resource_provider()->open_source(name.c_str());

		unsigned char v = input->read_uchar8();
		value = (v == 1);
		
		delete input;
		load_count = 0;
	}
	
	bool get_value() const { return value; }

	virtual void load() { load_count++; }
	virtual void unload() { load_count--; }
	virtual int get_load_count() { return load_count; }

	virtual void serialize_save(CL_OutputSource *output)
	{
		unsigned char v = value ? 1 : 0;
		output->write_uchar8(v);
	}

private:
	bool value;
	int load_count;
};

#endif
