import py

from py.__.rest.convert import convert_dot, latexformula2png

import sys
from docutils import nodes
from docutils.parsers.rst import directives, states, roles
from docutils.parsers.rst.directives import images

try:
    from docutils.utils import unescape # docutils version > 0.3.5
except ImportError:
    from docutils.parsers.rst.states import unescape # docutils 0.3.5

backend_to_image_format = {"html": "png", "latex": "pdf"}

class BackendStore(object):
    #XXX this whole class should not be there:
    #XXX it is only used to work around the inflexibility of docutils:
    # a directive does not know the path of the file it looks at, nor the
    # format
    def __init__(self, backend):
        self.backend = backend
        self.convert_to_format = backend_to_image_format[backend]
        directives.register_directive("graphviz", self.graphviz_directive)
        roles.register_canonical_role("latexformula", self.latexformula_role)

    def convert(self, fn, path):
        path = py.path.local(path).dirpath()
        dot = path.join(fn)
        result = convert_dot(dot, self.convert_to_format)
        return result.relto(path)

    def graphviz_directive(self, name, arguments, options, content,
                           lineno, content_offset, block_text, state,
                           state_machine):
        newname = self.convert(arguments[0], state.document.settings._source)
        text = block_text.replace("graphviz", "image", 1)
        text = text.replace(arguments[0], newname, 1)
        return images.image(u'image', [newname], options, content, lineno,
                            content_offset, text, state, state_machine)
    graphviz_directive.arguments = (1, 0, 1)
    graphviz_directive.options = {'alt': directives.unchanged,
                                  'height': directives.nonnegative_int,
                                  'width': directives.nonnegative_int,
                                  'scale': directives.nonnegative_int,
                                  'align': images.align,
                                  'target': directives.unchanged_required,
                                  'class': directives.class_option}

    def latexformula_role(self, name, rawtext, text, lineno, inliner,
                          options={}, content=[]):
        if self.backend == 'latex':
            options['format'] = 'latex'
            return roles.raw_role(name, rawtext, text, lineno, inliner,
                                  options, content)
        else:
            # XXX: make the place of the image directory configurable
            sourcedir = py.path.local(inliner.document.settings._source).dirpath()
            imagedir = sourcedir.join("img")
            if not imagedir.check():
                imagedir.mkdir()
            # create halfway senseful imagename:
            # use hash of formula + alphanumeric characters of it
            # could
            imagename = "%s_%s.png" % (
                hash(text), "".join([c for c in text if c.isalnum()]))
            image = imagedir.join(imagename)
            latexformula2png(unescape(text, True), image)
            imagenode = nodes.image(image.relto(sourcedir), uri=image.relto(sourcedir))
            return [imagenode], []
    latexformula_role.content = True
    latexformula_role.options = {}

