/**************************************************************************/
/*                                                                        */
/*  cpbk - a mirroring utility for backing up your files                  */
/*  Copyright (C) 1998 Kevin Lindsay <klindsay@mkintraweb.com>            */
/*  Copyright (C) 2001 Yuuki NINOMIYA <gm@debian.or.jp>                   */
/*                                                                        */
/*  This program is free software; you can redistribute it and/or modify  */
/*  it under the terms of the GNU General Public License as published by  */
/*  the Free Software Foundation; either version 2, or (at your option)   */
/*  any later version.                                                    */
/*                                                                        */
/*  This program is distributed in the hope that it will be useful,       */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/*  GNU General Public License for more details.                          */
/*                                                                        */
/*  You should have received a copy of the GNU General Public License     */
/*  along with this program; if not, write to the                         */
/*  Free Software Foundation, Inc., 59 Temple Place - Suite 330,          */
/*  Boston, MA 02111-1307, USA.                                           */
/*                                                                        */
/**************************************************************************/

/* $Id: cpbk.c,v 1.18 2001/03/30 12:04:55 gm Exp $ */

#define GLOBAL_VALUE_DEFINE

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <errno.h>
#include <limits.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef ENABLE_NLS
#  include <locale.h>
#endif

#include "intl.h"
#include "shhopt.h"
#include "strlib.h"
#include "parsecfg.h"
#include "variable.h"
#include "proto.h"


static void put_usage_and_exit(void);
static void put_version_and_exit(void);
static void set_default_config(void);
static void copy_command_line_option_to_config(void);


static optStruct option_table[] = {
	/* short long                type        var/func                            special */
	{ 'c', "config",             OPT_STRING, &command_line_option.config_file,         0 },
	{ 'e', "exclude",            OPT_STRING, &command_line_option.linear_exclude_path, 0 },
	{ 't', "trash-bin",          OPT_STRING, &command_line_option.trashbin_dir,        0 },
	{ 'l', "list",               OPT_FLAG,   &command_line_option.list,                0 },
	{ 'n', "nlist",              OPT_FLAG,   &command_line_option.nlist,               0 },
	{ 's', "simulate",           OPT_FLAG,   &command_line_option.simulate,            0 },
	{ 'f', "force",              OPT_FLAG,   &command_line_option.force,               0 },
	{ 'i', "inode-ctime",        OPT_FLAG,   &command_line_option.inode_ctime_check,   0 },
	{ 'I', "ignore-minor-error", OPT_FLAG,   &command_line_option.ignore_minor_error,  0 },
	{ 'p', "suppress-progress",  OPT_FLAG,   &command_line_option.suppress_progress,   0 },
	{ 'q', "quiet",              OPT_FLAG,   &command_line_option.quiet,               0 },
	{ 'v', "verbose",            OPT_FLAG,   &command_line_option.verbose,             0 },
	{ 'h', "help",               OPT_FLAG,   &put_usage_and_exit,           OPT_CALLFUNC },
	{ 'V', "version",            OPT_FLAG,   &put_version_and_exit,         OPT_CALLFUNC },
	{ 0, 0, OPT_END, 0, 0 }
};

static cfgStruct config_table[] = {
	/* parameter         type             address of variable */
	{ "Source",           CFG_STRING,      &config.source_dir         },
	{ "Dest",             CFG_STRING,      &config.dest_dir           },
	{ "Exclude",          CFG_STRING_LIST, &config.exclude_path       },
	{ "TrashBin",         CFG_STRING,      &config.trashbin_dir       },
	{ "Simulate",         CFG_BOOL,        &config.simulate           },
	{ "Force",            CFG_BOOL,        &config.force              },
	{ "InodeCtimeCheck",  CFG_BOOL,        &config.inode_ctime_check  },
	{ "FileSizeCheck",    CFG_BOOL,        &config.file_size_check    },
	{ "IgnoreMinorError", CFG_BOOL,        &config.ignore_minor_error },
	{ "SaveExclude",      CFG_BOOL,        &config.save_exclude_path  },
	{ "SuppressProgress", CFG_BOOL,        &config.suppress_progress  },
	{ "Quiet",            CFG_BOOL,        &config.quiet              },
	{ "Verbose",          CFG_BOOL,        &config.verbose            },
	{ NULL, CFG_END, NULL }
};


/* --- MAIN FUNCTIONS --- */

int main(int argc, char *argv[])
{
#ifdef ENABLE_NLS
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
#endif

	if (argc < 2) {
		put_usage_and_exit();
	}

	set_default_config();

	optParseOptions(&argc, argv, option_table, 0);
	if (argc > 3) {
		err_printf(CAN_PROCEED, NULL, _("Too many command line arguments.\n"));
	}

	if (command_line_option.config_file != NULL) {
		if (cfgParse(command_line_option.config_file, config_table, CFG_SIMPLE) == -1) {
			err_printf(ALWAYS_EXIT, _("\nAn error has occured while parsing configuration file `%s'.\n"), command_line_option.config_file);
		}
		if (argc > 2) {
			config.source_dir = str_dup(argv[1]);
			config.dest_dir = str_dup(argv[2]);
		}
	} else {
		if (argc < 2) {
			err_printf(ALWAYS_EXIT, NULL, _("Source and destination directory are not specified.\n"));
		}
		if (argc < 3) {
			err_printf(ALWAYS_EXIT, NULL, _("Destination directory is not specified.\n"));
		}
		config.source_dir = str_dup(argv[1]);
		config.dest_dir = str_dup(argv[2]);
	}
	parse_linear_exclude_path();

	copy_command_line_option_to_config();

	prepare_exclude_path();
	prepare_trashbin_dir();
	prepare_source_dir();
	prepare_dest_dir();

	copy_procedure();

        return (0);
}


void err_printf(ProcessOnError proc, const char *func_name, const char *format, ...)
{
	va_list ap;

	if (config.quiet || command_line_option.quiet) {
		if (proc == ALWAYS_EXIT || !config.ignore_minor_error || !command_line_option.ignore_minor_error) {
			exit(1);
		}
		return;
	}

	if (!config.suppress_progress) {
		printf("\n");
	}

	if (func_name != NULL) {
		perror(func_name);
	}
	if (format == NULL) {
		if (proc == ALWAYS_EXIT || !config.ignore_minor_error || !command_line_option.ignore_minor_error) {
			exit(1);
		}
		return;
	}
	va_start(ap, format);
	vfprintf(stderr, format, ap);
	va_end(ap);

	if (proc == ALWAYS_EXIT || !config.ignore_minor_error || !command_line_option.ignore_minor_error) {
		exit(1);
	}
}


void internal_error(const char *file, int line)
{
	fprintf(stderr, _("Internal error has occurred in %s at line %d.\n"), file, line);
	fprintf(stderr, _("It must be a bug. Please send a bug report to the author.\n"));
	exit(1);
}


/* --- PRIVATE FUNCTIONS --- */

static void put_usage_and_exit(void)
{
	printf(_("Backup Copy %s Copyright (C) 1998 Kevin Lindsay <klindsay@mkintraweb.com>\n"
		 "            %*s Copyright (C) 2001 Yuuki NINOMIYA <gm@debian.or.jp>\n\n"), VERSION, strlen(VERSION), "");
	printf(_("This program comes with ABSOLUTELY NO WARRANTY, to the extent permitted by law.\n"
		 "You may redistribute it under the terms of the GNU General Public License.\n"
		 "For more information about these matters, see the file named COPYING.\n\n"));
	printf(_("usage: cpbk [OPTIONS] SOURCE DEST\n"
		 "   or: cpbk [OPTIONS] -c FILE [SOURCE] [DEST]\n\n"));

	printf(_("options:\n"));
	printf(_("  -c, --config=FILE         Specify a configuration file.\n"));
	printf(_("  -e, --exclude=\"FILE,...\"  Exclude specified directories and/or files.\n"
		 "                            Excluded directories or files will be not removed\n"
		 "                            from destination directory.\n"));
	printf(_("  -t, --trash-bin=DIR       Move all overwritten or removed files from the last\n"
		 "                            backup to the specified trash bin directory.\n"));
	printf(_("  -l, --list                Display a listing of files needing update.\n"));
	printf(_("  -n, --nlist               Display the number of files needing update.\n"));
	printf(_("  -s, --simulate            Perform ordering simulation. (not actually change)\n"));
	printf(_("  -f, --force               Copy all files whether updated or not.\n"));
	printf(_("  -i, --inode-ctime         Enable using ctime that is changed by writing or\n"
		 "                            by setting inode information for file comparison\n"
		 "                            check.\n"));
	printf(_("  -I, --ignore-minor-error  Ignore minor error that process can proceed.\n"));
	printf(_("  -q, --quiet               Quiet mode. Suppress all error messages and\n"
                 "                            progress information.\n"));
	printf(_("  -p, --suppress-progress   Suppress progress information.\n"));
	printf(_("  -v, --verbose             Verbose mode. Display processed file names.\n"));
	printf(_("  -h, --help                Display this help and exit.\n"));
	printf(_("  -V, --version             Display version information and exit.\n\n"));

	exit(0);
}


static void put_version_and_exit(void)
{
	printf(_("Backup Copy %s\n"), VERSION);
	printf(_("Copyright (C) 1998 Kevin Lindsay <klindsay@mkintraweb.com>\n"));
	printf(_("Copyright (C) 2001 Yuuki NINOMIYA <gm@debian.or.jp>\n\n"));
	printf(_("This program comes with ABSOLUTELY NO WARRANTY, to the extent permitted by law.\n"
		 "You may redistribute it under the terms of the GNU General Public License.\n"
		 "For more information about these matters, see the file named COPYING.\n\n"));

        exit(0);
}


static void set_default_config(void)
{
	/* STRING */
	config.source_dir         = NULL;
	config.dest_dir           = NULL;
	config.trashbin_dir       = NULL;

	/* LINKED LIST */
	config.exclude_path = NULL;

	/* BOOL */
	config.simulate           = FALSE;
	config.inode_ctime_check  = FALSE;
	config.file_size_check    = TRUE;
	config.save_exclude_path  = TRUE;
	config.ignore_minor_error = FALSE;
	config.force              = FALSE;
	config.quiet              = FALSE;
	config.suppress_progress  = FALSE;
	config.verbose            = FALSE;


	/* STRING */
	command_line_option.config_file         = NULL;
	command_line_option.linear_exclude_path = NULL;
	command_line_option.trashbin_dir        = NULL;

	/* BOOL */
	command_line_option.list               = FALSE;
	command_line_option.nlist              = FALSE;
	command_line_option.simulate           = FALSE;
	command_line_option.inode_ctime_check  = FALSE;
	command_line_option.ignore_minor_error = FALSE;
	command_line_option.force              = FALSE;
	command_line_option.quiet              = FALSE;
	command_line_option.suppress_progress  = FALSE;
	command_line_option.verbose            = FALSE;
}


static void copy_command_line_option_to_config(void)
{
	if (command_line_option.trashbin_dir != NULL) config.trashbin_dir = command_line_option.trashbin_dir;
	if (command_line_option.simulate)           config.simulate           = TRUE;
	if (command_line_option.inode_ctime_check)  config.inode_ctime_check  = TRUE;
	if (command_line_option.ignore_minor_error) config.ignore_minor_error = TRUE;
	if (command_line_option.force)              config.force              = TRUE;
	if (command_line_option.quiet)              config.quiet              = TRUE;
	if (command_line_option.suppress_progress)  config.suppress_progress  = TRUE;
	if (command_line_option.verbose)            config.verbose            = TRUE;
}
