#ifndef SINFO_COLTILT_H
#define SINFO_COLTILT_H
/*
 * This file is part of the ESO SINFONI Pipeline
 * Copyright (C) 2004,2005 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */
/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: sinfo_coltilt.h,v 1.4 2007-06-06 07:10:45 amodigli Exp $"
*
* who       when      what
* --------  --------  ----------------------------------------------
* schreib  20/12/00  created
*/

/************************************************************************
 * sinfo_coltilt.h
 * routines to calculate and correct the spatial tilt of spectra in raw images
 *----------------------------------------------------------------------
 */

/*
 * header files
 */
#include <cpl.h> 
#include "sinfo_msg.h"
#include "sinfo_recipes.h"
#include "sinfo_wave_calibration.h"
#include "sinfo_spiffi_types.h"
/*----------------------------------------------------------------------------
 *                        Function ANSI C prototypes
 *--------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------
   @name   sinfo_new_slope_of_spectrum()
   @param  ns_image   image with at least one continuum spectrum of a pinhole
   @param  box_length width of the box in which the lines are fit by a Gaussian
   @param  fwhm       first guess of the full width at half maximum
   @param  min_amplitude_factor factor peak/background below given threshold 
           the fit is not carried through
   @return slope of a straight line fitted to the spectrum. -FLT_MAX if 
           something went wrong.
   @doc    determines the sub-pixel shifts of each row by using an image with 
           at least one continuum spectrum of a pinhole this is done by 
           searching the spectrum within the image then fitting the spectrum 
           along the rows within a given box by a Gaussian, so that the exact 
           position is determined for each row. Afterwards, a straight line is 
           fitted through the fitted positions. The slope of this linear fit 
           is returned.
*/

float 
sinfo_new_slope_of_spectrum(cpl_image * ns_image,
                       int        box_length,
                       float      fwhm,
                       float      min_amplitude_factor ) ;

/**
   @name sinfo_new_shift_rows()
   @param image   raw image in which the rows should be shifted
   @param slope   slope of a fitted straight line along a spectrum
                  output of sinfo_slopeOfSpectrum
   @param n_order order of the interpolation polynom
   @return resulting image with shifted rows
   @doc   shifts the rows of a raw image by using the output of
          sinfo_slopeOfSpectrum and applying polynomial interpolation
*/


cpl_image * 
sinfo_new_shift_rows(cpl_image * image,
             float      slope,
             int        n_order ) ;


/**
   @name  sinfo_new_parameter_to_ascii()
   @memo stores parameters in an ASCII file
   @param parameter float parameter array to be stored in an ASCII file
   @param number number of parameters
   @param filename filename of ASCII file
   @return void
*/


void 
sinfo_new_parameter_to_ascii ( float * parameter,
                        int n,
                        char * filename ) ;


/**
   @name  sinfo_new_ascii_to_parameter()
   @param filename filename of ASCII file
   @param n number of parameters in the parameter array array of parameters
   @doc writes parameters stored in an ASCII file in an float array
*/

float * 
sinfo_new_ascii_to_parameter ( char * filename,
                           int  * n ) ;

/**
   @name  sinfo_new_curvature_of_spectrum()
   @param ns_image   image with at least one continuum spectrum of a pinhole
   @param order      order of the fit polynomial
   @param box_length width of the box in which the lines are fit by a Gaussian
   @param left_pos
   @param right_pos left and right positions between which the spectrum should 
                    be located
   @param fwhm      first guess of the full width at half maximum
   @param min_amplitude_factor factor peak/background below given
                       threshold the fit is not carried through
   @return resulting polynomial coefficients.
   @doc    this routine determines the curvature of a spectrum by fitting
           a polynomial to a continuum spectrum. This is done by using
       an image with at least one continuum spectrum of a pinhole.
       this is done by searching the spectrum within the image
       then fitting the spectrum along the rows within a given box
       by a sinfo_gaussian, so that the exact position is determined for
       each row. Afterwards, a polynomial is fitted through the
       found positions. The polynomial coefficients are returned.
*/

double * 
sinfo_new_curvature_of_spectrum(cpl_image * ns_image,
                int        order, 
             int        box_length,
             int        left_pos,
                         int        right_pos,
                         float      fwhm,
                         float      min_amplitude_factor ) ;

/**
@name sinfo_new_image_warp_fits
@memo correct optical distortions
@param image distorted image
@param kernel_type type of kernel to correct distortions
@param poly_table table containing distortion coefficients
@return image distortion corrected
*/
cpl_image * 
sinfo_new_image_warp_fits(cpl_image * image,
                      char      * kernel_type, 
                      char      * poly_table );

#endif /*!SINFO_COLTILT_H*/

