// Description:
//
// Copyright (C) 2001 Frank Becker
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation;  either version 2 of the License,  or (at your option) any  later
// version.
//
// This program is distributed in the hope that it will be useful,  but  WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
//
#include <Trace.hpp>
#include <Timer.hpp>
#include <Random.hpp>

#include <Hero.hpp>
#include <Enemy.hpp>
#include <Constants.hpp>

#include <SoftSurface.hpp>

const float GAME_STEP_SIZE = 1.0f/50.0f;
const int MAX_GAME_STEPS = 10;

PuckMaze *maze;
Hero *hero;

int numEnemies = -1;
Enemy **enemies;
bool done = false;

bool keys[ SDLK_LAST];

void updateInput( void)
{
    static bool autoOn = false;
    SDL_Event event;
    while( SDL_PollEvent( &event ) ) 
    {
	bool isDown = false;
	switch( event.type ) 
	{
	    case SDL_KEYDOWN:
	        switch( event.key.keysym.sym)
		{
		    case SDLK_F1:
			autoOn = !autoOn;
			break;

		    case SDLK_F12:
		        SurfaceS()->Snapshot();
			break;

		    default:
			break;
		}
		isDown = true;
		//fall through
	    case SDL_KEYUP:
		keys[ event.key.keysym.sym] = isDown;
		break;

	    default:
		break;
	}
    }

    if( keys[ SDLK_ESCAPE])
    {
	done = true;
    }

    if( keys[ SDLK_x])
    {
	LOG_INFO << hero->X() << "," << hero->Y() << endl;
    }

    if( keys[ SDLK_d])
    {
	maze->Draw();
    }

    if( autoOn || keys[ SDLK_p])
    {
	hero->Track( POINT);
    }
    if( keys[ SDLK_e])
    {
	hero->Track( ENEMY);
    }
    if( keys[ SDLK_b])
    {
	hero->Track( POWERPOINT);
    }

    static int lastHDir = WallLT;
    if( keys[ SDLK_LEFT] ^ keys[ SDLK_RIGHT])
    {
	if( keys[ SDLK_LEFT])
	{
	    hero->Move( WallLT);
	    lastHDir = WallLT;
	}
	if( keys[ SDLK_RIGHT])
	{
	    hero->Move( WallRT);
	    lastHDir = WallRT;
	}
    }
    else if( hero->X()&3)
    {
	//align with grid
	hero->Move( lastHDir);
    }

    static int lastVDir = WallUP;
    if( keys[ SDLK_UP] ^ keys[ SDLK_DOWN])
    {
	if( keys[ SDLK_UP])
	{
	    hero->Move( WallUP);
	    lastVDir = WallUP;
	}
	if( keys[ SDLK_DOWN])
	{
	    hero->Move( WallDN);
	    lastVDir = WallDN;
	}
    }
    else if( hero->Y()&3)
    {
	//align with grid
	hero->Move( lastVDir);
    }
}

void updateEnemies( void)
{
    for( int i=0; i<numEnemies; i++)
    {
	if( enemies[ i] != NULL)
	{
	    if( enemies[ i]->Update( hero->X(), hero->Y(), hero->Power()))
	    {
		hero->Weaken( enemies[ i]->Power());

		if( hero->isAlive())
		{
		    delete enemies[ i];
		    enemies[ i] = NULL;
		}
		else
		{
//			    PlaySound( 0);
		}
	    }
	}
    }
}

void updateVideo( float /*frameFraction*/)
{
    for( int i=0; i<numEnemies; i++)
    {
	if( enemies[ i] != NULL)
	{
	    enemies[ i]->Remove();
	    enemies[ i]->Add();
	}
    }
    //ignore interpolation for now
    SurfaceS()->Flip();
}

int main( int argc, char *argv[])
{
    int i;

    for( i=1; i<argc; i++)
    {
	if( strcmp( argv[ i], "-e") == 0)
	{
	    i++;
	    if( i==argc)
	    {
		return 0;
	    }
	    sscanf( argv[ i], "%d", &numEnemies);
	}
    }

    BaseSurface::createSurface( SURFACE_SDL_SOFT, 640, 480);

    for( i=0; i<SDLK_LAST; i++)
    {
	keys[ i] = false;
    }

    SurfaceS()->SetColor( HEROCOLOR, 50,50,0);
    SurfaceS()->SetColor( BGCOLOR, 0,0,0);
    SurfaceS()->SetColor( POINTCOLOR, 40,40,50);
    SurfaceS()->SetColor( WALLCOLOR, 20,20,40);
    SurfaceS()->SetColor( ENEMYCOLOR, 63,0,0);
    SurfaceS()->SetColor( PPOINTCOLOR, 0,50,0);

    SurfaceS()->SetColor( DARKGREEN, 0,20,0);
    SurfaceS()->SetColor( MEDIUMGREEN, 0,40,0);
    SurfaceS()->SetColor( LIGHTGREEN, 0,60,0);

    SurfaceS()->SetColor( DARKYELLOW, 20,20,0);
    SurfaceS()->SetColor( MEDIUMYELLOW, 40,40,0);
    SurfaceS()->SetColor( LIGHTYELLOW, 60,60,0);

    SurfaceS()->SetColor( DARKRED, 20,0,0);
    SurfaceS()->SetColor( MEDIUMRED, 40,0,0);
    SurfaceS()->SetColor( LIGHTRED, 60,0,0);

    maze = new PuckMaze( SurfaceS()->Width()/4-5, SurfaceS()->Height()/4-5);
    //one powerpoint per enemy

    if( numEnemies < 0)
    {
	numEnemies = (maze->Width()*maze->Height()) / 1700;
    }

    maze->AddPowerpoints( numEnemies);
    maze->Draw();

    //enter hero into maze...
    hero = new Hero( maze);

    //enter enemies into maze...
    enemies = new Enemy*[ numEnemies];// = new EE[10]; //numEnemies]);
    for( i=0; i<numEnemies; i++)
    {
	enemies[ i] = new Enemy( maze);
	enemies[ i]->SetPosition( 
		(Random::random() % maze->Width()) <<2, 
		(Random::random() % maze->Height()) <<2);

	enemies[ i]->Add();
    }

    hero->SetPosition( (maze->Width()>>1)<<2, (maze->Height()>>1)<<2);
    hero->Add();
    hero->Check();

    updateVideo( 0.0f);
    //sleep a few seconds, so you don't die before your monitor had
    //a chance to switch video modes...
    SDL_Delay( 3000);

    float startOfStep = Timer::getTime();
    float gameTime = startOfStep;
    while( !done)
    {
	//update game logic
	int stepCount = 0;
	float currentTime = Timer::getTime();
	while( (currentTime - startOfStep) > GAME_STEP_SIZE)
	{
	    updateInput();
	    updateEnemies();
	    hero->Update();

	    if( !hero->isAlive()) done = true;
	    if( maze->Points() <= 0) done = true;

	    //advance to next start-of-game-step point in time
	    startOfStep += GAME_STEP_SIZE;
	    currentTime = Timer::getTime();
	    stepCount++;
	    if( stepCount > MAX_GAME_STEPS) break;
	}

	float frameFraction = (currentTime - startOfStep) / GAME_STEP_SIZE;
	updateVideo( frameFraction);
    }
    gameTime = Timer::getTime() - gameTime;

    for( i=0; i<numEnemies; i++)
    {
	if( enemies[ i] != NULL)
	{
	    delete enemies[ i];
	}
    }
    delete[] enemies;
    delete SurfaceS();

    int pucksEaten =  maze->Width() * maze->Height() - maze->Points();
    if( maze->Points())
    {
	LOG_INFO << "Well, you tried...\n";
	LOG_INFO << "You cleaned out " 
	         << 100 - maze->Points() *100/ (maze->Width() * maze->Height())
		 << "% of the points (" << pucksEaten
		 << "/" << maze->Width() * maze->Height() << ")\n";
    }
    else
    {
	LOG_INFO << "Wow, you sucked up all "
	         << maze->Width() * maze->Height()
		 << " points!!!\n";
    }
    LOG_INFO << "Your PPS (pucks per second) is: " 
             << (float)pucksEaten/gameTime << endl;

    delete hero;
    delete maze;

    return 0;
}

