/*
 * UnixCW - Unix CW (Morse code) training program
 * Copyright (C) 2001, 2002  Simon Baldwin (simonb@caldera.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * cwlib.h - Header and definitions for the CW library.
 *
 */

#ifndef _CWLIB_H
#define	_CWLIB_H

/* Needed include for struct timeval. */
#include <sys/time.h>

/* Limits on values of CW send and timing parameters */
#define	CW_MIN_SPEED		4	/* Lowest WPM allowed */
#define	CW_MAX_SPEED		60	/* Highest WPM allowed */
#define	CW_MIN_FREQUENCY	0	/* Lowest tone allowed (0=silent) */
#define	CW_MAX_FREQUENCY	4000	/* Highest tone allowed */
#define	CW_MIN_VOLUME		0	/* Quietest volume allowed (0=silent) */
#define	CW_MAX_VOLUME		100	/* Loudest volume allowed */
#define	CW_MIN_GAP		0	/* Lowest extra gap allowed */
#define	CW_MAX_GAP		20	/* Highest extra gap allowed */
#define	CW_MIN_TOLERANCE	0	/* Lowest receive tolerance allowed */
#define	CW_MAX_TOLERANCE	90	/* Highest receive tolerance allowed */
#define	CW_MIN_WEIGHTING	-50	/* Lowest weighting allowed */
#define	CW_MAX_WEIGHTING	50	/* Highest weighting allowed */

/*
 * Representation characters for Dot and Dash.  Only the following
 * characters are permitted in Morse representation strings.
 */
#define	CW_DOT_REPRESENTATION	'.'
#define	CW_DASH_REPRESENTATION	'-'

/* Debug levels definitions. */
#define	CW_DEBUG_SILENT		0x0001	/* Suppresses KIOCSOUND ioctls */
#define	CW_DEBUG_KEYING		0x0002	/* Print out keying control data */
#define	CW_DEBUG_SOUND		0x0004	/* Print out tone generation data */
#define	CW_DEBUG_TONE_QUEUE	0x0008	/* Print out tone queue data */
#define	CW_DEBUG_PARAMETERS	0x0010	/* Print out timing parameters */
#define	CW_DEBUG_RECEIVE_STATES	0x0020	/* Print out receive state changes */
#define	CW_DEBUG_KEYER_STATES	0x0040	/* Print out keyer information */
#define	CW_DEBUG_STRAIGHTKEY	0x0080	/* Print out straight key information */
#define	CW_DEBUG_LOOKUPS	0x0100	/* Print out table lookup results */
#define	CW_DEBUG_PTT		0x0200   /* print out ptt informations */ 
#define	CW_DEBUG_MASK		0x03FF	/* Bit mask of used debug bits */

/* CW library function return status codes. */
#define	CWLIB_SUCCESS		0
#define	CWLIB_ERROR		-1

/* Enable use with C++. */
#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

/* CW library function prototypes. */
extern	int	cw_version ();
extern	void	cw_license ();
extern	void	cw_set_debug_flags (int new_value);
extern	int	cw_get_debug_flags ();
extern	void	cw_get_speed_limits (int *min_speed, int *max_speed);
extern	void	cw_get_frequency_limits (int *min_frequency,
	       						int *max_frequency);
extern	void	cw_get_volume_limits (int *min_volume, int *max_volume);
extern	void	cw_get_gap_limits (int *min_gap, int *max_gap);
extern	void	cw_get_tolerance_limits (int *min_tolerance,
							int *max_tolerance);
extern	int	cw_set_send_speed (int new_value);
extern	int	cw_set_receive_speed (int new_value);
extern	int	cw_set_frequency (int new_value);
extern	int	cw_set_volume (int new_value);
extern	int	cw_set_gap (int new_value);
extern	int	cw_set_tolerance (int new_value);
extern  int 	cw_set_weighting (int new_value);
extern	int	cw_get_send_speed ();
extern	int	cw_get_receive_speed ();
extern	int	cw_get_frequency ();
extern	int	cw_get_volume ();
extern	int	cw_get_gap ();
extern	int	cw_get_tolerance ();
extern	void	cw_set_console_file (char *new_value);
extern	char	*cw_get_console_file ();
extern	void	cw_set_soundcard_file (char *new_value);
extern	char	*cw_get_soundcard_file ();
extern	void	cw_set_soundmixer_file (char *new_value);
extern	char	*cw_get_soundmixer_file ();
extern	int	cw_soundcard_possible ();
extern	int	cw_console_possible ();
extern	void	cw_enable_adaptive_receive ();
extern	void	cw_disable_adaptive_receive ();
extern	int	cw_get_adaptive_receive_state ();
extern	int	cw_set_noise_spike_threshold (int threshold);
extern	int	cw_get_noise_spike_threshold ();
extern	void	cw_enable_iambic_curtis_mode_b ();
extern	void	cw_disable_iambic_curtis_mode_b ();
extern	int	cw_get_iambic_curtis_mode_b_state ();
extern	void	cw_get_send_parameters	(int *dot_usecs, int *dash_usecs,
					 int *end_of_element_usecs,
					 int *end_of_character_usecs,
					 int *end_of_word_usecs,
					 int *additional_usecs,
					 int *adjustment_usecs);
extern	void	cw_get_receive_parameters
					(int *dot_usecs, int *dash_usecs,
					 int *dot_min_usecs,
					 int *dot_max_usecs,
					 int *dash_min_usecs,
					 int *dash_max_usecs,
					 int *end_of_element_min_usecs,
					 int *end_of_element_max_usecs,
					 int *end_of_character_min_usecs,
					 int *end_of_character_max_usecs,
					 int *adaptive_threshold);
extern	void	cw_set_console_sound (int sound_state);
extern	int	cw_get_console_sound ();
extern	void	cw_set_soundcard_sound (int sound_state);
extern	int	cw_get_soundcard_sound ();
extern	void	cw_keying_callback (void (*callback_func) (int));
extern  void	cw_ptt_callback(void (*callback_func) (int));
extern  void	cw_ptt_control(int requested_ptt_state);
extern	int	cw_tone_queue_low_callback (void (*callback_func) (void),
								int level);
extern	void	cw_block_callback (int block);
extern	int	cw_tone_busy ();
extern	int	cw_tone_wait ();
extern	int	cw_tone_queue_wait ();
extern	int	cw_tone_queue_wait_critical (int level);
extern	int	cw_tone_queue_full ();
extern	int	cw_get_tone_queue_capacity ();
extern	int	cw_get_tone_queue_length ();
extern	void	cw_flush_tone_queue ();
extern	int	cw_queue_tone (int usec, int frequency);
extern	int	cw_send_dot ();
extern	int	cw_send_dash ();
extern	int	cw_send_character_space ();
extern	int	cw_send_word_space ();
extern	int	cw_get_character_count ();
extern	void	cw_list_characters (unsigned char *list);
extern	int	cw_get_maximum_representation_length ();
extern	int	cw_lookup_character (unsigned char c, char *representation);
extern	int	cw_check_representation (const char *representation);
extern	int	cw_lookup_representation (const char *representation,
							unsigned char *c);
extern	int	cw_send_representation (const char *representation);
extern	int	cw_send_representation_partial (const char *representation);
extern	int	cw_check_character (unsigned char c);
extern	int	cw_send_character (unsigned char c);
extern	int	cw_send_character_partial (unsigned char c);
extern	int	cw_check_string (const unsigned char *string);
extern	int	cw_send_string (const unsigned char *string);
extern	int	cw_start_receive_tone (const struct timeval *timestamp);
extern	int	cw_end_receive_tone (const struct timeval *timestamp);
extern	int	cw_receive_buffer_dot  (const struct timeval *timestamp);
extern	int	cw_receive_buffer_dash (const struct timeval *timestamp);
extern	int	cw_receive_representation (const struct timeval *timestamp,
					char *representation, int *end_of_word,
					int *error_flag);
extern	int	cw_receive_character (const struct timeval *timestamp,
					unsigned char *c, int *end_of_word,
					int *error_flag);
extern	void	cw_clear_receive_buffer ();
extern	int	cw_get_receive_buffer_capacity ();
extern	int	cw_get_receive_buffer_length ();
extern	int	cw_keyer_paddle_event (int dot_paddle_state,
						int dash_paddle_state);
extern	int	cw_keyer_dot_paddle_event  (int dot_paddle_state);
extern	int	cw_keyer_dash_paddle_event (int dash_paddle_state);
extern	void	cw_get_keyer_paddles (int *dot_paddle_state,
						int *dash_paddle_state);
extern	void	cw_get_keyer_paddle_latches (int *dot_paddle_latch_state,
						int *dash_paddle_latch_state);
extern	int	cw_keyer_busy ();
extern	int	cw_keyer_element_wait ();
extern	int	cw_keyer_wait ();
extern	int	cw_straightkey_event (int key_state);
extern	int	cw_get_straightkey_state ();
extern	int	cw_straightkey_busy ();
extern	int	cw_self_test ();

#if defined(__cplusplus)
}
#endif /* __cplusplus */
#endif /* _CWLIB_H */
