/* -*- C++ -*-
 * Copyright 2019-2021 LibRaw LLC (info@libraw.org)
 *
 LibRaw uses code from dcraw.c -- Dave Coffin's raw photo decoder,
 dcraw.c is copyright 1997-2018 by Dave Coffin, dcoffin a cybercom o net.
 LibRaw do not use RESTRICTED code from dcraw.c

 LibRaw is free software; you can redistribute it and/or modify
 it under the terms of the one of two licenses as you choose:

1. GNU LESSER GENERAL PUBLIC LICENSE version 2.1
   (See file LICENSE.LGPL provided in LibRaw distribution archive for details).

2. COMMON DEVELOPMENT AND DISTRIBUTION LICENSE (CDDL) Version 1.0
   (See file LICENSE.CDDL provided in LibRaw distribution archive for details).

 */

#include "../../internal/dcraw_defs.h"

/*
   All matrices are from Adobe DNG Converter unless otherwise noted.
 */
int LibRaw::adobe_coeff(unsigned make_idx, const char *t_model,
                        int internal_only)
{
  // clang-format off
  static const struct
  {
	  unsigned m_idx;
	  const char *prefix;
	  int t_black, t_maximum, trans[12];
  } table[] = {
    { LIBRAW_CAMERAMAKER_Agfa, "DC-833m", 0, 0,
      { 11438,-3762,-1115,-2409,9914,2497,-1227,2295,5300 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Apple, "QuickTake", 0, 0,
      { 21392,-5653,-3353,2406,8010,-415,7166,1427,2078 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Broadcom, "RPi IMX219", 66, 0x3ff,
      { 5302,1083,-728,-5320,14112,1699,-863,2371,5136 } }, /* LibRaw */
    { LIBRAW_CAMERAMAKER_Broadcom, "RPi OV5647", 16, 0x3ff,
      { 12782,-4059,-379,-478,9066,1413,1340,1513,5176 } }, /* DJC */
	{ LIBRAW_CAMERAMAKER_Broadcom, "Pi", 16, 0x3ff,
	  { 12782,-4059,-379,-478,9066,1413,1340,1513,5176 } }, /* DJC */

#ifdef USE_6BY9RPI // this code normalizes model to LIBRAW_CAMERAMAKER_RaspberryPi
    { LIBRAW_CAMERAMAKER_RaspberryPi, "RP_imx477", 0, 0, // Do not set black, it is set at parser to 256 or 64
      { 5603, -1351, -600, -2872, 11180, 2132, 600, 453, 5821 } }, /* PyDNG */
    { LIBRAW_CAMERAMAKER_RaspberryPi, "RP_imx", 66, 0x3ff,
      { 5302,1083,-728,-5320,14112,1699,-863,2371,5136 } }, /* LibRaw */
    { LIBRAW_CAMERAMAKER_RaspberryPi, "ov5647", 16, 0x3ff,
      { 12782,-4059,-379,-478,9066,1413,1340,1513,5176 } }, /* DJC */
#endif
    { LIBRAW_CAMERAMAKER_Canon, "EOS D30", 0, 0,
      { 9900,-2771,-1324,-7072,14229,3140,-2790,3344,8861 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS D60", 0, 0xfa0,
      { 6211,-1358,-896,-8557,15766,3012,-3001,3507,8567 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 5DS", 0, 0x3c96, // same CMs: 5DS, "5DS R" */
      { 6250,-711,-808,-5153,12794,2636,-1249,2198,5610 } }, // v.2
    { LIBRAW_CAMERAMAKER_Canon, "EOS 5D Mark IV", 0, 0,
      { 6446,-366,-864,-4436,12204,2513,-952,2496,6348 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 5D Mark III", 0, 0x3c80,
      { 6722,-635,-963,-4287,12460,2028,-908,2162,5668 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 5D Mark II", 0, 0x3cf0,
      { 4716,603,-830,-7798,15474,2480,-1496,1937,6651 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 5D", 0, 0xe6c,
      { 6347,-479,-972,-8297,15954,2480,-1968,2131,7649 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 6D Mark II", 0, 0x38de,
      { 6875,-970,-932,-4691,12459,2501,-874,1953,5809 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 6D", 0, 0x3c82,
      { 7034,-804,-1014,-4420,12564,2058,-851,1994,5758 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 77D", 0, 0,
      { 7377,-742,-998,-4235,11981,2549,-673,1918,5538 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 7D Mark II", 0, 0x3510,
      { 7268,-1082,-969,-4186,11839,2663,-825,2029,5839 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 7D", 0, 0x3510,
      { 6844,-996,-856,-3876,11761,2396,-593,1772,6198 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 850D", 0, 0,
      { 9079,-1923,-1236,-4677,12454,2492,-922,2319,5565}},
    { LIBRAW_CAMERAMAKER_Canon, "EOS 800D", 0, 0,
      { 6970,-512,-968,-4425,12161,2553,-739,1982,5601 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 80D", 0, 0,
      { 7457,-671,-937,-4849,12495,2643,-1213,2354,5492 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 10D", 0, 0xfa0,
      { 8250,-2044,-1127,-8092,15606,2664,-2893,3453,8348 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 250D", 0, 0,
      { 9079,-1923,-1236,-4677,12454,2492,-922,2319,5565 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 200D", 0, 0,
      { 7377,-742,-998,-4235,11981,2549,-673,1918,5538 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 20Da", 0, 0,
      { 14155,-5065,-1382,-6550,14633,2039,-1623,1824,6561 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 20D", 0, 0xfff,
      { 6599,-537,-891,-8071,15783,2424,-1983,2234,7462 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 30D", 0, 0,
      { 6257,-303,-1000,-7880,15621,2396,-1714,1904,7046 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 40D", 0, 0x3f60,
      { 6071,-747,-856,-7653,15365,2441,-2025,2553,7315 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 50D", 0, 0x3d93,
      { 4920,616,-593,-6493,13964,2784,-1774,3178,7005 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 60Da", 0, 0x2ff7,
      { 17492,-7240,-2023,-1791,10323,1701,-186,1329,5406 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 60D", 0, 0x2ff7,
      { 6719,-994,-925,-4408,12426,2211,-887,2129,6051 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 70D", 0, 0x3bc7,
      { 7034,-804,-1014,-4420,12564,2058,-851,1994,5758 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 100D", 0, 0x350f,
      { 6602,-841,-939,-4472,12458,2247,-975,2039,6148 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 300D", 0, 0xfa0,
      { 8250,-2044,-1127,-8092,15606,2664,-2893,3453,8348 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 350D", 0, 0xfff,
      { 6018,-617,-965,-8645,15881,2975,-1530,1719,7642 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 3000D", 0, 0,
      { 6939,-1016,-866,-4428,12473,2177,-1175,2178,6162 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 400D", 0, 0xe8e,
      { 7054,-1501,-990,-8156,15544,2812,-1278,1414,7796 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 450D", 0, 0x390d,
      { 5784,-262,-821,-7539,15064,2672,-1982,2681,7427 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 500D", 0, 0x3479,
      { 4763,712,-646,-6821,14399,2640,-1921,3276,6561 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 550D", 0, 0x3dd7,
      { 6941,-1164,-857,-3825,11597,2534,-416,1540,6039 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 600D", 0, 0x3510,
      { 6461,-907,-882,-4300,12184,2378,-819,1944,5931 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 650D", 0, 0x354d,
      { 6602,-841,-939,-4472,12458,2247,-975,2039,6148 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 750D", 0, 0x3c00,
      { 6362,-823,-847,-4426,12109,2616,-743,1857,5635 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 760D", 0, 0x3c00,
      { 6362,-823,-847,-4426,12109,2616,-743,1857,5635 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 700D", 0, 0x3c00,
      { 6602,-841,-939,-4472,12458,2247,-975,2039,6148 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 90D", 0, 0,
      { 11498, -3759, -1516, -5073, 12954,  2349,  -892,  1867,  6118}},
    { LIBRAW_CAMERAMAKER_Canon, "EOS 1000D", 0, 0xe43,
      { 6771,-1139,-977,-7818,15123,2928,-1244,1437,7533 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 1100D", 0, 0x3510,
      { 6444,-904,-893,-4563,12308,2535,-903,2016,6728 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 1200D", 0, 0x37c2,
      { 6461,-907,-882,-4300,12184,2378,-819,1944,5931 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 1300D", 0, 0x37c2,
      { 6939,-1016,-866,-4428,12473,2177,-1175,2178,6162 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS 1500D", 0, 0,
      { 8300,-2110,-1120,-4917,12694,2482,-938,2141,5666 } }, // v.2

    { LIBRAW_CAMERAMAKER_Canon, "EOS RP", 0, 0,
      { 8608,-2097,-1178,-5425,13265,2383,-1149,2238,5680 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS R5", 0, 0,
      { 9766,-2953,-1254,-4276,12116,2433,-437,1336,5131 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS R6", 0, 0,
      { 8293,-1611,-1132,-4759,12711,2275,-1013,2415,5509 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS R", 0, 0,
      { 8293,-1789,-1094,-5025,12925,2327,-1199,2769,6108 } }, // v.2

    { LIBRAW_CAMERAMAKER_Canon, "EOS M6 Mark II", 0, 0,
      { 11498,-3759,-1516,-5073,12954,2349,-892,1867,6118 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS M6", 0, 0,
      { 8532,-701,-1167,-4095,11879,2508,-797,2424,7010 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS M50 Mark II", 0, 0,
      { 10463,-2173,-1437,-4856,12635,2482,-1216,2915,7237 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS M50", 0, 0,
      { 8532,-701,-1167,-4095,11879,2508,-797,2424,7010 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS M5", 0, 0,
      { 8532,-701,-1167,-4095,11879,2508,-797,2424,7010 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS M3", 0, 0,
      { 6362,-823,-847,-4426,12109,2616,-743,1857,5635 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS M200", 0, 0,
      { 10463,-2173,-1437,-4856,12635,2482,-1216,2915,7237 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS M2", 0, 0,
      { 6400,-480,-888,-5294,13416,2047,-1296,2203,6137 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS M100", 0, 0,
      { 8532,-701,-1167,-4095,11879,2508,-797,2424,7010 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS M10", 0, 0,
      { 6400,-480,-888,-5294,13416,2047,-1296,2203,6137 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS M", 0, 0,
      { 6602,-841,-939,-4472,12458,2247,-975,2039,6148 } },

    { LIBRAW_CAMERAMAKER_Canon, "EOS-1Ds Mark III", 0, 0x3bb0,
      { 5859,-211,-930,-8255,16017,2353,-1732,1887,7448 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1Ds Mark II", 0, 0xe80,
      { 6517,-602,-867,-8180,15926,2378,-1618,1771,7633 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1D Mark IV", 0, 0x3bb0,
      { 6014,-220,-795,-4109,12014,2361,-561,1824,5787 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1D Mark III", 0, 0x3bb0,
      { 6291,-540,-976,-8350,16145,2311,-1714,1858,7326 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1D Mark II N", 0, 0xe80,
      { 6240,-466,-822,-8180,15825,2500,-1801,1938,8042 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1D Mark II", 0, 0xe80,
      { 6264,-582,-724,-8312,15948,2504,-1744,1919,8664 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1Ds", 0, 0xe20,
      { 3925,4060,-1739,-8973,16552,2545,-3287,3945,8243 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1D C", 0, 0x3c4e,
      { 6847,-614,-1014,-4669,12737,2139,-1197,2488,6846 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1D X Mark III", 0, 0,
      {  8971, -2022, -1242, -5405, 13249,  2380, -1280,  2483,  6072}},
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1D X Mark II", 0, 0x3c4e,
      { 7596,-978,-967,-4808,12571,2503,-1398,2567,5752 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1D X", 0, 0x3c4e,
      { 6847,-614,-1014,-4669,12737,2139,-1197,2488,6846 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS-1D", 0, 0xe20,
      { 6806,-179,-1020,-8097,16415,1687,-3267,4236,7690 } },
    { LIBRAW_CAMERAMAKER_Canon, "EOS C500", 853, 0,
      { 17851,-10604,922,-7425,16662,763,-3660,3636,22278 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Canon, "IXUS 160", 0, 0,
      { 11657,-3781,-1136,-3544,11262,2283,-160,1219,4700 } }, /* DJC */
    {LIBRAW_CAMERAMAKER_Canon, "PowerShot 600", 0, 0,
      { -3822,10019,1311,4085,-157,3386,-5341,10829,4812,-1969,10969,1126 } },

    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A3300 IS", 0, 0,
      { 10826,-3654,-1023,-3215,11310,1906,0,999,4960 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A460", 0, 0,
      { 6493,-2338,-885,-1589,5934,697,-445,1368,2543 } }, // CHDK
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A470", 0, 0,
      { 12513,-4407,-1242,-2680,10276,2405,-878,2215,4734 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A530", 0, 0,
      { 7252,-2405,-1223,-2102,6560,523,-112,704,3007 } }, // CHDK
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A50", 0, 0,
      { -6233,10706,1825,3260,821,3980,-6512,10745,6287,-2539,12232,262 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A5", 0, 0,
      { -5707,10308,2002,2662,1829,4139,-6265,11063,6033,-2659,11911,593 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A610", 0, 0,
      { 15591,-6402,-1592,-5365,13198,2168,-1300,1824,5075 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A620", 0, 0,
      { 15265,-6193,-1558,-4125,12116,2010,-888,1639,5220 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A630", 0, 0,
      { 14201,-5308,-1757,-6087,14472,1617,-2191,3105,5348 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A640", 0, 0,
      { 13124,-5329,-1390,-3602,11658,1944,-1612,2863,4885 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A650 IS", 0, 0,
      { 9427,-3036,-959,-2581,10671,1911,-1039,1982,4430 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot A720 IS", 0, 0,
      { 14573,-5482,-1546,-1266,9799,1468,-1040,1912,3810 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Canon, "PowerShot D10", 127, 0,
      { 14052,-5229,-1156,-1325,9420,2252,-498,1957,4116 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G10", 0, 0,
      { 11093,-3906,-1028,-5047,12492,2879,-1003,1750,5561 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G11", 0, 0,
      { 12177,-4817,-1069,-1612,9864,2049,-98,850,4471 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G12", 0, 0,
      { 13244,-5501,-1248,-1508,9858,1935,-270,1083,4366 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G15", 0, 0,
      { 7474,-2301,-567,-4056,11456,2975,-222,716,4181 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G16", 0, 0,
      { 8020,-2687,-682,-3704,11879,2052,-965,1921,5556 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G1 X Mark III", 0, 0,
      { 8532,-701,-1167,-4095,11879,2508,-797,2424,7010 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G1 X Mark II", 0, 0,
      { 7378,-1255,-1043,-4088,12251,2048,-876,1946,5805 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G1 X", 0, 0,
      { 7378,-1255,-1043,-4088,12251,2048,-876,1946,5805 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G1", 0, 0,
      { -5686,10300,2223,4725,-1157,4383,-6128,10783,6163,-2688,12093,604 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G2", 0, 0,
      { 9194,-2787,-1059,-8098,15657,2608,-2610,3064,7867 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G3 X", 0, 0,
      { 9701,-3857,-921,-3149,11537,1817,-786,1817,5147 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G3", 0, 0,
      { 9326,-2882,-1084,-7940,15447,2677,-2620,3090,7740 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G5 X Mark II",0, 0,
      { 11629, -5713, -914, -2706, 11090, 1842, -206, 1225, 5515 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G5 X",0, 0,
      { 9602,-3823,-937,-2984,11495,1675,-407,1415,5049 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G5", 0, 0,
      { 9869,-2972,-942,-7314,15098,2369,-1898,2536,7282 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G6", 0, 0,
      { 9876,-3774,-871,-7613,14807,3071,-1448,1305,7485 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G7 X Mark III", 0, 0,
      { 11629, -5713, -914, -2706, 11090, 1842, -206, 1225, 5515 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G7 X Mark II", 0, 0,
      { 9602,-3823,-937,-2984,11495,1675,-407,1415,5049 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G7 X", 0, 0,
      { 9602,-3823,-937,-2984,11495,1675,-407,1415,5049 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G9 X Mark II", 0, 0,
      { 10056,-4131,-944,-2576,11143,1625,-238,1294,5179 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G9 X",0, 0,
      { 9602,-3823,-937,-2984,11495,1675,-407,1415,5049 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot G9", 0, 0,
      { 7368,-2141,-598,-5621,13254,2625,-1418,1696,5743 } },

    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S2 IS", 0, 0,
      { 5477,-1435,-992,-1868,6639,510,-58,792,2670 } }, // CHDK
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S3 IS", 0, 0,
      { 14062,-5199,-1446,-4712,12470,2243,-1286,2028,4836 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S30", 0, 0,
      { 10744,-3813,-1142,-7962,15966,2075,-2492,2805,7744 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S40", 0, 0,
      { 8606,-2573,-949,-8237,15489,2974,-2649,3076,9100 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S45", 0, 0, // +
      { 8251,-2410,-964,-8047,15430,2823,-2380,2824,8119 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S50", 0, 0,
      { 8979,-2658,-871,-7721,15500,2357,-1773,2366,6634 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S60", 0, 0,
      { 8794,-2482,-797,-7804,15403,2572,-1422,1996,7083 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S70", 0, 0,
      { 9976,-3810,-832,-7115,14463,2906,-901,989,7889 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S90", 0, 0,
      { 12374,-5016,-1049,-1677,9902,2078,-83,852,4683 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S95", 0, 0,
      { 13440,-5896,-1279,-1236,9598,1931,-180,1001,4651 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S120", 0, 0,
      { 6961,-1685,-695,-4625,12945,1836,-1114,2152,5518 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S110", 0, 0,
      { 8039,-2643,-654,-3783,11230,2930,-206,690,4194 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot S100", 0, 0,
      { 7968,-2565,-636,-2873,10697,2513,180,667,4211 } },

    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SD300", 0, 0,
      { 6526,-1720,-1075,-1390,5945,602,-90,820,2380 } }, // CHDK

    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX1 IS", 0, 0,
      { 6578,-259,-502,-5974,13030,3309,-308,1058,4970 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX20 IS", 0, 0,
      { 8275,-2904,-1260,-128,5305,505,51,481,2450 } }, // CHDK
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX30 IS", 0, 0,
      { 13014,-4698,-1026,-2001,9615,2386,-164,1423,3759 } }, // CHDK
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX40 HS", 0, 0,
      { 54480,-17404,-8039,-7505,44044,1136,-580,7158,11891 } }, // CHDK
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX50 HS", 0, 0,
      { 12432,-4753,-1247,-2110,10691,1629,-412,1623,4926 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX60 HS", 0, 0,
      { 13161,-5451,-1344,-1989,10654,1531,-47,1271,4955 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX70 HS", 0, 0,
      { 18285,-8907,-1951,-1845,10688,1323,364,1101,5139 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX110 IS", 0, 0,
      { 14134,-5576,-1527,-1991,10719,1273,-1158,1929,3581 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX120 IS", 0, 0,
      { 7286,-2242,-1047,41,4401,457,269,684,1864 } }, // CHDK
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX220 HS", 0, 0,
      { 13898,-5076,-1447,-1405,10109,1297,-244,1860,3687 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot SX710 HS", 0, 0,
      { 13161,-5451,-1344,-1989,10654,1531,-47,1271,4955 } },

    { LIBRAW_CAMERAMAKER_Canon, "PowerShot Pro1", 0, 0,
      { 10062,-3522,-1000,-7643,15117,2730,-765,817,7322 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot Pro70", 34, 0,
      { -5106,10695,1576,3820,53,4566,-6497,10736,6701,-3336,11887,1394 } },
    { LIBRAW_CAMERAMAKER_Canon, "PowerShot Pro90", 0, 0,
      { -5912,10768,2288,4612,-989,4333,-6153,10897,5944,-2907,12288,624 } },

    { LIBRAW_CAMERAMAKER_Casio, "EX-F1", 0, 0,
      { 9084,-2016,-848,-6711,14351,2570,-1059,1725,6135 } },
    { LIBRAW_CAMERAMAKER_Casio, "EX-FH100", 0, 0,
      { 12771,-4179,-1558,-2149,10938,1375,-453,1751,4494 } },
    { LIBRAW_CAMERAMAKER_Casio, "EX-S20", 0, 0,
      { 11634,-3924,-1128,-4968,12954,2015,-1588,2648,7206 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Casio, "EX-Z750", 0, 0,
      { 10819,-3873,-1099,-4903,13730,1175,-1755,3751,4632 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Casio, "EX-Z10", 128, 0xfff,
      { 9790,-3338,-603,-2321,10222,2099,-344,1273,4799 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_CINE, "650", 0, 0,
      { 3390,480,-500,-800,3610,340,-550,2336,1192 } },
    { LIBRAW_CAMERAMAKER_CINE, "660", 0, 0,
      { 3390,480,-500,-800,3610,340,-550,2336,1192 } },
	  { LIBRAW_CAMERAMAKER_CINE, "", 0, 0, /* empty camera name*/
      { 20183,-4295,-423,-3940,15330,3985,-280,4870,9800 } },

    { LIBRAW_CAMERAMAKER_Contax, "N Digital", 0, 0xf1e,
      { 7777,1285,-1053,-9280,16543,2916,-3677,5679,7060 } },

    { LIBRAW_CAMERAMAKER_DXO, "ONE", 0, 0,
      { 6596,-2079,-562,-4782,13016,1933,-970,1581,5181 } },

    { LIBRAW_CAMERAMAKER_Epson, "R-D1", 0, 0, // same CMs: R-D1, R-D1s, R-D1x
      { 6827,-1878,-732,-8429,16012,2564,-704,592,7145 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "DBP for GX680", -128, 0x0fff,
      { 12741,-4916,-1420,-8510,16791,1715,-1767,2302,7771 } }, /* temp, copy from S2Pro */

    { LIBRAW_CAMERAMAKER_Fujifilm, "E550", 0, 0,
      { 11044,-3888,-1120,-7248,15167,2208,-1531,2276,8069 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "E900", 0, 0,
      { 9183,-2526,-1078,-7461,15071,2574,-2022,2440,8639 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "F5", 0, 0, // F500EXR/F505EXR; F550EXR
      { 13690,-5358,-1474,-3369,11600,1998,-132,1554,4395 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "F6", 0, 0, // F600EXR/F605EXR; F660EXR
      { 13690,-5358,-1474,-3369,11600,1998,-132,1554,4395 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "F77", 0, 0xfe9, // F770EXR/F775EXR
      { 13690,-5358,-1474,-3369,11600,1998,-132,1554,4395 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "F7", 0, 0, // same CMs: F700, F710EXR
      { 10004,-3219,-1201,-7036,15047,2107,-1863,2565,7736 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "F810", 0, 0,
      { 11044,-3888,-1120,-7248,15167,2208,-1531,2276,8069 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "F8", 0, 0, // F800EXR
      { 13690,-5358,-1474,-3369,11600,1998,-132,1554,4395 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "F900EXR", 0, 0,
      { 12085,-4727,-953,-3257,11489,2002,-511,2046,4592 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "GFX 100", 0, 0, // same CMs: "GFX 100", "GFX 100S"/"GFX100S", "GFX 100 IR"
      { 16212,-8423,-1583,-4336,12583,1937,-195,726,6199 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "GFX 50", 0, 0,  // same CMs: "GFX 50S", "GFX 50R", "GFX 50S II"
      { 11756,-4754,-874,-3056,11045,2305,-381,1457,6006 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "HS10", 0, 0xf68,
      { 12440,-3954,-1183,-1123,9674,1708,-83,1614,4086 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "HS2", 0, 0, // HS20EXR/HS22EXR
      { 13690,-5358,-1474,-3369,11600,1998,-132,1554,4395 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "HS3", 0, 0, // HS30EXR/HS33EXR/HS35EXR
      { 13690,-5358,-1474,-3369,11600,1998,-132,1554,4395 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "HS50EXR", 0, 0,
      { 12085,-4727,-953,-3257,11489,2002,-511,2046,4592 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "IS-1", 0, 0,
      { 21461,-10807,-1441,-2332,10599,1999,289,875,7703 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "IS Pro", 0, 0,
      { 12300,-5110,-1304,-9117,17143,1998,-1947,2448,8100 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "S5000", 0, 0,
      { 8754,-2732,-1019,-7204,15069,2276,-1702,2334,6982 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "S5100", 0, 0,
      { 11940,-4431,-1255,-6766,14428,2542,-993,1165,7421 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "S5200", 0, 0, // S5200/S5600
      { 9636,-2804,-988,-7442,15040,2589,-1803,2311,8621 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "S6", 0, 0, // S6000fd/S6500fd
      { 12628,-4887,-1401,-6861,14996,1962,-2198,2782,7091 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "S7000", 0, 0,
      { 10190,-3506,-1312,-7153,15051,2238,-2003,2399,7505 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "S9000", 0, 0, // S9000/S9500
      { 10491,-3423,-1145,-7385,15027,2538,-1809,2275,8692 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "S9100", 0, 0, // S9100/S9600
      { 12343,-4515,-1285,-7165,14899,2435,-1895,2496,8800 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "S100FS", -514, 0,
      { 11521,-4355,-1065,-6524,13767,3058,-1466,1984,6045 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "S20Pro", 0, 0,
      { 10004,-3219,-1201,-7036,15047,2107,-1863,2565,7736 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "S20", -512, 0x3fff, // S200EXR/S205EXR
      { 11401,-4498,-1312,-5088,12751,2613,-838,1568,5941 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "SL1000", 0, 0,
      { 11705,-4262,-1107,-2282,10791,1709,-555,1713,4945 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "S1", 0, 0,
      { 12297,-4882,-1202,-2106,10691,1623,-88,1312,4790 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "S2Pro", -128, 0,
      { 12741,-4916,-1420,-8510,16791,1715,-1767,2302,7771 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "S3Pro", 0, 0,
      { 11807,-4612,-1294,-8927,16968,1988,-2120,2741,8006 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "S5Pro", 0, 0,
      { 12300,-5110,-1304,-9117,17143,1998,-1947,2448,8100 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "X100F", 0, 0,
      { 11434,-4948,-1210,-3746,12042,1903,-666,1479,5235 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X100S", 0, 0,
      { 10592,-4262,-1008,-3514,11355,2465,-870,2025,6386 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X100T", 0, 0,
      { 10592,-4262,-1008,-3514,11355,2465,-870,2025,6386 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X100V", 0, 0,
      { 13426,-6334,-1177,-4244,12136,2371,580,1303,5980 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X100", 0, 0,
      { 12161,-4457,-1069,-5034,12874,2400,-795,1724,6904 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "X10", 0, 0,
      { 13509,-6199,-1254,-4430,12733,1865,-331,1441,5022 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X20", 0, 0,
      { 11768,-4971,-1133,-4904,12927,2183,-480,1723,4605 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X30", 0, 0,
      { 12328,-5256,-1144,-4469,12927,1675,-87,1291,4351 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X70", 0, 0,
      { 10450,-4329,-878,-3217,11105,2421,-752,1758,6519 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "XF10", 0, 0,
      { 11673,-4760,-1041,-3988,12058,2166,-771,1417,5569 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "XF1", 0, 0,
      { 13509,-6199,-1254,-4430,12733,1865,-331,1441,5022 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "XQ", 0, 0,  // same CMs: XQ1, XQ2
      { 9252,-2704,-1064,-5893,14265,1717,-1101,2341,4349 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "X-Pro1", 0, 0,
      { 10413,-3996,-993,-3721,11640,2361,-733,1540,6011 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-Pro2", 0, 0,
      { 11434,-4948,-1210,-3746,12042,1903,-666,1479,5235 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-Pro3", 0, 0,
      { 13426,-6334,-1177,-4244,12136,2371,580,1303,5980 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "X-A10", 0, 0,
      { 11540,-4999,-991,-2949,10963,2278,-382,1049,5605} },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-A20", 0, 0,
      { 11540,-4999,-991,-2949,10963,2278,-382,1049,5605} },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-A1", 0, 0,
      { 11086,-4555,-839,-3512,11310,2517,-815,1341,5940 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-A2", 0, 0,
      { 10763,-4560,-917,-3346,11311,2322,-475,1135,5843 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-A3", 0, 0,
      { 12407,-5222,-1086,-2971,11116,2120,-294,1029,5284 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-A5", 0, 0,
      { 11673,-4760,-1041,-3988,12058,2166,-771,1417,5569 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-A7", 0, 0,
      { 15055,-7391,-1274,-4062,12071,2238,-610,1217,6147 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-E1", 0, 0,
      { 10413,-3996,-993,-3721,11640,2361,-733,1540,6011 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-E2S", 0, 0,
      { 11562,-5118,-961,-3022,11007,2311,-525,1569,6097 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-E2", 0, 0,
      { 8458,-2451,-855,-4597,12447,2407,-1475,2482,6526 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-E3", 0, 0,
      { 11434,-4948,-1210,-3746,12042,1903,-666,1479,5235 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-E4", 0, 0,
      { 13426, -6334, -1177, -4244, 12136, 2371, -580, 1303, 5980 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-H1", 0, 0,
      { 11434,-4948,-1210,-3746,12042,1903,-666,1479,5235 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-M1", 0, 0,
      { 10413,-3996,-993,-3721,11640,2361,-733,1540,6011 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-S10", 0, 0,
      { 13426,-6334,-1177,-4244,12136,2371,-580,1303,5980 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-S1", 0, 0,
      { 13509,-6199,-1254,-4430,12733,1865,-331,1441,5022 } },

    { LIBRAW_CAMERAMAKER_Fujifilm, "X-T100", 0, 0,
      { 11673,-4760,-1041,-3988,12058,2166,-771,1417,5569 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-T1", 0, 0, /* same CMs: X-T1, "X-T1IR", "X-T1 IR", X-T10 */
      { 8458,-2451,-855,-4597,12447,2407,-1475,2482,6526 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-T200", 0, 0,
      { 15055,-7391,-1274,-4062,12071,2238,-610,1217,6147 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-T2", 0, 0,  // same CMs: X-T2, X-T20
      { 11434,-4948,-1210,-3746,12042,1903,-666,1479,5235 } },
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-T3", 0, 0,   // same CMs: X-T3, X-T30
      { 13426,-6334,-1177,-4244,12136,2371,580,1303,5980 } }, // v.2
    { LIBRAW_CAMERAMAKER_Fujifilm, "X-T4", 0, 0,
      { 13426,-6334,-1177,-4244,12136,2371,580,1303,5980 } },

    { LIBRAW_CAMERAMAKER_GITUP, "G3DUO", 130, 62000,
       { 8489,-2583,-1036,-8051,15583,2643,-1307,1407,7354 } },

    { LIBRAW_CAMERAMAKER_GITUP, "GIT2P", 4160, 0,
      { 8489,-2583,-1036,-8051,15583,2643,-1307,1407,7354 } },
    { LIBRAW_CAMERAMAKER_GITUP, "GIT2", 3200, 0,
      { 8489,-2583,-1036,-8051,15583,2643,-1307,1407,7354 } },

    { LIBRAW_CAMERAMAKER_GoPro, "HERO5 Black", 0, 0,
	    { 10344,-4210,-620,-2315,10625,1948,93,1058,5541 } },

    {LIBRAW_CAMERAMAKER_Hasselblad, "L1D-20c", 0, 0,
      {  7310, -2746,  -646, -2991, 10847,  2469,   163,   585,  6324}},

    { LIBRAW_CAMERAMAKER_Hasselblad, "16-Uncoated-3FR", 0, 0,
      {  8519, -3260,  -280, -5081, 13459,  1738, -1449,  2960,  7809}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "16-Uncoated-FFF", 0, 0,
      {  8068, -2959,  -108, -5788, 13608,  2389, -1002,  2237,  8162}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "16-Uncoated", 0, 0,
      {  8519, -3260,  -280, -5081, 13459,  1738, -1449,  2960,  7809}},

    { LIBRAW_CAMERAMAKER_Hasselblad, "22-Uncoated-3FR", 0, 0,
      {  8523, -3257,  -280, -5078, 13458,  1743, -1449,  2961,  7809}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "22-Uncoated-FFF", 0, 0,
      {  8068, -2959,  -108, -5788, 13608,  2389, -1002,  2237,  8162}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "22-Uncoated", 0, 0,
      {  8519, -3260,  -280, -5081, 13459,  1738, -1449,  2960,  7809}},

    {LIBRAW_CAMERAMAKER_Hasselblad, "31-Uncoated-FFF", 0, 0,
      {  5155, -1201,   200, -5841, 13197,  2950, -1101,  2317,  6988}},
    {LIBRAW_CAMERAMAKER_Hasselblad, "31-Uncoated", 0, 0,
      {  5458, -1448,   145, -4479, 12338,  2401, -1659,  3086,  6710}},

    {LIBRAW_CAMERAMAKER_Hasselblad, "39-Uncoated-3FR", 0, 0,
      {  3904,  -100,   262, -4318, 12407,  2128, -1598,  3594,  6233}},
    {LIBRAW_CAMERAMAKER_Hasselblad, "39-Uncoated-FFF", 0, 0,
      {  4739,  -932,   295, -4829, 12220,  2952, -1027,  2341,  7083}},
    {LIBRAW_CAMERAMAKER_Hasselblad, "39-Uncoated", 0, 0,
      {  3894,  -110,   287, -4672, 12610,  2295, -2092,  4100,  6196}},

    { LIBRAW_CAMERAMAKER_Hasselblad, "39-Coated-3FR", 0, 0,
      {  5427, -1147,   173, -3834, 12073,  1969, -1444,  3320,  5621}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "39-Coated-FFF", 0, 0,
      {  5323, -1233,   399, -4926, 12362,  2894,  -856,  2471,  5961}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "39-Coated", 0, 0,
      {  3857,   452,   -46, -6008, 14477,  1596, -2627,  4481,  5718}},

    {LIBRAW_CAMERAMAKER_Hasselblad, "40-Coated5-3FR", 0, 0,
      {  7014, -2067,  -540, -4821, 13016,  1980, -1663,  3089,  6940}},
    {LIBRAW_CAMERAMAKER_Hasselblad, "40-Coated5-FFF", 0, 0,
      {  5963, -1357,  -172, -5439, 12762,  3007,  -964,  2222,  7172}},
    {LIBRAW_CAMERAMAKER_Hasselblad, "40-Coated5", 0, 0,
      {  6159, -1402,  -177, -5439, 12762,  3007,  -955,  2200,  7104}},

    { LIBRAW_CAMERAMAKER_Hasselblad, "40-Coated-3FR", 0, 0,
      {  6550, -1681,  -399, -4626, 12598,  2257, -1807,  3354,  6486}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "40-Coated-FFF", 0, 0,
      {  6041, -1375,  -174, -5439, 10000,  3007,  -930,  2145,  6923}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "40-Coated", 0, 0,
	    {  6159, -1402,  -177, -5439, 12762,  3007,  -955,  2200,  7104}},

    { LIBRAW_CAMERAMAKER_Hasselblad, "50-Coated5-3FR", 0, 0,
      {  5707,  -693,  -382, -4285, 12669,  1773, -1615,  3519,  5410}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "50-Coated5-FFF", 0, 0,
      {  5263,  -612,    39, -4950, 12426,  2843,  -935,  2423,  5941}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "50-Coated5", 0, 0,
      {  5656,  -659,  -346, -3923, 12306,  1791, -1602,  3509,  5442}},

    { LIBRAW_CAMERAMAKER_Hasselblad, "50-Coated-3FR", 0, 0,
      {  5656,  -659,  -346, -3923, 12305,  1790, -1602,  3509,  5442}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "50-Coated-FFF", 0, 0,
      {  5280,  -614,    39, -4950, 12426,  2843,  -939,  2434,  5968}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "50-Coated", 0, 0,
	    {  5656,  -659,  -346, -3923, 12306, 1791,  -1602,  3509,  5442}},

    { LIBRAW_CAMERAMAKER_Hasselblad, "50-15-Coated5-II-3FR", 0, 0,
      { 10887, -6152,  1034, -3564, 12412,  4224,    63,   626, 10123}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "50-15-Coated5-II-FFF", 0, 0,
      {  4932,  -835,   141, -4878, 11868,  3437, -1138,  1961,  7067}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "50-15-Coated5-II", 0, 0,
      {  8737, -4937,   830, -2860,  9961,  3390,    51,   502,  8124}},

    { LIBRAW_CAMERAMAKER_Hasselblad, "50-15-Coated5", 0, 0,
	    { 4932,-835,141,-4878,11868,3437,-1138,1961,7067 } },

    { LIBRAW_CAMERAMAKER_Hasselblad, "60-Coated-3FR", 0, 0,
      {  9296,   336, -1088, -6442, 14323,  2289, -1433,  2942,  5756}},
    { LIBRAW_CAMERAMAKER_Hasselblad, "60-Coated", 0, 0,
	    {  9662,  -684,  -279, -4903, 12293,  2950,  -344,  1669,  6024}},

    { LIBRAW_CAMERAMAKER_Hasselblad, "100-17-Coated5", 0, 0,
      {  5110, -1357,  -308, -5573, 12835,  3077, -1279,  2025,  7010}},

    { LIBRAW_CAMERAMAKER_HTC, "One A9", 64, 1023,
      { 101,-20,-2,-11,145,41,-24,1,56 } }, /* this is FM1 transposed */

    { LIBRAW_CAMERAMAKER_Imacon, "Ixpress", 0, 0,
      { 7025,-1415,-704,-5188,13765,1424,-1248,2742,6038 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Kodak, "NC2000", 0, 0, // AP Nikon
      { 13891,-6055,-803,-465,9919,642,2121,82,1291 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS315C", -8, 0,
      { 17523,-4827,-2510,756,8546,-137,6113,1649,2250 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS330C", -8, 0,
      { 20620,-7572,-2801,-103,10073,-396,3551,-233,2220 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS420", 0, 0,
      { 10868,-1852,-644,-1537,11083,484,2343,628,2216 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS46", 0, 0, // same CM as EOSDCS1 and DCS465 DB
      { 10592,-2206,-967,-1944,11685,230,2206,670,1273 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS520C", -178, 0, // same CamID: DCS520C, "EOS D2000C"
      { 24542,-10860,-3401,-1490,11370,-297,2858,-605,3225 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS560C", -177, 0, // same CamID: DCS560C, "EOS D6000C"
      { 20482,-7172,-3125,-1033,10410,-285,2542,226,3136 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS620C", -177, 0,
      { 23617,-10175,-3149,-2054,11749,-272,2586,-489,3453 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS620X", -176, 0,
      { 13095,-6231,154,12221,-21,-2137,895,4602,2258 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS660C", -173, 0,
      { 18244,-6351,-2739,-791,11193,-521,3711,-129,2802 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS720X", 0, 0,
      { 11775,-5884,950,9556,1846,-1286,-1019,6221,2728 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS760C", 0, 0,
      { 16623,-6309,-1411,-4344,13923,323,2285,274,2926 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS Pro SLR", 0, 0,
      { 5494,2393,-232,-6427,13850,2846,-1876,3997,5445 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS Pro 14nx", 0, 0,
      { 5494,2393,-232,-6427,13850,2846,-1876,3997,5445 } },
    { LIBRAW_CAMERAMAKER_Kodak, "DCS Pro 14", 0, 0, // same CamID: "DCS Pro 14N", "Photo Control Camerz ZDS 14"
      { 7791,3128,-776,-8588,16458,2039,-2455,4006,6198 } },
    { LIBRAW_CAMERAMAKER_Kodak, "EOSDCS1", 0, 0,
      { 10592,-2206,-967,-1944,11685,230,2206,670,1273 } },
    { LIBRAW_CAMERAMAKER_Kodak, "EOSDCS3", 0, 0,
      { 9898,-2700,-940,-2478,12219,206,1985,634,1031 } },
    { LIBRAW_CAMERAMAKER_Kodak, "ProBack645", 0, 0,
      { 16414,-6060,-1470,-3555,13037,473,2545,122,4948 } },
    { LIBRAW_CAMERAMAKER_Kodak, "ProBack", 0, 0,
      { 21179,-8316,-2918,-915,11019,-165,3477,-180,4210 } },

    {LIBRAW_CAMERAMAKER_Kodak, "PIXPRO AZ901", 0, 0,	// dng
      { 21875, -8006, -2558,   634,  8194,  1104,  1535,   951,  6969}},
    { LIBRAW_CAMERAMAKER_Kodak, "P712", 0, 3963,
      { 9658,-3314,-823,-5163,12695,2768,-1342,1843,6044 } },
    { LIBRAW_CAMERAMAKER_Kodak, "P850", 0, 3964,
      { 10511,-3836,-1102,-6946,14587,2558,-1481,1792,6246 } },
    { LIBRAW_CAMERAMAKER_Kodak, "P880", 0, 3963,
      { 12805,-4662,-1376,-7480,15267,2360,-1626,2194,7904 } },
    { LIBRAW_CAMERAMAKER_Kodak, "Z980", 0, 0,
      { 11313,-3559,-1101,-3893,11891,2257,-1214,2398,4908 } },
    { LIBRAW_CAMERAMAKER_Kodak, "Z981", 0, 0,
      { 12729,-4717,-1188,-1367,9187,2582,274,860,4411 } },
    { LIBRAW_CAMERAMAKER_Kodak, "Z990", 0, 0xfed,
      { 11749,-4048,-1309,-1867,10572,1489,-138,1449,4522 } },
    { LIBRAW_CAMERAMAKER_Kodak, "Z1015", 0, 0xef1,
      { 11265,-4286,-992,-4694,12343,2647,-1090,1523,5447 } },

    {LIBRAW_CAMERAMAKER_Leaf, "AFi 54S", 0, 0,
      {  8236,  1746, -1313, -8251, 15953,  2428, -3672,  5786,  5771}},
    {LIBRAW_CAMERAMAKER_Leaf, "AFi 65S", 0, 0,
      {  7914,  1414, -1190, -8776, 16582,  2280, -2811,  4605,  5562}},
    {LIBRAW_CAMERAMAKER_Leaf, "AFi 75S", 0, 0,
      {  7914,  1414, -1190, -8776, 16582,  2280, -2811,  4605,  5562}},
    {LIBRAW_CAMERAMAKER_Leaf, "Aptus 17", 0, 0,
      {  8236,  1746, -1313, -8251, 15953,  2428, -3672,  5786,  5771}},
    {LIBRAW_CAMERAMAKER_Leaf, "Aptus 22", 0, 0,
      {  8236,  1746, -1313, -8251, 15953,  2428, -3672,  5786,  5771}},
    {LIBRAW_CAMERAMAKER_Leaf, "Aptus 54S", 0, 0,
      {  8236,  1746, -1313, -8251, 15953,  2428, -3672,  5786,  5771}},
    {LIBRAW_CAMERAMAKER_Leaf, "Aptus 65S", 0, 0,
      {  7914,  1414, -1190, -8776, 16582,  2280, -2811,  4605,  5562}},
    {LIBRAW_CAMERAMAKER_Leaf, "Aptus 65", 0, 0,
      {  7914,  1414, -1190, -8776, 16582,  2280, -2811,  4605,  5562}},
    {LIBRAW_CAMERAMAKER_Leaf, "Aptus 75S", 0, 0,
      {  7914,  1414, -1190, -8776, 16582,  2280, -2811,  4605,  5562}},
    {LIBRAW_CAMERAMAKER_Leaf, "Aptus 75", 0, 0,
      {  7914,  1414, -1190, -8776, 16582,  2280, -2811,  4605,  5562}},
    {LIBRAW_CAMERAMAKER_Leaf, "C-Most", 0, 0,
      {  3952,  2188,   449, -6701, 14584,  2275, -4536,  7349,  6535}},
    {LIBRAW_CAMERAMAKER_Leaf, "Credo 40", 0, 0,
      {  8035,   435,  -962, -6001, 13872,  2320, -1159,  3065,  5434}},
    {LIBRAW_CAMERAMAKER_Leaf, "Credo 50", 0, 0, // emb
      { 10325,   845,  -604, -4113, 13385,   481, -1791,  4163,  6924}},
    {LIBRAW_CAMERAMAKER_Leaf, "Credo 60", 0, 0,
      {  8035,   435,  -962, -6001, 13872,  2320, -1159,  3065,  5434}},
    {LIBRAW_CAMERAMAKER_Leaf, "Credo 80", 0, 0,
      {  6294,   686,  -712, -5435, 13417,  2211, -1006,  2435,  5042}},
    {LIBRAW_CAMERAMAKER_Leaf, "Valeo 11", 0, 0,
      {  8236,  1746, -1313, -8251, 15953,  2428, -3672,  5786,  5771}},
    {LIBRAW_CAMERAMAKER_Leaf, "Valeo 17", 0, 0,
      {  8236,  1746, -1313, -8251, 15953,  2428, -3672,  5786,  5771}},
    {LIBRAW_CAMERAMAKER_Leaf, "Valeo 22", 0, 0,
      {  8236,  1746, -1313, -8251, 15953,  2428, -3672,  5786,  5771}},
    {LIBRAW_CAMERAMAKER_Leaf, "Valeo 6", 0, 0,
      {  3952,  2188,   449, -6701, 14584,  2275, -4536,  7349,  6535}},

//     { LIBRAW_CAMERAMAKER_Leaf, "AFi-II 6", 0, 0,
    { LIBRAW_CAMERAMAKER_Leaf, "AFi-II 7", 0, 0,
      { 7691,-108,-339,-6185,13627,2833,-2046,3899,5952 } },
    { LIBRAW_CAMERAMAKER_Leaf, "AFi-II 10", 0, 0,
      { 6719,1147,-148,-6929,14061,3176,-1781,3343,5424 } },

    { LIBRAW_CAMERAMAKER_Leaf, "Aptus-II 5", 0, 0,
      { 7914,1414,-1190,-8777,16582,2280,-2811,4605,5562 } },
    { LIBRAW_CAMERAMAKER_Leaf, "Aptus-II 6", 0, 0,
      { 7989,-113,-352,-6185,13627,2833,-2028,3866,5901 } },
    { LIBRAW_CAMERAMAKER_Leaf, "Aptus-II 7", 0, 0,
      { 8209,-116,-362,-6185,13627,2833,-1962,3740,5709 } },
    { LIBRAW_CAMERAMAKER_Leaf, "Aptus-II 8", 0, 0,
      { 7361,1257,-163,-6929,14061,3176,-1839,3454,5603 } },
    { LIBRAW_CAMERAMAKER_Leaf, "Aptus-II 10R", 0, 0,
      { 7167,1224,-158,-6929,14061,3176,-1826,3429,5562 } },
    { LIBRAW_CAMERAMAKER_Leaf, "Aptus-II 10", 0, 0,
      { 7527,1285,-166,-6929,14061,3176,-1995,3747,6077 } },
//     { LIBRAW_CAMERAMAKER_Leaf, "Aptus-II 12R", 0, 0,
    { LIBRAW_CAMERAMAKER_Leaf, "Aptus-II 12", 0, 0,
      { 7361,1257,-163,-6929,14061,3176,-1695,3182,5162 } },

    { LIBRAW_CAMERAMAKER_Leica, "CL", 0, 0,
      { 7743,-2896,-921,-4211,12271,2169,-697,1562,5491 } },

    { LIBRAW_CAMERAMAKER_Leica, "M8", 0, 0,
      { 7675,-2196,-305,-5860,14119,1856,-2425,4006,6578 } },
    { LIBRAW_CAMERAMAKER_Leica, "M9", 0, 0,
      { 6687,-1751,-291,-3556,11373,2492,-548,2204,7146 } },
    { LIBRAW_CAMERAMAKER_Leica, "M10", 0, 0, // same CMs: M10, M10-D, M10-P
      { 9090,-3342,-740,-4006,13456,493,-569,2266,6871 } },
    { LIBRAW_CAMERAMAKER_Leica, "M (Typ 2", 0, 0, // same CMs: "M (Typ 240)", "M (Typ 262)", "M-D (Typ 262)"
      { 7199,-2140,-712,-4005,13327,649,-810,2521,6673 } },

    { LIBRAW_CAMERAMAKER_Leica, "Q (Typ 116)", 0, 0,
      { 10068,-4043,-1068,-5319,14268,1044,-765,1701,6522 } },
    { LIBRAW_CAMERAMAKER_Leica, "Q2", 0, 0,
      { 12312,-5440,-1307,-6408,15499,824,-1075,1677,7220 } },

    { LIBRAW_CAMERAMAKER_Leica, "SL (Typ 601)", 0, 0,
      { 11865,-4523,-1441,-5423,14458,935,-1587,2687,4830 } },
    { LIBRAW_CAMERAMAKER_Leica, "S (Typ 007)", 0, 0,
      { 6063,-2234,-231,-5210,13787,1500,-1043,2866,6997 } },
    { LIBRAW_CAMERAMAKER_Leica, "S2", 0, 0,
      { 5627,-721,-447,-4423,12456,2192,-1048,2948,7379 } },
    { LIBRAW_CAMERAMAKER_Leica, "S3", 0, 0,
//      { 5147,-1464,-318,-5374,13263,2325,-1425,2918,6450 } },
      { 5092,-1630,-470,-6313,14297,2170,-1603,3135,5982 } },
    {LIBRAW_CAMERAMAKER_Leica, "S", 0, 0, // same CMs: "S-E (Typ 006)", "S (Typ 006)"
      { 5749,-1072,-382,-4274,12432,2048,-1166,3104,7105 } },

    { LIBRAW_CAMERAMAKER_Leica, "TL2", 0, 0,
      { 6375,-2062,-732,-4878,12838,2262,-877,1705,6204 } },
    { LIBRAW_CAMERAMAKER_Leica, "T", 0, 0, // same CMs: TL, "T (Typ 701)"
      { 6295,-1679,-475,-5586,13046,2837,-1410,1889,7075 } },

    { LIBRAW_CAMERAMAKER_Leica, "X2", 0, 0,
      { 8336,-2853,-699,-4425,11989,2760,-954,1625,6396 } },
    { LIBRAW_CAMERAMAKER_Leica, "X1", 0, 0,
      { 9055,-2611,-666,-4906,12652,2519,-555,1384,7417 } },
    { LIBRAW_CAMERAMAKER_Leica, "X", 0, 0, /*  same CMs: "X (Typ 113)", "X-U (Typ 113)", XV, "X Vario (Typ 107)" */
      { 9062,-3198,-828,-4065,11772,2603,-761,1468,6458 } },

    { LIBRAW_CAMERAMAKER_Mamiya,"ZD", 0, 0,
      { 7645,2579,-1363,-8689,16717,2015,-3712,5941,5961 } },

    { LIBRAW_CAMERAMAKER_Micron, "2010", 110, 0,
      { 16695,-3761,-2151,155,9682,163,3433,951,4904 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Minolta, "DiMAGE 5", 0, 0xf7d,
      { 9117,-3063,-973,-7949,15763,2306,-2752,3136,8093 } },
    { LIBRAW_CAMERAMAKER_Minolta, "DiMAGE 7Hi", 0, 0xf7d,
      { 11555,-4064,-1256,-7903,15633,2409,-2811,3320,7358 } },
    { LIBRAW_CAMERAMAKER_Minolta, "DiMAGE 7i", 0, 0xf7d,
      { 11050,-3791,-1199,-7875,15585,2434,-2797,3359,7560 } },
    { LIBRAW_CAMERAMAKER_Minolta, "DiMAGE 7", 0, 0xf7d,
      { 9258,-2879,-1008,-8076,15847,2351,-2806,3280,7821 } },
    { LIBRAW_CAMERAMAKER_Minolta, "DiMAGE A1", 0, 0xf8b,
      { 9274,-2548,-1167,-8220,16324,1943,-2273,2721,8340 } },
    { LIBRAW_CAMERAMAKER_Minolta, "DiMAGE A200", 0, 0,
      { 8560,-2487,-986,-8112,15535,2771,-1209,1324,7743 } },
    { LIBRAW_CAMERAMAKER_Minolta, "DiMAGE A2", 0, 0xf8f,
      { 9097,-2726,-1053,-8073,15506,2762,-966,981,7763 } },
    { LIBRAW_CAMERAMAKER_Minolta, "DiMAGE Z2", 0, 0,
      { 11280,-3564,-1370,-4655,12374,2282,-1423,2168,5396 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Minolta, "DG-5D", 0, 0xffb, // same CamID: "ALPHA 5D", "DYNAX 5D", "MAXXUM 5D", "Alpha Sweet Digital"
      { 10284,-3283,-1086,-7957,15762,2316,-829,882,6644 } },
    { LIBRAW_CAMERAMAKER_Minolta, "DG-7D", 0, 0xffb, // same CamID: "ALPHA 7D", "DYNAX 7D", "MAXXUM 7D"
      { 10239,-3104,-1099,-8037,15727,2451,-927,925,6871 } },

    { LIBRAW_CAMERAMAKER_Motorola, "PIXL", 0, 0,
      { 8898,-989,-1033,-3292,11619,1674,-661,3178,5216 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Nikon, "1 AW1", 0, 0,
      { 6588,-1305,-693,-3277,10987,2634,-355,2016,5106 } },
    { LIBRAW_CAMERAMAKER_Nikon, "1 J3", 0, 0,
      { 6588,-1305,-693,-3277,10987,2634,-355,2016,5106 } },
    { LIBRAW_CAMERAMAKER_Nikon, "1 J4", 0, 0,
      { 5958,-1559,-571,-4021,11453,2939,-634,1548,5087 } },
    { LIBRAW_CAMERAMAKER_Nikon, "1 J5", 0, 0,
      { 7520,-2518,-645,-3844,12102,1945,-913,2249,6835 } },
    { LIBRAW_CAMERAMAKER_Nikon, "1 S2", -200, 0,
      { 6612,-1342,-618,-3338,11055,2623,-174,1792,5075 } },
    { LIBRAW_CAMERAMAKER_Nikon, "1 V2", 0, 0,
      { 6588,-1305,-693,-3277,10987,2634,-355,2016,5106 } },
    { LIBRAW_CAMERAMAKER_Nikon, "1 V3", -200, 0,
      { 5958,-1559,-571,-4021,11453,2939,-634,1548,5087 } },
    { LIBRAW_CAMERAMAKER_Nikon, "1 ", 0, 0,      /* same CMs: "1 J1", "1 J2", "1 S1", "1 V1" */
      { 8994,-2667,-865,-4594,12324,2552,-699,1786,6260 } },

    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 2100", 0, 0, // a.k.a. E2100
      { 13142,-4152,-1596,-4655,12374,2282,-1769,2696,6711 } }, /* DJC, copied from Z2, new white balance */
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 2500", 0, 0, // a.k.a. E2500, possibly same CM as for E5000
      { -5547,11762,2189,5814,-558,3342,-4924,9840,5949,688,9083,96 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 3200", 0, 0, // a.k.a. E3200
      { 9846,-2085,-1019,-3278,11109,2170,-774,2134,5745 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 4300", 0, 0, // a.k.a. E4300
      { 11280,-3564,-1370,-4655,12374,2282,-1423,2168,5396 } }, /* DJC, copied from Minolta DiMAGE Z2 */
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 4500", 0, 0, // a.k.a. E4500, possibly same CM as for E5000
      { -5547,11762,2189,5814,-558,3342,-4924,9840,5949,688,9083,96 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 5000", 0, 0, // a.k.a. E5000
      { -6678,12805,2248,5725,-499,3375,-5903,10713,6034,-270,9976,134 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 5400", 0, 0, // a.k.a. E5400
      { 9349,-2988,-1001,-7918,15766,2266,-2097,2680,6839 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 5700", 0, 0, // a.k.a. E5700
      { -6475,12496,2428,5409,-16,3180,-5965,10912,5866,-177,9918,248 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 8400", 0, 0, // a.k.a. E8400
      { 7842,-2320,-992,-8154,15718,2599,-1098,1342,7560 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 8700", 0, 0, // a.k.a. E8700
      { 8489,-2583,-1036,-8051,15583,2643,-1307,1407,7354 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 8800", 0, 0, // a.k.a. E8800
      { 7971,-2314,-913,-8451,15762,2894,-1442,1520,7610 } },

    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 700", 0, 0x3dd, // a.k.a. E700
      { -3746,10611,1665,9621,-1734,2114,-2389,7082,3064,3406,6116,-244 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 800", 0, 0x3dd, // a.k.a. E800
      { -3746,10611,1665,9621,-1734,2114,-2389,7082,3064,3406,6116,-244 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 950", 0, 0x3dd, // a.k.a. E950
      { -3746,10611,1665,9621,-1734,2114,-2389,7082,3064,3406,6116,-244 } }, /* DJC */
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX 995", 0, 0,     // a.k.a. E995
      { -5547,11762,2189,5814,-558,3342,-4924,9840,5949,688,9083,96 } }, /* DJC, copied from E5000 */

    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX A1000", 0, 0,
      { 10601,-3487,-1127,-2931,11443,1676,-587,1740,5278 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX B700", 0, 0,
      { 14387,-6014,-1299,-1357,9975,1616,467,1047,4744 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX A", 0, 0,
      { 8198,-2239,-724,-4871,12389,2798,-1043,2050,7181 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX P1000", 0, 0,
      { 14294,-6116,-1333,-1628,10219,1637,-14,1158,5022 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX P6000", 0, 0,
      { 9698,-3367,-914,-4706,12584,2368,-837,968,5801 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX P7000", 0, 0,
      { 11432,-3679,-1111,-3169,11239,2202,-791,1380,4455 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX P7100", 0, 0,
      { 11053,-4269,-1024,-1976,10182,2088,-526,1263,4469 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX P7700", -3200, 0, // same CamID: "COOLPIX P7700", "COOLPIX Deneb"
      { 10321,-3920,-931,-2750,11146,1824,-442,1545,5539 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX P7800", -3200, 0, // same CamID: "COOLPIX P7800", "COOLPIX Kalon"
      { 10321,-3920,-931,-2750,11146,1824,-442,1545,5539 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX P330", -200, 0,
      { 10321,-3920,-931,-2750,11146,1824,-442,1545,5539 } },
    { LIBRAW_CAMERAMAKER_Nikon, "COOLPIX P340", -200, 0,
      { 10321,-3920,-931,-2750,11146,1824,-442,1545,5539 } },
    { LIBRAW_CAMERAMAKER_Nikon, "Coolpix P950", 0, 0,
      { 13307, -5641, -1290, -2048, 10581,  1689,   -64,  1222,  5176}},

    { LIBRAW_CAMERAMAKER_Nikon, "D3000", 0, 0,
      { 8736,-2458,-935,-9075,16894,2251,-1354,1242,8263 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D3100", 0, 0,
      { 7911,-2167,-813,-5327,13150,2408,-1288,2483,7968 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D3200", 0, 0xfb9,
      { 7013,-1408,-635,-5268,12902,2640,-1470,2801,7379 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D3300", 0, 0,
      { 6988,-1384,-714,-5631,13410,2447,-1485,2204,7318 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D3400", 0, 0,
      { 6988,-1384,-714,-5631,13410,2447,-1485,2204,7318 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D3500", 0, 0,
      { 8821,-2938,-785,-4178,12142,2287,-824,1651,6860 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D5000", 0, 0xf00,
      { 7309,-1403,-519,-8474,16008,2622,-2433,2826,8064 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D5100", 0, 0x3de6,
      { 8198,-2239,-724,-4871,12389,2798,-1043,2050,7181 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D5200", 0, 0,
      { 8322,-3112,-1047,-6367,14342,2179,-988,1638,6394 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D5300", 0, 0,
      { 6988,-1384,-714,-5631,13410,2447,-1485,2204,7318 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D5500", 0, 0,
      { 8821,-2938,-785,-4178,12142,2287,-824,1651,6860 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D5600", 0, 0,
      { 8821,-2938,-785,-4178,12142,2287,-824,1651,6860 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D7000", 0, 0,
      { 8198,-2239,-724,-4871,12389,2798,-1043,2050,7181 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D7100", 0, 0,
      { 8322,-3112,-1047,-6367,14342,2179,-988,1638,6394 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D7200", 0, 0,
      { 8322,-3112,-1047,-6367,14342,2179,-988,1638,6394 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D7500", 0, 0,
      { 8813,-3210,-1036,-4703,12868,2021,-1054,1940,6129 } },

    { LIBRAW_CAMERAMAKER_Nikon, "D100", 0, 0,
      { 5902,-933,-782,-8983,16719,2354,-1402,1455,6464 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D200", 0, 0xfbc,
      { 8367,-2248,-763,-8758,16447,2422,-1527,1550,8053 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D300", 0, 0,    // same CMs: D300, D300s
      { 9030,-1992,-715,-8465,16302,2255,-2689,3217,8069 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D500", 0, 0,
      { 8813,-3210,-1036,-4703,12868,2021,-1054,1940,6129 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D600", 0, 0x3e07,
      { 8178,-2245,-609,-4857,12394,2776,-1207,2086,7298 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D610",0, 0,
      { 8178,-2245,-609,-4857,12394,2776,-1207,2086,7298 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D700", 0, 0,
      { 8139,-2171,-663,-8747,16541,2295,-1925,2008,8093 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D750", -600, 0,
      { 9020,-2890,-715,-4535,12436,2348,-934,1919,7086 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D780", -600, 0,
      { 9943,-3269,-839,-5323,13269,2259,-1198,2083,7557 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D800", 0, 0,    // same CMs: D800, D800E
      { 7866,-2108,-555,-4869,12483,2681,-1176,2069,7501 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D810A", 0, 0,
      { 11973,-5685,-888,-1965,10326,1901,-115,1123,7169 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D810", 0, 0,
      { 9369,-3195,-791,-4488,12430,2301,-893,1796,6872 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D850", 0, 0,
      { 10405,-3755,-1270,-5461,13787,1793,-1040,2015,6785 } },

    { LIBRAW_CAMERAMAKER_Nikon, "D40X", 0, 0,
      { 8819,-2543,-911,-9025,16928,2151,-1329,1213,8449 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D40", 0, 0,
      { 6992,-1668,-806,-8138,15748,2543,-874,850,7897 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D50", 0, 0,
      { 7732,-2422,-789,-8238,15884,2498,-859,783,7330 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D60", 0, 0,
      { 8736,-2458,-935,-9075,16894,2251,-1354,1242,8263 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D70", 0, 0,     // same CMs: D70, D70s
      { 7732,-2422,-789,-8238,15884,2498,-859,783,7330 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D80", 0, 0,
      { 8629,-2410,-883,-9055,16940,2171,-1490,1363,8520 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D90", 0, 0xf00,
      { 7309,-1403,-519,-8474,16008,2622,-2434,2826,8064 } },

    { LIBRAW_CAMERAMAKER_Nikon, "D1H", 0, 0,
      { 7659,-2238,-935,-8942,16969,2004,-2701,3051,8690 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D1X", 0, 0,
      { 7702,-2245,-975,-9114,17242,1875,-2679,3055,8521 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D1", 0, 0,
//      { 16772,-4726,-2141,-7611,15713,1972,-2846,3494,9521 } }, /* multiplied by 2.218750, 1.0, 1.148438 */
      { 7637,-2199,-974,-9109,17099,2043,-2822,3306,8372 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D2H", 0, 0,     // same CMs: D2H, D2Hs
      { 5733,-911,-629,-7967,15987,2055,-3050,4013,7048 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D2X", 0, 0,     // same CMs: D2X, D2Xs
      { 10231,-2768,-1254,-8302,15900,2551,-797,681,7148 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D3S", 0, 0,
      { 8828,-2406,-694,-4874,12603,2541,-660,1509,7587 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D3X", 0, 0,
      { 7171,-1986,-648,-8085,15555,2718,-2170,2512,7457 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D3", 0, 0,
      { 8139,-2171,-663,-8747,16541,2295,-1925,2008,8093 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D4", 0, 0,      // same CMs: D4, D4S (and Df)
      { 8598,-2848,-857,-5618,13606,2195,-1002,1773,7137 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D5", 0, 0,
      { 9200,-3522,-992,-5755,13803,2117,-753,1486,6338 } },
    { LIBRAW_CAMERAMAKER_Nikon, "D6", 0, 0,
      { 9028,-3423,-1035,-6321,14265,2217,-1013,1683,6928 } },
    { LIBRAW_CAMERAMAKER_Nikon, "Df", 0, 0,
      { 8598,-2848,-857,-5618,13606,2195,-1002,1773,7137 } },

    { LIBRAW_CAMERAMAKER_Nikon, "Z 50", 0, 0,
      { 11640,-4829,-1079,-5107,13006,2325,-972,1711,7380 } },
    { LIBRAW_CAMERAMAKER_Nikon, "Z 5", 0, 0,
      { 8695,-2558,-648,-5015,12711,2575,-1279,2215,7514 } },
    { LIBRAW_CAMERAMAKER_Nikon, "Z 6", 0, 0,
      { 9943,-3269,-839,-5323,13269,2259,-1198,2083,7557 } },  // 'Z 6'(v.2), 'Z 6_2'
    { LIBRAW_CAMERAMAKER_Nikon, "Z 7", 0, 0,
      { 13705,-6004,-1400,-5464,13568,2062,-940,1706,7618 } }, // 'Z 7'(v.2), 'Z 7_2'
    { LIBRAW_CAMERAMAKER_Nikon, "Z fc", 0, 0,
      { 11640,-4829,-1079,-5107,13006,2325,-972,1711,7380 } },

    { LIBRAW_CAMERAMAKER_Olympus, "AIR A01", 0, 0xfe1,
      { 8992,-3093,-639,-2563,10721,2122,-437,1270,5473 } },

    { LIBRAW_CAMERAMAKER_Olympus, "C-5050Z", 0, 0,
      { 10633,-3234,-1285,-7460,15570,1967,-1917,2510,6299 } },
    { LIBRAW_CAMERAMAKER_Olympus, "C-5060WZ", 0, 0,
      { 10445,-3362,-1307,-7662,15690,2058,-1135,1176,7602 } },
    { LIBRAW_CAMERAMAKER_Olympus, "C-7070WZ", 0, 0,
      { 10252,-3531,-1095,-7114,14850,2436,-1451,1723,6365 } },
    { LIBRAW_CAMERAMAKER_Olympus, "C-7000Z", 0, 0,
      { 10793,-3791,-1146,-7498,15177,2488,-1390,1577,7321 } },
    { LIBRAW_CAMERAMAKER_Olympus, "C-8080WZ", 0, 0,
      { 8606,-2509,-1014,-8238,15714,2703,-942,979,7760 } },

    { LIBRAW_CAMERAMAKER_Olympus, "E-300", 0, 0,
      { 7828,-1761,-348,-5788,14071,1830,-2853,4518,6557 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-330", 0, 0,
      { 8961,-2473,-1084,-7979,15990,2067,-2319,3035,8249 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-400", 0, 0,
      { 6169,-1483,-21,-7107,14761,2536,-2904,3580,8568 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-410", 0, 0xf6a,
      { 8856,-2582,-1026,-7761,15766,2082,-2009,2575,7469 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-420", 0, 0xfd7,
      { 8746,-2425,-1095,-7594,15612,2073,-1780,2309,7416 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-450", 0, 0xfd2,
      { 8745,-2425,-1095,-7594,15613,2073,-1780,2309,7416 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-500", 0, 0,
      { 8136,-1968,-299,-5481,13742,1871,-2556,4205,6630 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-510", 0, 0xf6a,
      { 8785,-2529,-1033,-7639,15624,2112,-1783,2300,7817 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-520", 0, 0xfd2,
      { 8344,-2322,-1020,-7596,15635,2048,-1748,2269,7287 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-600", 0, 0xfaf,
      { 8453,-2198,-1092,-7609,15681,2008,-1725,2337,7824 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-620", 0, 0xfaf,
      { 8453,-2198,-1092,-7609,15681,2008,-1725,2337,7824 } },

    { LIBRAW_CAMERAMAKER_Olympus, "E-10", 0, 0x3ff,
      { 12970,-4703,-1433,-7466,15843,1644,-2191,2451,6668 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-20", 0, 0x3ff,  // model is "E-20,E-20N,E-20P"
      { 13414,-4950,-1517,-7166,15293,1960,-2325,2664,7212 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-30", 0, 0xfbc,
      { 8144,-1861,-1111,-7763,15894,1929,-1865,2542,7607 } },

    { LIBRAW_CAMERAMAKER_Olympus, "E-1", 0, 0,
      { 11846,-4767,-945,-7027,15878,1089,-2699,4122,8311 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-3", 0, 0xf99,
      { 9487,-2875,-1115,-7533,15606,2010,-1618,2100,7389 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-5", 0, 0xeec,
      { 11200,-3783,-1325,-4576,12593,2206,-695,1742,7504 } },

    { LIBRAW_CAMERAMAKER_Olympus, "E-P1", 0, 0xffd,
      { 8343,-2050,-1021,-7715,15705,2103,-1831,2380,8235 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-P2", 0, 0xffd,
      { 8343,-2050,-1021,-7715,15705,2103,-1831,2380,8235 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-P3", 0, 0,
      { 7575,-2159,-571,-3722,11341,2725,-1434,2819,6271 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-P5", 0, 0,
      { 8380,-2630,-639,-2887,10725,2496,-627,1427,5438 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-P7", 0, 0,
      { 9476,-3182,-765,-2613,10958,1893,-449,1315,5268 } },

    { LIBRAW_CAMERAMAKER_Olympus, "E-PL10", 0, 0,
      { 9197,-3190,-659,-2606,10830,2039,-458,1250,5458 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-PL1s", 0, 0,
      { 11409,-3872,-1393,-4572,12757,2003,-709,1810,7415 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-PL1", 0, 0,
      { 11408,-4289,-1215,-4286,12385,2118,-387,1467,7787 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-PL2", 0, 0xcf3,
      { 15030,-5552,-1806,-3987,12387,1767,-592,1670,7023 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-PL3", 0, 0,
      { 7575,-2159,-571,-3722,11341,2725,-1434,2819,6271 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-PL5", 0, 0xfcb,
      { 8380,-2630,-639,-2887,10725,2496,-627,1427,5438 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-PL6", 0, 0,
      { 8380,-2630,-639,-2887,10725,2496,-627,1427,5438 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-PL7", 0, 0,
      { 9197,-3190,-659,-2606,10830,2039,-458,1250,5458 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-PL8", 0, 0,
      { 9197,-3190,-659,-2606,10830,2039,-458,1250,5458 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-PL9", 0, 0,
      { 8380,-2630,-639,-2887,10725,2496,-627,1427,5438 } },

    { LIBRAW_CAMERAMAKER_Olympus, "E-PM1", 0, 0,
      { 7575,-2159,-571,-3722,11341,2725,-1434,2819,6271 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-PM2", 0, 0,
      { 8380,-2630,-639,-2887,10725,2496,-627,1427,5438 } },

    { LIBRAW_CAMERAMAKER_Olympus, "E-M10 Mark IV", 0, 0,
      { 9476,-3182,-765,-2613,10958,1893,-449,1315,5268 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-M10", 0, 0,	  // Same CMs: E-M10, E-M10 Mark II, E-M10 Mark III; "CLAUSS piX 5oo"
      { 8380,-2630,-639,-2887,10725,2496,-627,1427,5438 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-M1X", 0, 0,
      { 11896,-5110,-1076,-3181,11378,2048,-519,1224,5166 } },

    { LIBRAW_CAMERAMAKER_Olympus, "E-M1 Mark III", 0, 0,
      { 11896,-5110,-1076,-3181,11378,2048,-519,1224,5166 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-M1 Mark II", 0, 0,
      { 9383,-3170,-763,-2457,10702,2020,-384,1236,5552 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-M1", 0, 0,
      { 7687,-1984,-606,-4327,11928,2721,-1381,2339,6452 } },

    { LIBRAW_CAMERAMAKER_Olympus, "E-M5 Mark III", 0, 0,
      { 11896,-5110,-1076,-3181,11378,2048,-519,1224,5166 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-M5 Mark II", 0, 0,
      { 9422,-3258,-711,-2655,10898,2015,-512,1354,5512 } },
    { LIBRAW_CAMERAMAKER_Olympus, "E-M5", 0, 0xfe1,
      { 8380,-2630,-639,-2887,10725,2496,-627,1427,5438 } },

    { LIBRAW_CAMERAMAKER_Olympus, "PEN-F",0, 0,
      { 9476,-3182,-765,-2613,10958,1893,-449,1315,5268 } },

    { LIBRAW_CAMERAMAKER_Olympus, "SH-2", 0, 0, // same CamID: SH-2, SH-3
     { 10156,-3425,-1077,-2611,11177,1624,-385,1592,5080 } },

    { LIBRAW_CAMERAMAKER_Olympus, "SP-350", 0, 0,
      { 12078,-4836,-1069,-6671,14306,2578,-786,939,7418 } },
    { LIBRAW_CAMERAMAKER_Olympus, "SP-3", 0, 0,	// Same CMs: SP310, SP320
      { 11766,-4445,-1067,-6901,14421,2707,-1029,1217,7572 } },
    { LIBRAW_CAMERAMAKER_Olympus, "SP-500UZ", 0, 0xfff,
      { 9493,-3415,-666,-5211,12334,3260,-1548,2262,6482 } },
    { LIBRAW_CAMERAMAKER_Olympus, "SP-510UZ", 0, 0xffe,
      { 10593,-3607,-1010,-5881,13127,3084,-1200,1805,6721 } },
    { LIBRAW_CAMERAMAKER_Olympus, "SP-550UZ", 0, 0xffe,
      { 11597,-4006,-1049,-5432,12799,2957,-1029,1750,6516 } },
    { LIBRAW_CAMERAMAKER_Olympus, "SP-560UZ", 0, 0xff9,
      { 10915,-3677,-982,-5587,12986,2911,-1168,1968,6223 } },
    { LIBRAW_CAMERAMAKER_Olympus, "SP-565UZ", 0, 0,
      { 11856,-4469,-1159,-4814,12368,2756,-993,1779,5589 } },
    { LIBRAW_CAMERAMAKER_Olympus, "SP-570UZ", 0, 0,
      { 11522,-4044,-1146,-4736,12172,2904,-988,1829,6039 } },

    { LIBRAW_CAMERAMAKER_Olympus, "STYLUS 1",0, 0,	// Olympus "STYLUS 1 and STYLUS 1s have the same CamID, cameras are slightly different
      { 8360,-2420,-880,-3928,12353,1739,-1381,2416,5173 } },

    { LIBRAW_CAMERAMAKER_Olympus, "TG-4", 0, 0,
     { 11426,-4159,-1126,-2066,10678,1593,-120,1327,4998 } },
    { LIBRAW_CAMERAMAKER_Olympus, "TG-", 0, 0, // same CMs: TG-5, TG-6
     { 10899,-3833,-1082,-2112,10736,1575,-267,1452,5269 } },

    { LIBRAW_CAMERAMAKER_Olympus, "XZ-10", 0, 0,
      { 9777,-3483,-925,-2886,11297,1800,-602,1663,5134 } },
    { LIBRAW_CAMERAMAKER_Olympus, "XZ-1", 0, 0,
      { 10901,-4095,-1074,-1141,9208,2293,-62,1417,5158 } },
    { LIBRAW_CAMERAMAKER_Olympus, "XZ-2", 0, 0,
      { 9777,-3483,-925,-2886,11297,1800,-602,1663,5134 } },

	  { LIBRAW_CAMERAMAKER_Olympus, "OM-1", 0, 0,
		{ 9488, -3984, -714, -2887, 10945, 2229, -137, 960, 5786 } },

	{ LIBRAW_CAMERAMAKER_OmniVison, "", 16, 0x3ff,
      { 12782,-4059,-379,-478,9066,1413,1340,1513,5176 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Pentax, "*istDL2", 0, 0,
      { 10504,-2438,-1189,-8603,16207,2531,-1022,863,12242 } },
    { LIBRAW_CAMERAMAKER_Pentax, "*istDL", 0, 0,
      { 10829,-2838,-1115,-8339,15817,2696,-837,680,11939 } },
    { LIBRAW_CAMERAMAKER_Pentax, "*istDS2", 0, 0,
      { 10504,-2438,-1189,-8603,16207,2531,-1022,863,12242 } },
    { LIBRAW_CAMERAMAKER_Pentax, "*istDS", 0, 0,
      { 10371,-2333,-1206,-8688,16231,2602,-1230,1116,11282 } },
    { LIBRAW_CAMERAMAKER_Pentax, "*istD", 0, 0,
      { 9651,-2059,-1189,-8881,16512,2487,-1460,1345,10687 } },

    { LIBRAW_CAMERAMAKER_Pentax, "K-01", 0, 0,
      { 8134,-2728,-645,-4365,11987,2694,-838,1509,6498 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K10D", 0, 0,
      { 9679,-2965,-811,-8622,16514,2182,-975,883,9793 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K1", 0, 0, // same CMs: K100D, "K100D Super", K110D
      { 11095,-3157,-1324,-8377,15834,2720,-1108,947,11688 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K20D", 0, 0,
      { 9427,-2714,-868,-7493,16092,1373,-2199,3264,7180 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K200D", 0, 0,
      { 9186,-2678,-907,-8693,16517,2260,-1129,1094,8524 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-m", 0, 0,
      { 9730,-2989,-970,-8527,16258,2381,-1060,970,8362 } },
    { LIBRAW_CAMERAMAKER_Pentax, "KP", 0, 0,
      { 7825,-2160,-1403,-4841,13555,1349,-1559,2449,5814 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-x", 0, 0,
      { 8843,-2837,-625,-5025,12644,2668,-411,1234,7410 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-r", 0, 0,
      { 9895,-3077,-850,-5304,13035,2521,-883,1768,6936 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-1", 0, 0, // same CMs: K-1, "K-1 Mark II"
      { 8596,-2981,-639,-4202,12046,2431,-685,1424,6122 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-30", 0, 0,
      { 8134,-2728,-645,-4365,11987,2694,-838,1509,6498 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-3 Mark III", 0, 0,
      {  9251, -3817, -1069, -4627, 12667,  2175,  -798,  1660,  5633 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-3", 0, 0, // same CMs: K-3, "K-3 II"
      { 7415,-2052,-721,-5186,12788,2682,-1446,2157,6773 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-500", 0, 0,
      { 8109,-2740,-608,-4593,12175,2731,-1006,1515,6545 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-50", 0, 0,
      { 8109,-2740,-608,-4593,12175,2731,-1006,1515,6545 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-5 II", 0, 0, // same CMs: "K-5 II" and "K-5 IIs"
      { 8170,-2725,-639,-4440,12017,2744,-771,1465,6599 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-5", 0, 0,
      { 8713,-2833,-743,-4342,11900,2772,-722,1543,6247 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-70", 0, 0,
      { 8766,-3149,-747,-3976,11943,2292,-517,1259,5552 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-7", 0, 0,
      { 9142,-2947,-678,-8648,16967,1663,-2224,2898,8615 } },
    { LIBRAW_CAMERAMAKER_Pentax, "KP", 0, 0,
      { 8617,-3228,-1034,-4674,12821,2044,-803,1577,5728 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-S1", 0, 0,
      { 8512,-3211,-787,-4167,11966,2487,-638,1288,6054 } },
    { LIBRAW_CAMERAMAKER_Pentax, "K-S2", 0, 0,
      { 8662,-3280,-798,-3928,11771,2444,-586,1232,6054 } },

    { LIBRAW_CAMERAMAKER_Pentax, "Q-S1", 0, 0,
      { 12995,-5593,-1107,-1879,10139,2027,-64,1233,4919 } },
    { LIBRAW_CAMERAMAKER_Pentax, "Q7", 0, 0,
      { 10901,-3938,-1025,-2743,11210,1738,-823,1805,5344 } },
    { LIBRAW_CAMERAMAKER_Pentax, "Q10", 0, 0,
      { 11562,-4183,-1172,-2357,10919,1641,-582,1726,5112 } },
    { LIBRAW_CAMERAMAKER_Pentax, "Q", 0, 0,
      { 11731,-4169,-1267,-2015,10727,1473,-217,1492,4870 } },

    { LIBRAW_CAMERAMAKER_Pentax, "MX-1", 0, 0,
      { 9296,-3146,-888,-2860,11287,1783,-618,1698,5151 } },

    { LIBRAW_CAMERAMAKER_Pentax, "645D", 0, 0x3e00,
      { 10646,-3593,-1158,-3329,11699,1831,-667,2874,6287 } },
    { LIBRAW_CAMERAMAKER_Pentax, "645Z", 0, 0,
      { 9519,-3591,-664,-4074,11725,2671,-624,1501,6653 } },


    {LIBRAW_CAMERAMAKER_Panasonic, "DC-S1R", 0, 0,
      { 11822,-5321,-1249,-5958,15114,766,-614,1264,7043 } },
    {LIBRAW_CAMERAMAKER_Panasonic, "DC-S1H", 0, 0,
      { 9397,-3719,-805,-5425,13326,2309,-972,1715,6034 } },
    {LIBRAW_CAMERAMAKER_Panasonic, "DC-S1", 0, 0,
      { 9744,-3905,-779,-4899,12807,2324,-798,1630,5827 } },
    {LIBRAW_CAMERAMAKER_Panasonic, "DC-S5", 0, 0,
      { 9744,-3905,-779,-4899,12807,2324,-798,1630,5827 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-CM1", -15, 0, // same CMs: DMC-CM1, DMC-CM10
      { 8770,-3194,-820,-2871,11281,1803,-513,1552,4434 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DC-FZ1000M2", -15, 0,
      { 9803,-4185,-992,-4066,12578,1628,-838,1824,5288 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ1000", -15, 0,
      { 7830,-2696,-763,-3325,11667,1866,-641,1712,4824 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ2500", -15, 0,
      { 7386,-2443,-743,-3437,11864,1757,-608,1660,4766 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ100", -15, 0xfff,
      { 16197,-6146,-1761,-2393,10765,1869,366,2238,5248 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ150", -15, 0xfff,
      { 11904,-4541,-1189,-2355,10899,1662,-296,1586,4289 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ200", -15, 0xfff,
      { 8112,-2563,-740,-3730,11784,2197,-941,2075,4933 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ300", -15, 0xfff,
      { 8378,-2798,-769,-3068,11410,1877,-538,1792,4623 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ18", 0, 0,
      { 9932,-3060,-935,-5809,13331,2753,-1267,2155,5575 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ28", -15, 0xf96,
      { 10109,-3488,-993,-5412,12812,2916,-1305,2140,5543 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ30", 0, 0xf94,
      { 10976,-4029,-1141,-7918,15491,2600,-1670,2071,8246 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ35", -15, 0,
      { 9938,-2780,-890,-4604,12393,2480,-1117,2304,4620 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ40", -15, 0,
      { 13639,-5535,-1371,-1698,9633,2430,316,1152,4108 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ50", 0, 0,
      { 7906,-2709,-594,-6231,13351,3220,-1922,2631,6537 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ70", -15, 0,
      { 11532,-4324,-1066,-2375,10847,1749,-564,1699,4351 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-FZ80", -15, 0,
      { 8550,-2908,-842,-3195,11529,1881,-338,1603,4631 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FZ8", 0, 0xf7f,
      { 8986,-2755,-802,-6341,13575,3077,-1476,2144,6379 } },


    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-L10", -15, 0xf96,
      { 8025,-1942,-1050,-7920,15904,2100,-2456,3005,7039 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-L1", 0, 0xf7f,
      { 8054,-1885,-1025,-8349,16367,2040,-2805,3542,7629 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-LC1", 0, 0,
      { 11340,-4069,-1275,-7555,15266,2448,-2960,3426,7685 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-LF1", -15, 0,
      { 9379,-3267,-816,-3227,11560,1881,-926,1928,5340 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DC-LX100M2", -15, 0,
      { 8585,-3127,-833,-4005,12250,1953,-650,1494,4862 } }, // v.2
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-LX100", -15, 0,
      { 8844,-3538,-768,-3709,11762,2200,-698,1792,5220 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-LX1", 0, 0xf7f,
      { 10704,-4187,-1230,-8314,15952,2501,-920,945,8927 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-LX2", 0, 0,
      { 8048,-2810,-623,-6450,13519,3272,-1700,2146,7049 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-LX3", -15, 0,
      { 8128,-2668,-655,-6134,13307,3161,-1782,2568,6083 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-LX5", -15, 0,
      { 10909,-4295,-948,-1333,9306,2399,22,1738,4582 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-LX7", -15, 0,
      { 10148,-3743,-991,-2837,11366,1659,-701,1893,4899 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-LX9", -15, 0,
      { 7790,-2736,-755,-3452,11870,1769,-628,1647,4898 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-FX150", -15, 0xfff,
      { 9082,-2907,-925,-6119,13377,3058,-1797,2641,5609 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DC-G99", -15, 0,
      { 9657,-3963,-748,-3361,11378,2258,-568,1415,5158 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-G100", -15, 0,
      { 8370,-2869,-710,-3389,11372,2298,-640,1599,4887 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-G10", 0, 0,
      { 10113,-3400,-1114,-4765,12683,2317,-377,1437,6710 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-G1", -15, 0xf94,
      { 8199,-2065,-1056,-8124,16156,2033,-2458,3022,7220 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-G2", -15, 0xf3c,
      { 10113,-3400,-1114,-4765,12683,2317,-377,1437,6710 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-G3", -15, 0xfff,
      { 6763,-1919,-863,-3868,11515,2684,-1216,2387,5879 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-G5", -15, 0xfff,
      { 7798,-2562,-740,-3879,11584,2613,-1055,2248,5434 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-G6", -15, 0xfff,
      { 8294,-2891,-651,-3869,11590,2595,-1183,2267,5352 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-G7", -15, 0xfff,
      { 7610,-2780,-576,-4614,12195,2733,-1375,2393,6490 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-G8", -15, 0xfff,
      { 7610,-2780,-576,-4614,12195,2733,-1375,2393,6490 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-G9", -15, 0,
      { 7685,-2375,-634,-3687,11700,2249,-748,1546,5111 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GH1", -15, 0xf92,
      { 6299,-1466,-532,-6535,13852,2969,-2331,3112,5984 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GH2", -15, 0xf95,
      { 7780,-2410,-806,-3913,11724,2484,-1018,2390,5298 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GH3", -15, 0,
      { 6559,-1752,-491,-3672,11407,2586,-962,1875,5130 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GH4", -15, 0,
      { 7122,-2108,-512,-3155,11201,2231,-541,1423,5045 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-GH5s", -15, 0,
      { 6929,-2355,-708,-4192,12534,1828,-1097,1989,5195 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-GH5M2", 0, 0,
      {  9300, -3659,  -755, -2981, 10988,  2287,  -190,  1077,  5016 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-GH5", -15, 0,
      { 7641,-2336,-605,-3218,11299,2187,-485,1338,5121 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GM1", -15, 0,
      { 6770,-1895,-744,-5232,13145,2303,-1664,2691,5703 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GM5", -15, 0,
      { 8238,-3244,-679,-3921,11814,2384,-836,2022,5852 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DC-GF10", -15, 0,
      { 7610,-2780,-576,-4614,12195,2733,-1375,2393,6490 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GF1", -15, 0xf92,
      { 7888,-1902,-1011,-8106,16085,2099,-2353,2866,7330 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GF2", -15, 0xfff,
      { 7888,-1902,-1011,-8106,16085,2099,-2353,2866,7330 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GF3", -15, 0xfff,
      { 9051,-2468,-1204,-5212,13276,2121,-1197,2510,6890 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GF5", -15, 0xfff,
      { 8228,-2945,-660,-3938,11792,2430,-1094,2278,5793 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GF6", -15, 0,
      { 8130,-2801,-946,-3520,11289,2552,-1314,2511,5791 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GF7", -15, 0,
      { 7610,-2780,-576,-4614,12195,2733,-1375,2393,6490 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GF8", -15, 0,
      { 7610,-2780,-576,-4614,12195,2733,-1375,2393,6490 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-GF9", -15, 0,
      { 7610,-2780,-576,-4614,12195,2733,-1375,2393,6490 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GX85", -15, 0,
      { 7771,-3020,-629,-4029,11950,2345,-821,1977,6119 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GX1", -15, 0,
      { 6763,-1919,-863,-3868,11515,2684,-1216,2387,5879 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GX7", -15,0,
      { 7610,-2780,-576,-4614,12195,2733,-1375,2393,6490 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-GX8", -15,0,
      { 7564,-2263,-606,-3148,11239,2177,-540,1435,4853 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-GX9", -15, 0,
      { 7564,-2263,-606,-3148,11239,2177,-540,1435,4853 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-ZS100", -15, 0,
      { 7790,-2736,-755,-3452,11870,1769,-628,1647,4898 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-ZS200", -15, 0,
      { 7790,-2736,-755,-3452,11870,1769,-628,1647,4898 } },

    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-ZS40", -15, 0,
      { 8607,-2822,-808,-3755,11930,2049,-820,2060,5224 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-ZS50", -15, 0,
      { 8802,-3135,-789,-3151,11468,1904,-550,1745,4810 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DMC-ZS60", -15, 0,
      { 8550,-2908,-842,-3195,11529,1881,-338,1603,4631 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-ZS70", -15, 0,
      { 9052,-3117,-883,-3045,11346,1927,-205,1520,4730 } },
    { LIBRAW_CAMERAMAKER_Panasonic, "DC-ZS80", -15, 0,
      { 12194,-5340,-1329,-3035,11394,1858,-50,1418,5219 } },

    { LIBRAW_CAMERAMAKER_PhaseOne, "H20", 0, 0,
      { 3906,1422,-467,-9953,18472,1365,-3307,4496,6406 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "H25", 0, 0,
      { 2905,732,-237,-8134,16626,1476,-3038,4253,7517 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ4 150MP", 0, 0,
      {  6644, -2257,  -804, -6459, 14562,  2019, -1221,  1876,  6411}},
    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ140", 0, 0,
      { 8035,435,-962,-6001,13872,2320,-1159,3065,5434 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ150", 0, 0,
      {10325,845,-604,-4113,13385,481,-1791,4163,6924}}, /* temp */ /* emb */
//      { 3984,0,0,0,10000,0,0,0,7666 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ160", 0, 0,
      { 8035,435,-962,-6001,13872,2320,-1159,3065,5434 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ180", 0, 0,
      { 6294,686,-712,-5435,13417,2211,-1006,2435,5042 } },

    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ250",0, 0,
//    {3984,0,0,0,10000,0,0,0,7666}},
      {10325,845,-604,-4113,13385,481,-1791,4163,6924}}, /* emb */
    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ260", 0, 0,
      { 8035,435,-962,-6001,13872,2320,-1159,3065,5434 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ280", 0, 0,
      { 6294,686,-712,-5435,13417,2211,-1006,2435,5042 } },

    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ3 100MP", 0, 0,
//    {2423,0,0,0,9901,0,0,0,7989}},
      { 10999,354,-742,-4590,13342,937,-1060,2166,8120} }, /* emb */
    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ3 50MP", 0, 0,
//      { 3984,0,0,0,10000,0,0,0,7666 } },
      {10058,1079,-587,-4135,12903,944,-916,2726,7480}}, /* emb */
    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ3 60MP", 0, 0,
      { 8035,435,-962,-6001,13872,2320,-1159,3065,5434 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "IQ3 80MP", 0, 0,
      { 6294,686,-712,-5435,13417,2211,-1006,2435,5042 } },

    { LIBRAW_CAMERAMAKER_PhaseOne, "P21", 0, 0,
      { 6516,-2050,-507,-8217,16703,1479,-3492,4741,8489 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "P30", 0, 0,
      { 4516,-244,-36,-7020,14976,2174,-3206,4670,7087 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "P40", 0, 0,
      { 8035,435,-962,-6001,13872,2320,-1159,3065,5434 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "P45", 0, 0,
      { 5053,-24,-117,-5685,14077,1703,-2619,4491,5850 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "P65", 0, 0,
      { 8035,435,-962,-6001,13872,2320,-1159,3065,5434 } },
    { LIBRAW_CAMERAMAKER_PhaseOne, "P2", 0, 0,
      { 2905,732,-237,-8134,16626,1476,-3038,4253,7517 } },

    { LIBRAW_CAMERAMAKER_Photron, "BC2-HD", 0, 0,
      { 14603,-4122,-528,-1810,9794,2017,-297,2763,5936 } }, /* DJC */

	  { LIBRAW_CAMERAMAKER_Polaroid, "x530", 0, 0,
      { 13458,-2556,-510,-5444,15081,205,0,0,12120 } },

	  { LIBRAW_CAMERAMAKER_RED, "One", 704, 0xffff,
      { 21014,-7891,-2613,-3056,12201,856,-2203,5125,8042 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Ricoh, "S10 24-72mm F2.5-4.4 VC", 0, 0,
      { 10531,-4043,-878,-2038,10270,2052,-107,895,4577 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GR A12 50mm F2.5 MACRO", 0, 0,
      { 8849,-2560,-689,-5092,12831,2520,-507,1280,7104 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GR DIGITAL 2", 0, 0,
      { 8846,-2704,-729,-5265,12708,2871,-1471,1955,6218 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GR DIGITAL 3", 0, 0,
      { 8170,-2496,-655,-5147,13056,2312,-1367,1859,5265 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GR DIGITAL 4", 0, 0,
      { 8771,-3151,-837,-3097,11015,2389,-703,1343,4924 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GR III", 0, 0,
      { 6127,-1777,-585,-5913,13699,2428,-1088,1780,6017 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GR II", 0, 0,
      { 5329,-1459,-390,-5407,12930,2768,-1119,1772,6046 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GR", 0, 0,
      { 5329,-1459,-390,-5407,12930,2768,-1119,1772,6046 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GX200", 0, 0,
      { 8040,-2368,-626,-4659,12543,2363,-1125,1581,5660 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GXR Mount A12", 0, 0,
      { 7834,-2182,-739,-5453,13409,2241,-952,2005,6620 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GXR A12 50mm", 0, 0,
      { 8849,-2560,-689,-5092,12831,2520,-507,1280,7104 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GXR A12 28mm", 0, 0,
      { 10228,-3159,-933,-5304,13158,2371,-943,1873,6685 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GXR A16", 0, 0,
      { 7837,-2538,-730,-4370,12184,2461,-868,1648,5830 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GXR P10", 0, 0,
      { 13168,-5128,-1663,-3006,11569,1611,-373,1244,4907 } },
    { LIBRAW_CAMERAMAKER_Ricoh, "GXR S10", 0, 0,
      { 8963,-2926,-754,-4881,12921,2164,-1464,1944,4901 } },

    { LIBRAW_CAMERAMAKER_Samsung, "EX1", 0, 0x3e00,
      { 8898,-2498,-994,-3144,11328,2066,-760,1381,4576 } },
    { LIBRAW_CAMERAMAKER_Samsung, "EX2F", 0, 0x7ff,
      { 10648,-3897,-1055,-2022,10573,1668,-492,1611,4742 } },
//    { LIBRAW_CAMERAMAKER_Samsung, "GX20", 0, 0,
//      { 23213,-14575,-4840,-7077,16564,316,385,-1656,9398 } }, // Adobe DNG
//      { 27717,-17403,-5779,-8450,19778,377,459,1978,11221 } }, // Samsung DNG
//      { 9427,-2714,-868,-7493,16092,1373,-2199,3264,7180 } },  // Adobe DCP

//    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S6 Edge Rear Camera", 0, 0,
//    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S6 Rear Camera", 0, 0,
    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S6", 0, 0, // same CMs: "Galaxy S6", "Galaxy S6 Edge"
      { 13699,-5767,-1384,-4449,13879,499,-467,1691,5892 } },

//    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S7 Edge Rear Camera", 0, 0,
//    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S7 Rear Camera", 0, 0,
    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S7", 0, 0, // same CMs: "Galaxy S7", "Galaxy S7 Edge"
      { 9927,-3704,-1024,-3935,12758,1257,-389,1512,4993 } },

//    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S8+ Rear Camera", 0, 0,
//    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S8 Rear Camera", 0, 0,
    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S8", 0, 0, // same CMs: "Galaxy S8", "Galaxy S8+"
      { 9927,-3704,-1024,-3935,12758,1257,-389,1512,4993 } },

//    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S9+ Rear Camera", 0, 0,
//    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S9 Rear Camera", 0, 0,
    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy S9", 0, 0, // same CMs: "Galaxy S9", "Galaxy S9+"
      { 13292,-6142,-1268,-4095,12890,1283,-557,1930,5163 } },
//    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy Note 9 Rear Telephoto Camera", 0, 0,
    { LIBRAW_CAMERAMAKER_Samsung, "Galaxy Note 9 Rear Camera", 0, 0,
      { 13292,-6142,-1268,-4095,12890,1283,-557,1930,5163 } },

    { LIBRAW_CAMERAMAKER_Samsung, "NX U", 0, 0,
      { 7557,-2522,-739,-4679,12949,1894,-840,1777,5311 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX3300", 0, 0,
      { 8060,-2933,-761,-4504,12890,1762,-630,1489,5227 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX3000", 0, 0,
      { 8060,-2933,-761,-4504,12890,1762,-630,1489,5227 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX30", 0, 0, // same CMs: NX30, NX300, NX300M
      { 7557,-2522,-739,-4679,12949,1894,-840,1777,5311 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX2000", 0, 0,
      { 7557,-2522,-739,-4679,12949,1894,-840,1777,5311 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX2", 0, 0xfff, // same CMs: NX20, NX200, NX210
      { 6933,-2268,-753,-4921,13387,1647,-803,1641,6096 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX1000", 0, 0,
      { 6933,-2268,-753,-4921,13387,1647,-803,1641,6096 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX1100", 0, 0,
      { 6933,-2268,-753,-4921,13387,1647,-803,1641,6096 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX11", 0, 0,
      { 10332,-3234,-1168,-6111,14639,1520,-1352,2647,8331 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX10", 0, 0, // same CMs: NX10, NX100
      { 10332,-3234,-1168,-6111,14639,1520,-1352,2647,8331 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX500", 0, 0,
      { 10686,-4042,-1052,-3595,13238,276,-464,1259,5931 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX5", 0, 0,
      { 10332,-3234,-1168,-6111,14639,1520,-1352,2647,8331 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX1", 0, 0,
      { 10686,-4042,-1052,-3595,13238,276,-464,1259,5931 } },
    { LIBRAW_CAMERAMAKER_Samsung, "NX mini", 0, 0,
      { 5222,-1196,-550,-6540,14649,2009,-1666,2819,5657 } },

    { LIBRAW_CAMERAMAKER_Samsung, "WB2000", 0, 0xfff,
      { 12093,-3557,-1155,-1000,9534,1733,-22,1787,4576 } },
    { LIBRAW_CAMERAMAKER_Samsung, "WB5000", 0, 0,
      {  7675, -2195,  -305, -5860, 14118,  1857, -2425,  4007,  6578}},
    { LIBRAW_CAMERAMAKER_Samsung, "S85", 0, 0,
      { 11885,-3968,-1473,-4214,12299,1916,-835,1655,5549 } }, /* DJC */

// Foveon: LibRaw color data
    { LIBRAW_CAMERAMAKER_Sigma, "dp0 Quattro", 2047, 0,
      { 13801,-3390,-1016,5535,3802,877,1848,4245,3730 } },
    { LIBRAW_CAMERAMAKER_Sigma, "dp1 Quattro", 2047, 0,
      { 13801,-3390,-1016,5535,3802,877,1848,4245,3730 } },
    { LIBRAW_CAMERAMAKER_Sigma, "dp2 Quattro", 2047, 0,
      { 13801,-3390,-1016,5535,3802,877,1848,4245,3730 } },
    { LIBRAW_CAMERAMAKER_Sigma, "dp3 Quattro", 2047, 0,
      { 13801,-3390,-1016,5535,3802,877,1848,4245,3730 } },
    { LIBRAW_CAMERAMAKER_Sigma, "sd Quattro H", 256, 0,
      { 1295,108,-311,256,828,-65,-28,750,254 } }, /* temp */
    { LIBRAW_CAMERAMAKER_Sigma, "sd Quattro", 2047, 0,
      { 1295,108,-311,256,828,-65,-28,750,254 } }, /* temp */
    { LIBRAW_CAMERAMAKER_Sigma, "SD9", 15, 4095,
      { 13564,-2537,-751,-5465,15154,194,-67,116,10425 } },
    { LIBRAW_CAMERAMAKER_Sigma, "SD10", 15, 16383,
      { 6787,-1682,575,-3091,8357,160,217,-369,12314 } },
    { LIBRAW_CAMERAMAKER_Sigma, "SD14", 15, 16383,
      { 13589,-2509,-739,-5440,15104,193,-61,105,10554 } },
    { LIBRAW_CAMERAMAKER_Sigma, "SD15", 15, 4095,
      { 13556,-2537,-730,-5462,15144,195,-61,106,10577 } },
// Merrills + SD1
    { LIBRAW_CAMERAMAKER_Sigma, "SD1", 31, 4095,
      { 5133,-1895,-353,4978,744,144,3837,3069,2777 } }, /* LibRaw */
    { LIBRAW_CAMERAMAKER_Sigma, "DP1 Merrill", 31, 4095,
      { 5133,-1895,-353,4978,744,144,3837,3069,2777 } }, /* LibRaw */
    { LIBRAW_CAMERAMAKER_Sigma, "DP2 Merrill", 31, 4095,
      { 5133,-1895,-353,4978,744,144,3837,3069,2777 } }, /* LibRaw */
    { LIBRAW_CAMERAMAKER_Sigma, "DP3 Merrill", 31, 4095,
      { 5133,-1895,-353,4978,744,144,3837,3069,2777 } }, /* LibRaw */
// Sigma DP (non-Merrill Versions)
    { LIBRAW_CAMERAMAKER_Sigma, "DP1X", 0, 4095,
      { 13704,-2452,-857,-5413,15073,186,-89,151,9820 } },
    { LIBRAW_CAMERAMAKER_Sigma, "DP1", 0, 4095,
      { 12774,-2591,-394,-5333,14676,207,15,-21,12127 } },
    { LIBRAW_CAMERAMAKER_Sigma, "DP", 0, 4095,
      //  { 7401,-1169,-567,2059,3769,1510,664,3367,5328 } },
      { 13100,-3638,-847,6855,2369,580,2723,3218,3251 } }, /* LibRaw */

	{ LIBRAW_CAMERAMAKER_Sinar, "", 0, 0,
      { 16442,-2956,-2422,-2877,12128,750,-1136,6066,4559 } }, /* DJC */

    { LIBRAW_CAMERAMAKER_Sony, "DSC-F828", 0, 0,
      { 7924,-1910,-777,-8226,15459,2998,-1517,2199,6818,-7242,11401,3481 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-R1", 0, 0,
      { 8512,-2641,-694,-8042,15670,2526,-1821,2117,7414 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-V3", 0, 0,
      { 7511,-2571,-692,-7894,15088,3060,-948,1111,8128 } },

    { LIBRAW_CAMERAMAKER_Sony, "DSC-HX9", -800, 0, // same CMs: DSC-HX95, DSC-HX99
      { 13076,-5686,-1481,-4027,12851,1251,-167,725,4937 } },

    { LIBRAW_CAMERAMAKER_Sony, "ZV-1", -800, 0,
      {8280,-2987,-703,-3531,11645,2133,-550,1542,5312 } },

    { LIBRAW_CAMERAMAKER_Sony, "ZV-E10", 0, 0,
      { 7657,-2847,-607,-4083,11966,2389,-684,1418,5844 } }, // a la ILCE-6100

    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX100M7", -800, 0,
      {10315, -4390, -937, -4859, 12734, 2365, -734, 1537, 5997 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX100M6", -800, 0,
      { 7325,-2321,-596,-3494,11674,2055,-668,1562,5031 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX100M5A", -800, 0,
      { 11176,-4700,-965,-4004,12184,2032,-763,1726,5876 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX100M", -800, 0, // same CMs: DSC-RX100M2, DSC-RX100M3, DSC-RX100M4, DSC-RX100M5
      { 6596,-2079,-562,-4782,13016,1933,-970,1581,5181 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX100", 0, 0,
      { 8651,-2754,-1057,-3464,12207,1373,-568,1398,4434 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX10M4", -800, 0,
      { 7699,-2566,-629,-2967,11270,1928,-378,1286,4807 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX10",0, 0, // same CMs: DSC-RX10, DSC-RX10M2, DSC-RX10M3
      { 6679,-1825,-745,-5047,13256,1953,-1580,2422,5183 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX1RM2", 0, 0,
      { 6629,-1900,-483,-4618,12349,2550,-622,1381,6514 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX1R", 0, 0,
      { 6344,-1612,-462,-4863,12477,2681,-865,1786,6899 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX1", 0, 0,
      { 6344,-1612,-462,-4863,12477,2681,-865,1786,6899 } },

    { LIBRAW_CAMERAMAKER_Sony, "DSC-RX0", -800, 0, // same CMs: DSC-RX0, DSC-RX0M2
      { 9396,-3507,-843,-2497,11111,1572,-343,1355,5089 } },

    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A100", 0, 0xfeb,
      { 9437,-2811,-774,-8405,16215,2290,-710,596,7181 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A290", 0, 0,
      { 6038,-1484,-579,-9145,16746,2512,-875,746,7218 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A2", 0, 0, // same CMs: DSLR-A200, DSLR-A230
      { 9847,-3091,-928,-8485,16345,2225,-715,595,7103 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A300", 0, 0,
      { 9847,-3091,-928,-8485,16345,2225,-715,595,7103 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A330", 0, 0,
      { 9847,-3091,-929,-8485,16346,2225,-714,595,7103 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A3", 0, 0, // same CMs: DSLR-A350, DSLR-A380, DSLR-A390
      { 6038,-1484,-579,-9145,16746,2512,-875,746,7218 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A450", 0, 0, // close to 16596 if arw is 14-bit
      { 4950,-580,-103,-5228,12542,3029,-709,1435,7371 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A580", 0, 16596,
      { 5932,-1492,-411,-4813,12285,2856,-741,1524,6739 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A500", 0, 16596,
      { 6046,-1127,-278,-5574,13076,2786,-691,1419,7625 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A5", 0, 16596,  // same CMs: DSLR-A550, DSLR-A560
      { 4950,-580,-103,-5228,12542,3029,-709,1435,7371 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A700", 0, 0,
      { 5775,-805,-359,-8574,16295,2391,-1943,2341,7249 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A850", 0, 0,
      { 5413,-1162,-365,-5665,13098,2866,-608,1179,8440 } },
    { LIBRAW_CAMERAMAKER_Sony, "DSLR-A900", 0, 0,
      { 5209,-1072,-397,-8845,16120,2919,-1618,1803,8654 } },

    { LIBRAW_CAMERAMAKER_Sony, "ILCA-68", 0, 0,
      { 6435,-1903,-536,-4722,12449,2550,-663,1363,6517 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCA-77M2", 0, 0,
      { 5991,-1732,-443,-4100,11989,2381,-704,1467,5992 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCA-99M2", 0, 0,
      { 6660,-1918,-471,-4613,12398,2485,-649,1433,6447 } },

    { LIBRAW_CAMERAMAKER_Sony, "ILCE-1", 0, 0,
      { 8161, -2947, -739, -4811, 12668, 2389, -437, 1229, 6524}},
    { LIBRAW_CAMERAMAKER_Sony, "ILCE-7RM4", 0, 0, // same CMs: ILCE-7RM4, ILCE-7RM4A
      { 7662, -2686,-660,-5240, 12965,2530, -796, 1508, 6167 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCE-7RM3", 0, 0, // same CMs: ILCE-7RM3, ILCE-7RM3A
      { 6640,-1847,-503,-5238,13010,2474,-993,1673,6527 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCE-7RM2", 0, 0,
      { 6629,-1900,-483,-4618,12349,2550,-622,1381,6514 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCE-7R", 0, 0,
      { 4913,-541,-202,-6130,13513,2906,-1564,2151,7183 } },

    { LIBRAW_CAMERAMAKER_Sony, "ILCE-7SM3", 0, 0,
      { 6912,-2127,-469,-4470,12175,2587,-398,1478,6492 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCE-7S", 0, 0, // same CMs: ILCE-7S, ILCE-7SM2
      { 5838,-1430,-246,-3497,11477,2297,-748,1885,5778 } },

    { LIBRAW_CAMERAMAKER_Sony, "ILCE-7C", 0, 0,
      { 7374,-2389,-551,-5435,13162,2519,-1006,1795,6552 } },

    { LIBRAW_CAMERAMAKER_Sony, "ILCE-7M3", 0, 0,
      { 7374,-2389,-551,-5435,13162,2519,-1006,1795,6552 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCE-7", 0, 0, // same CMs: ILCE-7, ILCE-7M2
      { 5271,-712,-347,-6153,13653,2763,-1601,2366,7242 } },

    { LIBRAW_CAMERAMAKER_Sony, "ILCE-9", 0, 0, // same CMs: ILCE-9, ILCE-9M2
      { 6389,-1703,-378,-4562,12265,2587,-670,1489,6550 } },

    { LIBRAW_CAMERAMAKER_Sony, "ILCE-6100", 0, 0,
      { 7657,-2847,-607,-4083,11966,2389,-684,1418,5844 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCE-6300", 0, 0,
      { 5973,-1695,-419,-3826,11797,2293,-639,1398,5789 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCE-6400", 0, 0,
      { 7657,-2847,-607,-4083,11966,2389,-684,1418,5844 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCE-6500", 0, 0,
      { 5973,-1695,-419,-3826,11797,2293,-639,1398,5789 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCE-6600", 0, 0,
      { 7657,-2847,-607,-4083,11966,2389,-684,1418,5844 } },
    { LIBRAW_CAMERAMAKER_Sony, "ILCE", 0, 0, // same CMs: ILCE-3000, ILCE-5000, ILCE-5100, ILCE-6000, ILCE-QX1
      { 5991,-1456,-455,-4764,12135,2980,-707,1425,6701 } },

    { LIBRAW_CAMERAMAKER_Sony, "ILME-FX3", 0, 0,
      { 6912, -2127, -469, -4470, 12175, 2587, -398, 1478, 6492 } },

    { LIBRAW_CAMERAMAKER_Sony, "NEX-5N", 0, 0,
      { 5991,-1456,-455,-4764,12135,2980,-707,1425,6701 } },
    { LIBRAW_CAMERAMAKER_Sony, "NEX-5R", 0, 0,
      { 6129,-1545,-418,-4930,12490,2743,-977,1693,6615 } },
    { LIBRAW_CAMERAMAKER_Sony, "NEX-5T", 0, 0,
      { 6129,-1545,-418,-4930,12490,2743,-977,1693,6615 } },
    { LIBRAW_CAMERAMAKER_Sony, "NEX-5", 0, 0,
      { 6549,-1550,-436,-4880,12435,2753,-854,1868,6976 } },
    { LIBRAW_CAMERAMAKER_Sony, "NEX-3N", 0, 0,
      { 6129,-1545,-418,-4930,12490,2743,-977,1693,6615 } },
    { LIBRAW_CAMERAMAKER_Sony, "NEX-3", 0, 0,
      { 6549,-1550,-436,-4880,12435,2753,-854,1868,6976 } },
    { LIBRAW_CAMERAMAKER_Sony, "NEX-6", 0, 0,
      { 6129,-1545,-418,-4930,12490,2743,-977,1693,6615 } },
    { LIBRAW_CAMERAMAKER_Sony, "NEX-7", 0, 0,
      { 5491,-1192,-363,-4951,12342,2948,-911,1722,7192 } },
    { LIBRAW_CAMERAMAKER_Sony, "NEX-VG30", 0, 0,
      { 6129,-1545,-418,-4930,12490,2743,-977,1693,6615 } },
    { LIBRAW_CAMERAMAKER_Sony, "NEX-VG900", 0, 0,
      { 6344,-1612,-462,-4863,12477,2681,-865,1786,6899 } },
    { LIBRAW_CAMERAMAKER_Sony, "NEX", 0, 0, // same CMs: NEX-C3, NEX-F3, NEX-VG20
      { 5991,-1456,-455,-4764,12135,2980,-707,1425,6701 } },
    { LIBRAW_CAMERAMAKER_Sony, "SLT-A33", 0, 0,
      { 6069,-1221,-366,-5221,12779,2734,-1024,2066,6834 } },
    { LIBRAW_CAMERAMAKER_Sony, "SLT-A35", 0, 0,
      { 5986,-1618,-415,-4557,11820,3120,-681,1404,6971 } },
    { LIBRAW_CAMERAMAKER_Sony, "SLT-A37", 0, 0,
      { 5991,-1456,-455,-4764,12135,2980,-707,1425,6701 } },
    { LIBRAW_CAMERAMAKER_Sony, "SLT-A55", 0, 0,
      { 5932,-1492,-411,-4813,12285,2856,-741,1524,6739 } },
    { LIBRAW_CAMERAMAKER_Sony, "SLT-A5", 0, 0, // same CMs: SLT-A57, SLT-A58
      { 5991,-1456,-455,-4764,12135,2980,-707,1425,6701 } },
    { LIBRAW_CAMERAMAKER_Sony, "SLT-A65", 0, 0,
      { 5491,-1192,-363,-4951,12342,2948,-911,1722,7192 } },
    { LIBRAW_CAMERAMAKER_Sony, "SLT-A77", 0, 0,
      { 5491,-1192,-363,-4951,12342,2948,-911,1722,7192 } },
    { LIBRAW_CAMERAMAKER_Sony, "SLT-A99", 0, 0,
      { 6344,-1612,-462,-4863,12477,2681,-865,1786,6899 } },
    { LIBRAW_CAMERAMAKER_Sony, "MODEL-NAME", 0, 0,
      { 5491,-1192,-363,-4951,12342,2948,-911,1722,7192 } },

    { LIBRAW_CAMERAMAKER_YI, "M1", 0, 0,
      { 7712,-2059,-653,-3882,11494,2726,-710,1332,5958 } },
  };
  // clang-format on

  double cam_xyz[4][3];
  //char name[130];
  int i, j;

  if (colors > 4 || colors < 1)
    return 1;

  int bl4 = (cblack[0] + cblack[1] + cblack[2] + cblack[3]) / 4, bl64 = 0;
  if (cblack[4] * cblack[5] > 0)
  {
    for (unsigned c = 0; c < 4096 && c < cblack[4] * cblack[5]; c++)
      bl64 += cblack[c + 6];
    bl64 /= cblack[4] * cblack[5];
  }
  int rblack = black + bl4 + bl64;

  for (i = 0; i < int(sizeof table / sizeof *table); i++)
  {
	  if (table[i].m_idx == make_idx)
	  {
		  size_t l = strlen(table[i].prefix);
		  if (!l ||  !strncasecmp(t_model, table[i].prefix, l))
		  {
			  if (!dng_version)
			  {
				  if (table[i].t_black > 0)
				  {
					  black = (ushort)table[i].t_black;
					  memset(cblack, 0, sizeof(cblack));
				  }
				  else if (table[i].t_black < 0 && rblack == 0)
				  {
					  black = (ushort)(-table[i].t_black);
					  memset(cblack, 0, sizeof(cblack));
				  }
				  if (table[i].t_maximum)
					  maximum = (ushort)table[i].t_maximum;
			  }
			  if (table[i].trans[0])
			  {
				  for (raw_color = j = 0; j < 12; j++)
					  if (internal_only)
						  imgdata.color.cam_xyz[j / 3][j % 3] = table[i].trans[j] / 10000.f;
					  else
                          ((double *)cam_xyz)[j] = imgdata.color.cam_xyz[j / 3][j % 3] = table[i].trans[j] / 10000.f;
				  if (!internal_only)
					  cam_xyz_coeff(rgb_cam, cam_xyz);
			  }
			  return 1; // CM found
		  }
	  }
  }
  return 0; // CM not found
}
void LibRaw::simple_coeff(int index)
{
  static const float table[][12] = {
      /* index 0 -- all Foveon cameras */
      {1.4032, -0.2231, -0.1016, -0.5263, 1.4816, 0.017, -0.0112, 0.0183,
       0.9113},
      /* index 1 -- Kodak DC20 and DC25 */
      {2.25, 0.75, -1.75, -0.25, -0.25, 0.75, 0.75, -0.25, -0.25, -1.75, 0.75,
       2.25},
      /* index 2 -- Logitech Fotoman Pixtura */
      {1.893, -0.418, -0.476, -0.495, 1.773, -0.278, -1.017, -0.655, 2.672},
      /* index 3 -- Nikon E880, E900, and E990 */
      {-1.936280, 1.800443, -1.448486, 2.584324, 1.405365, -0.524955, -0.289090,
       0.408680, -1.204965, 1.082304, 2.941367, -1.818705}};
  int i, c;

  for (raw_color = i = 0; i < 3; i++)
    FORCC rgb_cam[i][c] = table[index][i * MIN(colors,4) + c];
}
