/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * display.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <glib.h>

#include "display.h"

#include "var.h"

/* this mutex provides a lock for stdout to avoid display conflict between thread */
G_LOCK_DEFINE_STATIC(stdout);

static FILE *log_fd=NULL;
static FILE *errlog_fd=NULL;

/* this is the list of strings prefixed to displayed messaged */
/* always 5 bytes + a ] at the end. They must be in the same */
/* order as in DISP_MSG_TYPE */
static const char *msg_str_type[]={
										"ERR  ]",
                              "DEBUG]",
                              "INFO ]",
                              "USER ]",
                              "OP   ]",
                              "USER+]",
                              "USER-]",
                              "CHAT ]",
                              "UINFO]",
                              "HUBNM]",
                              "SREST]",
                              "$USER]",
                              "$UL+ ]",
                              "$UL- ]",
                              "$UL# ]",
                              "$DL+ ]",
                              "$DL- ]",
                              "$DL# ]",
                              "HUB- ]",
                              "PRIV ]",
                              "XFERR]",
                              "XFERQ]",
										"HUBGO]",
                              "$DL~ ]",
										"CMDKB]",
                              "HUB+ ]",
                              "$LS+ ]",
                              "$LS- ]",
                              "$LS# ]",
										"FLST ]",
										"FLSTE]",
                              "$UL= ]",
                              "$DL= ]",
                              "$LS= ]",
                              "$LS~ ]",
										"ASTRT]",
										"ASTOP]",
										"RFRSH]",
										"BXFER]",
										"VAR  ]",
										"EXFER]",
										"PASWD]",
										"ADMIN]",
                              "$ULST]",
										"PRGBR]",
										"GQLSB]",
										"GQLSC]",
										"GQLSE]",
										"GLSTB]",
										"GLSTC]",
										"GLSTE]",
										"UALSB]",
										"UALSC]",
										"UALSE]",
										"UALS+]",
										"UALS-]",
										"DRLAY]",		/* this message is never displayed, only the message content is */
										"FLSTB]",
										"LUSER]",
                              };

#define ERR_TO_LOG(x)	if((errlog_fd!=NULL)&&(msg_type==ERR_MSG))	{x;}


/*******************************/
/* display a formatted message */
/**************************************************************************/
/* this function is thread safe, multiple disp_msg won't mix their output */
/* The displayed message will have this form:                             */
/* MSGTP] "fnc_name"xxxxx|yyyyy|                                          */
/* fnc_name can be NULL. one or more string can be provided. They will be */
/* append after the fnc_name and will be separated by |                   */
/**************************************************************************/
/* if produced_string is not NULL, the printed string is not freed but */
/* put inside.                                                         */
/***********************************************************************/
void disp_msg_full(DISP_MSG_TYPE msg_type,GString **produced_string,const char *fnc_name, ...)
{
	va_list ap;
	char *t;
	GString *o_str;
	int i;

	if(produced_string!=NULL)
		*produced_string=NULL;

	if((msg_type==DEBUG_MSG)&&(debug_mode==0))
		return;

	G_LOCK(stdout);
	o_str=g_string_new("");

	va_start(ap,fnc_name);

	if(msg_type==DISPLAY_RELAY)
	{
		t=va_arg(ap,char *);
		o_str=g_string_append(o_str,t);
		goto end_of_decode;
	}

	if(msg_str_type[msg_type]!=NULL)
	{
		g_string_sprintf(o_str,"%s",msg_str_type[msg_type]);
	}
	else
	{
		g_string_sprintf(o_str,"     ]");
	}

	if(fnc_name!=NULL)
	{
		g_string_sprintfa(o_str," \"%s\"",fnc_name);
	}
	else
	{
		g_string_sprintfa(o_str," \"\"");
	}

	t=va_arg(ap,char *);
	while(t!=NULL)
	{
		if(t[0]=='|')			/* we can't use % like printf because it can appears (in nickname for instance). I use | */
		{
			switch(t[1])
			{
				case 's':		g_string_sprintfa(o_str,"%s|",va_arg(ap,char *));
									break;

				case 'd':		g_string_sprintfa(o_str,"%d|",va_arg(ap,int));
									break;
				case 'u':		g_string_sprintfa(o_str,"%u|",va_arg(ap,unsigned int));
									break;

				case 'l':		switch(t[2])
									{
										case 'd':		g_string_sprintfa(o_str,"%ld|",va_arg(ap,long int));
															break;
										case 'u':		g_string_sprintfa(o_str,"%lu|",va_arg(ap,unsigned long int));
															break;

										default:	
															/* never display an error message during a display, stdout is locked */
															g_string_sprintfa(o_str,"unknown pattern: %s|",t);
															break;
									}
									break;

				default:			
									g_string_sprintfa(o_str,"unknown pattern: %s|",t);
									break;
			}
		}
		else
		{
			g_string_sprintfa(o_str,"%s|",t);
			/* we can use | without problem because it is a reserved character in DC, it will never appears anywhere */
		}

		t=va_arg(ap,char *);
	}

	end_of_decode:
	g_string_sprintfa(o_str,"\n");
	/* now, o_str is the string to display */
	
	if(keyb_fd==0)			/* display the message if term exists */
		printf("%s",o_str->str);

	/* if log file exists, put it into */
	if(log_fd!=NULL)
		fprintf(log_fd,"%s",o_str->str);

	/* if err log file exists and it is an ERR] message, put it into */
	ERR_TO_LOG(fprintf(errlog_fd,"%s",o_str->str))

	G_LOCK(local_client_socket);
	if((local_client_socket!=NULL)&&(local_client_socket->len!=0))
	{
		/* send the message to all client currently connected */
		for(i=local_client_socket->len-1;i>=0;i--)
		{
			int res;

			res=send(g_array_index(local_client_socket,int,i),o_str->str,o_str->len,0);

			if(res!=o_str->len)
			{	/* on error, the client socket is closed and the socket is removed from the list */
				int sck;

				sck=g_array_index(local_client_socket,int,i);
				shutdown(sck,2);
				close(sck);
				local_client_socket=g_array_remove_index_fast(local_client_socket,i);
			}
		}
	}
	G_UNLOCK(local_client_socket);

	if(produced_string==NULL)
		g_string_free(o_str,TRUE);
	else
		(*produced_string)=o_str;

	if(log_fd!=NULL)
		fflush(log_fd);
	if(errlog_fd!=NULL)
		fflush(errlog_fd);

	va_end(ap);

	fflush(stdout);			/* flush stdout before unlocking */
	G_UNLOCK(stdout);
}

/***********************************/
/* change the current log filename */
/************************************************/
/* if filename==NULL, log into file is disabled */
/************************************************/
void change_logfile(char *filename)
{
	/* to avoid potential access conflict, we lock the display */
	G_LOCK(stdout);

	/* close the previously existing FILE */
	if(log_fd!=NULL)
	{
		fclose(log_fd);
		log_fd=NULL;
	}

	/* and open the new one if a filename is provided */
	if(filename!=NULL)
	{
		log_fd=fopen(filename,"ab");
		if(log_fd==NULL)
		{
			disp_msg(ERR_MSG,"change_logfile",strerror(errno),NULL);
		}
	}
	G_UNLOCK(stdout);
}

/***************************************/
/* change the current ERR log filename */
/************************************************/
/* if filename==NULL, log into file is disabled */
/************************************************/
void change_errlogfile(char *filename)
{
	/* to avoid potential access conflict, we lock the display */
	G_LOCK(stdout);

	/* close the previously existing FILE */
	if(errlog_fd!=NULL)
	{
		fclose(errlog_fd);
		errlog_fd=NULL;
	}

	/* and open the new one if a filename is provided */
	if(filename!=NULL)
	{
		errlog_fd=fopen(filename,"ab");
		if(errlog_fd==NULL)
		{
			disp_msg(ERR_MSG,"change_errlogfile",strerror(errno),NULL);
		}
	}
	G_UNLOCK(stdout);
}

