/* pkginfo.c - Retrieve information about packages for deborphan.
   Copyright (C) 2000 Cris van Pelt

   Distributed under the terms of the Artistic License. */

/* This code is not nearly as advanced as dpkg's. It assumes a
   "perfect" statusfile. If the status-file is corrupted it may
   result in deborphan giving the wrong packages, or even crashing. */

#include <stdlib.h>
#include <string.h>
#include <regex.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <deborphan.h>
#include <set.h>

#ifdef USE_XALLOC
#  include <xalloc.h>
#endif

extern int options[];

static regex_t re_statusinst, re_statushold, re_namedev, re_gnugrepv;

void
init_pkg_regex()
{
    regcomp(&re_statusinst,
	    "^Status:.*[^-]installed$", REG_EXTENDED | REG_FLAGS);
    regcomp(&re_statushold,
	    "^Status:hold.*[^-]installed$", REG_EXTENDED | REG_FLAGS);

    if (options[GUESS]) {
	char guess[30];
	guess[0] = '\0';

	if (guess_chk(GUESS_PERL))
	    strcat(guess, "^lib.*-perl$|");
	if (guess_chk(GUESS_DEV))
	    strcat(guess, "-dev$|");
	if (guess_chk(GUESS_DEBUG))
	    strcat(guess, "-dbg$|");
	if (guess_chk(GUESS_SECTION)) {
	    strcat(guess, "^lib");
	    regcomp(&re_gnugrepv, "(-perl|-dev|-doc|-dbg)$", 
		    REG_EXTENDED | REG_FLAGS);
	}
	if (guess[strlen(guess)-1] == '|')
	    guess[strlen(guess)-1] = '\0';

	regcomp(&re_namedev, guess, REG_EXTENDED | REG_FLAGS);
    }
}

void
free_pkg_regex()
{
    regfree(&re_statusinst);
    regfree(&re_statushold);
    regfree(&re_namedev);
    regfree(&re_gnugrepv);
}

/* A similar "hack" was created by Paul Martin a while ago. It was not
 * implemented then for various reasons. This selects the function to
 * call to get the info, based on the first few characters.
 * Not as versatile as regular expressions, but it makes up for that in
 * speed.
 */
void
get_pkg_info(const char *line, pkg_info * package)
{
    const char c = upcase(line[0]);
    char t;

    switch (c) {
    case 'P':
	t = upcase(line[2]);
	switch (t) {
	case 'I':		/* PrIority */
	    get_pkg_priority(line, package);
	    break;
	case 'C':		/* PaCkage */
	    get_pkg_name(line, package);
	    break;
	case 'O':		/* PrOvides */
	    get_pkg_provides(line, package);
	    break;
	case 'E':		/* PrE-depends */
	    get_pkg_deps(line, package);
	    break;
	}
	break;
    case 'D':
	if (upcase(line[2]) == 'P')	/* DePends */
	    get_pkg_deps(line, package);
	break;
    case 'R':			/* ReCommends */
	if (options[NICE_MODE] && upcase(line[2]) == 'C')
	    get_pkg_deps(line, package);
	break;
    case 'S':
	t = upcase(line[1]);
	switch (t) {
	case 'E':		/* SEction */
	    get_pkg_section(line, package);
	    break;
	case 'T':		/* STatus */
	    get_pkg_status(line, package);
	    break;
	case 'U':		/* SUggests */
	    if (options[NICE_MODE])
		get_pkg_deps(line, package);
	    break;
	}
    }
}

void
get_pkg_deps(const char *line, pkg_info * package)
{
    char *tok, *line2, *version = NULL;
    static unsigned int num_deps;
    unsigned int i, dup = 0;
    dep d;

    line2 = strchr(line, ':')+1;

    if (!package->deps[0].name)
        num_deps = 0;

    for (; (tok = strsep(&line2, ",|")); num_deps++) {
	/* Versions are up to dpkg. */
        if ((version = strchr(tok, '(')))
            *version = '\0';

	set_dep(&d, tok);
	for (i = dup = 0; package->deps[i].name; i++) {
	    if (pkgcmp(package->deps[i], d)) {
		dup = 1;
  		break;
	    }
	}

	if (dup)
	    num_deps--;
	else
  	    package->deps[num_deps] = d;
    }

    package->deps[num_deps].name = NULL;
}

void
get_pkg_priority(const char *line, pkg_info * package)
{
    set_priority(package, strchr(line, ':')+1);
}

void
get_pkg_provides(const char *line, pkg_info * package)
{
    char *prov, *name;
    int i = 0;

    prov = strchr(line, ':') + 1;

    for (i = 0; (name = strsep(&prov, ",")) || !i; i++) {
	if (!name)
	    name = prov;
	set_provides(package, name, i);
    }
}

void
get_pkg_name(const char *line, pkg_info * package)
{
    set_dep(&(package->self), strchr(line, ':') + 1);
}

void
get_pkg_status(const char *line, pkg_info * package)
{
    if (!regexec(&re_statusinst, line, 0, NULL, 0))
	set_install(package);

    if (!options[FORCE_HOLD]) {
	if (!regexec(&re_statushold, line, 0, NULL, 0))
	    set_hold(package);
    }
}

/* Okay, this function does not really check the libraryness of a package,
 * but it checks wether the package should be checked (1) or not (0).
 */
unsigned int
is_library(pkg_info *package)
{
    if (options[ALL_PACKAGES])
	return 1;

    if (!package->section)
	return 0;

    
    if (!options[GUESS_ONLY]) {
	if (strstr(package->section, "/libs") || 
	    strstr(package->section, "/oldlibs"))
	    return 1;
    }

    /* This ugliness is because GNU libc does not support collating elements.
     */
    if (options[GUESS]) {
	if (!regexec(&re_namedev, package->self.name, 0, NULL, 0)) {
	    if (guess_chk(GUESS_SECTION) && !guess_chk(GUESS_ALL)) {
		if (regexec(&re_gnugrepv, package->self.name, 0, NULL, 0))
		    return 1;
	    } else {
		return 1;
	    }
	}
    }

    return 0;
}

void
get_pkg_section(const char *line, pkg_info * package)
{
    char *section;

    section = strchr(line, ':')+1;

    if (strchr(section, '/'))
	set_section(package, section, NULL);
    else
	set_section(package, section, "main");
}

