 # debpartial-mirror - partial debian mirror package tool
# (c) 2004 Otavio Salvador <otavio@debian.org>, Nat Budin <natb@brandeis.edu>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import unittest

from TestBase import TestBase
from debpartial_mirror import Config

class ConfigurationFilesTests(TestBase):
    def loadConfig(self, conf_file, exception):
        """ Generic test code to handle configuration file load and
        exception handling for it. """
        try:
            cnf = Config.Config(conf_file)
            return cnf
        except Config.InvalidOption, msg:
            if not exception or not isinstance(msg, exception):
                self.fail("""Wrongly intercept as Config.InvalidOption exception. %s was expected.
                section: %s
                option:  %s""" % (type(exception), msg.section, msg.option))

        except Config.InvalidSection, msg:
            if not exception or not isinstance(msg, exception):
                self.fail("""Wrongly intercept as Config.InvalidSection exception. %s was expected.
                section: %s""" % (type(exception), msg.section))

        except Config.RequiredOptionMissing, msg:
            if not exception or not isinstance(msg, exception):
                self.fail("""Wrongly intercept as Config.RequiredOptionMissing exception. %s was expected.
                section: %s
                option:  %s""" % (type(exception), msg.section, msg.option))
        except Exception:
            self.fail("Wrongly intercept a strange exception. %s was expected." % type(exception))

        if not exception:
            self.fail("Fail to raise the need exception. %s was expected." % type(exception))
        
    def testGoodConfigurationFile(self):
        """Config: valid configuration file"""
        self.loadConfig(self.aux_file('TestConfig-good.conf'), None)

    def testMissingRequiredOption(self):
        """Config: missing required option"""
        self.loadConfig(self.aux_file('TestConfig-missing-required.conf'), Config.RequiredOptionMissing)

    def testInvalidOption(self):
        """Config: invalid option"""
        self.loadConfig(self.aux_file('TestConfig-invalid-option.conf'), Config.InvalidOption)

    def testInvalidOptionValue(self):
        """Config: invalid option value"""
        self.loadConfig(self.aux_file('TestConfig-invalid-option-value.conf'), Config.InvalidOption)
    
    def testSectionType(self):
        """Config: invalid option due a section type"""
        self.loadConfig(self.aux_file('TestConfig-section-types.conf'), Config.InvalidOption)

    def testInvalidFilter(self):
        """Config: invalid filter"""
        self.loadConfig(self.aux_file('TestConfig-invalid-filter.conf'), Config.InvalidOption)

    def testInvalidSection(self):
        """Config: invalid section"""
        self.loadConfig(self.aux_file('TestConfig-invalid-section.conf'), Config.InvalidSection)

    def testTypeCast(self):
        """Config: type casting"""
        def castCompare(option, section, right):
            self.failUnless(cnf.get_option(option, section) == right,
                            "We were expecting to receive '%s' but we received '%s'."
                            % (right, cnf.get_option(option, section)))

        cnf = self.loadConfig(self.aux_file('TestConfig-typecast.conf'), None)

        # List conversion
        castCompare('architectures', 'GLOBAL', ['i386', 'powerpc'])
        castCompare('components', 'GLOBAL', ['main'])
        castCompare('distributions', 'GLOBAL', ['stable'])

        # Boolean conversion
        castCompare('get_suggests', 'GLOBAL', False)
        castCompare('get_recommends', 'GLOBAL', True)
        castCompare('get_provides', 'GLOBAL', True)
        castCompare('get_packages', 'GLOBAL', True)
        castCompare('get_sources', 'GLOBAL', False)

def suite():
    suite = unittest.TestSuite()

    suite.addTest(unittest.makeSuite(ConfigurationFilesTests, 'test'))

    return suite

if __name__ == '__main__':
    log = logging.getLogger()
    log_handler = logging.FileHandler(sys.argv[0][:-3] + '.log')
    log.setLevel(logging.DEBUG)
    log.addHandler(log_handler)
    unittest.main(defaultTest='suite')