/*
 * Detect a Library for hardware detection
 *
 * Copyright (C) 1998-2000 MandrakeSoft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "serial.h"


#include "detect.h"
#include "utils.h"

#define SERIAL_DEVICES "/dev/ttyS"

#define SERIAL_DOS_EQUIVALENT "COM"

#define NUMBER_OF_VALID_SERIAL_PORTS 4

/* there are two possible bytes to signify the start of a PnP ID      */
/* string                                                             */
#define BeginPnP1 0x28
#define BeginPnP2 0x08

/* Likewise, two possible stop bytes                                  */
#define EndPnP1   0x29
#define EndPnP2   0x09

/* these chars indicate extensions to the base dev id exist           */
#define ExtendPnP1 0x5c
#define ExtendPnP2 0x3c

#define PNP_COM_MAXLEN 256


struct serial_info *serial_detect(struct cards_lst *lst){
  char *testing_port = (char *)NULL;
  struct serial_info *result = (struct serial_info *)NULL;
  static struct serial_info *first = (struct serial_info *)NULL;

  struct cards_lst *bkup_lst;
  char *dev_id; 
  int i;
  int fd;
  int test, found;
  int serial_attr;
  struct termios origattr;
  struct pnp_com_id pnp_id;
  char *string;

  if(debug)
    fprintf(stdout, "\nProbing serial ports...\n");

  /********************************************************************/
  /********************** SERIAL PORT DETECTION ***********************/
  /********************************************************************/

  /* Need to try to open /dev/ttyS<i> ports */
  for(i=0; i < NUMBER_OF_VALID_SERIAL_PORTS; i++){
    testing_port = (char *)
                   my_malloc((strlen(SERIAL_DEVICES) + 2)*sizeof(char));
    sprintf(testing_port, "%s%d", SERIAL_DEVICES, i);
    
    if((fd = open_serial_port(testing_port)) < 0){
      free(testing_port);
      continue;
    }/*endif*/
    serial_attr = get_serial_attr(fd, &origattr);
    if(serial_attr < 0){
      free(testing_port);
      continue;
    }/*endif*/
    if(!first){
      first = result = (struct serial_info *)
                                  my_malloc(sizeof(struct serial_info));
    }else{
      result->next = (struct serial_info *)
                                  my_malloc(sizeof(struct serial_info));
      result = result->next;
    }/*endif*/
    result->next = (struct serial_info *)NULL;
    
    result->device = testing_port;
    result->dos_equivalent = (char *)
            my_malloc((strlen(SERIAL_DOS_EQUIVALENT) + 3)*sizeof(char));
    sprintf(result->dos_equivalent, "%s%d", SERIAL_DOS_EQUIVALENT, i);
    result->vendor = s_unknown;
    result->model = s_unknown;
    result->modulename = s_unknown;
    result->type = UNKNOWN_DEVICE;
    result->speed = -1;
    result->dev_id = s_unknown;

    test = find_legacy_modem(fd);
    if(test == 3){  /* we found a modem */
      string = modem_response(fd, "ATI9\r");
      if(parse_pnp_string(string, strlen(string), &pnp_id) == 0){
        dev_id = (char *)my_malloc(9);
        sprintf(dev_id, "%3s%4s", pnp_id.eisa_id, 
                                  pnp_id.product_id);
        found = 0;
        for(bkup_lst = lst; bkup_lst; bkup_lst = bkup_lst->next){
          if(bkup_lst->bus == SERIAL){
            if(!strcmp(dev_id, bkup_lst->dev_id)){
              result->vendor = bkup_lst->vendor;
              result->model = bkup_lst->model;
              result->modulename = bkup_lst->modulename;
              result->type = MODEM;
              result->dev_id = bkup_lst->dev_id;
              found = 1;
            }/*endif*/
          }/*endif*/
        }/*next bkup_lst*/
        if(!found){
          result->dev_id = dev_id;
        }/*endif*/
      }/*endif*/
      result->speed = modem_speed(fd);
/*
      c = modem_capabilities(fd);
      if(c & MODEM_V90)
        strcat(cap, "V.90/");
      if(c & MODEM_VX2)
        strcat(cap, "V.X2/");
      if(c & MODEM_V34BS)
        strcat(cap, "V.34BS/");
      if(c & MODEM_V34S)
        strcat(cap, "V.34S/");
      if(c & MODEM_V34B)
        strcat(cap, "V.34B/");
      if(c & MODEM_V34)
        strcat(cap, "V.34/");
      if(c & MODEM_V32B)
        strcat(cap, "V.32B/");
      if(c & MODEM_V32)
        strcat(cap, "V.32/");
      if(c & MODEM_FAX1)
        strcat(cap, "FAX Class 1/");
      if(c & MODEM_FAX2)
        strcat(cap, "FAX Class 2/");
      if(c & MODEM_MNP5)
        strcat(cap, "MNP5/");
      if(c & MODEM_V42BIS)
        strcat(cap, "V.42bis/");
      cap[strlen(cap) - 1] = 0;
*/

    }/*endif*/
    set_serial_attr(fd, &origattr);
    close_serial_port(fd);
    /* I don't know why the following two lines are needed...
       but if they're left out, detect doesn't find the modem
       if it's started again. */
    if((fd = open_serial_port(testing_port)) < 0){
      fprintf(stderr,
              _("\nmodem_detect(): Couldn't open previously open "
                "port %s\n"), result->device);
      exit(1);
    }else{
      close_serial_port(fd);
    }/*endif*/
 
    if(debug)
      fprintf(stdout, "\t\tFound [%s]\n", testing_port);
  }/*next i*/
  
  return first;
}/*endfunc serial_detect*/


/**********************************************************************/
/* wait_for_input:  Wait for a reaction on a device attached to a     */
/*                  file descriptor.                                  */
/*                                                                    */
/*     fd: file descriptor.                                           */
/*     timeout: maximum time to wait.                                 */
/**********************************************************************/
extern int wait_for_input(int fd, struct timeval *timeout){
  fd_set ready;
  int n;
  
  FD_ZERO(&ready);
  FD_SET(fd, &ready);
  
  n = select(fd + 1, &ready, NULL, &ready, timeout);
  
  return n;
}/*endfunc wait_for_input*/


extern int open_serial_port(char *port){
  int fd;

  fd = open(port, O_RDWR | O_NONBLOCK);
  if(fd < 0){
    return fd;
  }/*endif*/
  if(fcntl(fd, F_SETFL, 0) < 0){
    close(fd);
    return -1;
  }/*endif*/
  return fd;
}/*endfunc open_serial_port*/


extern int close_serial_port(int fd){
  return close(fd);
}/*endfunc close_serial_port*/


extern int get_serial_lines(int fd){
  int modem_lines;
  
  ioctl(fd, TIOCMGET, &modem_lines);
  return modem_lines;
}/*endfunc get_serial_lines*/


extern int set_serial_lines(int fd, int modem_lines){
  return ioctl(fd, TIOCMSET, &modem_lines);
}/*endfunc set_serial_lines*/


extern int get_serial_attr(int fd, struct termios *attr){
  return tcgetattr(fd, attr);
}/*endfunc get_serial_lines*/


extern int set_serial_attr(int fd, struct termios *attr){
  return tcsetattr(fd, TCSANOW, attr);
}/*endfunc set_serial_attr*/


extern int setup_serial_port(int fd, int nbits, struct termios *attr){
  attr->c_iflag = IGNBRK | IGNPAR;
  attr->c_cflag = 0;
  attr->c_cflag &= ~(CSIZE | CSTOPB | PARENB | PARODD | PARENB);
  attr->c_cflag |= CREAD | CLOCAL;
  if(nbits == 7){
    attr->c_cflag |= CS7 | CSTOPB;
  }else{
    attr->c_cflag |= CS8;
  }/*endif*/
  attr->c_oflag = 0;
  attr->c_lflag = 0;
  
  attr->c_cc[VMIN] = 1;
  attr->c_cc[VTIME] = 5;
  
  cfsetospeed(attr, B1200);
  cfsetispeed(attr, B1200);
  return set_serial_attr(fd, attr);
}/*endfunc setup_serial_port*/


extern struct termios *init_serial_port(int fd){
  int modem_lines, temp;
  struct termios *portattr = (struct termios *) 
                                      my_malloc(sizeof(struct termios));
  
  /*set port to 1200 baud, 8 bits, no parity, 1 stop bit */
  temp = get_serial_attr(fd, portattr);
  if(temp < 0){
    return (struct termios *)NULL;
  }/*endif*/
  
  /* goto 1200 baud, 8 bits */
  temp = setup_serial_port(fd, 8, portattr);
  if(temp < 0){
    return (struct termios *)NULL;
  }/*endif*/
  
  /*set on DTR and RTS */
  modem_lines = get_serial_lines(fd);
  modem_lines |= TIOCM_RTS | TIOCM_DTR;
  set_serial_lines(fd, modem_lines);
  usleep(200000);
  return portattr;
}/*endfunc init_serial_port*/


/**********************************************************************/
/* parse_pnp_string:  parse the PnP ID string into components         */
/*                                                                    */
/*    pnp_id_string: the PnP string to be parsed                      */
/*    pnp_len: length of the PnP string                               */
/*    pnp_id: pnp_com_id struct to be filled in                       */
/**********************************************************************/
int parse_pnp_string(unsigned char *pnp_id_string, int pnp_len,
                                             struct pnp_com_id *pnp_id){
  unsigned char *p1, *p2;
  unsigned char *start;
  unsigned char *end;
  unsigned char *curpos;
  unsigned char *endfield;
  unsigned char *temppos;
  unsigned char *pnp_string;
  unsigned char end_char;

  int no_more_extensions = 0;
  int stage;
  int len;
  unsigned short int checksum = 0;
  char hex_checksum[5];

  char extension_delims[] = {
    EndPnP1, 
    EndPnP2, 
    ExtendPnP1, 
    ExtendPnP2, 
    0
  };
  char end_delims[] = {
    EndPnP1, 
    EndPnP2, 
    0
  };

  /* clear out pnp_id                                                 */
  memset(pnp_id, 0, sizeof(*pnp_id));
  
  /* copy pnp_string to temp space                                    */
  pnp_string = alloca(pnp_len+1);
  memcpy(pnp_string, pnp_id_string, pnp_len+1);
  
  /* first find the start of the PnP part of string                   */
  p1 = memchr(pnp_string, BeginPnP1, pnp_len);
  p2 = memchr(pnp_string, BeginPnP2, pnp_len);
  
  /* use the one which points nearest to start of the string          */
  /* and is actually defined                                          */
  if(p1 && p2){
    start = (p1 < p2) ? p1 : p2;
  }else if(p1){
    start = p1;
  }else if(p2){
    start = p2;
  }else{
    start = NULL;
  }/*endif*/
  
  /* if no start then we're done */
  if(!start){
    return -1;
  }/*endif*/
    
  /* the length of the initial part cannot be more than 17 bytes      */
  if((start - pnp_string) > 17){
    return -1;
  }/*endif*/
  
  /* setup end character we are looking for based on the start        */
  /* character                                                        */
  if(start == p2){
    pnp_id->xlate_6bit = 1;
    end_char = EndPnP2;
    /* we need to xlate data in PnP fields.                           */
    /* remember to skip the revision fields (bytes 1 and 2 after      */
    /* start)                                                         */
    temppos=start;
    while(1){
      if(*temppos == EndPnP2){
        *temppos += 0x20;
        break;
      }else if(temppos != start+1 && temppos != start+2){
        *temppos += 0x20;
      }/*endif*/
      temppos++;
    }/*endwhile*/
  }else{
    pnp_id->xlate_6bit = 0;
    end_char = EndPnP1;
  }/*endif*/
  
  /* move everything before the start of the PnP block                */
  memcpy(pnp_id->other_id, pnp_string, start-pnp_string);
  pnp_id->other_len = start - pnp_string;
  
  /* now we get the PnP fields - all were zero'd out above            */
  curpos = start+1;
  memcpy(pnp_id->pnp_rev, curpos, 2);
  curpos += 2;
  memcpy(pnp_id->eisa_id, curpos, 3);
  curpos += 3;
  memcpy(pnp_id->product_id, curpos, 4);
  curpos += 4;
  /* now we see if have extension fields */
  no_more_extensions = 0;
  stage = 0;
  while(!no_more_extensions){
    if(*curpos == ExtendPnP1 || *curpos == ExtendPnP2){
      curpos++;
      endfield = strpbrk(curpos, extension_delims);
      if(!endfield){
        return -1;
      }/*endif*/
      /* if we reached the end of all PnP data, back off */
      /* cause there is a checksum at the end of extension data */
      if(*endfield == EndPnP1 || *endfield == EndPnP2){
        endfield -= 2;
      }else{
        break;
      }/*endif*/
      len = endfield - curpos;
      switch(stage){
      case 0:
        if(len != 8 && len != 0){
          return -1;
        }/*endif*/    
        memcpy(pnp_id->serial_number, curpos, len);
        curpos += len;
        break;
      case 1:
        if(len > 33){
          return -1;
        }/*endif*/
        memcpy(pnp_id->class_name, curpos, len);
        curpos = endfield;
        break;
      case 2:
        if(len > 41){
          return -1;
        }/*endif*/
        memcpy(pnp_id->driver_id, curpos, len);
        curpos = endfield;
        break;
      case 3:
        if(len > 41){
          return -1;
        }/*endif*/
        memcpy(pnp_id->user_name, curpos, len);
        curpos = endfield;
        break;
      }/*endswitch*/
      stage++;
    }/*endif*/
    
    /* now find the end of all PnP data                               */
    end = strpbrk(curpos, end_delims);
    if(!end){
      return -1;
    }/*endif*/
    
    /* if we had any extensions, we expect an checksum                */
    if(stage != 0){
      /* copy checksum into struct                                    */
      memcpy(pnp_id->checksum, curpos, 2);
      /* compute the checksum as the sum of all PnP bytes, excluding  */
      /* the two byte checksum.                                       */
      checksum = 0;
      for(temppos=start; temppos <= end; temppos++){
        /* skip checksum in calculation */
        if(temppos == (end-2) || temppos == (end-1)){
          continue;
          /* dont xlate the revision at start */
          if(temppos != (start+1) && temppos != (start+2)){
            checksum += *temppos - ((pnp_id->xlate_6bit) ? 0x20 : 0);
          }else{
            checksum += *temppos;
          }/*endif*/
        }/*endif*/
      }/*next temppos*/
    }/*endif*/
    sprintf(hex_checksum, "%.2X", checksum & 0xff);
    if(strncmp(hex_checksum, pnp_id->checksum, 2)){
      return -1;
    }/*endif*/
  }/*endwhile*/
  /* checksum was ok, so we're done                                   */
  return 0;
}/*endfunc parse_pnp_string*/


/**********************************************************************/
/* print_pnp_id:  UNUSED except for debugging                         */
/**********************************************************************/
void print_pnp_id(struct pnp_com_id id){
  int i;
  int extensions_exist;
  int revision_temp;
  
  if (id.other_len != 0) {
    printf("Detected non-PnP data stream at start.\n");
    printf("  Length   = 0x%x\n",id.other_len);
    printf("  Contents =");
    for(i=0; i<id.other_len; i++){
      printf(" 0x%x",id.other_id[i]);
    }/*next i*/
    printf("\n");
  }else{
    printf("Non-PnP data stream not detected at start.\n");
  }/*endif*/
  
  /* parse PnP revision bytes into a string values (eg. "1.00") */
  revision_temp = ((id.pnp_rev[0]&0x3f) << 6)+(id.pnp_rev[1]&0x3f);
  sprintf(id.pnp_rev_str, "%d.%d", revision_temp/100,
                                   revision_temp % 100);
  printf("\nPnP Required fields:\n");
  printf("    Revision       = %s\n", id.pnp_rev_str);
  printf("    Manufacturer   = %s\n", id.eisa_id);
  printf("    Product ID     = %s\n", id.product_id);
  extensions_exist = id.serial_number[0] || id.class_name[0] ||
                     id.driver_id[0]     || id.user_name[0];
  if(extensions_exist){
    printf("\nPnP extension field(s) exist:\n");
    if(id.serial_number[0]){
      printf("    Serial Number   = %s\n", id.serial_number);
    }/*endif*/
    if(id.class_name[0]){
      printf("    PnP class name  = %s\n", id.class_name);
    }/*endif*/
    if (id.driver_id[0]){
      printf("    PnP Compatible  = %s\n", id.driver_id);
    }/*endif*/
    if (id.user_name[0]){
      printf("    PnP Description = %s\n", id.user_name);
    }/*endif*/
  }/*endif*/
}/*endfunc print_pnp_id*/

