/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/annotations.hpp
 *
 * $Id: front_end.hpp,v 1.20 2005/06/23 09:54:22 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta] Conditional Macros for annotating code.  This
 * file is meant as a quick interface -- a library should come with
 * its own annotations file (a template exists as
 * lib_annotations.hpp), such that it can be used at a specific build
 * level and with a specific configuration.
 */
#ifndef DIAGNOSTICS__ANNOTATIONS__INCLUDE_GUARD
#define DIAGNOSTICS__ANNOTATIONS__INCLUDE_GUARD

#include <diagnostics/macros/check_annotation.hpp>
#include <diagnostics/macros/assert_annotation.hpp>
#include <diagnostics/macros/block_annotation.hpp>
#include <diagnostics/macros/invariance_annotation.hpp>
#include <diagnostics/macros/exceptionless_block_annotation.hpp>
#include <diagnostics/macros/relation_modifier.hpp>

// ATTENTION: defined a second time in diagnostics/instrumentation.hpp
#ifndef DIAGNOSTICS_DEBUG__LEVEL__
#  define DIAGNOSTICS_DEBUG__LEVEL__ 2
#endif

#ifndef DIAGNOSTICS_INLINE
#  if DIAGNOSTICS_DEBUG__LEVEL__ == 0
#    define DIAGNOSTICS_INLINE inline 
#  else
#    define DIAGNOSTICS_INLINE 
#  endif
#endif


#ifndef DIAGNOSTICS_BASE_EXCEPTION_TYPE
#  define DIAGNOSTICS_BASE_EXCEPTION_TYPE ::diagnostics::Exception
#  include <diagnostics/frame/exception.hpp>
#  define DIAGNOSTICS_BASE_EXCEPTION_STR_WHAT(EX) (::std::string("EXCEPTION=\"") + (EX).name() + "\" WHAT=\"" + (EX).what() + "\"")
#endif

#ifndef DIAGNOSTICS_HANDLE_TEST_EXCEPTION_EXPLICITLY
#  define DIAGNOSTICS_HANDLE_TEST_EXCEPTION_EXPLICITLY 0
#endif


#ifndef DIAGNOSTICS_FAILED_ASSERT_ACTION
#  define DIAGNOSTICS_FAILED_ASSERT_ACTION(LEVEL,NAME,COND) throw NAME(#COND)
#endif

#ifndef DIAGNOSTICS_FAILED_ASSERT_ACTION1
#  define DIAGNOSTICS_FAILED_ASSERT_ACTION1(LEVEL,NAME,COND,EXCEP_WHAT) throw NAME(EXCEP_WHAT)
#endif

#ifndef DIAGNOSTICS_FAILED_CHECK_ACTION
#  define DIAGNOSTICS_FAILED_CHECK_ACTION(LEVEL,NAME,COND) throw NAME(#COND)
#endif

#ifndef DIAGNOSTICS_FAILED_CHECK_ACTION1
#  define DIAGNOSTICS_FAILED_CHECK_ACTION1(LEVEL,NAME,COND,EXCEP_WHAT) throw NAME(EXCEP_WHAT)
#endif


#ifndef DIAGNOSTICS_NR_WHAT_DEFAULT
#  define DIAGNOSTICS_NR_WHAT_DEFAULT 0
#endif 

////////////////////////////////////////////////////////////////////////////////

// derived config


#if DIAGNOSTICS_HANDLE_TEST_EXCEPTION_EXPLICITLY == 1
#  include <diagnostics/unittest/test_exception.hpp>
#endif

#ifndef DIAGNOSTICS_BASE_EXCEPTION_STR_WHAT
#  warning DIAGNOSTICS_BASE_EXCEPTION_STR_WHAT not defined
#endif

////////////////////////////////////////////////////////////////////////////////

// AUDIT

#if DIAGNOSTICS_DEBUG__LEVEL__ > 1

#  define DIAGNOSTICS_AUDIT_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_AUDIT,::diagnostics::TYPE_TRACE,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))
#  define DIAGNOSTICS_AUDIT_BINARY_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_AUDIT,::diagnostics::TYPE_TRACE_BINARY,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))

#  define DIAGNOSTICS_AUDIT_BLOCK_ENTER(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))
#  define DIAGNOSTICS_AUDIT_BLOCK_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define DIAGNOSTICS_AUDIT_BLOCK_GUARD(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))

#  define DIAGNOSTICS_AUDIT_PROCEDURE_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))
#  define DIAGNOSTICS_AUDIT_PROCEDURE_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define DIAGNOSTICS_AUDIT_PROCEDURE_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))

#  define DIAGNOSTICS_AUDIT_METHOD_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("MEHTOD=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))
#  define DIAGNOSTICS_AUDIT_METHOD_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define DIAGNOSTICS_AUDIT_METHOD_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("METHOD=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))

#  define DIAGNOSTICS_AUDIT_ASSERT(NAME,COND) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_AUDIT,\
                            DIAGNOSTICS_NR_WHAT_DEFAULT,\
                            "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                            (COND),\
                            DIAGNOSTICS_FAILED_ASSERT_ACTION(::diagnostics::LEVEL_AUDIT,NAME,COND))
#  define DIAGNOSTICS_AUDIT_ASSERT1(NAME,COND,EXCEP_WHAT) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_AUDIT,\
                            DIAGNOSTICS_NR_WHAT_DEFAULT,\
                            ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (EXCEP_WHAT) + "\"",\
                            (COND),\
                            DIAGNOSTICS_FAILED_ASSERT_ACTION1(::diagnostics::LEVEL_AUDIT,NAME,COND,EXCEP_WHAT))
#  define DIAGNOSTICS_AUDIT_ASSERT_LOG(NAME,COND) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_AUDIT,NAME,\
                            DIAGNOSTICS_NR_WHAT_DEFAULT,\
                            "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                            (COND),\
                            ((void)0))
#  define DIAGNOSTICS_AUDIT_ASSERT_LOG1(NAME,COND,EXCEP_WHAT) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_AUDIT,NAME,\
                            DIAGNOSTICS_NR_WHAT_DEFAULT,\
                            ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (EXCEP_WHAT) + "\"",\
                            (COND),\
                            ((void)0))
#  define DIAGNOSTICS_AUDIT_CHECK(NAME,COND) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_AUDIT,\
                           DIAGNOSTICS_NR_WHAT_DEFAULT,\
                           "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                           (COND),\
                           DIAGNOSTICS_FAILED_CHECK_ACTION(::diagnostics::LEVEL_AUDIT,NAME,COND))
#  define DIAGNOSTICS_AUDIT_CHECK1(NAME,COND,STR_WHAT) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_AUDIT,\
                           DIAGNOSTICS_NR_WHAT_DEFAULT,\
                           ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (STR_WHAT) + "\"",\
                           (COND),\
                           DIAGNOSTICS_FAILED_CHECK_ACTION1(::diagnostics::LEVEL_AUDIT,NAME,COND,STR_WHAT))

#  define DIAGNOSTICS_AUDIT_CLASS_THROWING_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER
#  define DIAGNOSTICS_AUDIT_CLASS_THROWING_INVARIANCE_EXIT  \
    DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(::diagnostics::LEVEL_AUDIT,\
                                           DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                           DIAGNOSTICS_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                           DIAGNOSTICS_BASE_EXCEPTION_TYPE,\
                                           DIAGNOSTICS_BASE_EXCEPTION_STR_WHAT)

#  define DIAGNOSTICS_AUDIT_CLASS_INVARIANCE_GUARD DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD
#  define DIAGNOSTICS_AUDIT_CLASS_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_INVARIANCE_ENTER
#  define DIAGNOSTICS_AUDIT_CLASS_INVARIANCE_EXIT  DIAGNOSTICS_BASE_CLASS_INVARIANCE_EXIT

#  define DIAGNOSTICS_AUDIT_CLASS_INVARIANCE_ASSERT  DIAGNOSTICS_BASE_CLASS_INVARIANCE_ASSERT

#  define DIAGNOSTICS_AUDIT_EXCEPTIONLESS_BLOCK_ENTER  DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_ENTER
#  define DIAGNOSTICS_AUDIT_EXCEPTIONLESS_BLOCK_EXIT  \
    DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_EXIT(::diagnostics::LEVEL_AUDIT,\
                                              DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                              true,\
                                              DIAGNOSTICS_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                              DIAGNOSTICS_BASE_EXCEPTION_TYPE,\
                                              DIAGNOSTICS_BASE_EXCEPTION_STR_WHAT)

#else

#  define DIAGNOSTICS_AUDIT_TRACE(STR_WHAT)        ((void)0)
#  define DIAGNOSTICS_AUDIT_BINARY_TRACE(STR_WHAT) ((void)0)

#  define DIAGNOSTICS_AUDIT_BLOCK_ENTER(STR_WHAT) do {
#  define DIAGNOSTICS_AUDIT_BLOCK_EXIT            } while(false)
#  define DIAGNOSTICS_AUDIT_BLOCK_GUARD(STR_WHAT) ((void)0)

#  define DIAGNOSTICS_AUDIT_PROCEDURE_ENTER(STR_WHAT) do {
#  define DIAGNOSTICS_AUDIT_PROCEDURE_EXIT            } while(false)
#  define DIAGNOSTICS_AUDIT_PROCEDURE_GUARD(STR_WHAT) ((void)0)

#  define DIAGNOSTICS_AUDIT_METHOD_ENTER(STR_WHAT) do {
#  define DIAGNOSTICS_AUDIT_METHOD_EXIT            } while(false)
#  define DIAGNOSTICS_AUDIT_METHOD_GUARD(STR_WHAT) ((void)0)

#  define DIAGNOSTICS_AUDIT_ASSERT(NAME,COND)           ((void)0)
#  define DIAGNOSTICS_AUDIT_ASSERT1(NAME,COND,STR_WHAT) ((void)0)
#  define DIAGNOSTICS_AUDIT_CHECK(NAME,COND)            ((void)0)
#  define DIAGNOSTICS_AUDIT_CHECK1(NAME,COND,STR_WHAT)  ((void)0)

#  define DIAGNOSTICS_AUDIT_CLASS_THROWING_INVARIANCE_ENTER do {
#  define DIAGNOSTICS_AUDIT_CLASS_THROWING_INVARIANCE_EXIT  } while(false)

#  define DIAGNOSTICS_AUDIT_CLASS_INVARIANCE_GUARD ((void)0)
#  define DIAGNOSTICS_AUDIT_CLASS_INVARIANCE_ENTER do {
#  define DIAGNOSTICS_AUDIT_CLASS_INVARIANCE_EXIT  } while(false)

#  define DIAGNOSTICS_AUDIT_CLASS_INVARIANCE_ASSERT  ((void)0)


#  define DIAGNOSTICS_AUDIT_EXCEPTIONLESS_BLOCK_ENTER do {
#  define DIAGNOSTICS_AUDIT_EXCEPTIONLESS_BLOCK_EXIT  } while(false)

#endif


////////////////////////////////////////////////////////////////////////////////

#if DIAGNOSTICS_DEBUG__LEVEL__ > 0

#  define DIAGNOSTICS_DEBUG_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_DEBUG,::diagnostics::TYPE_TRACE,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))
#  define DIAGNOSTICS_DEBUG_BINARY_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_DEBUG,::diagnostics::TYPE_TRACE_BINARY,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))

#  define DIAGNOSTICS_DEBUG_BLOCK_ENTER(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))
#  define DIAGNOSTICS_DEBUG_BLOCK_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define DIAGNOSTICS_DEBUG_BLOCK_GUARD(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))

#  define DIAGNOSTICS_DEBUG_PROCEDURE_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))
#  define DIAGNOSTICS_DEBUG_PROCEDURE_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define DIAGNOSTICS_DEBUG_PROCEDURE_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))

#  define DIAGNOSTICS_DEBUG_METHOD_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("MEHTOD=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))
#  define DIAGNOSTICS_DEBUG_METHOD_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define DIAGNOSTICS_DEBUG_METHOD_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("METHOD=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))

#  define DIAGNOSTICS_DEBUG_ASSERT(NAME,COND) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_DEBUG,\
                            DIAGNOSTICS_NR_WHAT_DEFAULT,\
                            "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                            (COND),\
                            DIAGNOSTICS_FAILED_ASSERT_ACTION(::diagnostics::LEVEL_DEBUG,NAME,COND))
#  define DIAGNOSTICS_DEBUG_ASSERT1(NAME,COND,STR_WHAT) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_DEBUG,\
                            DIAGNOSTICS_NR_WHAT_DEFAULT,\
                            ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (STR_WHAT) + "\"",\
                            (COND),\
                            DIAGNOSTICS_FAILED_ASSERT_ACTION1(::diagnostics::LEVEL_DEBUG,NAME,COND,STR_WHAT))
#  define DIAGNOSTICS_DEBUG_CHECK(NAME,COND) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_DEBUG,\
                           DIAGNOSTICS_NR_WHAT_DEFAULT,\
                           "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                           (COND),\
                           DIAGNOSTICS_FAILED_CHECK_ACTION(::diagnostics::LEVEL_DEBUG,NAME,COND))
#  define DIAGNOSTICS_DEBUG_CHECK1(NAME,COND,STR_WHAT) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_DEBUG,\
                           DIAGNOSTICS_NR_WHAT_DEFAULT,\
                           ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (STR_WHAT) + "\"",\
                           (COND),\
                           DIAGNOSTICS_FAILED_CHECK_ACTION1(::diagnostics::LEVEL_DEBUG,NAME,COND,STR_WHAT))

#  define DIAGNOSTICS_DEBUG_CLASS_THROWING_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER
#  define DIAGNOSTICS_DEBUG_CLASS_THROWING_INVARIANCE_EXIT  \
    DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(::diagnostics::LEVEL_DEBUG,\
                                           DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                           DIAGNOSTICS_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                           DIAGNOSTICS_BASE_EXCEPTION_TYPE,\
                                           DIAGNOSTICS_BASE_EXCEPTION_STR_WHAT)

#  define DIAGNOSTICS_DEBUG_CLASS_INVARIANCE_GUARD DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD
#  define DIAGNOSTICS_DEBUG_CLASS_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_INVARIANCE_ENTER
#  define DIAGNOSTICS_DEBUG_CLASS_INVARIANCE_EXIT  DIAGNOSTICS_BASE_CLASS_INVARIANCE_EXIT

#  define DIAGNOSTICS_DEBUG_CLASS_INVARIANCE_ASSERT DIAGNOSTICS_BASE_CLASS_INVARIANCE_ASSERT


#  define DIAGNOSTICS_DEBUG_EXCEPTIONLESS_BLOCK_ENTER  DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_ENTER
#  define DIAGNOSTICS_DEBUG_EXCEPTIONLESS_BLOCK_EXIT  \
    DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_EXIT(::diagnostics::LEVEL_DEBUG,\
                                              DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                              true,\
                                              DIAGNOSTICS_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                              DIAGNOSTICS_BASE_EXCEPTION_TYPE,\
                                              DIAGNOSTICS_BASE_EXCEPTION_STR_WHAT)

#else

#  define DIAGNOSTICS_DEBUG_TRACE(STR_WHAT)        ((void)0)
#  define DIAGNOSTICS_DEBUG_BINARY_TRACE(STR_WHAT) ((void)0)

#  define DIAGNOSTICS_DEBUG_BLOCK_ENTER(STR_WHAT) do {
#  define DIAGNOSTICS_DEBUG_BLOCK_EXIT            } while(false)
#  define DIAGNOSTICS_DEBUG_BLOCK_GUARD(STR_WHAT) ((void)0)

#  define DIAGNOSTICS_DEBUG_PROCEDURE_ENTER(STR_WHAT) do {
#  define DIAGNOSTICS_DEBUG_PROCEDURE_EXIT            } while(false)
#  define DIAGNOSTICS_DEBUG_PROCEDURE_GUARD(STR_WHAT) ((void)0)

#  define DIAGNOSTICS_DEBUG_METHOD_ENTER(STR_WHAT) do {
#  define DIAGNOSTICS_DEBUG_METHOD_EXIT            } while(false)
#  define DIAGNOSTICS_DEBUG_METHOD_GUARD(STR_WHAT) ((void)0)

#  define DIAGNOSTICS_DEBUG_ASSERT(NAME,COND)           ((void)0)
#  define DIAGNOSTICS_DEBUG_ASSERT1(NAME,COND,STR_WHAT) ((void)0)
#  define DIAGNOSTICS_DEBUG_CHECK(NAME,COND)            ((void)0)
#  define DIAGNOSTICS_DEBUG_CHECK1(NAME,COND,STR_WHAT)  ((void)0)

#  define DIAGNOSTICS_DEBUG_CLASS_THROWING_INVARIANCE_ENTER do {
#  define DIAGNOSTICS_DEBUG_CLASS_THROWING_INVARIANCE_EXIT  } while(false)

#  define DIAGNOSTICS_DEBUG_CLASS_INVARIANCE_GUARD ((void)0)
#  define DIAGNOSTICS_DEBUG_CLASS_INVARIANCE_ENTER do {
#  define DIAGNOSTICS_DEBUG_CLASS_INVARIANCE_EXIT  } while(false)

#  define DIAGNOSTICS_DEBUG_CLASS_INVARIANCE_ASSERT  ((void)0)


#  define DIAGNOSTICS_DEBUG_EXCEPTIONLESS_BLOCK_ENTER do {
#  define DIAGNOSTICS_DEBUG_EXCEPTIONLESS_BLOCK_EXIT  } while(false)

#endif


////////////////////////////////////////////////////////////////////////////////


#define DIAGNOSTICS_PROD_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_PROD,::diagnostics::TYPE_TRACE,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))
#define DIAGNOSTICS_PROD_BINARY_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_PROD,::diagnostics::TYPE_TRACE_BINARY,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))

#define DIAGNOSTICS_PROD_TRACE1(NR_WHAT,STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_PROD,::diagnostics::TYPE_TRACE,(NR_WHAT),(STR_WHAT))
#define DIAGNOSTICS_PROD_BINARY_TRACE1(NR_WHAT,STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_PROD,::diagnostics::TYPE_TRACE_BINARY,(NR_WHAT),(STR_WHAT))


#define DIAGNOSTICS_PROD_BLOCK_ENTER(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))
#define DIAGNOSTICS_PROD_BLOCK_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#define DIAGNOSTICS_PROD_BLOCK_GUARD(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT,(STR_WHAT))

#define DIAGNOSTICS_PROD_PROCEDURE_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))
#define DIAGNOSTICS_PROD_PROCEDURE_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#define DIAGNOSTICS_PROD_PROCEDURE_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))

#define DIAGNOSTICS_PROD_METHOD_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("MEHTOD=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))
#define DIAGNOSTICS_PROD_METHOD_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#define DIAGNOSTICS_PROD_METHOD_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                 ::std::string("METHOD=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))

#define DIAGNOSTICS_PROD_ASSERT(NAME,COND) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_PROD,\
                            DIAGNOSTICS_NR_WHAT_DEFAULT,\
                            "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                            (COND),\
                            DIAGNOSTICS_FAILED_ASSERT_ACTION(::diagnostics::LEVEL_PROD,NAME,COND))
#define DIAGNOSTICS_PROD_ASSERT1(NAME,COND,STR_WHAT) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_PROD,\
                            DIAGNOSTICS_NR_WHAT_DEFAULT,\
                            ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (STR_WHAT) + "\"",\
                            (COND),\
                            DIAGNOSTICS_FAILED_ASSERT_ACTION1(::diagnostics::LEVEL_PROD,NAME,COND,STR_WHAT))

#define DIAGNOSTICS_PROD_CLASS_THROWING_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER
#define DIAGNOSTICS_PROD_CLASS_THROWING_INVARIANCE_EXIT  \
    DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(::diagnostics::LEVEL_PROD,\
                                           DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                           DIAGNOSTICS_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                           DIAGNOSTICS_BASE_EXCEPTION_TYPE,\
                                           DIAGNOSTICS_BASE_EXCEPTION_STR_WHAT)

#define DIAGNOSTICS_PROD_CLASS_INVARIANCE_GUARD DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD
#define DIAGNOSTICS_PROD_CLASS_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_INVARIANCE_ENTER
#define DIAGNOSTICS_PROD_CLASS_INVARIANCE_EXIT  DIAGNOSTICS_BASE_CLASS_INVARIANCE_EXIT

#define DIAGNOSTICS_PROD_CLASS_INVARIANCE_ASSERT  DIAGNOSTICS_BASE_CLASS_INVARIANCE_ASSERT

#define DIAGNOSTICS_PROD_EXCEPTIONLESS_BLOCK_ENTER  DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_ENTER
#define DIAGNOSTICS_PROD_EXCEPTIONLESS_BLOCK_EXIT  \
    DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_EXIT(::diagnostics::LEVEL_PROD,\
                                              DIAGNOSTICS_NR_WHAT_DEFAULT, \
                                              true,\
                                              DIAGNOSTICS_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                              DIAGNOSTICS_BASE_EXCEPTION_TYPE,\
                                              DIAGNOSTICS_BASE_EXCEPTION_STR_WHAT)


#if DIAGNOSTICS_DEBUG__LEVEL__ > 0

#  define DIAGNOSTICS_PROD_CHECK(NAME,COND) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_PROD,\
                           DIAGNOSTICS_NR_WHAT_DEFAULT,\
                           "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                           (COND),\
                           DIAGNOSTICS_FAILED_CHECK_ACTION(::diagnostics::LEVEL_PROD,NAME,COND))
#  define DIAGNOSTICS_PROD_CHECK1(NAME,COND,STR_WHAT) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_PROD,\
                           DIAGNOSTICS_NR_WHAT_DEFAULT,\
                           ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (STR_WHAT) + "\"",\
                           (COND),\
                           DIAGNOSTICS_FAILED_CHECK_ACTION1(::diagnostics::LEVEL_PROD,NAME,COND,STR_WHAT))

#else

#  define DIAGNOSTICS_PROD_CHECK(NAME,COND) \
    do { if(!(COND)) { DIAGNOSTICS_FAILED_CHECK_ACTION(::diagnostics::LEVEL_PROD,NAME,COND); } } while(false)
#  define DIAGNOSTICS_PROD_CHECK1(NAME,COND,STR_WHAT) \
    do { if(!(COND)) { DIAGNOSTICS_FAILED_CHECK_ACTION1(::diagnostics::LEVEL_PROD,NAME,COND,STR_WHAT); } } while(false)

#endif

#define DIAGNOSTICS_METHOD_GUARD(STR_WHAT) DIAGNOSTICS_AUDIT_METHOD_GUARD(STR_WHAT); DIAGNOSTICS_AUDIT_CLASS_INVARIANCE_GUARD 


#define DIAGNOSTICS_AUDIT_ASSERT_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(DIAGNOSTICS_AUDIT_ASSERT1,NAME,OP1,REL,OP2)
#define DIAGNOSTICS_AUDIT_CHECK_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(DIAGNOSTICS_AUDIT_CHECK1,NAME,OP1,REL,OP2)

#define DIAGNOSTICS_DEBUG_ASSERT_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(DIAGNOSTICS_DEBUG_ASSERT1,NAME,OP1,REL,OP2)
#define DIAGNOSTICS_DEBUG_CHECK_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(DIAGNOSTICS_DEBUG_CHECK1,NAME,OP1,REL,OP2)

#define DIAGNOSTICS_PROD_ASSERT_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(DIAGNOSTICS_PROD_ASSERT1,NAME,OP1,REL,OP2)
#define DIAGNOSTICS_PROD_CHECK_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(DIAGNOSTICS_PROD_CHECK1,NAME,OP1,REL,OP2)


#endif

// vim:ts=4:sw=4
