/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/std_test_case.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref diagnostics::unittest::Std_Test_Case
 *
 * $Id: std_test_case.t.cpp,v 1.12 2005/06/23 09:54:24 esdentem Exp $
 *
 * @author Christian Schallhart
 */

#include <diagnostics/unittest.hpp>

#include <diagnostics/unittest/std_test_case.hpp>

// used components
#include <diagnostics/unittest/test_system_exception.hpp>
#include <diagnostics/unittest/name_separators.hpp>
#include <diagnostics/util/to_string.hpp>

// test support
#include <diagnostics/util/dummy_test_case.ts.hpp>

#define TEST_COMPONENT_NAME Std_Test_Case
#define TEST_COMPONENT_NAMESPACE diagnostics::unittest

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;


/**
 * @brief [PRO] fct/normal: constructor testing -- timeout == 0
 */
void timeout_eq_0(Test_Data & test_data) 
{
    Dummy_Test_Case t("name",0,false,false,false,false);
    TEST_ASSERT(t.name()=="name");
    TEST_ASSERT(t.timeout()==0);
    TEST_ASSERT(t.is_rerunable()==false);
    TEST_ASSERT(t.is_testable_at(LEVEL_PROD)==false);
    TEST_ASSERT(t.is_testable_at(LEVEL_DEBUG)==false);
    TEST_ASSERT(t.is_testable_at(LEVEL_AUDIT)==false);
}

/**
 * @brief [PRO] fct/normal: constructor testing -- timeout > 0
 */
void timeout_gt_0(Test_Data & test_data) 
{
    Dummy_Test_Case t("name1",2,true,true,false,false);
    TEST_ASSERT(t.name()=="name1");
    TEST_ASSERT(t.timeout()==2);
    TEST_ASSERT(t.is_rerunable()==true);
    TEST_ASSERT(t.is_testable_at(LEVEL_PROD)==true);
    TEST_ASSERT(t.is_testable_at(LEVEL_DEBUG)==false);
    TEST_ASSERT(t.is_testable_at(LEVEL_AUDIT)==false);
}

/**
 * @brief [PRO] fct/normal: constructor testing -- all variations for is_testable_at
 */
void all_is_testable_at_combinations(Test_Data & test_data) 
{
    int i;
    for(i=0;i<8;++i) {
	Dummy_Test_Case t("name1",2,true,i&(1<<LEVEL_PROD),i&(1<<LEVEL_DEBUG),i&(1<<LEVEL_AUDIT));
	TEST_ASSERT(t.is_testable_at(LEVEL_PROD)==static_cast<bool>(i&(1<<LEVEL_PROD)));
	TEST_ASSERT(t.is_testable_at(LEVEL_DEBUG)==static_cast<bool>(i&(1<<LEVEL_DEBUG)));
	TEST_ASSERT(t.is_testable_at(LEVEL_AUDIT)==static_cast<bool>(i&(1<<LEVEL_AUDIT)));
    }
}

/**
 * @brief [PRO] fct/abnormal: constructor testing -- invalid name (test_case separator)
 */
void case_seperator(Test_Data & test_data) 	
{
    // at the end
    TEST_THROWING_BLOCK_ENTER;
    Dummy_Test_Case t("name1" DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR,2,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // at the beginning
    TEST_THROWING_BLOCK_ENTER;
    Dummy_Test_Case t(DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR "name1",2,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // in the middle
    TEST_THROWING_BLOCK_ENTER;
    Dummy_Test_Case t("name" DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR "name1",2,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // solely the separator
    TEST_THROWING_BLOCK_ENTER;
    Dummy_Test_Case t(DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR "name1",2,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}

/**
 * @brief [PRO] fct/abnormal: constructor testing -- invalid name (test_data separator)
 */
void data_seperator(Test_Data & test_data) 	
{
    // at the end
    TEST_THROWING_BLOCK_ENTER;
    Dummy_Test_Case t("name1" DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR,2,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // at the beginning
    TEST_THROWING_BLOCK_ENTER;
    Dummy_Test_Case t(DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR "name1",2,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // in the middle
    TEST_THROWING_BLOCK_ENTER;
    Dummy_Test_Case t("name" DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR "name1",2,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // solely the separator
    TEST_THROWING_BLOCK_ENTER;
    Dummy_Test_Case t(DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR "name1",2,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}

/**
 * @brief [PRO] fct/abnormal: constructor testing -- timeout < 0
 */
void timeout_lt_0(Test_Data & test_data) 	
{
    TEST_THROWING_BLOCK_ENTER;
    Dummy_Test_Case t("name1",-1,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}

/**
 * @brief [PRO] fct/abnormal: constructor testing -- empty name
 */
void empty_name(Test_Data & test_data) 	
{
    TEST_THROWING_BLOCK_ENTER;
    Dummy_Test_Case t("",0,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}

/**
 * @brief [PRO] fct/abnormal: is_testable_at with LEVEL_TEST
 */
void is_testable_at_LEVEL_TEST(Test_Data & test_data) 	
{
    Dummy_Test_Case t("name1",2,true,true,false,false);
    TEST_THROWING_BLOCK_ENTER;
    t.is_testable_at(LEVEL_TEST);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&timeout_eq_0,LEVEL_PROD);
TEST_NORMAL_CASE(&timeout_gt_0,LEVEL_PROD);
TEST_NORMAL_CASE(&all_is_testable_at_combinations,LEVEL_PROD);
TEST_ABNORMAL_CASE(&case_seperator,LEVEL_PROD);
TEST_ABNORMAL_CASE(&empty_name,LEVEL_PROD);
TEST_ABNORMAL_CASE(&data_seperator,LEVEL_PROD);
TEST_ABNORMAL_CASE(&timeout_lt_0,LEVEL_PROD);
TEST_ABNORMAL_CASE(&is_testable_at_LEVEL_TEST,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;
// vim:ts=4:sw=4
