/* playlist.c -- routines for manipulating playlists
 *
 * Copyright (c) 1998-2002  Mike Oliphant <oliphant@gtk.org>
 *
 *   http://www.nostatic.org/grip
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "ddj.h"
#include "mp3db.h"
#include "playlist.h"

static void PlaylistSelectRow(GtkWidget *widget,gint row,gint column,
			      GdkEventButton *event,gpointer data);
static void OkClicked(GtkWidget *widget,gpointer data);


/* Store a playlist in the mysql database, overwriting it if asked to */
gboolean StorePlaylistToDB(MYSQL *mysql,PlaylistInfo *list_info,IDList *idlist,
			   gboolean overwrite)
{
  char query[1024];
  char esc_name[1024];
  MYSQL_RES *res;
  MYSQL_ROW row;
  int pos;
  char *type;

  if(list_info->id==-1) {
    mysql_escape_string(esc_name,list_info->name,strlen(list_info->name));

    switch(list_info->type) {
    case LIST_TYPE_SONG:
      type="SONG";
      break;
    case LIST_TYPE_ARTIST:
      type="ARTIST";
      break;
    case LIST_TYPE_DISC:
      type="DISC";
      break;
    case LIST_TYPE_GENRE:
      type="GENRE";
      break;
    }
    
    /* First check if this playlist name exists */
    g_snprintf(query,1024,"SELECT id FROM playlist_data WHERE name='%s' "
	       "AND type='%s'",esc_name,type);
    
    printf("query is [%s]\n",query);

    if(mysql_query(mysql,query)) {
      printf("Query error\n");
      
      return FALSE;
    }
    
    if(!(res = mysql_store_result(mysql))) {
      printf("Query error\n");
      
      return FALSE;
    }
    
    if(mysql_num_rows(res)) {
      row=mysql_fetch_row(res);
      list_info->id=atoi(row[0]);

      mysql_free_result(res);
    }
  }

  /* Bail if we weren't asked to overwrite */
  if(list_info->id!=-1 && !overwrite) {
    mysql_free_result(res);
    
    return FALSE;
  }
      
  if(list_info->id!=-1) {
    g_snprintf(query,1024,"UPDATE playlist_data SET size=%d WHERE id=%d",
	       idlist->num_ids,list_info->id);
    
    printf("query is [%s]\n",query);
    
    SQLQuery(mysql,query);
  }
  else {
    g_snprintf(query,1024,"INSERT INTO playlist_data (name,type,size) "
	       "VALUES ('%s','%s',%d)",esc_name,type,idlist->num_ids);
    
    SQLQuery(mysql,query);
    
    list_info->id=mysql_insert_id(mysql);
  }

  /* Delete any ids in the pre-existing playlist */
  g_snprintf(query,1024,"DELETE FROM playlist_items WHERE playlist_id=%d",
	     list_info->id);
  
  SQLQuery(mysql,query);
  
  /* Insert the ids into the database */
  for(pos=0;pos<idlist->num_ids;pos++) {
    g_snprintf(query,1024,"INSERT INTO playlist_items "
	       "(playlist_id,item_id,position) VALUES (%d,%d,%d)",
	       list_info->id,idlist->ids[pos],pos);
    
    SQLQuery(mysql,query);
  }
  
  return TRUE;
}

int PlaylistTypeFromStr(char *type_str)
{
  int list_type;

  if(!strcmp("SONG",type_str)) {
    list_type=LIST_TYPE_SONG;
  }
  else if(!strcmp("ARTIST",type_str)) {
    list_type=LIST_TYPE_ARTIST;
  }
  else if(!strcmp("DISC",type_str)) {
    list_type=LIST_TYPE_DISC;
  }
  else if(!strcmp("GENRE",type_str)) {
    list_type=LIST_TYPE_GENRE;
  }

  return list_type;
}

gboolean GetPlaylistInfoFromDB(MYSQL *mysql,int list_id,PlaylistInfo *pinfo)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  char query[1024];
  int id,size,pos;

  g_snprintf(query,1024,
	     "SELECT id,size,name,type FROM playlist_data WHERE id=%d",
	     list_id);

  if(mysql_query(mysql,query)) {
    printf("Query error\n");

    return FALSE;
  }

  if(!(res=mysql_store_result(mysql))) {
    printf("Query error\n");

    return FALSE;
  }

  if(!mysql_num_rows(res)) {
    mysql_free_result(res);

    return FALSE;
  }

  row=mysql_fetch_row(res);

  pinfo->id=atoi(row[0]);
  pinfo->size=atoi(row[1]);
  g_snprintf(pinfo->name,256,"%s",row[2]);
  pinfo->type=PlaylistTypeFromStr(row[3]);

  mysql_free_result(res);

  return TRUE;
}

/* Load a playlist from the mysql database */
gboolean LoadPlaylistFromDB(MYSQL *mysql,int list_id,IDList *idlist)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  char query[1024];
  int id,size,pos;

  g_snprintf(query,1024,
	     "SELECT id,size FROM playlist_data WHERE id=%d",
	     list_id);

  if(mysql_query(mysql,query)) {
    printf("Query error\n");

    return FALSE;
  }

  if(!(res=mysql_store_result(mysql))) {
    printf("Query error\n");

    return FALSE;
  }

  if(!mysql_num_rows(res)) {
    mysql_free_result(res);

    return FALSE;
  }

  row=mysql_fetch_row(res);

  id=atoi(row[0]);
  size=atoi(row[1]);

  mysql_free_result(res);

  idlist->ids=g_new(int,size);
  if(!idlist->ids) {
    printf("Unable to allocate memory\n");

    return FALSE;
  }

  idlist->num_ids=size;

  g_snprintf(query,1024,"SELECT item_id FROM playlist_items WHERE "
	     "playlist_id=%d ORDER BY position",id);

  if(mysql_query(mysql,query)) {
    printf("Query error\n");

    return FALSE;
  }

  if(!(res = mysql_store_result(mysql))) {
    printf("Query error\n");

    return FALSE;
  }

  pos=0;

  while((row=mysql_fetch_row(res))) {
    id=atoi(row[0]);

    idlist->ids[pos++]=id;
  }

  idlist->current_id=0;

  return TRUE;
}

static gchar *playlist_titles[]={"Title","Size","Type"};

/* Create a clist with the current playlists in it */
GtkWidget *PlaylistCList(MYSQL *mysql,GtkWidget **in_clist)
{
  GtkWidget *clist;
  GtkWidget *scroll;
  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char query[1024];
  int pos;

  clist=gtk_clist_new_with_titles(3,playlist_titles);
 
  gtk_clist_set_column_width(GTK_CLIST(clist),0,200);
  gtk_clist_set_column_width(GTK_CLIST(clist),1,50);
  gtk_clist_set_column_width(GTK_CLIST(clist),2,55);

  gtk_clist_column_titles_passive(GTK_CLIST(clist));

  gtk_clist_set_reorderable(GTK_CLIST(clist),FALSE);
    
  scroll=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll),clist);


  g_snprintf(query,1024,
	     "SELECT name,size,type,id FROM playlist_data ORDER BY name");

  if(mysql_query(mysql,query)||
     !(res = mysql_store_result(mysql))) {
    printf("Query error");
  }
  else {
    pos=0;

    while((row=mysql_fetch_row(res))) {
      gtk_clist_append(GTK_CLIST(clist),row);

      gtk_clist_set_row_data(GTK_CLIST(clist),pos++,(gpointer)atoi(row[3]));
    }

    mysql_free_result(res);
  }

  gtk_widget_show(clist);

  *in_clist=clist;

  return scroll;
}

static void PlaylistSelectRow(GtkWidget *widget,gint row,gint column,
			      GdkEventButton *event,gpointer data)
{
  char *str;
  PlaylistInfo *list_info;

  list_info=(PlaylistInfo *)data;
  gtk_clist_get_text(GTK_CLIST(widget),row,0,&str);

  if(list_info->entry) {
    gtk_entry_set_text(GTK_ENTRY(list_info->entry),str);
  }

  gtk_clist_get_text(GTK_CLIST(widget),row,1,&str);
  list_info->size=atoi(str);

  gtk_clist_get_text(GTK_CLIST(widget),row,2,&str);

  list_info->type=PlaylistTypeFromStr(str);

  list_info->id=(int)gtk_clist_get_row_data(GTK_CLIST(widget),row);
}

static void OkClicked(GtkWidget *widget,gpointer data)
{
  PlaylistInfo *list_info;

  list_info=(PlaylistInfo *)data;

  strcpy(list_info->name,gtk_entry_get_text(GTK_ENTRY(list_info->entry)));
}

/* Allow the user to select a playlist name */
void SelectPlaylist(MYSQL *mysql,gboolean allow_new,
		    GtkSignalFunc resultfunc,PlaylistInfo *list_info)
{
  GtkWidget *dialog;
  GtkWidget *listobj;
  GtkWidget *clist;
  GtkWidget *vbox;
  GtkWidget *entry;
  GtkWidget *label;
  GtkWidget *hbox;
  GtkWidget *doitbutton;
  GtkWidget *cancelbutton;

  list_info->id=-1;
  list_info->entry=NULL;

  dialog=gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog),"Playlist Selection");

  gtk_widget_set_usize(dialog,360,200);

  vbox=GTK_DIALOG(dialog)->vbox;

  gtk_container_border_width(GTK_CONTAINER(vbox),5);

  listobj=PlaylistCList(mysql,&clist);

  gtk_box_pack_start(GTK_BOX(vbox),listobj,TRUE,TRUE,0);
  gtk_widget_show(listobj);

  if(allow_new) {
    hbox=gtk_hbox_new(FALSE,5);

    label=gtk_label_new("Playlist: ");
    gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_LEFT);
    gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
    gtk_widget_show(label);
    
    entry=gtk_entry_new_with_max_length(255);
    gtk_box_pack_start(GTK_BOX(hbox),entry,TRUE,TRUE,0);
    gtk_widget_show(entry);

    /*if(!allow_new) gtk_entry_set_editable(GTK_ENTRY(entry),FALSE);*/

    gtk_box_pack_start(GTK_BOX(vbox),hbox,TRUE,TRUE,0);
    gtk_widget_show(hbox);

    list_info->entry=entry;
  }

  gtk_signal_connect(GTK_OBJECT(clist),"select_row",
		     GTK_SIGNAL_FUNC(PlaylistSelectRow),
		     (gpointer)list_info);

  doitbutton=gtk_button_new_with_label("OK");
  if(list_info->entry)
    gtk_signal_connect(GTK_OBJECT(doitbutton),"clicked",
		       OkClicked,(gpointer)list_info);
  gtk_signal_connect(GTK_OBJECT(doitbutton),"clicked",
		     resultfunc,(gpointer)list_info);
  gtk_signal_connect_object(GTK_OBJECT(doitbutton),"clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(dialog));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),doitbutton,
		     TRUE,TRUE,0);
  gtk_widget_show(doitbutton);

  cancelbutton=gtk_button_new_with_label("Cancel");
  gtk_signal_connect_object(GTK_OBJECT(cancelbutton),"clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(dialog));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),cancelbutton,
		     TRUE,TRUE,0);
  gtk_widget_show(cancelbutton);

  gtk_widget_show(dialog);

  gtk_grab_add(dialog);

  return;
}
