/*
 * sunlite.c
 *
 * Copyright (C) Michael Stickel <michael@cubic.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "dmxsunlite.h"

MODULE_AUTHOR("(c) 2001 Michael Stickel <michael@cubic.org> http://llg.cubic.org");
MODULE_DESCRIPTION("Driver for the Sunlite-DMX512 interface version " DMXVERSION);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,17)
MODULE_LICENSE("GPL");
#endif


static int inputrate = 30;
MODULE_PARM(inputrate,"i");
MODULE_PARM_DESC(inputrate,"input universe is read <inputrate> times per second (default 30)");


DMXFamily *usbdmx2_family = NULL;


/*-----------------------------------------------------------
 *-- dmx_write_universe
 *--
 *-- Used to write a couple of slot-values to the universe.
 *---------------------------------------------------------*/
static int dmx_write_universe (DMXUniverse *u, off_t offs, DMXSlotType *buff, size_t size)
{
#ifdef DEBUG
  printk(KERN_INFO "dmx_write_universe called\n");
#endif

  if (u && size>0 && offs+size <= 512)
    {
      struct dmxusb_universe *u2d_u = (struct dmxusb_universe *)u->user_data;
      if (u2d_u && u2d_u->dmx_if)
	{
	  /* copy the data and tell the thread something has changed */

	  memcpy ((u2d_u->buffer)+offs, (void *)buff, size);

	  udmx_do_write (u2d_u->dmx_if, offs, size, buff);

	  u2d_u->dmx_if->data_pending = 1;
	  wake_up (&u2d_u->dmx_if->waitqueue);

	  u2d_u->data_avail=1;
	  u->signal_changed (u, offs, size);

	  return 0;
	}
    }
  return -EINVAL;
}



/*-----------------------------------------------------------
 *-- usbdmx2_data_available
 *--
 *--
 *---------------------------------------------------------*/
int  usbdmx2_data_available (DMXUniverse *u, uint start, uint size)
{

  if (u && u->user_data)
    {
      struct dmxusb_universe *u2d_u = (struct dmxusb_universe *)u->user_data;
      if (u2d_u)
	return u2d_u->data_avail;
    }
  return 0;
}




/*-----------------------------------------------------------
 *-- dmx_read_universe
 *--
 *-- Used to write a couple of slot-values to the universe.
 *---------------------------------------------------------*/
static int dmx_read_universe (DMXUniverse *u, off_t offs, DMXSlotType *buff, size_t size)
{
#if DEBUGLEVEL(USBDMX2,1)
  printk(KERN_INFO "dmx_read_universe called\n");
#endif

  if (u && size>0 && offs+size <= 512)
    {
      struct dmxusb_universe *u2d_u = (struct dmxusb_universe *)u->user_data;

#if DEBUGLEVEL(USBDMX2,2)
      printk(KERN_INFO "dmx_read_universei: call is valid\n");
#endif
      if (u2d_u && u2d_u->dmx_if)
	{
          unsigned char *inbuffer = u2d_u->buffer;

#if DEBUGLEVEL(USBDMX2,2)
          printk(KERN_INFO "dmx_read_universe: usbdmx2 universe data valid\n");
#endif

          if (u2d_u->data_avail)
            {
	      u2d_u->data_avail=0;

	      if (offs+size>512)
	        size = 512-offs;

	      memcpy (buff, inbuffer+offs, size);

	      return size;
            }
#if DEBUGLEVEL(USBDMX2,2)
          else
            printk(KERN_INFO "dmx_read_universe: no change at universe\n");
#endif
          return 0;
	}
#if DEBUGLEVEL(USBDMX2,2)
      else
        printk(KERN_INFO "dmx_read_universe: usbdmx2 universe data INVALID\n");
#endif
    }
#if DEBUGLEVEL(USBDMX2,2)
  else
    printk(KERN_INFO "dmx_read_universei: call is INVALID: offs=%lu, size=%d\n", offs, size);
#endif
  return -EINVAL;
}




/*-----------------------------------------------------------
 *-- usbdmx2_delete_universe
 *--
 *-- Delete the universe of a USBDMX2 interface.
 *---------------------------------------------------------*/
static int usbdmx2_delete_universe (DMXUniverse *u)
{
  printk ("usbdmx2_delete_universe (DMXUniverse *u=0x%p) " _FUNCFRAME_ "\n", u, FUNCFRAME_(usbdmx2_delete_universe));
  if (u && u->user_data)
    {
      struct dmxusb_universe *u2d_u = (struct dmxusb_universe *)u->user_data;
      /* lock the universe */

      u2d_u->dmx_if->universe = NULL;
      u2d_u->dmx_if = NULL;
      u2d_u->universe = NULL;

      printk ("freeing universe->user_data = 0x%p\n", u->user_data);
      FREE(u->user_data);
      u->user_data = NULL;
      printk ("freeing universe->user_data done\n");
    }
  printk ("after usbdmx2_delete_universe (DMXUniverse *u=0x%p)\n", u);
  return 0;
}



/*-----------------------------------------------------------
 *-- usbdmx2_create_universe
 *--
 *-- Create the universe for the USBDMX2 interface.
 *---------------------------------------------------------*/
static int usbdmx2_create_universe (DMXUniverse *u, DMXPropList *pl)
{
  printk ("usbdmx2_create_universe (DMXUniverse *u=0x%p, DMXPropList *pl=0x%p)" _FUNCFRAME_ "\n", u, pl, FUNCFRAME_(usbdmx2_create_universe));
  if (u && u->interface)
    {
      struct dmxusb_universe *u2d_u = NULL;

      u2d_u = DMX_ALLOC(struct dmxusb_universe);
      if (u2d_u)
	{
	  memset (u2d_u->buffer, 0, 512);

	  u->user_data  = (void *)u2d_u;

	  u->user_delete = usbdmx2_delete_universe;
	  u->read_slots  = dmx_read_universe;
	  u->data_available = usbdmx2_data_available;
	  if (u->kind == 0)
	    {
	      /* output universe */
	      u->write_slots = dmx_write_universe;
	      strcpy (u->connector, "OUT");
	      u->conn_id = 0;
	      u2d_u->universe_id = 0;
	    }
	  else if (u->kind == 1)
	    {
	      /* input universe */
	      strcpy (u->connector, "IN");
	      u->conn_id = 1;
	      u2d_u->universe_id = 0;
	    }
	  else
	    return -1;

	  u2d_u->dmx_if = (struct dmxusb_interface *)u->interface->user_data;
	  u2d_u->universe    = u;
	  u2d_u->dmx_if->universe = u2d_u;
	}
      return 0;
    }
  return -1;
}



/*-----------------------------------------------------------
 *-- usbdmx2_delete_interface
 *--
 *--
 *---------------------------------------------------------*/
static int usbdmx2_delete_interface (DMXInterface *dif)
{
  printk ("usbdmx2_delete_interface (DMXInterface *dif=%p)" _FUNCFRAME_ "\n", dif, FUNCFRAME_(usbdmx2_delete_interface));

  if (dif && dif->user_data)
    {
      int waitpid_result = 0;
      int ret = 0;
      struct dmxusb_interface *u2d_if = (struct dmxusb_interface *)dif->user_data;

      /* lock the interface */

      if (u2d_if->thread_pid > 0) /* we don't want to kill init */
	{
	  printk ("attempting to kill usbdmx2-thread pid=%d\n", u2d_if->thread_pid);
	  ret = kill_proc(u2d_if->thread_pid, SIGTERM, 1);
	  if (ret)
	    {
	      printk(KERN_ERR "usbdmx2: unable to signal thread\n");
	      return -1;
	    }
          THREAD_SEM_WAIT_COMPLETE(&u2d_if->thr_exited);
	  /* down(&u2d_if->thr_exited); */
	  waitpid_result = waitpid (u2d_if->thread_pid, NULL, __WCLONE|WNOHANG);
	  printk("usbdmx2 thread has been stopped\n");
	}


      /* be paranoyd and wait for a while */
      schedule_timeout(HZ/5);


      if (u2d_if->universe)
	u2d_if->universe->dmx_if = NULL;
      u2d_if->universe = NULL;
      u2d_if->interface = NULL;

      printk ("freeing interface->user_data = 0x%p\n", dif->user_data);
      FREE(dif->user_data);
      dif->user_data = NULL;
      printk ("freeing interface->user_data done\n");
    }
  return 0;
}




/*-----------------------------------------------------------
 *-- usbdmx2_event
 *--
 *-- compute an event from the userspace.
 *-- This function must only be called from the
 *-- usbdmx2_interface_thread.
 *---------------------------------------------------------*/
static void usbdmx2_tx_event(struct dmxusb_interface *u2d_if)
{
  if (u2d_if->data_pending)
    {
      struct dmxusb_universe *u2d_u = u2d_if->universe;

#ifdef DEBUG
      printk (KERN_INFO "data-pending - sending it\n");
#endif

      u2d_if->data_pending = 0;


      udmx_update (u2d_if);


      // if failed ->
      // u2d_if->data_pending = 1;
#ifdef DEBUG
      //else
      //	printk(KERN_INFO "wrote %d bytes to usbdmx2\n", size);
#endif
      if (u2d_u)
	{
	  u2d_u->data_avail = 1;
	  if (u2d_u->universe && u2d_u->universe->signal_changed)
	    u2d_u->universe->signal_changed (u2d_u->universe, 0, 512);
	}
    }
}



/*-----------------------------------------------------------
 *-- u2d_interface_thread
 *--
 *-- The thread that sends the data to the USBDMX2 interface.
 *---------------------------------------------------------*/
static int u2d_interface_thread (void *user_data)
{
  int timeout = HZ;
  struct dmxusb_interface *u2d_if = (struct dmxusb_interface *)user_data;
  struct urb* purb;
  unsigned char inbuffer[1024];
  int status;

  printk ("u2d_interface_thread (void *user_data=0x%p)" _FUNCFRAME_ "\n" , user_data, FUNCFRAME_(u2d_interface_thread));

  if(inputrate<1)
    inputrate=1;

  printk ("u2d_interface_thread inputrate=%d\n", inputrate);

  timeout=HZ/inputrate;

  if (!user_data)
    {
      printk ("u2d_interface_thread: user_data = NULL -> exiting\n");
      THREAD_SEM_EXIT (&u2d_if->thr_exited, 1);
    }


  lock_kernel();

  daemonize ();
  spin_lock_irq(&current->sigmask_lock);
  sigemptyset(&current->blocked);
  recalc_sigpending(current);
  spin_unlock_irq(&current->sigmask_lock);

  strncpy (current->comm, "usbdmx2d", sizeof(current->comm) - 1);
  current->comm[sizeof(current->comm) - 1] = '\0';

  u2d_if->running = 1;

  printk ("starting usbdmx2d...\n");

  /* set_current_state(TASK_INTERRUPTIBLE); */
  /* schedule_timeout(HZ*2); */

  /* USBDMX-IN */
  if (u2d_if->dmx_dev->descriptor.idProduct == USB_PRODUCT_EZ_USB3a) {

    atomic_set(&u2d_if->urb_submit_pending, 1);

    purb = usb_alloc_urb(0);
    if (purb) {
      usb_fill_bulk_urb (purb, u2d_if->dmx_dev, usb_rcvbulkpipe(u2d_if->dmx_dev, 2), inbuffer, sizeof(inbuffer), bulk_urb_complete, u2d_if);
      status=usb_submit_urb(purb);
      if (status)
	info("urb submit error: %d", status);
    }

    do {

      status = interruptible_sleep_on_timeout(&u2d_if->waitqueue, HZ/inputrate);

      if (purb && !atomic_read(&u2d_if->urb_submit_pending)) {

	usb_fill_bulk_urb (purb, u2d_if->dmx_dev, usb_rcvbulkpipe(u2d_if->dmx_dev, 2), inbuffer, sizeof(inbuffer), bulk_urb_complete, u2d_if);
	status=usb_submit_urb(purb);
	if (status)
	  info("urb submit error: %d", status);

	atomic_set(&u2d_if->urb_submit_pending, 1);
      }

    } while (!signal_pending(current));

    if (atomic_read(&u2d_if->urb_submit_pending))
      info("waiting for urb submit completion");

    while (purb && atomic_read(&u2d_if->urb_submit_pending))
      mdelay(5);

    if (purb)
      usb_free_urb(purb);

  }

  /* USBDMX2 */
  if (u2d_if->dmx_dev->descriptor.idProduct == USB_PRODUCT_EZ_USB2a) {
  do
    {
      int stat = 1;

      /*
       * Check for pending data before going to sleep and wait for a signal that
       * data is pending is nessesary here, because the interruptible_sleep_on_timeout
       * only returns for signals arived while it is executed. If we don't check data_pending
       * here, we may loose some updates.
       */
      if (u2d_if->data_pending == 0)
        stat = interruptible_sleep_on_timeout(&u2d_if->waitqueue, HZ);

      /*
       * data is pending -> update it to the interface
       */
      if (stat)
	usbdmx2_tx_event(u2d_if);

    } while (!signal_pending(current));

  }

  printk ("usbdmx2 thread is exiting\n");
  u2d_if->running = 0;

  THREAD_SEM_EXIT (&u2d_if->thr_exited, 0);

  return 0;
}


/*-----------------------------------------------------------
 *-- usbdmx2_create_interface
 *--
 *-- Creates a dmx-interface for USBDMX2 hardware.
 *---------------------------------------------------------*/
static int usbdmx2_create_interface (DMXInterface *dif, DMXPropList *pl)
{
  printk ("usbdmx2_create_interface (DMXInterface *dif=%p, DMXPropList *pl=%p)" _FUNCFRAME_ "\n", dif, pl, FUNCFRAME_(usbdmx2_create_interface));
  if (dif)
    {
      struct dmxusb_interface  *u2d_if=NULL;
      struct usb_device        *usbdev = NULL;
#if 0
      unsigned int               ifnum = 0;
#endif

      if (pl && pl->find)
	{
	  DMXProperty *p = pl->find(pl, "usbdev");
	  if (p)
	    p->get_long (p, (unsigned long *)&usbdev);
	}

      if (!usbdev) /* fail to create, if no usbdevice has been given */
	{
	  printk("usbdmx2: failed to evaluate usbdev parameter\n");
	  return -1;
	}

      if (usbdev->actconfig && usbdev->actconfig->iConfiguration)
	{
	  char str[128];
	  if(usb_string(usbdev, usbdev->actconfig->iConfiguration, str, sizeof(str)) >= 0 )
	    info ("current configuration is \"%s\"", str);
	}

      u2d_if = DMX_ALLOC(struct dmxusb_interface);
      if (u2d_if)
	{

	  dif->user_data = (void *)u2d_if;
	  dif->user_delete = usbdmx2_delete_interface;

	  u2d_if->universe = NULL;
	  u2d_if->interface = dif;
	  u2d_if->dmx_dev   = usbdev; /* usb-device handle */
	  init_waitqueue_head (&u2d_if->waitqueue);
	  THREAD_SEM_INIT (&u2d_if->thr_exited);

	  /* u2d_if->rx_frames = 0; */
	  u2d_if->do_read = 0;
	  init_waitqueue_head (&u2d_if->read_waitqueue);
	  u2d_if->running = 0; /* set by the thread */
	  u2d_if->data_pending = 0; /* no data pending at the beginning */


	  /*printk ("initize usbdmx interface.\n");*/

	  /* udmx_init (u2d_if);*/

	  if(usbdev->descriptor.idProduct == USB_PRODUCT_EZ_USB2a) {
	    printk ("initize bulkout-buffer.\n");
	    udmx_init_packet (u2d_if->bulkout_buffer);
	  }

	  if(usbdev->descriptor.idProduct == USB_PRODUCT_EZ_USB3a) {
	    printk ("initize USBDMX-IN hardware\n");
	    udmx_in_init (u2d_if);
	  }

	  printk ("starting usbdmx2 thread\n");
	  u2d_if->thread_pid = kernel_thread(u2d_interface_thread, (void *)u2d_if, CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
	  if (u2d_if->thread_pid >= 0)
	    {
	      printk ("usbdmx2 thread successfully started\n");
	      return 0;
	    }
	  FREE(u2d_if);
	}
    }
  return -1;
}







/*-----------------------------------------------------------
 *-- Some declaration for USBDMX2.
 *---------------------------------------------------------*/
static void *usbdmx2_probe (struct usb_device *dev, unsigned int ifnum, const struct usb_device_id *id /* since 2.4.2 */);
static void usbdmx2_disconnect(struct usb_device *dev, void *ptr);

typedef struct
{
  unsigned short  vendor;
  unsigned short  device;
  char  *         name;
  short           universes_out;
  short           universes_in;
  const struct sunlite_hex_record *firmware;
} dmxsunlite_device_info;

#include "dmxsunlite_firmware.h"


dmxsunlite_device_info sunlite_device_info[] =
{
  { USB_VENDOR_DIGITALARTSYSTEM, USB_PRODUCT_EZ_USB,   "Sunlite USBDMX1",            1, 0, NULL},
  { USB_VENDOR_DIGITALARTSYSTEM, USB_PRODUCT_EZ_USB2,  "Sunlite USBDMX2-firmware",   0, 0, sunlite_firmware},
  { USB_VENDOR_DIGITALARTSYSTEM, USB_PRODUCT_EZ_USB2a, "Sunlite USBDMX2",            1, 0, NULL},
  { USB_VENDOR_DIGITALARTSYSTEM, USB_PRODUCT_EZ_USB3,  "Sunlite USBDMX-IN-firmware", 0, 0, sunlite_firmware_in},
  { USB_VENDOR_DIGITALARTSYSTEM, USB_PRODUCT_EZ_USB3a, "Sunlite USBDMX-IN",          0, 1, NULL},
#if 0
  { 0xb334,                      0x1,                  "Twilight",                   0, 0, sunlite_firmware},
#endif
  { 0, 0, NULL, 0, 0, NULL}
};


static struct usb_device_id usbdmx2_id_table [] =
{
  /* USBDMX1 is currently not supported. */
  { USB_DEVICE(USB_VENDOR_DIGITALARTSYSTEM, USB_PRODUCT_EZ_USB),   driver_info: 0 },  /* USBDMX1 */
  { USB_DEVICE(USB_VENDOR_DIGITALARTSYSTEM, USB_PRODUCT_EZ_USB2),  driver_info: 1 },  /* USBDMX2 */
  { USB_DEVICE(USB_VENDOR_DIGITALARTSYSTEM, USB_PRODUCT_EZ_USB2a), driver_info: 2 },  /* USBDMX2 firmware */
  { USB_DEVICE(USB_VENDOR_DIGITALARTSYSTEM, USB_PRODUCT_EZ_USB3),  driver_info: 3 },  /* USBDMX-IN firmware */
  { USB_DEVICE(USB_VENDOR_DIGITALARTSYSTEM, USB_PRODUCT_EZ_USB3a), driver_info: 4 },  /* USBDMX-IN */
#if 0
  { USB_DEVICE(0xb334, 0x1), driver_info: 6 },  /* Twilight */
#endif
  { }                                           /* Terminating entry */
};
MODULE_DEVICE_TABLE(usb, usbdmx2_id_table);




/*-----------------------------------------------------------
 *-- This is used to register the usb-driver.
 *---------------------------------------------------------*/
static struct usb_driver dmx_driver =
{
  name:           "USBDMX",
  probe:          usbdmx2_probe,
  disconnect:     usbdmx2_disconnect,
  driver_list:    LIST_HEAD_INIT(dmx_driver.driver_list),
  id_table:       usbdmx2_id_table,
/*
  struct semaphore serialize;
  void (*suspend)(struct usb_device *dev);
  void (*resume)(struct usb_device *dev);
*/
};








/*-----------------------------------------------------------
 *-- usbdmx2_probe
 *--
 *-- Is called after a USB device has been added
 *-- to the bus and check wether it is a USBDMX2.
 *-- If so, it initializes it and creates as much
 *-- DMX universes as the interface provides.
 *---------------------------------------------------------*/
static void *usbdmx2_probe (struct usb_device *dev, unsigned int ifnum,
			    const struct usb_device_id *id /* since 2.4.2 */
			    )
{
  const int sunlite_info_size = sizeof(sunlite_device_info) / sizeof(sunlite_device_info[0]);
  int device_info_idx = id->driver_info;
  dmxsunlite_device_info * device_info = (device_info_idx>=0 && device_info_idx < sunlite_info_size) ? &sunlite_device_info[device_info_idx] : NULL;

  DMXDriver *drv = NULL;
  char str[128];
  int res;

  printk ("usbdmx2_probe (struct usb_device *dev=0x%p, unsigned int ifnum=%u, const struct usb_device_id *id=0x%p" _FUNCFRAME_ "\n", dev, ifnum, id, FUNCFRAME_(usbdmx2_probe));

  if (dev == NULL)
    {
      printk ("usbdmx2_probe: FATAL: dev is NULL\n");
      return NULL;
    }

  if (device_info == NULL)
    return NULL;

  printk ("usbdmx2_probe:  VENDOR: 0x%X   PRODUCT: 0x%X\n",
	  dev->descriptor.idVendor,
	  dev->descriptor.idProduct);


  /*
   * Check if it is a known USBDMX2 release.
   */
  if (dev->descriptor.idVendor == USB_VENDOR_DIGITALARTSYSTEM)
    {
      if (dev->descriptor.idProduct == USB_PRODUCT_EZ_USB)
	info("Sunlite USBDMX1 found at address %d", dev->devnum);

      else if(dev->descriptor.idProduct == USB_PRODUCT_EZ_USB2 )
	info("Sunlite USBDMX2-firmware device found at address %d", dev->devnum);

      else if(dev->descriptor.idProduct == USB_PRODUCT_EZ_USB2a )
	info("Sunlite USBDMX2 device found at address %d", dev->devnum);

      else if(dev->descriptor.idProduct == USB_PRODUCT_EZ_USB3 )
	info("Sunlite USBDMX-IN-firmware device found at address %d", dev->devnum);

      else if(dev->descriptor.idProduct == USB_PRODUCT_EZ_USB3a )
	info("Sunlite USBDMX-IN device found at address %d", dev->devnum);

      else
	{
	  warn(KERN_INFO "Sunlite USBDMX model not supported/tested. (id=%d)", dev->descriptor.idProduct);
	  return NULL;
	}
    }
  else
    return NULL;


  /*------------------------------------------
  //-- Upload the firmware if nessesary.
  //------------------------------------------
  // device_info->firmware is not NULL, then upload the firmware */

  if ( (dev->descriptor.idVendor == USB_VENDOR_DIGITALARTSYSTEM) &&
       (dev->descriptor.idProduct == USB_PRODUCT_EZ_USB2 || dev->descriptor.idProduct == USB_PRODUCT_EZ_USB3 ) )
    {
      int err;
      info (DRIVERNAME " loading firmware...");
      err = USBDMX2_startup (dev, dev->descriptor.idProduct, device_info->firmware);
      if (err)
	  info (DRIVERNAME "...failed");
      else
	info (DRIVERNAME "...succeded");
      return (void *)0xFA57FEED;
    }
  else if (dev->descriptor.idVendor == 0xb334 && dev->descriptor.idProduct == 0x01)
    {
      int err;
      info (DRIVERNAME " loading firmware...");
      err = USBDMX2_startup (dev, 0x1, device_info->firmware);
      if (err)
          info (DRIVERNAME "...failed");
      else
        info (DRIVERNAME "...succeded");
      return (void *)0xFA57FEED;
    }


  /* if device_info->universes_out or device_info->universes_in is not
     0 then we have an input or an output or both and then we
     initialize the device and create one or more universes. */

  if (dev->actconfig)
    {
      if (dev->actconfig->iConfiguration)
	{
	  if((res = usb_string(dev, dev->actconfig->iConfiguration, str, sizeof(str))) < 0 )
	    err ("reading string for current config failed (error=%i)", res);
	  else
	    info ("configuration changed to \"%s\"", str);
	}
      else
	warn ("dev->actconfig->iConfiguration = NULL");
    }
  else
    warn ("dev->actconfig = NULL");

  /*
   * Create a USBDMX2 specific record and a DMX universe.
   */
  drv = dmx_find_driver (usbdmx2_family, "usbdmx2");
  if (drv)
    {
      DMXInterface *dmx_if = drv->create_interface (drv, dmxproplist_vacreate ("usbdev=%l", (long)dev));
      if (dmx_if && dmx_if->user_data)
	{
	  DMXUniverse  *uout=NULL;

	  if (dev->descriptor.idProduct == USB_PRODUCT_EZ_USB3a)
	    uout = dmx_if->create_universe (dmx_if, 1, NULL); /* input */
	  if (dev->descriptor.idProduct == USB_PRODUCT_EZ_USB2a || dev->descriptor.idProduct == USB_PRODUCT_EZ_USB)
	    uout = dmx_if->create_universe (dmx_if, 0, NULL); /* output */

	  if (uout)
	    {
	      printk("input/output-universes created, usb.private=%08lX\n", (unsigned long)dmx_if);
	      return dmx_if;
	    }
	  dmx_if->delete(dmx_if);
	}
      return dmx_if;
    }
  else
    printk (KERN_INFO "unable to find driver for dmx-family usbdmx2.usbdmx2\n");

  printk("returning NULL as usb.private\n");
  /*
   * Something failed.
   */
  return NULL;
}




/*-----------------------------------------------------------
 *-- usbdmx2_disconnect
 *--
 *-- disconnect method for usbdmx2.
 *-- It is called after a USBDMX2 device has been
 *-- removed from the USB bus.
 *---------------------------------------------------------*/
static void usbdmx2_disconnect (struct usb_device *dev, void *ptr)
{
  if (ptr == (void *)0xFA57FEED)
    printk ("disconnecting fake firmware loader\n");
  else if (ptr == (void *)0xDEADBEEF)
    printk("disconnecting from 0xDEADBEEF\n");
  else
    {
      DMXInterface *dmx = (DMXInterface *)ptr;
      printk ("usbdmx2_disconnect(struct usb_device *dev=0x%p, void *ptr=0x%p" _FUNCFRAME_ "\n", dev, ptr, FUNCFRAME_(usbdmx2_disconnect));
      if (dmx)
	{
#if 0
	  wait(dmx->sem); /* wait for the interface to be unused and block it (forever :-) */
#endif
	  printk (KERN_INFO "delete usbdmx2 interface\n");
	  if (dmx && dmx->delete)
	    dmx->delete (dmx);
          /* does a cascaded delete on the universes of that interface */
	}
    }
  printk ("after usbdmx2_disconnect\n");
}




/*-----------------------------------------------------------
 *-- usbdmx2_init
 *--
 *-- Called after the module has been loaded.
 *---------------------------------------------------------*/
static int __init usbdmx2_init(void)
{
  DMXDriver *drv = NULL;

  usbdmx2_family = dmx_create_family("USB");
  if (!usbdmx2_family)
    {
      printk (KERN_INFO "unable to register dmx-family USBDMX2\n");
      return -1;
    }

  drv = usbdmx2_family->create_driver (usbdmx2_family, "usbdmx2", usbdmx2_create_universe, NULL);
  if (!drv)
    {
      usbdmx2_family->delete(usbdmx2_family, 0);
      info("failed to create usbdmx2 driver\n");
      return -1;
    }
  drv->num_out_universes = 1;
  drv->num_in_universes = 1;
  drv->user_create_interface = usbdmx2_create_interface;

  if (usb_register(&dmx_driver) < 0)
    {
      usbdmx2_family->delete(usbdmx2_family, 0);
      info("failed to register USBDMX2\n");
      return -1;
    }

  info("USBDMX2 registered.");
  return 0;
}


/*-----------------------------------------------------------
 *-- usbdmx2_cleanup
 *--
 *-- Called to do the cleanup.
 *---------------------------------------------------------*/
static void __exit usbdmx2_cleanup(void)
{
  usb_deregister(&dmx_driver);
  mdelay(5);
  if (usbdmx2_family)
    usbdmx2_family->delete(usbdmx2_family, 0);
}



module_init(usbdmx2_init);
module_exit(usbdmx2_cleanup);

EXPORT_NO_SYMBOLS;
