/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "urlnavigatorbutton.h"
#include <kurl.h>
#include <qpainter.h>
#include <qtooltip.h>
#include <qcursor.h>
#include <qfontmetrics.h>
#include <kglobalsettings.h>
#include <kurldrag.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <klocale.h>

#include "urlnavigator.h"
#include "dolphinview.h"
#include "dolphin.h"

URLNavigatorButton::URLNavigatorButton(int index, URLNavigator* parent) :
    URLButton(parent),
    m_index(-1),
    m_urlNavigator(parent)
{
    setAcceptDrops(true);
    setMinimumWidth(arrowWidth());
    setIndex(index);
    connect(this, SIGNAL(clicked()), this, SLOT(updateNavigatorURL()));
}

URLNavigatorButton::~URLNavigatorButton()
{
}

void URLNavigatorButton::setIndex(int index)
{
    if (index < 0) {
        index = 0;
    }

    m_index = index;
    QString path(urlNavigator()->url().prettyURL());
    setText(path.section('/', index, index));

    // Check whether the button indicates the full path of the URL. If
    // this is the case, the button is marked as 'active'.
    ++index;
    QFont adjustedFont(font());
    if (path.section('/', index, index).isEmpty()) {
        setDisplayHintEnabled(ActivatedHint, true);
        adjustedFont.setBold(true);
    }
    else {
        setDisplayHintEnabled(ActivatedHint, false);
        adjustedFont.setBold(false);
    }

    setFont(adjustedFont);
    update();
}

int URLNavigatorButton::index() const
{
    return m_index;
}

void URLNavigatorButton::drawButton(QPainter* painter)
{
    const int buttonWidth  = width();
    const int buttonHeight = height();

    QColor backgroundColor;
    QColor foregroundColor;
    const bool isHighlighted = isDisplayHintEnabled(EnteredHint) ||
                               isDisplayHintEnabled(DraggedHint);
    if (isHighlighted) {
        backgroundColor = KGlobalSettings::highlightColor();
        foregroundColor = KGlobalSettings::highlightedTextColor();
    }
    else {
        backgroundColor = colorGroup().background();
        foregroundColor = KGlobalSettings::buttonTextColor();
    }

    // dimm the colors if the parent view does not have the focus
    const DolphinView* parentView = urlNavigator()->dolphinView();
    const Dolphin& dolphin = Dolphin::mainWin();

    const bool isActive = (dolphin.activeView() == parentView);
    if (!isActive) {
        QColor dimmColor(colorGroup().background());
        foregroundColor = mixColors(foregroundColor, dimmColor);
        if (isHighlighted) {
            backgroundColor = mixColors(backgroundColor, dimmColor);
        }
    }

    // draw button background
    painter->setPen(NoPen);
    painter->setBrush(backgroundColor);
    painter->drawRect(0, 0, buttonWidth, buttonHeight);

    int textWidth = buttonWidth;
    if (isDisplayHintEnabled(ActivatedHint) && isActive || isHighlighted) {
        painter->setPen(foregroundColor);
    }
    else {
        // dimm the foreground color by mixing it with the background
        foregroundColor = mixColors(foregroundColor, backgroundColor);
        painter->setPen(foregroundColor);
    }

    if (!isDisplayHintEnabled(ActivatedHint)) {
        // draw arrow
        const int middleY = height() / 2;
        const int width = arrowWidth();
        const int startX = (buttonWidth - width) - 1;
        const int startTopY = middleY - (width - 1);
        const int startBottomY = middleY + (width - 1);
        for (int i = 0; i < width; ++i) {
            painter->drawLine(startX, startTopY + i, startX + i, startTopY + i);
            painter->drawLine(startX, startBottomY - i, startX + i, startBottomY - i);
        }

        textWidth = startX - 1;
    }

    const bool clipped = isTextClipped();
    const int align = clipped ? Qt::AlignVCenter : Qt::AlignCenter;
    painter->drawText(QRect(0, 0, textWidth, buttonHeight), align, text());

    if (clipped) {
        // Blend the right area of the text with the background, as the
        // text is clipped.
        // TODO: use alpha blending in Qt4 instead of drawing the text that often
        const int blendSteps = 16;

        QColor blendColor(backgroundColor);
        const int redInc   = (foregroundColor.red()   - backgroundColor.red())   / blendSteps;
        const int greenInc = (foregroundColor.green() - backgroundColor.green()) / blendSteps;
        const int blueInc  = (foregroundColor.blue()  - backgroundColor.blue())  / blendSteps;
        for (int i = 0; i < blendSteps; ++i) {
            painter->setClipRect(QRect(textWidth - i, 0, 1, buttonHeight));
            painter->setPen(blendColor);
            painter->drawText(QRect(0, 0, textWidth, buttonHeight), align, text());

            blendColor.setRgb(blendColor.red()   + redInc,
                              blendColor.green() + greenInc,
                              blendColor.blue()  + blueInc);
        }
    }
}

void URLNavigatorButton::enterEvent(QEvent* event)
{
    URLButton::enterEvent(event);

    // if the text is clipped due to a small window width, the text should
    // be shown as tooltip
    if (isTextClipped()) {
        QToolTip::add(this, text());
    }
}

void URLNavigatorButton::leaveEvent(QEvent* event)
{
    URLButton::leaveEvent(event);
    QToolTip::remove(this);
}
void URLNavigatorButton::dropEvent(QDropEvent* event)
{
    KURL::List urls;
    if (KURLDrag::decode(event, urls) && !urls.isEmpty()) {
        setDisplayHintEnabled(DraggedHint, true);

        QString path(urlNavigator()->url().prettyURL());
        path = path.section('/', 0, m_index);

        Dolphin::mainWin().dropURLs(urls, KURL(path));

        setDisplayHintEnabled(DraggedHint, false);
        update();
    }
}

void URLNavigatorButton::dragEnterEvent(QDragEnterEvent* event)
{
    event->accept(KURLDrag::canDecode(event));

    setDisplayHintEnabled(DraggedHint, true);
    update();
}

void URLNavigatorButton::dragLeaveEvent(QDragLeaveEvent* event)
{
    URLButton::dragLeaveEvent(event);

    setDisplayHintEnabled(DraggedHint, false);
    update();
}


void URLNavigatorButton::updateNavigatorURL()
{
    QObject* p = parent();
    if ((p != 0) && p->inherits("URLNavigator")) {
        URLNavigator* navigator = static_cast<URLNavigator*>(p);
        QString old_path(navigator->url().prettyURL());
        QString new_path(old_path.section('/', 0, m_index));
        if (new_path != old_path) {
            const int len = new_path.length();
            if (new_path.at(len) != QChar('/')) {
                // Assure that paths always end with a slash. This prevents
                // a special handling of the root directory.
                new_path.append('/');
            }
            navigator->setURL(KURL(new_path));
        }
    }
}

int URLNavigatorButton::arrowWidth() const
{
    int width = (height() / 2) - 7;
    if (width < 4) {
        width = 4;
    }
    return width;
}

bool URLNavigatorButton::isTextClipped() const
{

    int availableWidth = width();
    if (!isDisplayHintEnabled(ActivatedHint)) {
        availableWidth -= arrowWidth() + 1;
    }

    QFontMetrics fontMetrics(font());
    return fontMetrics.width(text()) >= availableWidth;
}
