/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


/*  pulldown functions v.1
    Copyright (C) 2001  Tristan Willy
    - merged into Drip source code tree by Jarl van Katwijk - 20-11-2001

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	Notes
	=====
    	To convert a 29.97fps video to a 24fps video we need to prefrom a inverse 3:2 pulldown. What happens with film is that
    	it is shot at 24fps and is needed to be converted to 30fps for your TV. There are multiple problems with this and the
    	first one is that if the video is played progressivly (IE: computer monitor) at 29.97fps it will endure a interlacing
    	effect. This is because two frames out of five played are "mixed" together and if there is any motion durring this mixing
    	a choppy frame is produced. Another problem is that the audio is slowed down because the 3:2 pulldown maked a 30fps
    	video and the playrate is at 29.97fps. This is not normally a problem but can be if one is not careful about audio sync.

    	What we need to do is revert the 3:2 pulldown. How pulldown is done is shown below.

					Frame number
    	Film		1	2	3	4

		Video		1	2	3	4	5
			Odd		1	2	2	3	4
			Even	1	2	3	4	4

		To revert this process we take frame 3 and frame 4 of the video and make the odd and even fields frame 3 of the film.
		We let the others simply pass through. That's all there is to reverting 3:2 pulldown.
    
    Usage
    =====
    	To use properly, set your output to 24fps. Then call pulldown_init with the width and height. For tweaking there is
    	the odd and even field start setting and the start frame offset. Usually the default works. Then pass a 3byte frame
    	to inverse_pulldown along with the dimentions and the number of frames encoded with the first frame passed as frame zero.
		inverse_pulldown will return a frame pointer to the converted frame. If this pointer is the same as passed,
    	then simply do nothing. If it is NULL, skip the encoding of the passed frame. If it is different, copy the returned frame
    	to your current one. A sample piece would be like this.

			unsigned char* tmpbuffer = inverse_pulldown(framebuffer, width, height, pulldown_framenumber++);
			if(tmpbuffer != framebuffer && tmpbuffer){
				fast_memcpy(framebuffer, tmpbuffer, width * height * 3);
			}

			if(tmpbuffer)
				encode(framebuffer);

	   When finished encoding simply call pulldown_cleanup.

	Bugs
	====
		1.) For some reason I still see some slight interlacing effects every now and then. It would be great to know why this is happening.
		2.) Some videos produce a jittery effect to where it is like a new frame is late in arrival.
*/
#include <string.h>
#include "fast_memcpy.hh"
#include "pulldown.hh"


guchar *framethree, *outframe;
bool inited = false;
bool oddeven;
int framenumOffset;

int pulldown_init(int width, int height, bool oddfirst, int foffset){
        oddfirst = false;
        foffset = 0;
	if((framethree = new guchar[width * height * 3]) == NULL)
		return -1;
	if((outframe = new guchar[width * height * 3]) == NULL)
		return -1;

	oddeven = oddfirst;
	framenumOffset = foffset;
	inited = true;
		
	return 0;
}


unsigned char *inverse_pulldown(guchar *framebuffer, int width, int height, glong totalframes){
	glong framenumber;

	if(inited == false){
		pulldown_init(width, height,false,0);
		inited = true;
	}

	/* Get the pulldown frame number. */
	framenumber = (totalframes % 5) + 1 + framenumOffset;

	/* If the frame is 1, 2, or 5 then we just spit thoes out. */
	if(framenumber == 1 || framenumber == 2 || framenumber == 5)
		return framebuffer;

	/* The 24fps frame number 3 is split into fram 3 and 4 of a 30fps film.
	 * Here we try to store the 3rd 30fps frame and take the even field of it
	 * and the odd field of the 4th frame to make our true 3rd 24fps frame. */
	if(framenumber == 3){
		fast_memcpy(framethree, framebuffer, width * height * 3);
		return NULL;
	}

	/* This is the 30fps frame number 4 for sure, lets combine the fields.
	 * Take the odd from frame 4 and the even from frame 3 if we are oddeven ordered. */
	if(oddeven){
		for(unsigned int i = 0; i < height; i++){
			if(!(i % 2))
				fast_memcpy(outframe + (i * width * 3), framebuffer + (i * width * 3), width * 3);
			else 
				fast_memcpy(outframe + (i * width * 3), framethree + (i * width * 3), width * 3);
		}
	} else {
		for(unsigned int i = 0; i < height; i++){
			if(i % 2)
				fast_memcpy(outframe + (i * width * 3), framebuffer + (i * width * 3), width * 3);
			else 
				fast_memcpy(outframe + (i * width * 3), framethree + (i * width * 3), width * 3);
		}
	}
	
	return outframe;
}

void pulldown_cleanup(void){
	delete[] outframe;
	delete[] framethree;
}
