// This file is part of Dust Racing 2D.
// Copyright (C) 2011 Jussi Lind <jussi.lind@iki.fi>
//
// Dust Racing 2D is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// Dust Racing 2D is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Dust Racing 2D. If not, see <http://www.gnu.org/licenses/>.

#include "timing.hpp"
#include "car.hpp"

#include <QString>

#include <cassert>

Timing::Timing(unsigned int cars, QObject *parent)
: QObject(parent)
, m_times(cars, Timing::Times())
, m_time(0)
, m_started(false)
, m_lapRecord(-1)
, m_raceRecord(-1)
{
}

void Timing::setLapCompleted(unsigned int index, bool isHuman)
{
    Timing::Times & times = m_times.at(index);
    times.lap++;

    const int elapsed = m_time;
    times.lastLapTime = elapsed - times.raceTime;
    times.raceTime    = elapsed;

    // Check if a new personal record achieved.
    if (times.lastLapTime < times.recordLapTime ||
        times.recordLapTime == -1)
    {
        times.recordLapTime = times.lastLapTime;
    }

    // Check if a new lap record achieved.
    // Accept new lap records only by human players.
    if (isHuman)
    {
        if (times.lastLapTime < m_lapRecord || m_lapRecord == -1)
        {
            m_lapRecord = times.lastLapTime;

            emit lapRecordAchieved(m_lapRecord);
        }

        emit lapCompleted(index, times.lastLapTime);
    }
}

void Timing::setRaceCompleted(unsigned int index, bool state, bool isHuman)
{
    Timing::Times & times = m_times.at(index);
    times.raceCompleted = state;
    times.raceTime = m_time;

    if (isHuman)
    {
        if (times.raceTime < m_raceRecord || m_raceRecord == -1)
        {
            m_raceRecord = times.raceTime;

            emit raceRecordAchieved(m_raceRecord);
        }
    }
}

bool Timing::raceCompleted(unsigned int index) const
{
    const Timing::Times & times = m_times.at(index);
    return times.raceCompleted;
}

void Timing::setIsActive(unsigned int index, bool state)
{
    Timing::Times & times = m_times.at(index);
    times.isActive = state;
}

bool Timing::isActive(unsigned int index) const
{
    const Timing::Times & times = m_times.at(index);
    return times.isActive;
}

int Timing::lap(unsigned int index) const
{
    return m_times.at(index).lap;
}

int Timing::leadersLap() const
{
    int maxLap = 0;

    for (unsigned int index = 0; index < m_times.size(); index++)
    {
        const int lap = m_times.at(index).lap;
        if (lap > maxLap)
        {
            maxLap = lap;
        }
    }

    return maxLap;
}

int Timing::currentLapTime(unsigned int index) const
{
    if (!m_started)
    {
        return 0;
    }

    const Timing::Times & times = m_times.at(index);
    return m_time - times.raceTime;
}

int Timing::recordLapTime(unsigned int index) const
{
    if (!m_started)
    {
        return -1;
    }

    return m_times.at(index).recordLapTime;
}

int Timing::raceTime() const
{
    if (!m_started)
    {
        return 0;
    }

    return m_time;
}

int Timing::raceTime(unsigned int index) const
{
    if (!m_times.at(index).raceCompleted)
    {
        return raceTime();
    }
    else
    {
        return m_times.at(index).raceTime;
    }
}

int Timing::recordRaceTime(unsigned int index) const
{
    if (!m_started)
    {
        return -1;
    }

    return m_times.at(index).recordRaceTime;
}

int Timing::lapRecord() const
{
    return m_lapRecord;
}

void Timing::setLapRecord(int msecs)
{
    m_lapRecord = msecs;
}

int Timing::raceRecord() const
{
    return m_raceRecord;
}

void Timing::setRaceRecord(int msecs)
{
    m_raceRecord = msecs;
}

int Timing::lastLapTime(unsigned int index) const
{
    if (!m_started)
    {
        return -1;
    }

    return m_times.at(index).lastLapTime;
}

void Timing::start()
{
    m_time = 0;
    m_started = true;
}

void Timing::stop()
{
    m_started = false;
}

void Timing::reset()
{
    m_time = 0;
    m_started = false;

    for (Timing::Times & time : m_times)
    {
        time = Timing::Times();
    }
}

void Timing::tick()
{
    if (m_started)
    {
        m_time += 1000 / 60;
    }
}

std::wstring Timing::msecsToString(int msec)
{
    if (msec < 0)
    {
        return L"--:--.--";
    }

    const int hr = msec % 3600000;
    const int mm = hr   / 60000;
    const int mr = hr   % 60000;
    const int ss = mr   / 1000;
    const int ms = mr   % 1000;

    return QString().sprintf("%02d:%02d.%02d", mm, ss, ms / 10).toStdWString();
}

