/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Defines tools/types for 3 wire bus memory card handling
 *
 * Author :         Frank Thater (FTH)
 *
 * Last modified:   04/26/2000
 *
 *****************************************************************************/

#include <string.h>

#include "mc_3wbp.h"

int MC_3WBPInit(struct eco5000_t *ctx)

{
    struct memorycard_t *pt;
    int rc = OK;
    
    ctx->CTModFunc = (CTModFunc_t) MC_3WBPProcess;
    
    pt = (struct memorycard_t *)malloc(sizeof(struct memorycard_t));
    
    ctx->Data.card = pt;
    
    pt->SelectedFile = NONE;
    pt->Path = 0x00;
    pt->PINVerified = FALSE;
   
    pt->CardID = SLE4428;
   
    rc = DetermineVariables(ctx);
   
    if (rc < 0)
        return rc;
   
    rc = MC3WBP_UpdateBuffer(ctx, 0x00, -1);
   
    if (rc < 0)
        return rc;
   
#ifdef DEBUG
    {
    int help;
    
    printf("\nMemory Dump:\n");
    for(help=0;help < pt->Number_of_Data_Units;help++) {
        printf("%02x", pt->Memory[help]);
    }
    }
#endif 
   
    return rc;
}



int MC_3WBPProcess(struct eco5000_t *ctx,
                   unsigned int lc,
                   unsigned char *cmd,
                   unsigned int *lr,
                   unsigned char *rsp)

{
    int rc = OK;
    
#ifdef DEBUG
    printf("\nMC_3WBPProcess called...\n");
#endif
   
    /* Try calling the basic functions ... */
    
    rc = MemoryCardProcess(ctx, lc, cmd, lr, rsp);
    
    if(rc == OK)  /* command completed */
        return OK;
    
    if(rc < 0)    /* error processing command */
        return rc;
    
    
    /* "specialized" interindustry Commands */
    switch(cmd[1]) {
        case 0xA4:
            rc = MC3WBP_Select_File(ctx, lc, cmd, lr, rsp);
            break;
        case 0xB0:
            rc = MC3WBP_Read_Binary(ctx, lc, cmd, lr, rsp);
            break;
        case 0xD6:
            rc = MC3WBP_Update_Binary(ctx, lc, cmd, lr, rsp);
            break;
        case 0x20:
            rc = MC3WBP_Verify(ctx, lc, cmd, lr, rsp);
            break;
        case 0x24:
            rc = MC3WBP_Change_Verification_Data(ctx, lc, cmd, lr, rsp);
            break;
        default:
            *lr = 2;
            rsp[0] = HIGH(CLASS_NOT_SUPPORTED);
            rsp[1] = LOW(CLASS_NOT_SUPPORTED);
    }
  
    return rc;
    
}



int MC3WBP_Select_File(struct eco5000_t *ctx,
                       unsigned int lc,
                       unsigned char *cmd,
                       unsigned int *lr,
                       unsigned char *rsp)
{
    int rc = OK;

    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    *lr = 2;
    
    return rc;
}



int MC3WBP_Read_Binary(struct eco5000_t *ctx,
                       unsigned int lc,
                       unsigned char *cmd,
                       unsigned int *lr,
                       unsigned char *rsp)
{
    int rc = OK;

    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    *lr = 2;
    
    return rc;
}



/*
 * Update the memory area of a SLE4418/SLE4428 chip
 *
 */

int MC3WBP_Update_Binary(struct eco5000_t *ctx,
                         unsigned int lc,
                         unsigned char *cmd,
                         unsigned int *lr,
                         unsigned char *rsp)
{
    struct memorycard_t *pt;
    unsigned char command[3];
    unsigned int  offset, length;
    unsigned int  addr, i;
    unsigned char *data,*po;
    int rc;

    pt = ctx->Data.card;

    if (pt->SelectedFile != MF) {       /* Generic command only for MF       */
        rsp[0] = 0x69;
        rsp[1] = 0x86;
        *lr = 2;
        return 0;
    }
        
    offset = cmd[2] << 8 | cmd[3];      /* Decode file offset from P1 and P2 */

    if (offset >= pt->Number_of_Data_Units) {  /* Offset out of range        */
        rsp[0] = 0x6B;
        rsp[1] = 0x00;
        *lr = 2;
        return 0;
    }
                                        /* Decode lc and data field          */
    rc = DecodeAPDU(lc, cmd, &length, &data, NULL);

    if ((rc != 3) && (rc != 6)) {       /* Must the case 3S or 3E            */
        rsp[0] = HIGH(COMMUNICATION_NOT_POSSIBLE);
        rsp[1] = LOW(COMMUNICATION_NOT_POSSIBLE);
        *lr = 2;
        return 0;
    }
   
    if (offset + length >= pt->Number_of_Data_Units) {
        rsp[0] = HIGH(WRONG_LENGTH);                  /* Wrong lc                          */
        rsp[1] = LOW(WRONG_LENGTH);
        *lr = 2;
        return 0;
    }

    po = data;
    addr = offset;
    i = length;

    while (i--) {
        command[0] = UPDATE_MEMORY | (unsigned char)((addr >> 2) & 0xC0);
        command[1] = (unsigned char)(addr & 0xff);
        command[2] = *po;
                
        rc = MC3WBP_Command(ctx, command, NULL, 0);
                
        if (rc < 0) {
            rsp[0] = HIGH(COMMUNICATION_NOT_POSSIBLE);
            rsp[1] = LOW(COMMUNICATION_NOT_POSSIBLE);
            *lr = 2;
            return rc;
        }
                
        addr++;
        po++;
    }

    rc = MC3WBP_UpdateBuffer(ctx, offset, length);

    if (rc < 0)
        return rc;
                                        /* Verify update process             */
    if (memcmp(data, pt->Memory + offset, length)) {
        rsp[0] = HIGH(DATA_CORRUPTED);
        rsp[1] = LOW(DATA_CORRUPTED);
        *lr = 2;
        return 0;
    }   
    
    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    *lr = 2;

    return 0;
}



int MC3WBP_Verify(struct eco5000_t *ctx,
                  unsigned int lc,
                  unsigned char *cmd,
                  unsigned int *lr,
                  unsigned char *rsp)

{
    int rc = OK;
    unsigned char pin[2];
    unsigned char command[3];
    unsigned char buffer[16];
    unsigned char ec;
    struct memorycard_t *pt = ctx->Data.card;
    int toRead;

    memcpy(&pin, cmd + 5, 0x02); /* Two Bytes of command must be pin !*/
  
    *lr = 0;

    /* Get error counter */

    command[0] = READ_MEMORY | 0xC0;
    command[1] = 0xFD;
    command[2] = 0;

    toRead = 0x01;
    rc = MC3WBP_Command(ctx, command, buffer, &toRead);

    if (rc < 0)
        return rc;

    ec = buffer[0];                /* get error counter    */
  
    if (ec == 0) {                         /* Nothing left in error counter     */
        rsp[0] = HIGH(VERIFICATION_METHOD_BLOCK);
        rsp[1] = LOW(VERIFICATION_METHOD_BLOCK);
        *lr = 2;
        return OK;
    }
  
    ec <<= 1;   /* count down error counter   */

    command[0] = UPDATE_ERROR_COUNTER | 0xC0;
    command[1] = 0xFD;
    command[2] = ec;

    rc = MC3WBP_Command(ctx, command, NULL, 0);

    if (rc < 0)
        return rc;

    command[0] = VERIFY_PSC | 0xC0;
    command[1] = 0xFE;
    command[2] = pin[0];

    rc = MC3WBP_Command(ctx, command, NULL, 0);

    if (rc < 0)
        return rc;

    command[0] = VERIFY_PSC | 0xC0;
    command[1] = 0xFF;
    command[2] = pin[1];

    rc = MC3WBP_Command(ctx, command, NULL, 0);

    if (rc < 0)
        return rc;


    command[0] = UPDATE_MEMORY | 0xC0;
    command[1] = 0xFD;
    command[2] = 0xFF;

    rc = MC3WBP_Command(ctx, command, NULL, 0);

    if (rc < 0)
        return rc;


    /* Get error counter */
    command[0] = READ_MEMORY | 0xC0;
    command[1] = 0xFD;
    command[2] = 0;

    toRead = 0x01;
    rc = MC3WBP_Command(ctx, command, buffer, &toRead);

    if (rc < 0)
        return rc;

    ec = buffer[0];                /* get error counter    */
  
    if (ec == 0) {                         /* Nothing left in error counter     */
        rsp[0] = HIGH(VERIFICATION_METHOD_BLOCK);
        rsp[1] = LOW(VERIFICATION_METHOD_BLOCK);
        *lr = 2;
        return OK;
    }

    if (ec != 0xFF) {    /* wrong pin */
        rsp[0] = 0x63;
        rsp[1] = 0xC1;
        *lr = 2;
        return OK;
    }

    pt->PINVerified = TRUE;

    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    *lr = 2;


    return rc;
}



int MC3WBP_Change_Verification_Data(struct eco5000_t *ctx,
                                    unsigned int lc,
                                    unsigned char *cmd,
                                    unsigned int *lr,
                                    unsigned char *rsp)

{
    int rc = OK;
    struct memorycard_t *pt = ctx->Data.card;
    unsigned char newpin[3];
    unsigned char command[3];
 
#ifdef DEBUG
    int i = 0;
#endif

    memcpy(&newpin, cmd + 7, 2);

    if (pt->PINVerified != TRUE) {
        MC3WBP_Verify(ctx, lc - 2, cmd, lr, rsp);

        if (rsp[0] != 0x90) {
            return OK;
        }
    }

#ifdef DEBUG
    for(i=0;i<2;i++)
        printf("\n%02x", newpin[i]);
#endif
    *lr = 0;

    
    command[0] = UPDATE_ERROR_COUNTER | 0xC0;
    command[1] = 0xFD;
    command[2] = 0xFF;

    rc = MC3WBP_Command(ctx, command, NULL, 0);

    if (rc < 0)
        return rc;

    command[0] = UPDATE_MEMORY | 0xC0;
    command[1] = 0xFE;
    command[2] = newpin[0];

    rc = MC3WBP_Command(ctx, command, NULL, 0);

    if (rc < 0)
        return rc;

    command[0] = UPDATE_MEMORY | 0xC0;
    command[1] = 0xFF;
    command[2] = newpin[1];

    rc = MC3WBP_Command(ctx, command, NULL, 0);

    if (rc < 0)
        return rc;


    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    *lr = 2;

    return rc;

}



/*
 * MC3WBP_Command
 *
 * Send a command to the 3WBP-card and either
 * read the response (rsp != NULL) or clock the chip until
 * command completes
 *
 */

int MC3WBP_Command(struct eco5000_t *ctx,
                   unsigned char *cmd3wbp,
                   unsigned char *rsp,
                   int *length)

{
    int rc;
    unsigned char ack;
    
    rc = ecoCommand(ctx, COMMAND_4428, 3, cmd3wbp, 0, NULL);

    if (rc != ACK_CVCC_ON) {
#ifdef DEBUG
        printf("[MC3WBP_Command] Sending command rc=%d [%02x]\n", rc, rc);
#endif
        return ERR_TRANS;
    }

    if (rsp)
        rc = ecoCommand(ctx, READ_4428, 1, (unsigned char *)length, *length, rsp);
    else
        rc = ecoCommand(ctx, CLOCK_4428, 0, NULL, 1, &ack);

    if (rc != ACK_CVCC_ON) {
#ifdef DEBUG
        printf("[MC3WBP_Command] Getting response rc=%d [%02x]\n", rc, rc);
#endif
        return ERR_TRANS;
    }

    return 0;
}



/*
 * MC3WBP_UpdateBuffer
 * copies the content of the card memory into the memory image 
 *
 * length = -1 && soffset = 0x00 -> Read whole card
 * length = 0xff && soffset = 0x10 -> Read 256 bytes at offset 0x10
 *
 */

int MC3WBP_UpdateBuffer(struct eco5000_t *ctx, unsigned int soffset, int length) 

{
    struct memorycard_t *pt = ctx->Data.card;
    unsigned char *pointer = pt->Memory;
    int toSend;
    unsigned char cmd[3];
    int expected_bytes;
    unsigned int offset;
    int rc;

    if (length == -1)  {
        expected_bytes = 1024;
        offset = 0x00;
    }
    else {
        expected_bytes = length;
        offset = soffset;
    }

    pointer += offset;

    while (expected_bytes > 0)   {
    
        cmd[0] = READ_MEMORY | (unsigned char)((offset >> 2) & 0xC0);
        cmd[1] = (unsigned char)(offset & 0xff);
        cmd[2] = 0x00;
     
        if (expected_bytes > 0xff) {
            toSend = 0xff;
            rc = MC3WBP_Command(ctx, cmd, pointer, &toSend);
            pointer += 255;
            offset += 255;
            expected_bytes -= 255;
        }
        else {
            rc = MC3WBP_Command(ctx, cmd, pointer, &expected_bytes);    
            expected_bytes = 0x00;
        }
    
        if (rc < 0) {
            return rc;
        }

    }

    return OK;
}















