# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
UPnP MediaRenderer
"""

__maintainer__ = "Philippe Normand <philippe@fluendo.com>"

from elisa.base_components import service_provider
from elisa.core import common, component
from elisa.core.application import ComponentsLoadedMessage
from elisa.core import player, media_uri

from twisted.spread import pb

plugin_registry = common.application.plugin_registry
CoherencePluginMessage = plugin_registry.get_component_class('coherence:coherence_plugin_message')
PBReferenceableMessage = plugin_registry.get_component_class('twisted:pbreferenceable_message')
PlayerModelMessage = plugin_registry.get_component_class('base:player_model_message')

class UPnPMediaRenderer(pb.Referenceable, service_provider.ServiceProvider):
    """
    DOCME
    """

    def initialize(self):
        self._player_model = None
        common.application.bus.register(self._bus_message_received,
                                        PlayerModelMessage,
                                        ComponentsLoadedMessage)

    def _bus_message_received(self, msg, sender):
        if isinstance(msg, PlayerModelMessage):
            self._player_model = msg.player_model
        else:
            # register myself to the PB service
            msg = PBReferenceableMessage('get_player', self)
            common.application.bus.send_message(msg)

            args = {}#'host':'internal'}
            msg = CoherencePluginMessage('ElisaPlayer', args)
            common.application.bus.send_message(msg)


    def remote_set_uri(self, uri):
        if not self._player_model:
            return
        if uri:
            uri = media_uri.MediaUri(uri)
            self.debug("Playing %r", uri)
            self._player_model.uri = uri
        else:
            self.warning("Not usable URI: %r", uri)

    def remote_get_readable_state(self):
        state = 'STOPPED'
        if self._player_model and self._player_model.state:
            state = str(self._player_model.state)

        return state

    def remote_get_volume(self):
        volume = 0
        if self._player_model:
            volume = self._player_model.volume
            volume = (volume * 100) / 10.
        return int(volume)


    def remote_set_volume(self, volume):
        if self._player_model:
            if int(volume) in range(101):
                volume = int(volume)
                self.info("Setting volume to %s" % volume)
                volume = (volume * 10) / 100.
                if volume < 0.:
                    volume = 0.
                elif volume > 10.:
                    volume = 10.
                self._player_model.volume = volume

    def remote_get_status(self):
        position, duration = 0, 0
        if self._player_model:
            position = self._player_model.position
            duration = self._player_model.duration
        return (position, duration)

    def remote_get_mute(self):
        muted = False
        if self._player_model:
            muted = self._player_model.muted
        return muted

    def remote_mute(self):
        if self._player_model:
            self._player_model.muted = True

    def remote_un_mute(self):
        if self._player_model:
            self._player_model.muted = False

    def remote_play(self):
        if self._player_model:
            self._player_model.state = player.STATES.PLAYING
        return True

    def remote_pause(self):
        if self._player_model:
            self._player_model.state = player.STATES.PAUSED
        return True

    def remote_stop(self):
        if self._player_model:
            self._player_model.state = player.STATES.STOPPED
        return True


"""
remote_play_uri = play_uri_once
remote_toggle_play_pause = toggle_play_pause
remote_seek_forward = seek_forward
remote_seek_backward = seek_backward
remote_increment_playback_speed = increment_playback_speed
remote_decrement_playback_speed = decrement_playback_speed
remote_next = next
remote_get_current_item = get_current_item
remote_get_status_informations = get_status_informations
"""
