#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "guiutils.h"
#include "cdialog.h"

#include "cfg.h"
#include "cfgfio.h"
#include "url.h"
#include "edvtypes.h"
#include "edvhistory.h"
#include "edvobj.h"
#include "edvdevices.h"
#include "edvdevicesfio.h"
#include "edvmimetypes.h"
#include "edvmimetypesfio.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "browser.h"
#include "browserdirtree.h"
#include "browseropcb.h"
#include "imbr.h"
#include "imbropcb.h"
#include "archiver.h"
#include "archiveropcb.h"
#include "propdlg.h"
#include "recbin.h"
#include "recbincontents.h"
#include "recbinopcb.h"
#include "findwin.h"
#include "historywin.h"
#include "optwin.h"
#include "optwinop.h"
#include "edvoptions.h"
#include "edvcustomize.h"
#include "rundlg.h"
#include "deviceswin.h"
#include "mimetypeswin.h"
#include "aboutdlg.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvhelp.h"
#include "edvop.h"
#include "edvcfglist.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "config.h"

#include "images/icon_trash_32x32.xpm"


/* New/Map Window Nexus */
gint EDVNewWindowNexus(
	edv_core_struct *core_ptr,
	const gchar *name,              /* Name of window */
	const gchar *path,              /* Initial object path */
	const gchar *extra,
	GtkWidget *toplevel
);


/* New Windows */
gint EDVNewBrowser(edv_core_struct *core_ptr);
gint EDVNewImbr(edv_core_struct *core_ptr);
gint EDVNewArchiverOpen(edv_core_struct *core_ptr, const gchar *path);
gint EDVNewArchiver(edv_core_struct *core_ptr);

gint EDVNewPropertiesDialogPage(
	edv_core_struct *core_ptr, edv_object_struct *obj,
	const gchar *page_name,
	GtkWidget *toplevel
);
gint EDVNewPropertiesDialog(
	edv_core_struct *core_ptr, edv_object_struct *obj,
	GtkWidget *toplevel
);


/* Running */
void EDVRunDeviceCheck(
	edv_core_struct *core_ptr, edv_device_struct *dev,
	GtkWidget *toplevel
);
void EDVRunDeviceTools(
	edv_core_struct *core_ptr, edv_device_struct *dev,
	GtkWidget *toplevel
);
void EDVRunDeviceFormat(
	edv_core_struct *core_ptr, edv_device_struct *dev,
	GtkWidget *toplevel
);

void EDVRunTerminal(
	edv_core_struct *core_ptr, const gchar *cmd,
	GtkWidget *toplevel
);


/* Window Mapping */
void EDVMapRecBinDeskIcon(edv_core_struct *core_ptr);
void EDVMapRecBin(edv_core_struct *core_ptr);

void EDVMapHistoryListWin(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);

void EDVMapOptionsWinPage(
	edv_core_struct *core_ptr,
	const gchar *page_name,
	GtkWidget *toplevel
);
void EDVMapOptionsWin(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);

void EDVMapCustomizeWin(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);

void EDVMapDevicesListWin(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);

void EDVMapMIMETypesListWin(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);
void EDVMapRunDialogCommand(
	edv_core_struct *core_ptr,
	const gchar *command,
	const gchar *working_dir,
	GtkWidget *toplevel
);
void EDVMapRunDialog(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);

void EDVMapBrowserFindWin(
	edv_core_struct *core_ptr, edv_browser_struct *browser
);
void EDVMapImbrFindWin(
	edv_core_struct *core_ptr, edv_imbr_struct *imbr
);
void EDVMapArchiverFindWin(
	edv_core_struct *core_ptr, edv_archiver_struct *archiver
);
void EDVMapRecBinFindWin(
	edv_core_struct *core_ptr, edv_recbin_struct *recbin
);


/* About */
void EDVAboutPage(
	edv_core_struct *core_ptr,
	const gchar *page_name,
	GtkWidget *toplevel
);
void EDVAbout(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);


/* Update Resources & Widgets */
void EDVUpdateRCStyles(edv_core_struct *core_ptr);
void EDVUpdateIDPULists(edv_core_struct *core_ptr);
void EDVUpdateDevicesPUList(edv_core_struct *core_ptr);
void EDVUpdateMIMETypeHintIndices(edv_core_struct *core_ptr);
void EDVUpdateOpenWithPUList(edv_core_struct *core_ptr);


/* Refresh & Reset */
void EDVRefresh(edv_core_struct *core_ptr);
void EDVReset(edv_core_struct *core_ptr);


/* Sync */
void EDVSyncDisks(edv_core_struct *core_ptr);


/* Clearing */
void EDVClearHistoryEvents(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);
void EDVClearLocationsHistory(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);
void EDVClearRunHistory(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);


/* Network */
void EDVInternetDownloadObject(
	edv_core_struct *core_ptr,
	const url_struct *url,			/* Source */
	const gchar *target_path,		/* Target */
	GtkWidget *toplevel
);


/* Recycle Bin Size Check */
void EDVRecycleBinSizeCheck(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Nexus for creating a new window.
 */
gint EDVNewWindowNexus(
	edv_core_struct *core_ptr,
	const gchar *name,		/* Name of window */
	const gchar *path,		/* Initial object path */
	const gchar *extra,		/* Topic, page, tab, etc */
	GtkWidget *toplevel
)
{
	gchar *s, *dname;

	if((core_ptr == NULL) || STRISEMPTY(name))
	    return(-1);

	/* Make a copy of the window's name */
	dname = STRDUP(name);

	/* Replace all '-' and blank characters in dname with '_' for
	 * easier matching
	 */
	for(s = dname; *s != '\0'; s++)
	{
	    if((*s == '-') || ISBLANK(*s))
		*s = '_';
	}


	/* Begin checking which window we should create by checking
	 * dname
	 *
	 * Note that dname uses '_' characters to replace '-' or
	 * blank characters
	 */

	/* File Browser */
	if(!g_strcasecmp(dname, "file_browser") ||
	   !g_strcasecmp(dname, "filebrowser") ||
	   !g_strcasecmp(dname, "browser")
	)
	{
	    EDVNewBrowser(core_ptr);
	}
	/* Image Browser */
	else if(!g_strcasecmp(dname, "image_browser") ||
	        !g_strcasecmp(dname, "imagebrowser") ||
	        !g_strcasecmp(dname, "imbr")
	)
	{
	    EDVNewImbr(core_ptr);
	}
	/* Archiver */
	else if(!g_strcasecmp(dname, "archiver"))
	{
	    EDVNewArchiverOpen(core_ptr, path);
	}
	/* Recycle Bin */
	else if(!g_strcasecmp(dname, "recycle_bin") ||
	        !g_strcasecmp(dname, "recyclebin") ||
		!g_strcasecmp(dname, "rec_bin") ||
	        !g_strcasecmp(dname, "recbin")
	)
	{
	    EDVMapRecBin(core_ptr);
	}
	/* Properties Dialog */
	else if(!g_strcasecmp(dname, "properties_dialog") ||
		!g_strcasecmp(dname, "properties") ||
		!g_strcasecmp(dname, "prop_win") ||
		!g_strcasecmp(dname, "prop_dlg") ||
		!g_strcasecmp(dname, "propdlg") ||
	        !g_strcasecmp(dname, "propdlg")
	)
	{
	    struct stat lstat_buf;
	    edv_object_struct *obj = EDVObjectNew();
	    if(!STRISEMPTY(path))
	    {
		EDVObjectSetPath(obj, path);
		if(!lstat(path, &lstat_buf))
		    EDVObjectSetStat(obj, &lstat_buf);
	    }
	    EDVNewPropertiesDialogPage(core_ptr, obj, extra, NULL);
	    EDVObjectDelete(obj);
	}
	/* Options Window */
	else if(!g_strcasecmp(dname, "options_window") ||
		!g_strcasecmp(dname, "options")
	)
	{
	    EDVMapOptionsWinPage(core_ptr, extra, NULL);
	}
	/* Customize Window */
	else if(!g_strcasecmp(dname, "customize_window") ||
		!g_strcasecmp(dname, "customize")
	)
	{
	    EDVMapCustomizeWin(core_ptr, NULL);
	}
	/* Devices Window */
	else if(!g_strcasecmp(dname, "devices_window") ||
		!g_strcasecmp(dname, "devices")
	)
	{
	    EDVMapDevicesListWin(core_ptr, NULL);
	}
	/* MIME Types Window */
	else if(!g_strcasecmp(dname, "mimetypes_window") ||
		!g_strcasecmp(dname, "mimetypes")
	)
	{
	    EDVMapMIMETypesListWin(core_ptr, NULL);
	}
	/* Run Dialog */
	else if(!g_strcasecmp(dname, "run_dialog") ||
		!g_strcasecmp(dname, "run_dlg") ||
		!g_strcasecmp(dname, "rundlg")
	)
	{
	    EDVMapRunDialogCommand(
		core_ptr,
		((path == NULL) && (extra != NULL)) ?
		    extra : path,
		extra,
		NULL
	    );
	}
	/* About Dialog */
	else if(!g_strcasecmp(dname, "about_dialog") ||
		!g_strcasecmp(dname, "about_dlg") ||
		!g_strcasecmp(dname, "aboutdlg")
	)
	{
	    EDVAboutPage(core_ptr, extra, NULL);
	}
	/* Help */
	else if(!g_strcasecmp(dname, "help"))
	{
	    EDVHelp(core_ptr, extra, NULL);
	}


	g_free(dname);

	return(0);
}

/*
 *	Procedure to create a new browser and map it. The new browser
 *	will be appended to the core_ptr or an existing (unmapped)
 *	browser will be used.
 *
 *	Returns the index of the browser on the core structure or -1
 *	on error.
 */
gint EDVNewBrowser(edv_core_struct *core_ptr)
{
	gint i, existing_windows = 0;
	cfg_item_struct *cfg_list;
	edv_browser_struct *browser;

	if(core_ptr == NULL)
	    return(-1);

	cfg_list = core_ptr->cfg_list;

	/* Count existing windows */
	for(i = 0; i < core_ptr->total_browsers; i++)
	{
	    if(core_ptr->browser[i] != NULL)
		existing_windows++;
	}

	/* Look for available pointer in the array */
	for(i = 0; i < core_ptr->total_browsers; i++)
	{
	    if(core_ptr->browser[i] == NULL)
		break;
	}
	if(i >= core_ptr->total_browsers)
	{
	    /* Allocate more pointers */
	    i = MAX(core_ptr->total_browsers, 0);
	    core_ptr->total_browsers = i + 1;

	    core_ptr->browser = (edv_browser_struct **)g_realloc(
		core_ptr->browser,
		core_ptr->total_browsers * sizeof(edv_browser_struct *)
	    );
	    if(core_ptr->browser == NULL)
	    {
		core_ptr->total_browsers = 0;
		return(-1);
	    }
	    core_ptr->browser[i] = NULL;
	}

	/* Shift window position in options if there are existing
	 * windows
	 */
	if(existing_windows > 0)
	{
	    gint	offset_x = EDV_GET_I(
			    EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_X
			),
			offset_y = EDV_GET_I(
			    EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_Y
			),
			x = EDV_GET_I(
			    EDV_CFG_PARM_BROWSER_X
			) + offset_x,
			y = EDV_GET_I(
			    EDV_CFG_PARM_BROWSER_Y
			) + offset_y;
	    EDV_SET_I(EDV_CFG_PARM_BROWSER_X, x);
	    EDV_SET_I(EDV_CFG_PARM_BROWSER_Y, y);
	}

	/* Create new browser */
	core_ptr->browser[i] = browser = EDVBrowserNew(core_ptr);
	if(browser != NULL)
	{
	    gchar *startup_dir = EDVEvaluatePath(
		NULL, EDV_GET_S(EDV_CFG_PARM_DIR_START_UP)
            );
	    GtkWidget *toplevel = browser->toplevel;
	    GtkCList *clist;

	    EDVBrowserSetBusy(browser, TRUE);
	    GUIBlockInput(toplevel, TRUE);

	    /* Need to map it immediatly after creation, before any
	     * timeouts check on it
	     */
	    EDVBrowserMap(browser);

	    /* Get directory tree and create initial toplevel nodes */
	    clist = GTK_CLIST(browser->directory_ctree);
	    if(clist != NULL)
	    {
		/* Set directory tree origin as needed */
		if(!EDVBrowserIsPathFromDirTreeOrigin(browser, startup_dir))
		{
		    EDVBrowserDirTreeSetOriginPath(browser, startup_dir);
		}
		else
		{
		    /* Create toplevels */
		    EDVBrowserDirTreeCreateToplevels(browser);

		    /* Select startup directory */
		    EDVBrowserDirTreeSelectPath(browser, startup_dir);
		}
	    }

	    g_free(startup_dir);

	    GUIBlockInput(toplevel, FALSE);
	    EDVBrowserSetBusy(browser, FALSE);

	    return(i);
	}
	else
	{
	    return(-1);
	}
}

/*
 *      Procedure to create a new image browser and map it. The new
 *	image browser will be appended to the core_ptr or an existing
 *	(unmapped)image browser will be used.
 *
 *      Returns the index of the browser on the core structure or -1
 *      on error.
 */
gint EDVNewImbr(edv_core_struct *core_ptr)
{
	gint i, existing_windows = 0;
	cfg_item_struct *cfg_list;
	edv_imbr_struct *imbr;

	if(core_ptr == NULL)
	    return(-1);

	cfg_list = core_ptr->cfg_list;

	/* Count existing windows */
	for(i = 0; i < core_ptr->total_imbrs; i++)
	{
	    if(core_ptr->imbr[i] != NULL)
		existing_windows++;
	}

	/* Look for available pointer in the array */
	for(i = 0; i < core_ptr->total_imbrs; i++)
	{
	    if(core_ptr->imbr[i] == NULL)
		break;
	}
	if(i >= core_ptr->total_imbrs)
	{
	    /* Allocate more pointers */
	    i = MAX(core_ptr->total_imbrs, 0);
	    core_ptr->total_imbrs = i + 1;

	    core_ptr->imbr = (edv_imbr_struct **)g_realloc(
		core_ptr->imbr,
		core_ptr->total_imbrs * sizeof(edv_imbr_struct *)
	    );
	    if(core_ptr->imbr == NULL)
	    {
		core_ptr->total_imbrs = 0;
		return(-1);
	    }
	    core_ptr->imbr[i] = NULL;
	}

	/* Shift window position in options if there are existing
	 * windows
	 */
	if(existing_windows > 0)
	{
	    gint	offset_x = EDV_GET_I(
			    EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_X
			),
			offset_y = EDV_GET_I(
			    EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_Y
			),
			x = EDV_GET_I(
			    EDV_CFG_PARM_IMBR_X
			) + offset_x,
			y = EDV_GET_I(
			    EDV_CFG_PARM_IMBR_Y
			) + offset_y;
	    EDV_SET_I(EDV_CFG_PARM_IMBR_X, x);
	    EDV_SET_I(EDV_CFG_PARM_IMBR_Y, y);
	}

	/* Create new image browser */
	core_ptr->imbr[i] = imbr = EDVImbrNew(core_ptr);
	if(imbr != NULL)
	{
	    gchar *startup_dir = EDVEvaluatePath(
		NULL, EDV_GET_S(EDV_CFG_PARM_DIR_START_UP)
            );
	    GtkWidget *toplevel = imbr->toplevel;

	    EDVImbrSetBusy(imbr, TRUE);
	    GUIBlockInput(toplevel, TRUE);

	    /* Need to map it immediatly after creation, before any
	     * timeouts check on it
	     */
	    EDVImbrMap(imbr);

	    /* Switch to startup directory */
	    EDVImbrSelectPath(imbr, startup_dir);

	    EDVImbrUpdateMenus(imbr);

	    g_free(startup_dir);

            GUIBlockInput(toplevel, FALSE);
            EDVImbrSetBusy(imbr, FALSE);

	    return(i);
	}
	else
	{
	    return(-1);
	}
}

/*
 *      Procedure to create a new archiver and map it. The new
 *	archiver will be appended to the core_ptr or an existing
 *      (unmapped) archiver will be used.
 *
 *      Returns the index of the archiver on the core structure or -1
 *      on error.
 */
gint EDVNewArchiverOpen(edv_core_struct *core_ptr, const gchar *path)
{
	gint i, existing_windows = 0;
	cfg_item_struct *cfg_list;
	edv_archiver_struct *archiver;

	if(core_ptr == NULL)
	    return(-1);

	cfg_list = core_ptr->cfg_list;

	/* Count existing windows */
	for(i = 0; i < core_ptr->total_archivers; i++)
	{
	    if(core_ptr->archiver[i] != NULL)
		existing_windows++;
	}

	/* Look for available pointer in the array */
	for(i = 0; i < core_ptr->total_archivers; i++)
	{
	    if(core_ptr->archiver[i] == NULL)
		break;
	}
	if(i >= core_ptr->total_archivers)
	{
	    /* Allocate more pointers */
	    i = MAX(core_ptr->total_archivers, 0);
	    core_ptr->total_archivers = i + 1;

	    core_ptr->archiver = (edv_archiver_struct **)g_realloc(
		core_ptr->archiver,
		core_ptr->total_archivers * sizeof(edv_archiver_struct *)
	    );
	    if(core_ptr->archiver == NULL)
	    {
		core_ptr->total_archivers = 0;
		return(-1);
	    }
	    core_ptr->archiver[i] = NULL;
	}

	/* Shift window position in options if there are existing
	 * windows
	 */
	if(existing_windows > 0)
	{
	    const gint	offset_x = EDV_GET_I(
			    EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_X
			),
			offset_y = EDV_GET_I(
			    EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_Y
			),
			x = EDV_GET_I(
			    EDV_CFG_PARM_ARCHIVER_X
			) + offset_x,
			y = EDV_GET_I(
			    EDV_CFG_PARM_ARCHIVER_Y
			) + offset_y;
	    EDV_SET_I(EDV_CFG_PARM_ARCHIVER_X, x);
	    EDV_SET_I(EDV_CFG_PARM_ARCHIVER_Y, y);
	}

	/* Create new archiver */
	core_ptr->archiver[i] = archiver = EDVArchiverNew(core_ptr);
	if(archiver != NULL)
	{
	    EDVArchiverSetBusy(archiver, TRUE);
	    GUIBlockInput(archiver->toplevel, TRUE);

	    /* Need to map it immediatly after creation, before any
	     * timeouts check on it
	     */
	    EDVArchiverMap(archiver);

	    if(path != NULL)
	    {
		/* Clear the contents clist and load the listing of the
		 * new archive specified by path
		 */
		EDVArchiverSelectArchive(
		    archiver, path,
		    EDVArchiverCurrentPassword(archiver)
		);
	    }

	    EDVArchiverUpdateMenus(archiver);

	    GUIBlockInput(archiver->toplevel, FALSE);
	    EDVArchiverSetBusy(archiver, FALSE);

	    return(i);
	}
	else
	{
	    return(-1);
	}
}

gint EDVNewArchiver(edv_core_struct *core_ptr)
{
	return(EDVNewArchiverOpen(core_ptr, NULL));
}


/*
 *	Creates a new Properties Dialog and maps it displaying the
 *	information about the object specified by obj and switches to
 *	the page specified by page_name.
 *
 *	Returns the index of the new Properties Dialog or -1 on error.
 */
gint EDVNewPropertiesDialogPage(
	edv_core_struct *core_ptr, edv_object_struct *obj,
	const gchar *page_name,
	GtkWidget *toplevel
)
{
	gint i, existing_windows = 0;
	edv_propdlg_struct *propdlg;
	cfg_item_struct *cfg_list;

	if(core_ptr == NULL)
	    return(-1);

	cfg_list = core_ptr->cfg_list;

	/* Count existing windows */
	for(i = 0; i < core_ptr->total_propdlgs; i++)
	{
	    if(core_ptr->propdlg[i] != NULL)
		existing_windows++;
	}

	/* Look for available pointer in the array */
	for(i = 0; i < core_ptr->total_propdlgs; i++)
	{
	    if(core_ptr->propdlg[i] == NULL)
		break;
	}
	if(i >= core_ptr->total_propdlgs)
	{
	    /* Allocate more pointers */
	    i = MAX(core_ptr->total_propdlgs, 0);
	    core_ptr->total_propdlgs = i + 1;

	    core_ptr->propdlg = (edv_propdlg_struct **)g_realloc(
		core_ptr->propdlg,
		core_ptr->total_propdlgs * sizeof(edv_propdlg_struct *)
	    );
	    if(core_ptr->propdlg == NULL)
	    {
		core_ptr->total_propdlgs = 0;
		return(-1);
	    }

	    core_ptr->propdlg[i] = NULL;
	}

	/* Create new property window */
	core_ptr->propdlg[i] = propdlg = EDVPropDlgNew(core_ptr, obj);
	if(propdlg != NULL)
	{
	    GtkNotebook *notebook = GTK_NOTEBOOK(propdlg->notebook);

	    /* Select tab? */
	    if((page_name != NULL) && (notebook != NULL))
	    {
		gint	page_num = -1,
			total_pages = g_list_length(notebook->first_tab);

		/* Check which tab is to be switched to by matching
		 * its name
		 */
		if(!g_strcasecmp(page_name, "General"))
		    page_num = 0;
		else if(!g_strcasecmp(page_name, "Details"))
		    page_num = MAX(total_pages - 1, 1);
		else if(!g_strcasecmp(page_name, "Link"))
		    page_num = 1;
		else if(!g_strcasecmp(page_name, "Device"))
		    page_num = 1;
		else if(!g_strcasecmp(page_name, "Device Node"))
		    page_num = (total_pages >= 4) ? 2 : 1;

		/* Matched page to be switched to? */
		if(page_num > -1)
		    gtk_notebook_set_page(notebook, page_num);
	    }

	    /* Shift window position if there are existing windows */
	    if(existing_windows > 0)
	    {
		gint	offset_x = CFGItemListGetValueI(
			    cfg_list,
			    EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_X
			) * existing_windows,
			offset_y = CFGItemListGetValueI(
			    cfg_list,
			    EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_Y
			) * existing_windows;
		EDVCenterWindowToWindowOffset(
		    toplevel, propdlg->toplevel,
		    offset_x, offset_y
		);
	    }
	    else
	    {
		EDVCenterWindowToWindow(
		    toplevel, propdlg->toplevel
		);
	    }

	    EDVPropDlgMap(propdlg);
	    EDVPropDlgResetHasChanges(propdlg, FALSE);
	    EDVPropDlgUpdateMenus(propdlg);

	    return(i);
	}
	else
	{
	    return(-1);
	}
}

gint EDVNewPropertiesDialog(
	edv_core_struct *core_ptr, edv_object_struct *obj,
	GtkWidget *toplevel
)
{
	return(
	    EDVNewPropertiesDialogPage(core_ptr, obj, NULL, toplevel)
	);
}


/*
 *	Runs the device check command on the given device reference.
 */
void EDVRunDeviceCheck(
	edv_core_struct *core_ptr, edv_device_struct *dev,
	GtkWidget *toplevel
)
{
	pid_t p;
	const gchar *cmd;

	if((core_ptr == NULL) || (dev == NULL))
	    return;

	cmd = dev->command_check;
	if(cmd == NULL)
	    return;

	/* Seek past spaces in command string and make sure it is not
	 * empty
	 */
	while(ISBLANK(*cmd))
	    cmd++;
	if(*cmd == '\0')
	    return;

	p = Exec(cmd);
	if(p > 0)
	{
	    dev->last_check_time = (gulong)time(NULL);
	}
	else
	{
	    gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute command:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar la orden:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'ordre:\n\
\n\
    %s\n",
#endif
		cmd
	    );
	    EDVPlaySoundError(core_ptr);
	    EDVMessageError(
#ifdef PROG_LANGUAGE_ENGLISH
"Cannot Run Program",
		    buf,
"An error was encountered while attempting to run the\n\
specified program. Please verify that the command\n\
is specified properly by going to Device->Devices...\n\
and edit the device reference.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"No Puede Correr El Programa",
		    buf,
"Un error se encontr al procurar para correr el\n\
el programa especificado. Verifique por favor que la\n\
orden es especificado apropiadamente yendo\n\
Artefacto->Aartefactos... y redacta la referencia de\n\
artefacto.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ne Peut Pas Courir Le Programme",
		    buf,
"Une erreur a t rencontre en tentant pour courir\n\
le programme spcifi. S'il vous plat vrifier que\n\
l'ordre est convenablement spcifi en allant\n\
Appareil->Appareils... et dite la rfrence d'appareil.",
#endif
		toplevel
	    );
	    g_free(buf);
	}
}

/*
 *      Runs the device tools command on the given device reference.
 */
void EDVRunDeviceTools(
	edv_core_struct *core_ptr, edv_device_struct *dev,
	GtkWidget *toplevel
)
{
	pid_t p;
	const gchar *cmd;


	if((core_ptr == NULL) || (dev == NULL))
	    return;

	cmd = dev->command_tools;
	if(cmd == NULL)
	    return;

	/* Seek past spaces in command string and make sure it is not
	 * empty.
	 */
	while(ISBLANK(*cmd))
	    cmd++;
	if(*cmd == '\0')
	    return;

	p = Exec(cmd);
	if(p == 0)
	{
	    gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute command:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar la orden:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'ordre:\n\
\n\
    %s\n",
#endif
		cmd
	    );
	    EDVPlaySoundError(core_ptr);
	    EDVMessageError(
#ifdef PROG_LANGUAGE_ENGLISH
"Cannot Run Program",
		    buf,
"An error was encountered while attempting to run the\n\
specified program. Please verify that the command\n\
is specified properly by going to Device->Devices...\n\
and edit the device reference.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"No Puede Correr El Programa",
		    buf,
"Un error se encontr al procurar para correr el\n\
el programa especificado. Verifique por favor que la\n\
orden es especificado apropiadamente yendo\n\
Artefacto->Aartefactos... y redacta la referencia de\n\
artefacto.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ne Peut Pas Courir Le Programme",
		    buf,
"Une erreur a t rencontre en tentant pour courir\n\
le programme spcifi. S'il vous plat vrifier que\n\
l'ordre est convenablement spcifi en allant\n\
Appareil->Appareils... et dite la rfrence d'appareil.",
#endif
		toplevel
	    );
	    g_free(buf);
	}
}

/*
 *      Runs the device format command on the given device reference.
 */
void EDVRunDeviceFormat(
	edv_core_struct *core_ptr, edv_device_struct *dev,
	GtkWidget *toplevel
)
{
	pid_t p;
	const gchar *cmd;

	if((core_ptr == NULL) || (dev == NULL))
	    return;

	cmd = dev->command_format;
	if(cmd == NULL)
	    return;

	/* Seek past spaces in command string and make sure it is not
	 * empty.
	 */
	while(ISBLANK(*cmd))
	    cmd++;
	if(*cmd == '\0')
	    return;

	p = Exec(cmd);
	if(p == 0)
	{
	    gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute command:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar la orden:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'ordre:\n\
\n\
    %s\n",
#endif
		cmd
	    );
	    EDVPlaySoundError(core_ptr);
	    EDVMessageError(
#ifdef PROG_LANGUAGE_ENGLISH
"Cannot Run Program",
		    buf,
"An error was encountered while attempting to run the\n\
specified program. Please verify that the command\n\
is specified properly by going to Device->Devices...\n\
and edit the device reference.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"No Puede Correr El Programa",
		    buf,
"Un error se encontr al procurar para correr el\n\
el programa especificado. Verifique por favor que la\n\
orden es especificado apropiadamente yendo\n\
Artefacto->Aartefactos... y redacta la referencia de\n\
artefacto.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ne Peut Pas Courir Le Programme",
		    buf,
"Une erreur a t rencontre en tentant pour courir\n\
le programme spcifi. S'il vous plat vrifier que\n\
l'ordre est convenablement spcifi en allant\n\
Appareil->Appareils... et dite la rfrence d'appareil.",
#endif
		toplevel
	    );
	    g_free(buf);
	}
}

/*
 *	Runs the terminal defined on the configuration list on the
 *	given core structure. If cmd is not NULL, then the command
 *	specified by cmd will be appended to the terminal run command
 *	defined on the core structure.
 */
void EDVRunTerminal(
	edv_core_struct *core_ptr, const gchar *cmd,
	GtkWidget *toplevel
)
{
	const cfg_item_struct *cfg_list;

	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;

	if(!STRISEMPTY(cmd))
	{
	    /* Get terminal command for running a program with a
	     * terminal, format the command, then run the command
	     */
	    const gchar *cmd_src = EDV_GET_S(
		EDV_CFG_PARM_PROG_TERMINAL_RUN
	    );
	    if(!STRISEMPTY(cmd_src))
	    {
		gchar *cmd_tar = g_strdup_printf(
		    "%s %s",
		    cmd_src,
		    cmd
		);
		Exec(cmd_tar);
		g_free(cmd_tar);
	    }
	}
	else
	{
	    /* No command given, just run terminal */
	    const gchar *cmd_src = EDV_GET_S(
		EDV_CFG_PARM_PROG_TERMINAL
	    );
	    if(!STRISEMPTY(cmd_src))
		Exec(cmd_src);
	}
}


/*
 *	Maps the Recycle Bin Desktop Icon, creating it as needed.
 */
void EDVMapRecBinDeskIcon(edv_core_struct *core_ptr)
{
	if(core_ptr == NULL)
	    return;

	if(core_ptr->recbin_deskicon == NULL)
	    core_ptr->recbin_deskicon = EDVRecBinDeskIconNew(core_ptr);
	EDVRecBinDeskIconMap(core_ptr->recbin_deskicon);
	EDVRecBinDeskIconUpdate(core_ptr->recbin_deskicon);
}

/*
 *	Creates the recycle bin window as needed and maps it.
 */
void EDVMapRecBin(edv_core_struct *core_ptr)
{
	GtkWidget *toplevel;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_recbin_struct *recbin;

	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;

	/* Get pointer to Recycle Bin window, create it as needed */
	recbin = core_ptr->recbin;
	if(recbin == NULL)
	    core_ptr->recbin = recbin = EDVRecBinNew(core_ptr);
	if(recbin == NULL)
	    return;

	toplevel = recbin->toplevel;

	/* Need to map it immediatly after it is created so that it does
	 * not get deleted in the timeout callback when the recycle bin
	 * is updated further below
	 */
	EDVRecBinMap(recbin);

	EDVRecBinSetBusy(recbin, TRUE);
	GUIBlockInput(toplevel, TRUE);

	clist = (GtkCList *)recbin->contents_clist;
	if(clist != NULL)
	{
	    gtk_clist_freeze(clist);
	    EDVRecBinContentsDoUpdate(
		recbin,
		EDV_GET_B(EDV_CFG_PARM_LISTS_ANIMATE_UPDATES)
	    );
	    gtk_clist_thaw(clist);
	}

	GUIBlockInput(toplevel, FALSE);
	EDVRecBinSetBusy(recbin, FALSE);

	EDVRecBinUpdateMenus(recbin);
}

/*
 *      Creates the history list window as needed and maps it.
 */
void EDVMapHistoryListWin(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	edv_history_listwin_struct *lw;


	if(core_ptr == NULL)
	    return;

	/* Get pointer to history list window, create it as needed */
	lw = core_ptr->history_listwin;
	if(lw == NULL)
	    core_ptr->history_listwin = lw = EDVHistoryListWinNew(
		core_ptr
	    );
	if(lw == NULL)
	    return;

	EDVCenterWindowToWindow(toplevel, lw->toplevel);

	EDVHistoryListWinSetBusy(lw, TRUE);

	EDVHistoryListWinMap(lw);

	EDVHistoryListWinUpdateList(lw);
	EDVHistoryListWinUpdateMenus(lw);

	EDVHistoryListWinSetBusy(lw, FALSE);
}


/*
 *	Maps the Options Window (creating it as needed) and switches
 *	to the page specified by page_name.
 */
void EDVMapOptionsWinPage(
	edv_core_struct *core_ptr,
	const gchar *page_name,
	GtkWidget *toplevel
)
{
	optwin_struct *optwin;

	if(core_ptr == NULL)
	    return;

	optwin = core_ptr->options_window;
	if(optwin == NULL)
	    core_ptr->options_window = optwin = EDVCreateOptionsWindow(
		core_ptr
	    );
	if(optwin == NULL)
	    return;

	EDVCenterWindowToWindow(
	    toplevel, optwin->toplevel
	);

	OptWinDoFetch(optwin);

	if(!STRISEMPTY(page_name))
	    EDVOptionsWindowSelectTab(optwin, page_name);

	OptWinMap(optwin);
	OptWinResetHasChanges(optwin, FALSE);
	OptWinUpdateMenus(optwin);
}

/*
 *	Creates the options window as needed and maps it.
 */
void EDVMapOptionsWin(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	EDVMapOptionsWinPage(core_ptr, NULL, toplevel);
}

/*
 *      Creates the options window as needed and maps it.
 */
void EDVMapCustomizeWin(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	gint	browser_num,
		imbr_num,
		archiver_num,
		recbin_num;
	const gchar *page_name;
	optwin_struct *optwin;
	if(core_ptr == NULL)
	    return;


	/* Get Customize Window, create it as needed */
	optwin = core_ptr->customize_window;
	if(optwin == NULL)
	    core_ptr->customize_window = optwin = EDVCustomizeWindowNew(
		core_ptr
	    );
	if(optwin == NULL)
	    return;

	/* Center Customize Window */
	EDVCenterWindowToWindow(toplevel, optwin->toplevel);

	/* Get values from configuration and set to Customize Window's
	 * widgets
	 */
	OptWinDoFetch(optwin);

	/* Match the given toplevel with the corresponding window */
	if(EDVMatchWindowFromToplevel(
	    core_ptr,
	    toplevel,
	    &browser_num,
	    &imbr_num,
	    &archiver_num,
	    &recbin_num
	))
	{
	    if(browser_num > -1)
		page_name = "File Browser";
	    else if(imbr_num > -1)
		page_name = "Image Browser";
	    else if(archiver_num > -1)
		page_name = "Archiver";
	    else if(recbin_num > -1)
		page_name = "Recycle Bin";
	    else
		page_name = NULL;
	}
	else
	{
	    page_name = NULL;
	}

	/* Switch page */
	EDVCustomizeWindowSelectTab(optwin, page_name);

	OptWinMap(optwin);
	OptWinResetHasChanges(optwin, FALSE);
	OptWinUpdateMenus(optwin);
}

/*
 *      Creates the devices list window as needed and maps it.
 */
void EDVMapDevicesListWin(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	edv_device_listwin_struct *lw;


	if(core_ptr == NULL)
	    return;

	/* Get pointer to devices list window, create it as needed */
	lw = core_ptr->device_listwin;
	if(lw == NULL)
	    core_ptr->device_listwin = lw = EDVDevicesListWinNew(
		core_ptr
	    );
	if(lw == NULL)
	    return;

	EDVCenterWindowToWindow(toplevel, lw->toplevel);
	EDVDevicesListSetBusy(lw, TRUE);
	EDVDevicesListWinMap(lw);
	EDVDevicesListWinFetchValues(lw);
	EDVDevicesListSetBusy(lw, FALSE);
}

/*
 *	Creates the MIME Types list window as needed and maps it.
 */
void EDVMapMIMETypesListWin(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	edv_mimetype_listwin_struct *lw;


	if(core_ptr == NULL)
	    return;

	/* Get pointer to MIME Types list window, create it as needed */
	lw = core_ptr->mimetype_listwin;
	if(lw == NULL)
	    core_ptr->mimetype_listwin = lw = EDVMimeTypesListWinNew(
		core_ptr
	    );
	if(lw == NULL)
	    return;

	EDVCenterWindowToWindow(toplevel, lw->toplevel);
	EDVMimeTypesListWinSetBusy(lw, TRUE);
	EDVMimeTypesListWinMap(lw);
	EDVMimeTypesListWinFetchValues(lw);
	EDVMimeTypesListWinSetBusy(lw, FALSE);

}

/*
 *	Maps the Run Dialog (creating it as needed) with the specified
 *	command and working directory.
 */
void EDVMapRunDialogCommand(
	edv_core_struct *core_ptr,
	const gchar *command,
	const gchar *working_dir,
	GtkWidget *toplevel
)
{
	edv_run_dlg_struct *dlg;


	if(core_ptr == NULL)
	    return;

	/* Create run dialog as needed */
	dlg = core_ptr->run_dlg;
	if(dlg == NULL)
	{
	    core_ptr->run_dlg = dlg = EDVRunDlgNew(core_ptr);
	    EDVRunDlgFetchValues(dlg);
	}
	if(dlg == NULL)
	    return;

	/* Set command as needed */
	if(!STRISEMPTY(command))
	    EDVRunDlgSetCommand(dlg, command);

	/* Set working directory as needed */
	if(!STRISEMPTY(working_dir))
	    EDVRunDlgSetWorkingDir(dlg, working_dir);

	EDVRunDlgMap(dlg);
}

void EDVMapRunDialog(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	EDVMapRunDialogCommand(core_ptr, NULL, NULL, NULL);
}


/*
 *	Maps the find window relative to the given browser.
 */
void EDVMapBrowserFindWin(
	edv_core_struct *core_ptr, edv_browser_struct *browser
)
{
	gint i, browser_num = -1;
	edv_findwin_struct *fw;


	if((core_ptr == NULL) || (browser == NULL))
	    return;

	/* Get pointer to find window and create the find window if it has
	 * not been created yet.
	 */
	fw = core_ptr->findwin;
	if(fw == NULL)
	    core_ptr->findwin = fw = EDVFindWinNew(core_ptr);
	if(fw == NULL)
	    return;

	/* Find browser index on core structure that matches the pointer
	 * of the given browser and record as browser_num.
	 */
	for(i = 0; i < core_ptr->total_browsers; i++)
	{
	    if(core_ptr->browser[i] == browser)
	    {
		browser_num = i;
		break;
	    }
	}

	/* Set reference window for find window, only one input may be
	 * non-negative.
	 */
	EDVFindWinSetReferenceWindow(
	    fw,
	    browser_num,
	    -1,
	    -1,
	    -1
	);

	/* Set initial location of find window to match the current
	 * location of the given browser.
	 */
	EDVFindWinSetLocation(
	    fw,
	    EDVBrowserCurrentLocation(browser),
	    FALSE
	);

	EDVCenterWindowToWindow(browser->toplevel, fw->toplevel);
	EDVFindWinUpdateMenus(fw);
	EDVFindWinMap(fw);
}

/*
 *      Maps the find window relative to the given image browser.
 */
void EDVMapImbrFindWin(
	edv_core_struct *core_ptr, edv_imbr_struct *imbr
)
{
	gint i, imbr_num = -1;
	edv_findwin_struct *fw;


	if((core_ptr == NULL) || (imbr == NULL))
	    return;

	/* Get pointer to find window and create the find window if it has
	 * not been created yet.
	 */
	fw = core_ptr->findwin;
	if(fw == NULL)
	    core_ptr->findwin = fw = EDVFindWinNew(core_ptr);
	if(fw == NULL)
	    return;

	/* Find image browser index on core structure that matches the
	 * pointer of the given image browser and record as imbr_num.
	 */
	for(i = 0; i < core_ptr->total_imbrs; i++)
	{
	    if(core_ptr->imbr[i] == imbr)
	    {
		imbr_num = i;
		break;
	    }
	}

	/* Set reference window for find window, only one input may be
	 * non-negative.
	 */
	EDVFindWinSetReferenceWindow(
	    fw,
	    -1,
	    imbr_num,
	    -1,
	    -1
	);

	/* Set initial location of find window to match the current
	 * location of the given image browser.
	 */
	EDVFindWinSetLocation(
	    fw,
	    EDVImbrCurrentLocation(imbr),
	    FALSE
	);

	EDVCenterWindowToWindow(imbr->toplevel, fw->toplevel);
	EDVFindWinUpdateMenus(fw);
	EDVFindWinMap(fw);
}

/*
 *	Maps the find window relative to the given archiver.
 */
void EDVMapArchiverFindWin(
	edv_core_struct *core_ptr, edv_archiver_struct *archiver
)
{
	gint i, archiver_num = -1;
	edv_findwin_struct *fw;


	if((core_ptr == NULL) || (archiver == NULL))
	    return;

	/* Get pointer to find window and create the find window if it has
	 * not been created yet.
	 */
	fw = core_ptr->findwin;
	if(fw == NULL)
	    core_ptr->findwin = fw = EDVFindWinNew(core_ptr);
	if(fw == NULL)
	    return;

	/* Find archiver index on core structure that matches the
	 * pointer of the given archiver and record as archiver_num.
	 */
	for(i = 0; i < core_ptr->total_archivers; i++)
	{
	    if(core_ptr->archiver[i] == archiver)
	    {
		archiver_num = i;
		break;
	    }
	}

	/* Set reference window for find window, only one input may be
	 * non-negative.
	 */
	EDVFindWinSetReferenceWindow(
	    fw,
	    -1,
	    -1,
	    -1,
	    archiver_num
	);

	/* Set initial location of find window to match the current
	 * location of the given archiver.
	 */
	EDVFindWinSetLocation(
	    fw,
	    EDVArchiverCurrentLocation(archiver),
	    FALSE
	);

	EDVCenterWindowToWindow(archiver->toplevel, fw->toplevel);
	EDVFindWinUpdateMenus(fw);
	EDVFindWinMap(fw);
}

/*
 *      Maps the find window relative to the given recycle bin.
 */
void EDVMapRecBinFindWin(
	edv_core_struct *core_ptr, edv_recbin_struct *recbin
)
{
	const gchar *recycled_index_file;
	gchar *recycled_dir;
	const cfg_item_struct *cfg_list;
	edv_findwin_struct *fw;

	if((core_ptr == NULL) || (recbin == NULL))
	    return;

	cfg_list = core_ptr->cfg_list;

	/* Get pointer to find window and create the find window if it has
	 * not been created yet
	 */
	fw = core_ptr->findwin;
	if(fw == NULL)
	    core_ptr->findwin = fw = EDVFindWinNew(core_ptr);
	if(fw == NULL)
	    return;

	/* Set reference window for find window, only one input may be
	 * non-negative
	 */
	EDVFindWinSetReferenceWindow(
	    fw,
	    -1,
	    -1,
	    0,		/* First and only recycle bin */
	    -1
	);


	/* Get the path to the Recycled Objects Index File */
	recycled_index_file = EDV_GET_S(
	    EDV_CFG_PARM_FILE_RECYCLED_INDEX
	);

	/* Get the path to the Recycled Objects Directory */
	recycled_dir = (recycled_index_file != NULL) ?
	    g_dirname(recycled_index_file) : NULL;

	/* Set initial location of find window to be the Recycled
	 * Objects Directory
	 */
	if(recycled_dir != NULL)
	    EDVFindWinSetLocation(fw, recycled_dir, FALSE);

	EDVCenterWindowToWindow(recbin->toplevel, fw->toplevel);
	EDVFindWinUpdateMenus(fw);
	EDVFindWinMap(fw);

	g_free(recycled_dir);
}


/*
 *	Maps the About dialog and switches to the page specified by
 *	page_name.
 */
void EDVAboutPage(
	edv_core_struct *core_ptr,
	const gchar *page_name,
	GtkWidget *toplevel
)
{
	GtkNotebook *notebook;
	about_dlg_struct *d;

	if(core_ptr == NULL)
	    return;

	/* Get the About Dialog (create it as needed) */
	d = core_ptr->about_dlg;
	if(d == NULL)
	    core_ptr->about_dlg = d = AboutDlgNew(core_ptr);
	if(d == NULL)
	    return;

	/* Select tab */
	notebook = GTK_NOTEBOOK(d->notebook);
	if(!STRISEMPTY(page_name) && (notebook != NULL))
	{
	    gint	page_num = -1;
/*			total_pages = g_list_length(notebook->first_tab); */

	    /* Check which tab is to be switched to by matching
	     * its name
	     */
	    if(!g_strcasecmp(page_name, "Credits"))
		page_num = 0;
	    else if(!g_strcasecmp(page_name, "Copyright"))
		page_num = 1;
	    else if(!g_strcasecmp(page_name, "Version"))
		page_num = 2;

	    /* Matched page to be switched to? */
	    if(page_num > -1)
		gtk_notebook_set_page(notebook, page_num);
	}

	/* Map dialog */
	AboutDlgMap(d);
}

void EDVAbout(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	EDVAboutPage(core_ptr, NULL, toplevel);
}


/*
 *	Updates the RC styles.
 */
void EDVUpdateRCStyles(edv_core_struct *core_ptr)
{
	const cfg_item_struct *cfg_list;

	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;

	/* Override GTK style? */
	if(CFGItemListGetValueI(
	    cfg_list, EDV_CFG_PARM_GTK_STYLE_OVERRIDE
	))
	{
	    GtkRcStyle *rcstyle = EDVCreateRCStyleFromCfg(
		CFGItemListGetValueStyle(
		    cfg_list, EDV_CFG_PARM_STYLE_STANDARD
		)
	    );
	    GTK_RC_STYLE_UNREF(core_ptr->standard_rcstyle)
	    core_ptr->standard_rcstyle = rcstyle;

	    rcstyle = EDVCreateRCStyleFromCfg(
		CFGItemListGetValueStyle(
		    cfg_list, EDV_CFG_PARM_STYLE_LISTS
		)
	    );
	    GTK_RC_STYLE_UNREF(core_ptr->lists_rcstyle)
	    core_ptr->lists_rcstyle = rcstyle;
	}
	else
	{
	    gboolean style_defined;
	    GtkStateType state;
	    GtkRcStyle *rcstyle;
	    const gchar	*font_name,
			*fg_color_name[5],
			*bg_color_name[5],
			*text_color_name[5],
			*base_color_name[5],
			*bg_pixmap_path[5];

	    font_name = core_ptr->font_name;

	    state = GTK_STATE_NORMAL;
	    fg_color_name[state] = core_ptr->fg_color_name;
	    bg_color_name[state] = core_ptr->bg_color_name;
	    text_color_name[state] = core_ptr->fg_color_name;
	    base_color_name[state] = core_ptr->bg_color_name;
	    bg_pixmap_path[state] = core_ptr->bg_pixmap_path;

	    state = GTK_STATE_ACTIVE;
	    fg_color_name[state] = core_ptr->fg_color_name;
	    bg_color_name[state] = core_ptr->bg_color_name;
	    text_color_name[state] = core_ptr->fg_color_name;
	    base_color_name[state] = core_ptr->bg_color_name;
	    bg_pixmap_path[state] = core_ptr->bg_pixmap_path;

	    state = GTK_STATE_PRELIGHT;
	    fg_color_name[state] = core_ptr->fg_color_name;
	    bg_color_name[state] = core_ptr->bg_color_name;
	    text_color_name[state] = core_ptr->fg_color_name;
	    base_color_name[state] = core_ptr->bg_color_name;
	    bg_pixmap_path[state] = core_ptr->bg_pixmap_path;

	    state = GTK_STATE_SELECTED;
	    fg_color_name[state] = core_ptr->sfg_color_name;
	    bg_color_name[state] = core_ptr->sbg_color_name;
	    text_color_name[state] = core_ptr->sfg_color_name;
	    base_color_name[state] = core_ptr->sbg_color_name;
	    bg_pixmap_path[state] = core_ptr->sbg_pixmap_path;

	    state = GTK_STATE_INSENSITIVE;
	    fg_color_name[state] = core_ptr->fg_color_name;
	    bg_color_name[state] = core_ptr->bg_color_name;
	    text_color_name[state] = core_ptr->fg_color_name;
	    base_color_name[state] = core_ptr->bg_color_name;
	    bg_pixmap_path[state] = core_ptr->bg_pixmap_path;

#define SET_RCSTYLE(_s_)	{			\
 gint i;						\
 const gchar *s;					\
							\
 style_defined = FALSE;					\
							\
 s = font_name;						\
 if(!STRISEMPTY(s)) {					\
  (_s_)->font_name = STRDUP(s);				\
  style_defined = TRUE;					\
 }							\
							\
 for(i = 0; i < 5; i++) {				\
  s = fg_color_name[i];					\
  if(!STRISEMPTY(s)) {					\
   (_s_)->color_flags[i] |= GTK_RC_FG;			\
   gdk_color_parse(s, &((_s_)->fg[i]));			\
   style_defined = TRUE;				\
  }							\
  s = bg_color_name[i];					\
  if(!STRISEMPTY(s)) {					\
   (_s_)->color_flags[i] |= GTK_RC_BG;			\
   gdk_color_parse(s, &((_s_)->bg[i]));			\
   style_defined = TRUE;				\
  }                                                     \
  s = text_color_name[i];				\
  if(!STRISEMPTY(s)) {					\
   (_s_)->color_flags[i] |= GTK_RC_TEXT;		\
   gdk_color_parse(s, &((_s_)->text[i]));		\
   style_defined = TRUE;				\
  }							\
  s = base_color_name[i];				\
  if(!STRISEMPTY(s)) {					\
   (_s_)->color_flags[i] |= GTK_RC_BASE;		\
   gdk_color_parse(s, &((_s_)->base[i]));		\
   style_defined = TRUE;				\
  }							\
							\
  s = bg_pixmap_path[i];				\
  if(!STRISEMPTY(s)) {					\
   (_s_)->bg_pixmap_name[i] = STRDUP(s);		\
   style_defined = TRUE;				\
  }							\
 }							\
}

	    /* Standard RC style */
	    rcstyle = gtk_rc_style_new();
	    SET_RCSTYLE(rcstyle)
	    GTK_RC_STYLE_UNREF(core_ptr->standard_rcstyle)
	    if(style_defined)
	    {
		core_ptr->standard_rcstyle = rcstyle;
	    }
	    else
	    {
		GTK_RC_STYLE_UNREF(rcstyle)
		core_ptr->standard_rcstyle = NULL;
	    }

	    /* Lists RC style */
	    rcstyle = gtk_rc_style_new();
	    SET_RCSTYLE(rcstyle)
	    GTK_RC_STYLE_UNREF(core_ptr->lists_rcstyle)
	    if(style_defined)
	    {
		core_ptr->lists_rcstyle = rcstyle;
	    }
	    else
	    {
		GTK_RC_STYLE_UNREF(rcstyle)
		core_ptr->lists_rcstyle = NULL;  
	    }

#undef SET_RCSTYLE
	}
}


/*
 *	Updates the Users & Groups Popup Lists.
 *
 *	If the Popup Lists do not exist then they will be created.
 */
void EDVUpdateIDPULists(edv_core_struct *core_ptr)
{
	gint i, n;
	GtkRcStyle *rcstyle;
	GtkWidget *w;
	const edv_uid_struct *uid;
	const edv_gid_struct *gid;
	pulist_struct *pulist;

	if(core_ptr == NULL)
	    return;

	/* Get the Users Popup List, create it as needed */
	pulist = core_ptr->users_pulist;
	if(pulist == NULL)
	    core_ptr->users_pulist = pulist = PUListNew();
	if(pulist == NULL)
	    return;

	/* Update style */
	rcstyle = core_ptr->lists_rcstyle;
	w = PUListGetCList(pulist);
	if((w != NULL) && (rcstyle != NULL))
	    gtk_widget_modify_style(w, rcstyle);

	/* Delete all existing items from the Popup List */
	PUListClear(pulist);

	/* Add Users to the Popup List */
	for(i = 0; i < core_ptr->total_uids; i++)
	{
	    uid = core_ptr->uid[i];
	    if((uid != NULL) ? STRISEMPTY(uid->name) : TRUE)
		continue;

	    n = PUListAddItem(pulist, uid->name);
	    if(n < 0)
		continue;

	    PUListSetItemData(pulist, n, (gpointer)i);
	}


	/* Get the Groups Popup List, create it as needed */
	pulist = core_ptr->groups_pulist;
	if(pulist == NULL)
	    core_ptr->groups_pulist = pulist = PUListNew();
	if(pulist == NULL)
	    return;

	/* Update style */
	rcstyle = core_ptr->lists_rcstyle;
	w = PUListGetCList(pulist);
	if((w != NULL) && (rcstyle != NULL))
	    gtk_widget_modify_style(w, rcstyle);

	/* Delete all existing items from the Popup List */
	PUListClear(pulist);

	/* Add Groups to the Popup List */
	for(i = 0; i < core_ptr->total_gids; i++)
	{                                        
	    gid = core_ptr->gid[i];
	    if((gid != NULL) ? STRISEMPTY(gid->name) : TRUE)
		continue;   

	    n = PUListAddItem(pulist, gid->name);
	    if(n < 0)
		continue;

	    PUListSetItemData(pulist, n, (gpointer)i);
	}
}

/*
 *	Updates the Devices Popup List.
 *
 *	If the Popup List does not exist then it will be created.
 */
void EDVUpdateDevicesPUList(edv_core_struct *core_ptr)
{
	gint i, n;
	const gchar *dev_name;
	GtkRcStyle *rcstyle;
	GtkWidget *w;
	edv_device_icon_state state = EDV_DEVICE_ICON_STATE_STANDARD;
	edv_device_struct *dev;
	pulist_struct *pulist;

	if(core_ptr == NULL)
	    return;

	/* Get the Devices Popup List, create it as needed */
	pulist = core_ptr->devices_pulist;
	if(pulist == NULL)
	    core_ptr->devices_pulist = pulist = PUListNew();
	if(pulist == NULL)
	    return;

	/* Update style */
	rcstyle = core_ptr->lists_rcstyle;
	w = PUListGetCList(pulist);
	if((w != NULL) && (rcstyle != NULL))
	    gtk_widget_modify_style(w, rcstyle);

	/* Delete all existing items from the Popup List */
	PUListClear(pulist);

	/* Add Devices to the Popup List */
	for(i = 0; i < core_ptr->total_devices; i++)
	{
	    dev = core_ptr->device[i];
	    if(dev == NULL)
		continue;

	    /* Skip devices who are marked as unlisted, in which case 
	     * they should not be listed on the devices popup list
	     */
	    if(EDV_DEVICE_IS_UNLISTED(dev))
		continue;

	    /* Realize device, loading icons as needed */
	    EDVDeviceRealize(dev, FALSE);

	    /* Get device name */
	    dev_name = dev->name;
	    if((dev_name == NULL) && (dev->device_path != NULL))
		dev_name = EDVGetPathName(dev->device_path);
	    if(dev_name == NULL)
		dev_name = "(null)";

	    /* Add new item to devices popup list */
	    n = PUListAddItemPixText(
		pulist, dev_name,
		dev->small_pixmap[state], dev->small_mask[state]
	    );
	    if(n < 0)
		continue;

	    PUListSetItemData(pulist, n, (gpointer)i);
	}
}

/*
 *	Updates the MIME Type class hint indices.
 */
void EDVUpdateMIMETypeHintIndices(edv_core_struct *core_ptr)
{
	gint i;
	edv_mimetype_struct *mt_ptr;

	if(core_ptr == NULL)
	    return;

	/* Reset all indexes for all MIME Type classes */
	core_ptr->mimetype_system_index_first = -1;
	core_ptr->mimetype_system_index_last = -1;
	core_ptr->mimetype_format_index_first = -1;
	core_ptr->mimetype_format_index_last = -1;
	core_ptr->mimetype_program_index_first = -1;
	core_ptr->mimetype_program_index_last = -1;
	core_ptr->mimetype_unique_index_first = -1;
	core_ptr->mimetype_unique_index_last = -1;

	/* Iterate through all MIME Types and update each class hint
	 * index
	 */
	for(i = 0; i < core_ptr->total_mimetypes; i++)
	{
	    mt_ptr = core_ptr->mimetype[i];
	    if(mt_ptr == NULL)
		continue;

	    switch(mt_ptr->mt_class)
	    {
	      case EDV_MIMETYPE_CLASS_SYSTEM:
		if(core_ptr->mimetype_system_index_first < 0)
		    core_ptr->mimetype_system_index_first = i;
		core_ptr->mimetype_system_index_last = i;
		break;

	      case EDV_MIMETYPE_CLASS_FORMAT:
		if(core_ptr->mimetype_format_index_first < 0)
		    core_ptr->mimetype_format_index_first = i;
		core_ptr->mimetype_format_index_last = i;
		break;

	      case EDV_MIMETYPE_CLASS_PROGRAM:
		if(core_ptr->mimetype_program_index_first < 0)
		    core_ptr->mimetype_program_index_first = i;
		core_ptr->mimetype_program_index_last = i;
		break;

	      case EDV_MIMETYPE_CLASS_UNIQUE:
		if(core_ptr->mimetype_unique_index_first < 0)
		    core_ptr->mimetype_unique_index_first = i;
		core_ptr->mimetype_unique_index_last = i;
		break;
	    }
	}
}

/*
 *	Updates the Open With Popup List.
 *
 *	If the Popup List does not exist then it will be created.
 */
void EDVUpdateOpenWithPUList(edv_core_struct *core_ptr)
{
	gint i, n;
	GtkRcStyle *rcstyle;
	GtkWidget *w;
	edv_mimetype_icon_state state = EDV_MIMETYPE_ICON_STATE_STANDARD;
	edv_mimetype_struct *mt_ptr;
	pulist_struct *pulist;

	if(core_ptr == NULL)
	    return;

	/* Get the Open With Popup List, create it as needed */
	pulist = core_ptr->openwith_pulist;
	if(pulist == NULL)
	    core_ptr->openwith_pulist = pulist = PUListNew();
	if(pulist == NULL)
	    return;

	/* Update style */
	rcstyle = core_ptr->lists_rcstyle;
	w = PUListGetCList(pulist);
	if((w != NULL) && (rcstyle != NULL))
	    gtk_widget_modify_style(w, rcstyle);

	/* Delete all existing items from the Popup List */
	PUListClear(pulist);

	/* Add MIME Types of class application to the Popup List */
	for(i = 0; i < core_ptr->total_mimetypes; i++)
	{
	    mt_ptr = core_ptr->mimetype[i];
	    if(mt_ptr == NULL)
		continue;

	    if(STRISEMPTY(mt_ptr->type))
		continue;

	    /* Skip MIME Type if it is not of class application */
	    if(mt_ptr->mt_class != EDV_MIMETYPE_CLASS_PROGRAM)
		continue;

	    /* Realize MIME Type, loading icons as needed */
	    EDVMimeTypeRealize(mt_ptr, FALSE);

	    /* Add new item to the Popup List */
	    n = PUListAddItemPixText(
		pulist,
		(mt_ptr->description != NULL) ?
		    mt_ptr->description : mt_ptr->type,
		mt_ptr->small_pixmap[state], mt_ptr->small_mask[state]
	    );
	    if(n < 0)
		continue;

	    PUListSetItemData(pulist, n, (gpointer)i);
	}
}


/*
 *	Refreshes all of Endeavour's resources.
 */
void EDVRefresh(edv_core_struct *core_ptr)
{
	gint i;
	edv_browser_struct *browser;
	edv_imbr_struct *imbr;
	edv_archiver_struct *archiver;
	edv_recbin_struct *recbin;
	edv_device_listwin_struct *device_list;
	edv_mimetype_listwin_struct *mimetype_list;
	optwin_struct *optwin;

	if(core_ptr == NULL)
	    return;

	/* Refresh device mount states and stats */
	EDVDevicesListUpdateMountStates(
	    core_ptr->device, core_ptr->total_devices
	);
	EDVDevicesListUpdateStats(
	    core_ptr->device, core_ptr->total_devices
	);

	/* Refresh File Browsers */
	for(i = 0; i < core_ptr->total_browsers; i++)
	{                                            
	    browser = core_ptr->browser[i];
	    if(browser == NULL)
		continue;

	    EDVBrowserOPRefresh(browser);
	}

	/* Refresh Image Browsers */
	for(i = 0; i < core_ptr->total_imbrs; i++)
	{                                         
	    imbr = core_ptr->imbr[i];
	    if(imbr == NULL)
		continue;

	    EDVImbrOPRefresh(imbr);
	}

	/* Refresh Archivers */
	for(i = 0; i < core_ptr->total_archivers; i++)
	{
	    archiver = core_ptr->archiver[i];
	    if(archiver == NULL)  
		continue;

	    EDVArchiverOPRefresh(archiver);
	}

	/* Refresh Recycle Bin */
	recbin = core_ptr->recbin;
	if(recbin != NULL)
	{
	    EDVRecBinOPRefresh(recbin);
	}

	/* Devices List */
	device_list = core_ptr->device_listwin;
	if(EDVDevicesListWinIsMapped(device_list))
	{
	    EDVDevicesListWinFetchValues(device_list);
	}

	/* MIME Types List */
	mimetype_list = core_ptr->mimetype_listwin;
	if(EDVMimeTypesListWinIsMapped(mimetype_list))
	{
	    EDVMimeTypesListWinFetchValues(mimetype_list);
	}

	/* Options Window */
	optwin = core_ptr->options_window;
	if(OptWinIsMapped(optwin))
	{
	    OptWinDoFetch(optwin);
	}

	/* Customize Window */
	optwin = core_ptr->customize_window;
	if(OptWinIsMapped(optwin))
	{
	    OptWinDoFetch(optwin);
	}   


}

/*
 *	Reloads all of Endeavour's configurations from file and
 *	refreshes resources.
 *
 *	Any unsaved configuration in memory will be discarded.
 */
void EDVReset(edv_core_struct *core_ptr)
{
	gint i;
	const gchar *s;
	gchar hostname[64 + 1];
	cfg_item_struct *cfg_list;

	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;

	/* Get the Host's Name */
	if(gethostname((char *)hostname, sizeof(hostname)))
	    *hostname = '\0';
	else
	    hostname[sizeof(hostname) - 1] = '\0';

	/* Get User & Group IDs */
	for(i = 0; i < core_ptr->total_uids; i++)
	    EDVUIDDelete(core_ptr->uid[i]);
	g_free(core_ptr->uid);
	core_ptr->uid = EDVUIDGetSystem(&core_ptr->total_uids);
	for(i = 0; i < core_ptr->total_gids; i++)
	    EDVGIDDelete(core_ptr->gid[i]);
	g_free(core_ptr->gid);
	core_ptr->gid = EDVGIDGetSystem(&core_ptr->total_gids);

	/* Get Effective User ID string */
	s = EDVUIDGetNameFromUID(
	    core_ptr->uid, core_ptr->total_uids,
	    core_ptr->effective_user_id, NULL
	);
	g_free(core_ptr->effective_user_id_str);
	core_ptr->effective_user_id_str = STRDUP(
	    !STRISEMPTY(s) ? s : "anonymous"
	);

	/* Get Effective User ID & Hostname string */
	g_free(core_ptr->effective_user_id_host_str);
	core_ptr->effective_user_id_host_str = g_strdup_printf(
	    "%s@%s",
	    !STRISEMPTY(s) ? s : "anonymous",
	    !STRISEMPTY(hostname) ? hostname : "unknown"
	);

	/* Reload configuration */
	if(core_ptr->cfg_file != NULL)
	    CFGLoadFromFile(core_ptr->cfg_file, cfg_list);

	/* Reload Devices */
	for(i = 0; i < core_ptr->total_devices; i++)
	    EDVDeviceDelete(core_ptr->device[i]);
	g_free(core_ptr->device);
	core_ptr->device = NULL;
	core_ptr->total_devices = 0;
	EDVDevicesListLoadFromSystem(
	    &core_ptr->device, &core_ptr->total_devices,
	    NULL, core_ptr
	);
	EDVDevicesListLoadFromFile(
	    EDV_GET_S(EDV_CFG_PARM_FILE_DEVICES),
	    &core_ptr->device, &core_ptr->total_devices,
	    NULL, core_ptr
	);

	/* Reload MIME Types */
	for(i = 0; i < core_ptr->total_mimetypes; i++)
	    EDVMimeTypeDelete(core_ptr->mimetype[i]);
	g_free(core_ptr->mimetype);
	core_ptr->mimetype = NULL;
	core_ptr->total_mimetypes = 0;
	EDVDMimeTypesListLoadFromSystem(
	    &core_ptr->mimetype, &core_ptr->total_mimetypes,
	    -1,			/* Append */
	    NULL, core_ptr,
	    NULL, core_ptr,
	    TRUE		/* Read only */
	);
	EDVMimeTypeListLoadFromFile(
	    EDV_GET_S(EDV_CFG_PARM_FILE_MIME_TYPES),
	    &core_ptr->mimetype, &core_ptr->total_mimetypes,
	    -1,			/* Append */     
	    FALSE, FALSE,
	    NULL, core_ptr,
	    NULL, core_ptr,
	    FALSE		/* Not read only */
	);
	EDVMimeTypeListLoadFromFile(
	    EDV_GET_S(EDV_CFG_PARM_FILE_MIME_TYPES_GLOBAL),
	    &core_ptr->mimetype, &core_ptr->total_mimetypes,
	    -1,			/* Append */     
	    FALSE, FALSE,
	    NULL, core_ptr,
	    NULL, core_ptr,
	    TRUE		/* Read only */
	);
	/* Update MIME Types class list index hints */
	EDVUpdateMIMETypeHintIndices(core_ptr);

	/* Notify all resources about the change in the configuration */
	EDVReconfiguredEmit(core_ptr);
	EDVWriteProtectChangedEmit(
	    core_ptr, EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT)
	);
	EDVRefresh(core_ptr);
}


/*
 *	Syncs disks and saves all of Endeavour's configuration in
 *	memory to file.
 */
void EDVSyncDisks(edv_core_struct *core_ptr)
{
	gulong time_start = (gulong)time(NULL);
	gboolean object_existed;
	const gchar *s;
	struct stat lstat_buf;
	cfg_item_struct *cfg_list;

	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;

	/* Begin saving all of Endeavour's configuration in memory to
	 * file
	 */

	/* MIME Types */
	s = CFGItemListGetValueS(
	    cfg_list, EDV_CFG_PARM_FILE_MIME_TYPES
	);
	if(!STRISEMPTY(s))
	{
	    gchar *path = STRDUP(s);
	    object_existed = access(path, F_OK) ? FALSE : TRUE;
	    EDVMimeTypeListSaveToFile(
		path, core_ptr->mimetype, core_ptr->total_mimetypes,
		FALSE,		/* Do not write MIME Types marked read_only */
		NULL, core_ptr
	    );
	    if(!lstat(path, &lstat_buf))
	    {
		if(object_existed)
		    EDVObjectModifiedEmit(core_ptr, path, NULL, &lstat_buf);
		else
		    EDVObjectAddedEmit(core_ptr, path, &lstat_buf);
	    }
	    g_free(path);
	}

	/* Devices */
	s = CFGItemListGetValueS(
	    cfg_list, EDV_CFG_PARM_FILE_DEVICES
	);
	if(!STRISEMPTY(s))
	{
	    gchar *path = STRDUP(s);
	    object_existed = access(path, F_OK) ? FALSE : TRUE;
	    EDVDeviceListSaveToFile(
		path, core_ptr->device, core_ptr->total_devices,
		NULL, core_ptr
	    );
	    if(!lstat(path, &lstat_buf))
	    {
		if(object_existed)
		    EDVObjectModifiedEmit(core_ptr, path, NULL, &lstat_buf);
		else
		    EDVObjectAddedEmit(core_ptr, path, &lstat_buf);
	    }
	    g_free(path);
	}

	/* Configuration */
	if(core_ptr->cfg_file != NULL)
	{
	    gchar *path = STRDUP(core_ptr->cfg_file);
	    object_existed = access(path, F_OK) ? FALSE : TRUE;
	    CFGSaveToFile(path, cfg_list);
	    if(!lstat(path, &lstat_buf))
	    {
		if(object_existed)
		    EDVObjectModifiedEmit(core_ptr, path, NULL, &lstat_buf);
		else
		    EDVObjectAddedEmit(core_ptr, path, &lstat_buf);
	    }
	    g_free(path);
	}

	/* Flush all disk operations */
	sync();

	/* Record history */
	EDVAppendHistory(
	    core_ptr,
	    EDV_HISTORY_SYNC_DISKS,
	    time_start, (gulong)time(NULL),
	    0,
	    NULL,		/* Source */
	    NULL,		/* Target */
	    NULL		/* Comment */
	);
}


/*
 *	Clears all history events.
 *
 *	Clears the History List Window and deletes the history events
 *	log file.
 */
void EDVClearHistoryEvents(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	if(core_ptr == NULL)
	    return;

	/* Map History List Window */
	EDVMapHistoryListWin(core_ptr, toplevel);

	/* Clear history events by using the History List Window
	 * to delet the history events log file, the History List
	 * Window will also be cleared
	 */
	EDVHistoryListWinClear(core_ptr->history_listwin);
}

/*
 *	Clears the locations history on all windows.
 */
void EDVClearLocationsHistory(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	gint i, response;
	const gchar *path;
	edv_browser_struct *browser;
	edv_imbr_struct *imbr;
	edv_archiver_struct *archiver;
	const cfg_item_struct *cfg_list;


	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;

	/* Confirm */
	EDVPlaySoundQuestion(core_ptr);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Claro",
"Usted est seguro que usted quiere borrar la historia de\n\
ubicaciones en todas las ventanas?\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Clair",
"Etes-vous sr que vous voulez effacer l'histoire d'emplacements\n\
sur toutes les fentres?\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie Klar",
"Sind sie sicher sie die orte geschichte auf allen fenstern\n\
wollen lschen?\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Chiaro",
"Lei sono sicuro che lei vuole cancellare la storia di posizioni\n\
su tutte le finestre?\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Helder",
"ent u zeker u de plaatzen geschiedenis op alle ramen wil\n\
schrappen?\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Claro",
"Esto seguro quer anular a histria de localidades em todas\n\
as janelas?\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Klar",
"Er de sikker de stryker plasserings historien p alle vinduer?\n",
#else
"Confirm Clear",
"Are you sure you want to delete the locations history on\n\
all the windows?\n",
#endif
	    NULL,
	    CDIALOG_ICON_QUESTION,
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
	    CDIALOG_BTNFLAG_YES
	);
	CDialogSetTransientFor(NULL);
	if(response != CDIALOG_RESPONSE_YES)
	    return;


	/* Clear File Browser Locations List */
	for(i = 0; i < core_ptr->total_browsers; i++)
	{
	    browser = core_ptr->browser[i];
	    if(browser == NULL)
		continue;

	    GUIComboSetList(browser->location_combo, NULL);
	}

	/* Delete File Browser Locations History File */
	path = CFGItemListGetValueS(
	    cfg_list, EDV_CFG_PARM_FILE_BROWSER_LOCATION_HISTORY
	);
	if(!STRISEMPTY(path))
	{
	    if(!unlink((const char *)path))
		EDVObjectRemovedEmit(core_ptr, path);
	}


	/* Clear Image Browser Locations List */
	for(i = 0; i < core_ptr->total_imbrs; i++)
	{
	    imbr = core_ptr->imbr[i];
	    if(imbr == NULL)
		continue;

	    GUIComboSetList(imbr->location_combo, NULL);
	}

	/* Delete Image Browser Locations History File */
	path = CFGItemListGetValueS(
	    cfg_list, EDV_CFG_PARM_FILE_IMBR_LOCATION_HISTORY
	);
	if(!STRISEMPTY(path))
	{
	    if(!unlink((const char *)path))
		EDVObjectRemovedEmit(core_ptr, path);
	}


	/* Clear Archiver Locations List */
	for(i = 0; i < core_ptr->total_archivers; i++)
	{
	    archiver = core_ptr->archiver[i];
	    if(archiver == NULL)
		continue;

	    GUIComboSetList(archiver->location_combo, NULL);
	}

	/* Delete Image Browser Locations History File */
	path = CFGItemListGetValueS(
	    cfg_list, EDV_CFG_PARM_FILE_ARCHIVER_LOCATION_HISTORY
	);
	if(!STRISEMPTY(path))
	{
	    if(!unlink((const char *)path))
		EDVObjectRemovedEmit(core_ptr, path);
	}

	EDVPlaySoundCompleted(core_ptr);
}

/*
 *	Clear the Run Dialog history and deletes the run history log
 *	file.
 */
void EDVClearRunHistory(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	gint response;
	const gchar *path;
	edv_run_dlg_struct *d;
	const cfg_item_struct *cfg_list;


	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;

	/* Confirm */
	EDVPlaySoundQuestion(core_ptr);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Claro",
"Usted est seguro que usted quiere borrar el corre la historia?\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Clair",
"Etes-vous sr que vous voulez effacer l'histoire de course?\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie Klar",
"Sind Sie sicher Sie die Lauf Geschichte wollen lschen?\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Chiaro",
"Lei sono sicuro che lei vuole cancellare il ha corso la storia?\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Helder",
"Bent u zeker u de tocht geschiedenis wil schrappen?\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Claro",
"Esto seguro quer anular a histria de corrida?\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Klar",
"Er De sikker De stryker lphistorien?\n",
#else
"Confirm Clear",
"Are you sure you want to delete the run history?\n",
#endif
	    NULL,
	    CDIALOG_ICON_QUESTION,
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
	    CDIALOG_BTNFLAG_YES
	);
	CDialogSetTransientFor(NULL);
	if(response != CDIALOG_RESPONSE_YES)
	    return;


	/* Clear Run Dialog Run History */
	d = core_ptr->run_dlg;
	if(d != NULL)
	{
	    GUIComboSetList(d->run_combo, NULL);
	}

	/* Delete Run Dialog Run History File */
	path = CFGItemListGetValueS(
	    cfg_list, EDV_CFG_PARM_FILE_RUNDLG_HISTORY
	);
	if(!STRISEMPTY(path))
	{
	    if(!unlink((const char *)path))
		EDVObjectRemovedEmit(core_ptr, path);
	}

	EDVPlaySoundCompleted(core_ptr);
}


/*
 *	Calls the Endeavour Download Front End program to download the
 *	given URL described in dnd_obj to the directory target_path.
 *
 *	If the dnd_obj is not given then the download program will still
 *	be runned but used to query where the user wants to download
 *	to.
 */
void EDVInternetDownloadObject(
	edv_core_struct *core_ptr,
	const url_struct *url,			/* Source */
	const gchar *target_path,		/* Target */
	GtkWidget *toplevel
)
{
	gboolean confirm;
	const gchar *prog;
	const cfg_item_struct *cfg_list;

	if((core_ptr == NULL) || STRISEMPTY(target_path))
	    return;

	cfg_list = core_ptr->cfg_list;

	/* Get configuration */
	confirm = CFGItemListGetValueI(
	    cfg_list, EDV_CFG_PARM_CONFIRM_DOWNLOAD
	);
	prog = CFGItemListGetValueS(
	    cfg_list, EDV_CFG_PARM_PROG_NET_DOWNLOAD
	);

	/* Download Front End program not set? */
	if(STRISEMPTY(prog))
	{
	    EDVPlaySoundError(core_ptr);
	    EDVMessageError(
		"No Download Program",
"Download Front End program not set",
"To set the network download program, please go\n\
to Settings->Options->Programs.",
		toplevel
	    );
	}
	else
	{
	    gint p;
	    gchar *url_str, *cmd;
	    gulong time_start = time(NULL);

	    /* Format url */
	    if(url == NULL)
		url_str = STRDUP("");
	    else if(url->port > 0)
		url_str = g_strdup_printf(
		    "%s://%s%s%s%s%s:%i%s",
		    (url->protocol != NULL) ? url->protocol : "file",
		    (url->user != NULL) ? url->user : "",
		    (url->password != NULL) ? ":" : "",
		    (url->password != NULL) ? url->password : "",
		    (url->user != NULL) ? "@" : "",
		    (url->host != NULL) ? url->host : "localhost",
		    url->port,
		    url->path
		);
	    else
		url_str = g_strdup_printf(
		    "%s://%s%s%s%s%s%s",
		    (url->protocol != NULL) ? url->protocol : "file",
		    (url->user != NULL) ? url->user : "",
		    (url->password != NULL) ? ":" : "",
		    (url->password != NULL) ? url->password : "",
		    (url->user != NULL) ? "@" : "",
		    (url->host != NULL) ? url->host : "localhost",
		    url->path
		);

	    /* Format command, the program should accept arguments being
	     * the first argument(s) are the URLs to download and the
	     * last argument is the destination path
	     */
	    cmd = g_strdup_printf(
		"%s %s %s \"%s\" \"%s\"",
		prog,
		"-b",
		confirm ? "-c" : "",
		url_str, target_path
	    );

	    /* Execute download command */
	    p = Exec(cmd);
	    if(p <= 0)
	    {
		gchar *buf = g_strdup_printf(
"Unable to execute command:\n\n    %s",
		    cmd
		);
		EDVPlaySoundError(core_ptr);
		EDVMessageError(
		    "Download Failed",
		    buf,
"Make sure that the network download program is\n\
defined correctly, please go to\n\
Settings->Options->Programs.",
		    toplevel
		);
		g_free(buf);
	    }

	    /* Record history */
	    EDVAppendHistory(
		core_ptr,
		EDV_HISTORY_DISK_OBJECT_DOWNLOAD,
		time_start,
		time(NULL),
		(p <= 0) ? -1 : 0,
		url_str,		/* Source */
		target_path,		/* Target */
		NULL			/* Comment */
	    );

	    g_free(cmd);
	    g_free(url_str);
	}
}


/*
 *	Tabulates the total size of all objects in the recycled objects
 *	directory and checks if it has exceeded the configuration
 *	specified recycle bin size warn size. If it has exceeded then
 *	the user will be warned.
 */
void EDVRecycleBinSizeCheck(
	edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	const gchar *recycled_index_file;
	const cfg_item_struct *cfg_list;
	edv_recbin_index_struct *rbi_ptr;
	edv_recbin_object_struct *obj;
	gulong recbin_size_warn = 0, total_size = 0;


	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;

	/* Get path to recycled objects index file */
	recycled_index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLED_INDEX);
	if(recycled_index_file == NULL)
	    return;

	/* Get recycle bin size warn */
	recbin_size_warn = EDV_GET_UL(EDV_CFG_PARM_RECBIN_SIZE_WARN);
	/* If the warn size is 0, then this implies no warning should
	 * be given (in which case do not bother to check anything)
	 */
	if(recbin_size_warn == 0)
	    return;

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	rbi_ptr = EDVRecBinIndexOpen(recycled_index_file);
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    /* Get recycled object obtained from the recycled objects
	     * directory index and do not modify it
	     */
	    obj = rbi_ptr->obj;
	    if(obj == NULL)
		continue;

	    /* Increase total size of recycled objects by this recycled
	     * object
	     */
	    total_size += obj->size;
	}
	EDVRecBinIndexClose(rbi_ptr);
	rbi_ptr = NULL;


	/* At this point the total size of all recycled objects has been
	 * obtained. Now check if total_size reaches or exceeds the
	 * configuration set size to warn
	 */
	if(total_size >= recbin_size_warn)
	{
	    gchar	*total_size_s = STRDUP(
		EDVGetObjectSizeStr(core_ptr, total_size)
			),
			*recbin_size_warn_s = STRDUP(
		EDVGetObjectSizeStr(core_ptr, recbin_size_warn)
			),
			*buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"The size of the recycle bin is now %s bytes,\n\
it has exceeded the user specified size of %s bytes\n\
at which this warning is to be shown.\n\
\n\
To purge all the existing recycled objects, first\n\
go to the Recycle Bin and then go to File->Purge All.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El tamao del cajn de la recirculacin es ahora %s bytes,\n\
ha excedido al usuario el tamao especificado de\n\
%s bytes en que este advertir deber ser mostrado.\n\
\n\
Al purge todos los objetos existentes de recycled, van\n\
primero al Cajn de la Recirculacin y entonces van a\n\
File->Purge All.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"La taille du recycle l'huche est maintenant %s bytes,\n\
il a dpass l'utilisateur la taille spcifie de\n\
%s bytes  qui cet avertissement devrait tre montr.\n\
\n\
Purger tout l'existant recycl objets, premirement va\n\
au Recycle Huche et alors va File->Purge All.",
#endif
		total_size_s, recbin_size_warn_s
	    );

	    EDVPlaySoundWarning(core_ptr);
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponseIconData(
#ifdef PROG_LANGUAGE_ENGLISH
		"Recycle Bin Size Warning",
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Advertir De Cajn De Recirculacin",
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Recycler l'Avertissement d'Huche",
#endif
		buf, NULL,
		(guint8 **)icon_trash_32x32_xpm,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    g_free(buf);
	    g_free(total_size_s);
	    g_free(recbin_size_warn_s);
	}
}
