# -*- coding: utf-8 -*-

# Copyright (c) 2005 - 2006 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to enter filetype associations for the project.
"""

import sys
import os

from qt import *
from FiletypeAssociationForm import FiletypeAssociationForm

class FiletypeAssociationDialog(FiletypeAssociationForm):
    """
    Class implementing a dialog to enter filetype associations for the project.
    """
    def __init__(self, project, parent = None):
        """
        Constructor
        
        @param project reference to the project object
        @param parent reference to the parent widget (QWidget)
        """
        FiletypeAssociationForm.__init__(self, parent, None, 1)

        # keep these lists in sync
        self.filetypes = ["SOURCES", "FORMS", "INTERFACES", "OTHERS"]
        self.filetypeStrings = [self.trUtf8("Sources"), self.trUtf8("Forms"),
                                self.trUtf8("Interfaces"), self.trUtf8("Others")]
        self.filetypesList = QStringList()
        for fts in self.filetypeStrings:
            self.filetypesList.append(fts)
        self.filetypeCombo.insertStringList(self.filetypesList)
        
        self.project = project
        for pattern, filetype in self.project.pdata["FILETYPES"].items():
            try:
                index = self.filetypes.index(filetype)
                itm = QListViewItem(self.filetypeAssociationList, pattern, 
                    self.filetypeStrings[index])
            except ValueError:
                pass    # silently discard entries of unknown type
        
    def handleAssociationSelected(self, itm):
        """
        Private slot to handle the clicked signal of the association list.
        
        @param itm reference to the selecte item (QListViewItem)
        """
        if itm is None:
            self.filePatternEdit.clear()
            self.filetypeCombo.setCurrentItem(0)
            self.deleteAssociationButton.setEnabled(0)
        else:
            self.filePatternEdit.setText(itm.text(0))
            self.filetypeCombo.setCurrentText(itm.text(1))
            self.deleteAssociationButton.setEnabled(1)

    def handleAssociationAdd(self):
        """
        Private slot to add the association displayed to the listbox.
        """
        pattern = self.filePatternEdit.text()
        filetype = self.filetypeCombo.currentText()
        if not pattern.isEmpty():
            itm = self.filetypeAssociationList.findItem(pattern, 0)
            if itm:
                self.filetypeAssociationList.takeItem(itm)
                del itm
            itm = QListViewItem(self.filetypeAssociationList, pattern, filetype)
            self.filePatternEdit.clear()
            self.filetypeCombo.setCurrentItem(0)
            self.filetypeAssociationList.setSelected(itm, 1)

    def handleAssociationDelete(self):
        """
        Private slot to delete the currently selected association of the listbox.
        """
        itm = self.filetypeAssociationList.selectedItem()
        if itm is not None:
            self.filetypeAssociationList.takeItem(itm)
            del itm
            
            self.filetypeAssociationList.clearSelection()
            self.filePatternEdit.clear()
            self.filetypeCombo.setCurrentItem(0)

    def handlePatternTextChanged(self, txt):
        """
        Private slot to handle the textChanged signal of the pattern lineedit.
        
        @param txt text of the lineedit (QString)
        """
        if txt.isEmpty():
            self.addAssociationButton.setEnabled(0)
            self.deleteAssociationButton.setEnabled(0)
        else:
            self.addAssociationButton.setEnabled(1)
            if self.filetypeAssociationList.selectedItem() is None:
                self.deleteAssociationButton.setEnabled(0)
            else:
                self.deleteAssociationButton.setEnabled(1)

    def transferData(self):
        """
        Public slot to transfer the associations into the projects data structure.
        """
        self.project.pdata["FILETYPES"] = {}
        itm = self.filetypeAssociationList.firstChild()
        while itm is not None:
            pattern = unicode(itm.text(0))
            index = self.filetypeStrings.index(itm.text(1))
            self.project.pdata["FILETYPES"][pattern] = self.filetypes[index]
            itm = itm.nextSibling()
        
