# -*- coding: utf-8 -*-

# Copyright (c) 2005 - 2006 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing base classes used by the various class browsers.
"""

class _ClbrBase:
    """
    Class implementing the base of all class browser objects.
    """
    def __init__(self, module, name, file, lineno):
        """
        Constructor
        
        @param module name of the module containing this class
        @param name name of this class
        @param file filename containing this class
        @param lineno linenumber of the class definition
        """
        self.module = module
        self.name = name
        self.file = file
        self.lineno = lineno

class ClbrBase(_ClbrBase):
    """
    Class implementing the base of all complex class browser objects.
    """
    def __init__(self, module, name, file, lineno):
        """
        Constructor
        
        @param module name of the module containing this class
        @param name name of this class
        @param file filename containing this class
        @param lineno linenumber of the class definition
        """
        _ClbrBase.__init__(self, module, name, file, lineno)
        self.methods = {}
        self.attributes = {}
        self.classes = {}
        
    def _addmethod(self, name, function):
        """
        Method to add information about a method.
        
        @param name name of method to be added (string)
        @param function Function object to be added
        """
        self.methods[name] = function
        
    def _getmethod(self, name):
        """
        Method to retrieve a method by name.
        
        @param name name of the method (string)
        @return the named method or None
        """
        try:
            return self.methods[name]
        except KeyError:
            return None
        
    def _addattribute(self, attr):
        """
        Method to add information about attributes.
        
        @param attr Attribute object to be added (string)
        """
        if not self.attributes.has_key(attr.name):
            self.attributes[attr.name] = attr
        
    def _getattribute(self, name):
        """
        Method to retrieve an attribute by name.
        
        @param name name of the attribute (string)
        @return the named attribute or None
        """
        try:
            return self.attributes[name]
        except KeyError:
            return None
        
    def _addclass(self, name, _class):
        """
        Protected method to add a nested class to this class.
        
        @param name name of the class
        @param _class Class object to be added
        """
        self.classes[name] = _class
        
class ClbrVisibilityMixinBase:
    """
    Class implementing the base class of all visibility mixins.
    """
    def isPrivate(self):
        """
        Method to check, if the visibility is Private.
        
        @return flag indicating Private visibility (boolean)
        """
        return self.visibility == 0
        
    def isProtected(self):
        """
        Method to check, if the visibility is Protected.
        
        @return flag indicating Protected visibility (boolean)
        """
        return self.visibility == 1
        
    def isPublic(self):
        """
        Method to check, if the visibility is Public.
        
        @return flag indicating Public visibility (boolean)
        """
        return self.visibility == 2
        
    def setPrivate(self):
        """
        Method to set the visibility to Private.
        """
        self.visibility = 0
        
    def setProtected(self):
        """
        Method to set the visibility to Protected.
        """
        self.visibility = 1
        
    def setPublic(self):
        """
        Method to set the visibility to Public.
        """
        self.visibility = 2

class Attribute(_ClbrBase):
    """
    Class to represent an attribute.
    """
    def __init__(self, name):
        """
        Constructor
        
        @param name name of this attribute
        """
        _ClbrBase.__init__(self, None, name, None, None)

class Class(ClbrBase):
    """
    Class to represent a class.
    """
    def __init__(self, module, name, super, file, lineno):
        """
        Constructor
        
        @param module name of the module containing this class
        @param name name of this class
        @param super list of class names this class is inherited from
        @param file filename containing this class
        @param lineno linenumber of the class definition
        """
        ClbrBase.__init__(self, module, name, file, lineno)
        if super is None:
            super = []
        self.super = super

class Module(ClbrBase):
    """
    Class to represent a module.
    """
    def __init__(self, module, name, file, lineno):
        """
        Constructor
        
        @param module name of the module containing this class
        @param name name of this class
        @param file filename containing this class
        @param lineno linenumber of the class definition
        """
        ClbrBase.__init__(self, module, name, file, lineno)

class Function(ClbrBase):
    """
    Class to represent a function or method.
    """
    def __init__(self, module, name, file, lineno, signature = '', separator = ','):
        """
        Constructor
        
        @param module name of the module containing this function
        @param name name of this function
        @param file filename containing this class
        @param lineno linenumber of the class definition
        @param signature parameterlist of the method
        @param separator string separating the parameters
        """
        ClbrBase.__init__(self, module, name, file, lineno)
        self.parameters = [e.strip() for e in signature.split(separator)]
