/////////////////////////////////////////////////////////////
//                                                         //
// Copyright (c) 2003-2011 by The University of Queensland //
// Earth Systems Science Computational Centre (ESSCC)      //
// http://www.uq.edu.au/esscc                              //
//                                                         //
// Primary Business: Brisbane, Queensland, Australia       //
// Licensed under the Open Software License version 3.0    //
// http://www.opensource.org/licenses/osl-3.0.php          //
//                                                         //
/////////////////////////////////////////////////////////////


#include <boost/version.hpp>
#include <boost/python.hpp>
#include "Foundation/StringUtil.h"
#include "Python/esys/lsm/CheckPointParamsPy.h"

namespace esys
{
  namespace lsm
  {
    CheckPointPrmsPy::CheckPointPrmsPy(
      const std::string &fileNamePrefix,
      int beginTimeStep,
      int endTimeStep,
      int timeStepIncr
    )
      : m_fileNamePrefix(fileNamePrefix),
        m_beginTimeStep(beginTimeStep),
        m_endTimeStep(endTimeStep),
        m_timeStepIncr(timeStepIncr)
    {
    }

    boost::python::list CheckPointPrmsPy::getFileNameList() const
    {
      boost::python::list fileNameList;
      for (int i = getBeginTimeStep(); i < getEndTimeStep(); i += getTimeStepIncr())
        {
          fileNameList.append(getFileName(i));
        }
      fileNameList.append(getFileName(getEndTimeStep()));
      return fileNameList;
    }

    std::string CheckPointPrmsPy::getFileName(int timeStep, int rank) const
    {
      return (getFileNamePrefix()+ "_t=" +
              StringUtil::toString(timeStep) + "_" +
              StringUtil::toString(rank) + ".txt");
    }

    using boost::python::arg;
    void exportCheckPointPrms()
    {
      // Check that Boost 1.34.0 or higher is being used.
      // If so, disable auto-generation of C++ signatures for Epydoc
      // (which stumbles over indentation in the auto-generated strings).
      #if ((BOOST_VERSION / 100000 >= 1) \
          && (BOOST_VERSION / 100 % 1000 >= 34)) \
          || (BOOST_VERSION / 100000 >= 2)
        boost::python::docstring_options no_autogen(true,false);
      #endif

      boost::python::class_<CheckPointPrmsPy>(
        "CheckPointPrms",
        "Parameters for specifying check-pointing intervals.",
        boost::python::init<const std::string &, int, int, int>(
          (
            arg("fileNamePrefix"),
            arg("beginTimeStep"),
            arg("endTimeStep"),
            arg("timeStepIncr")
          ),
          "@type fileNamePrefix: string\n"
          "@kwarg fileNamePrefix: prefix of files where checkpoint data"
          " is saved.\n"
          "@type beginTimeStep: int\n"
          "@kwarg beginTimeStep: time step when checkpoint saving begins.\n"
          "@type endTimeStep: int\n"
          "@kwarg endTimeStep: time step when checkpoint saving stops.\n"
          "@type timeStepIncr: int\n"
          "@kwarg timeStepIncr: a checkpoint is made every"
          "C{timeStepIncr} time-steps.\n"
        )
      )
        .def(
          "getFileNamePrefix",
          &CheckPointPrmsPy::getFileNamePrefix,
          "@rtype: string\n"
          "@return: prefix of files where checkpoint data is saved.\n"
        )
        .def(
          "getBeginTimeStep",
          &CheckPointPrmsPy::getBeginTimeStep,
          "@rtype: int"
        )
        .def(
          "getEndTimeStep",
          &CheckPointPrmsPy::getEndTimeStep,
          "@rtype: int"
        )
        .def(
          "getTimeStepIncr",
          &CheckPointPrmsPy::getTimeStepIncr,
          "@rtype: int"
        )
        .def(
          "getFileNameList",
          &CheckPointPrmsPy::getFileNameList,
          "@rtype: list\n"
          "@return: List of 'master' check-point file names."
        )
        ;
    }
  }
}
