/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module:  orm_discover.c
 *
 *
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <linux/evms/evms_os2.h>
#include <plugin.h>
#include "os2regmgr.h"
#include "orm_options.h"
#include "orm_discover.h"


/*
 *  Function:  free_get_info_memory
 *
 *  frees memory associated with a get_info call.
 *
 */
static void free_get_info_memory( extended_info_array_t  *info )
{
    int i;

    if ( info ) {

        for ( i = 0; i < info->count; i++ ) {

            if ( info->info[i].type == EVMS_Type_String ) {

                if ( info->info[i].value.s ) {
                    ORM_EngFncs->engine_free( info->info[i].value.s );
                }
            }
        }

        ORM_EngFncs->engine_free( info );
    }
}


/*
 *  Function:  get_DLAT_volume_name
 *
 *  get the OS/2 Volume name as found in the DLA Table.
 *
 */
static int get_DLAT_volume_name( storage_object_t *segment, char *volume_name )
{
    int     rc = EINVAL;
    int     i;
    extended_info_array_t  *info        = NULL;

    if (( segment ) && ( volume_name )) {

        volume_name[0] = '\0';

        rc = (( struct plugin_functions_s * )( segment->plugin->functions.plugin ))->get_info( segment, NULL, &info );

        if ( rc == 0 ) {

            if ( info ) {

                for( i = 0; i < info->count; i++ ) {

                    if ( strncmp( info->info[i].name, "Volume Name", 11 ) == 0 ) {
                        strcpy( volume_name, info->info[i].value.s );
                        rc = 0;
                        break;
                    }
                }

                free_get_info_memory( info );
            }
            else {
                LOG_ERROR( "get_info for segment returned no info\n" );
            }
        }
        else {
            LOG_ERROR( "get_info for segment failed, rc == %d\n", rc );
        }
    }

    return rc;
}


/*
 *  Function:  get_DLAT_drive_letter
 *
 *  get the OS/2 Volume drive letter as found in the DLA Table.
 *
 */
static int get_DLAT_drive_letter( storage_object_t *segment, char *drive_letter )
{
    int     rc = EINVAL;
    int     i;
    extended_info_array_t  *info        = NULL;

    if (( segment ) && ( drive_letter )) {

        drive_letter[0] = '\0';
        rc = (( struct plugin_functions_s * )( segment->plugin->functions.plugin ))->get_info( segment, NULL, &info );

        if ( rc == 0 ) {

            if ( info ) {

                rc = 1;
                for( i = 0; i < info->count; i++ ) {

                    if ( strncmp( info->info[i].name, "Drive Letter", 12 ) == 0 ) {
                        if ( info->info[i].value.s[0] >= 'C'  && info->info[i].value.s[0] <= 'Z' ) {
                            drive_letter[0] = info->info[i].value.s[0];
                            drive_letter[1] = '\0';
                            rc = 0;
                        }
                        break;
                    }
                }

                free_get_info_memory( info );
            }
            else {
                LOG_ERROR( "get_info for segment returned no info\n" );
            }
        }
        else {
            LOG_ERROR( "get_info for segment failed, rc == %d\n", rc );
        }
    }

    return rc;
}


/*
 *  Function:  free_os2_drivelinks
 *
 *  Free the chain of drive link structures and the link table and
 *    BBR table each one points to.
 *
 */
static void free_os2_drivelinks( os2_drivelink_t  *drive_link )
{
    os2_drivelink_t  * cur_link, * next_link;

    LOGENTRY();

    if ( drive_link ) {

        cur_link = drive_link;
        while ( cur_link ) {

            next_link = cur_link->next;

            if ( cur_link->link_data )
                ORM_EngFncs->engine_free( cur_link->link_data );

            if ( cur_link->bbr_data )
                ORM_EngFncs->engine_free( cur_link->bbr_data );

            ORM_EngFncs->engine_free( cur_link );
            cur_link = next_link;
        }
    }

    LOGEXIT();
}


/*
 *  Function:  free_os2_region
 *
 *  Free the storage_region structure plus any other items associated
 *    with it.
 *
 */
void free_os2_region( storage_object_t *region )
{
    os2_drivelink_t  * first_link;

    LOGENTRY();

    if ( region ) {

        if ( region->private_data ) {

            first_link = (( orm_private_data_t  * )region->private_data )->drive_link;
            if ( first_link )
                free_os2_drivelinks( first_link );

            ORM_EngFncs->engine_free( region->private_data );
        }

        ORM_EngFncs->free_region( region );
    }

    LOGEXIT();
}


/*
 *  Function:  new_os2_region
 *
 *  Allocate a new storage object that will become the new region.
 *
 */
static storage_object_t * new_os2_region( char  * region_name )
{
    int rc;
    storage_object_t    * region = NULL;
    orm_private_data_t  * region_private_data;

    LOGENTRY();
    rc = ORM_EngFncs->allocate_region( region_name, &region );

    if ( rc == 0 ) {

        region->size = 0;

        region->plugin = &ORM_PluginRecord;

        // allocate region private data area
        region->private_data = ORM_EngFncs->engine_alloc( sizeof( orm_private_data_t ));

        if ( region->private_data ) {

            // initialize the private data for the region...
            region_private_data = ( orm_private_data_t * )region->private_data;
            region_private_data->signature            = ORM_PDATA_SIGNATURE;
            region_private_data->flags                = 0;
            region_private_data->size_in_sectors      = 0;
            region_private_data->Volume_Serial_Number = 0;
            region_private_data->drive_link_count     = 0;
            region_private_data->drive_link           = NULL;
        }
        else {
            LOG_ERROR( "engine_alloc could not allocate private data area\n" );
            free_os2_region( region );
            region = NULL;
        }
    }
    else {
        LOG_ERROR( "allocate_region failed, rc == %d\n", rc );
    }

    LOGEXIT();
    return region;
}


/*
 *  Function:  consume_segment
 *
 *  Consume the segment storage object and place in the region object.
 *
 */
static int consume_segment( storage_object_t * region,
                            storage_object_t * segment )
{
    int     rc;
    void  * handle;

    if ( region->private_data ) {

        // consumed object is inserted into our child DLIST
        rc = InsertObject( segment->parent_objects,
                           sizeof( storage_object_t ),
                           region,
                           REGION_TAG,
                           NULL,
                           InsertAtStart,
                           FALSE,
                           ( void** )&handle );
        LOG_DEBUG( "return code from segment parent InsertObject == %d \n", rc );

        if ( rc == DLIST_SUCCESS ) {

            rc = InsertObject( region->child_objects,
                               sizeof( storage_object_t ),
                               segment,
                               SEGMENT_TAG,
                               NULL,
                               AppendToList,
                               FALSE,
                               ( void** )&handle );
            LOG_DEBUG( "return code from region child InsertObject == %d \n", rc );
        }
    }
    else {
        LOG_ERROR( "region object has no private data area\n" );
        rc = ENOMEM;
    }

    return rc;
}


/*
 *  Function:  has_lvm_signature_sector
 *
 *  Quick Function to validate the existence of the LVM Signature Sector.
 *   There are 3 criteria to validate:
 *    1. The two signatures must be correct,
 *    2. The CRC must be correct,
 *    3. The recorded partition size must match the actual segment size.
 *
 */
static BOOLEAN has_lvm_signature_sector( storage_object_t *obj )
{
    int                     rc;
    char                    buffer[OS2_BYTES_PER_SECTOR];
    u_int32_t               old_crc;
    u_int32_t               calculated_crc;
    LVM_Signature_Sector   *lvm = ( LVM_Signature_Sector * )buffer;

    LOGENTRY();

    rc = (( struct plugin_functions_s * )( obj->plugin->functions.plugin ))->read( obj, obj->size-1, 1, buffer );
    if ( rc == 0 ) {

        lvm = ( LVM_Signature_Sector * )buffer;

        if (( DISK_TO_CPU32( lvm->LVM_Signature1 ) == OS2LVM_PRIMARY_SIGNATURE ) &&
            ( DISK_TO_CPU32( lvm->LVM_Signature2 ) == OS2LVM_SECONDARY_SIGNATURE )) {

            old_crc = DISK_TO_CPU32( lvm->Signature_Sector_CRC );
            lvm->Signature_Sector_CRC = 0;

            calculated_crc = ORM_EngFncs->calculate_CRC( EVMS_INITIAL_CRC, lvm, OS2_BYTES_PER_SECTOR );

            if ( calculated_crc == old_crc ) {

                if ( DISK_TO_CPU32( lvm->Partition_Sector_Count ) == obj->size ) {
                    LOGEXIT();
                    return TRUE;
                }
                else {
                    LOG_DEBUG( "size check failed, SignatureSector size == 0x%X  Segment size == 0x%X\n", lvm->Partition_Sector_Count, obj->size );
                }
            }
            else {
                LOG_DEBUG( "CRC failed, Original CRC == 0x%X  Calculated CRC == 0x%X\n", old_crc, calculated_crc );
            }
        }
        else {
            LOG_DEBUG( "signature check failed, Primary Signature == 0x%X  Secondary Signature == 0x%X\n", lvm->LVM_Signature1, lvm->LVM_Signature2 );
        }
    }

    LOGEXIT();
    return FALSE;
}


/*
 *  Function:  isa_lvm_segment
 *
 *  Called to query the segment manager for information about this storage object.
 *  If it turns out to be part of an LVM volume then return TRUE.
 */
static BOOLEAN isa_lvm_segment( storage_object_t *obj )
{
    extended_info_array_t  *info        = NULL;
    char                    DriveLetter[4];
    u_int32_t               PartitionSN = 0;
    u_int32_t               VolumeSN    = 0;
    u_int32_t               Type        = 0;
    int                     rc;
    int                     i;

    LOGENTRY();

    DriveLetter[0] = '\0';

    rc = (( struct plugin_functions_s * )( obj->plugin->functions.plugin ))->get_info( obj, NULL, &info );

    if ( rc == DLIST_SUCCESS ) {

        if ( info ) {

            for( i = 0; i < info->count; i++ ) {

                if ( strncmp( info->info[i].name, "Partition SN", 12 ) == 0 ) {
                    PartitionSN = info->info[i].value.ui32;
                }
                else if ( strncmp( info->info[i].name, "Volume SN", 9 ) == 0 ) {
                    VolumeSN = info->info[i].value.ui32;
                }
                else if ( strncmp( info->info[i].name, "Type", 4 ) == 0 ) {
                    Type = ( u_int32_t )info->info[i].value.uc;
                }
                else if ( strncmp( info->info[i].name, "Drive Letter", 12 ) == 0 ) {
                    if ( info->info[i].value.s[0] >= 'C'  && info->info[i].value.s[0] <= 'Z' ) {
                        DriveLetter[0] = info->info[i].value.s[0];
                        DriveLetter[1] = '\0';
                    }
                }
            }

            free_get_info_memory( info );

/* LOG_DEBUG("checking results, DriveLetter= %X  VolumeSN= %X  PartitionSN= %X  Type= %X\n", DriveLetter, VolumeSN, PartitionSN, Type);*/

            if (( VolumeSN != 0 ) &&
                ( PartitionSN != 0 ) &&
                ( Type == LVM_PARTITION_INDICATOR ) &&
                ( has_lvm_signature_sector( obj ) == TRUE )) {

                    LOG_DEBUG("Found OS/2 LVM segment:  DriveLetter= %s  VolumeSN= %X  PartitionSN= %X  Type= %X\n", DriveLetter, VolumeSN, PartitionSN, Type );
                    return TRUE;
            }
        }
    }

    LOGEXIT();
    return FALSE;
}


/*
 *  Function:  isa_os2_compatibility_segment
 *
 *  Called to query the segment manager for information about this storage object.
 *   If it turns out to be an OS/2 compatibility segment then return TRUE.
 */
static BOOLEAN isa_os2_compatibility_segment( storage_object_t *obj )
{
    extended_info_array_t  *info        = NULL;
    char                    DriveLetter[4];
    u_int32_t               PartitionSN = 0;
    u_int32_t               VolumeSN    = 0;
    u_int32_t               Type        = 0;
    int                     rc;
    int                     i;


    LOGENTRY();

    DriveLetter[0] = '\0';

    rc = (( struct plugin_functions_s * )( obj->plugin->functions.plugin ))->get_info( obj, NULL, &info );

    if ( rc == DLIST_SUCCESS ) {

        if ( info ) {

            for( i = 0; i < info->count; i++ ) {

                if ( strncmp( info->info[i].name, "Partition SN", 12 ) == 0 ) {
                    PartitionSN = info->info[i].value.ui32;
                }
                else if ( strncmp( info->info[i].name, "Volume SN", 9 ) == 0 ) {
                    VolumeSN = info->info[i].value.ui32;
                }
                else if ( strncmp( info->info[i].name, "Type", 4 ) == 0 ) {
                    Type = ( u_int32_t )info->info[i].value.uc;
                }
                else if ( strncmp( info->info[i].name, "Drive Letter", 12 ) == 0 ) {
                    if ( info->info[i].value.s[0] >= 'C'  && info->info[i].value.s[0] <= 'Z' ) {
                        DriveLetter[0] = info->info[i].value.s[0];
                        DriveLetter[1] = '\0';
                    }
                }
            }

            free_get_info_memory( info );

// LOG_DEBUG("checking results, DriveLetter= %X  VolumeSN= %X  PartitionSN= %X  Type= %X\n", DriveLetter, VolumeSN, PartitionSN, Type );

            if (( strlen( DriveLetter )) &&
                ( VolumeSN != 0 ) &&
                ( PartitionSN != 0 ) &&
                ( Type != LVM_PARTITION_INDICATOR )) {

                    LOG_DEBUG( "Found OS/2 Compatibility Volume:  DriveLetter= %s  VolumeSN= %X  PartitionSN= %X  Type= %X\n", DriveLetter, VolumeSN, PartitionSN, Type );
                    LOGEXIT();
                    return TRUE;
            }
        }
        else {
            LOG_ERROR( "get_info for segment returned no info\n" );
        }
    }
    else {
        LOG_ERROR( "get_info for segment failed, rc == %d\n", rc );
    }

    LOGEXIT();
    return FALSE;
}


/*
 *  Function:  commit_os2_lvm_region
 *
 *  Time to get committed.  All the checks passed, now consume the OS/2
 *   LVM segments into the region.  For each segment passed in, check if
 *   it is in the Drive Link metadata.  If so, remove it from the list
 *   we are passed.
 *
 *  Returns: TRUE if object is Ok and can be removed from its original child
 *           object DLIST.
 *
 */
static BOOLEAN commit_os2_lvm_region( void     *Object,
                                      TAG       ObjectTag,
                                      uint      ObjectSize,
                                      void     *ObjectHandle,
                                      void     *Parameters,
                                      BOOLEAN  *FreeMemory,
                                      uint     *Error )
{
    storage_object_t  *segment  = ( storage_object_t * )Object;
    storage_object_t  *region   = ( storage_object_t * )Parameters;
    os2_drivelink_t   *cur_link = NULL, *next_link = NULL;
    BOOLEAN    rc;

    LOGENTRY();

    *FreeMemory = FALSE;           /* tells dlist not to free any memory */
    *Error      = DLIST_SUCCESS;   /* tells dlist we were successful     */
    rc          = FALSE;           /* tells dlist not to prune item from list */

    cur_link = (( orm_private_data_t * )region->private_data )->drive_link;

    while ( cur_link ) {

        next_link = cur_link->next;

        if ( cur_link->object == segment ) {

            LOG_DEBUG( "Found drive link:  PartitionSN= %X - will make it a child of this region\n", cur_link->Partition_Serial_Number );
            *Error = consume_segment( region, segment );

            if ( !( *Error )) {
                LOGEXIT();
                return TRUE;   // prune segment from list cuz the OS/2 region consumed it
            }
            else {
                LOG_ERROR( "consume_segment failed, rc == %d\n", *Error );
                break;
            }
        }

        cur_link = next_link;
    }

    LOGEXIT();
    return rc;
}


/*
 *  Function:  purge_os2_lvm_incomplete_region
 *
 *  This function is used to purge the segments that were not found from
 *   the valid object list and therefore avoid trying to revalidate them.
 *
 *  Returns: TRUE if object is Ok and can be removed from its original child
 *           object DLIST.
 *
 */
static BOOLEAN purge_os2_lvm_incomplete_region( void     *Object,
                                                TAG       ObjectTag,
                                                uint      ObjectSize,
                                                void     *ObjectHandle,
                                                void     *Parameters,
                                                BOOLEAN  *FreeMemory,
                                                uint     *Error )
{
    storage_object_t  *segment  = ( storage_object_t * )Object;
    storage_object_t  *region   = ( storage_object_t * )Parameters;
    os2_drivelink_t   *cur_link = NULL, *next_link = NULL;
    BOOLEAN    rc;

    LOGENTRY();

    *FreeMemory = FALSE;           /* tells dlist not to free any memory */
    *Error      = DLIST_SUCCESS;   /* tells dlist we were successful     */
    rc          = FALSE;           /* tells dlist not to prune item from list */

    cur_link = (( orm_private_data_t * )region->private_data )->drive_link;

    while ( cur_link ) {

        next_link = cur_link->next;

        if ( cur_link->object == segment ) {

            LOG_DEBUG( "Found drive link:  PartitionSN= %X - will remove from the input list\n", cur_link->Partition_Serial_Number );

            LOGEXIT();
            return TRUE;   // prune segment from list cuz the OS/2 region consumed it
        }

        cur_link = next_link;
    }

    LOGEXIT();
    return rc;
}


/*
 * Function:  find_link_data
 *
 *  Find the Drive Link metadata that matches the partition serial number.
 *    Remove it from the link_list passed in.
 *
 */
static os2_drivelink_t  * find_link_data( os2_drivelink_t  ** link_list,
                                          u_int32_t        partitionser )
{
    os2_drivelink_t  * curlink = *link_list, * prevlink = NULL;

    while ( curlink ) {
        if ( curlink->Partition_Serial_Number == partitionser ) {
            if ( prevlink ) {
                prevlink->next = curlink->next;
            }
            else  {
                *link_list = curlink->next;
            }
            curlink->next = NULL;
            return curlink;
        }
        prevlink = curlink;
        curlink = prevlink->next;
    }

    return NULL;
}


/*
 *  Function:  add_os2link
 *
 *  Called to add a drive link structure to the region.  The drive link
 *   structures are always added at the end.
 *
 */
static void add_os2link( os2_drivelink_t     * newlink,
                         orm_private_data_t  * region_private_data )
{
    os2_drivelink_t   * curlink = region_private_data->drive_link, * nextlink;

    if ( curlink ) {
        nextlink = curlink->next;
        while ( nextlink ) {
                curlink = nextlink;
                nextlink = curlink->next;
        }
        curlink->next = newlink;
    }
    else {
        region_private_data->drive_link = newlink;
    }
}


/*
 *  Function:  validate_os2_lvm_region
 *
 *  Called during feature discovery to validate the OS/2 LVM region.  The
 *   drive link metadata will be ordered according to the Drive Linking table
 *   already in storage.  If Ok, return 0.  If not return 1.
 *
 *  Returns:  0 if object is Ok and 1 if not.
 *
 */
static int validate_os2_lvm_region( storage_object_t  * region )
{
    int i;
    sector_count_t   partition_offset = 0;
    char  * sect_ptr;
    orm_private_data_t  * region_private_data = ( orm_private_data_t * )region->private_data;
    LVM_Link_Table_First_Sector  * psector1;
    u_int32_t  numlinks, countlinks, linkser;
    os2_drivelink_t  *link_list, *link_hold;

    LOGENTRY();

    LOG_DEBUG( "Validating drive links for region:  VolumeSN= %X ...\n", region_private_data->Volume_Serial_Number );

    link_list = region_private_data->drive_link;
    region_private_data->drive_link = NULL;

    // Access the link data to order the drive links
    psector1 = ( LVM_Link_Table_First_Sector * )link_list->link_data;
    numlinks = DISK_TO_CPU32( psector1->Links_In_Use );
    LOG_DEBUG( "There are %d drive links to validate\n", numlinks );
    region_private_data->drive_link_count = numlinks;

    if ( numlinks > LINKS_IN_FIRST_SECTOR ) {
        countlinks = LINKS_IN_FIRST_SECTOR;
        numlinks -= LINKS_IN_FIRST_SECTOR;
    }
    else {
        countlinks = numlinks;
        numlinks = 0;
    }

    for ( i = 0; i < countlinks; i++ ) {

        linkser = DISK_TO_CPU32( psector1->Link_Table[i].Partition_Serial_Number );
        if ( ( link_hold = find_link_data( &link_list, linkser )) ) {

            // Add this partition to its parent Volume
            add_os2link( link_hold, region_private_data );
            LOG_DEBUG( "Link Table entry %i, start_RBA == %Ld , sector_count == %Ld\n", i, partition_offset, link_hold->sector_count );
            link_hold->start_sector = partition_offset;
            partition_offset += link_hold->sector_count;
        }
        else {
            LOG_ERROR( "Link Table entry %i, PartitionSN %X metadata missing\n", i, linkser );
            LOGEXIT();
            return 1;
        }
    }

    sect_ptr = ( char * )psector1;

    while ( numlinks ) {

        if ( numlinks > LINKS_IN_NEXT_SECTOR ) {
            countlinks = LINKS_IN_NEXT_SECTOR;
            numlinks -= LINKS_IN_NEXT_SECTOR;
        }
        else {
            countlinks = numlinks;
            numlinks = 0;
        }
        sect_ptr += OS2_BYTES_PER_SECTOR;

        for ( i = 0; i < countlinks; i++ ) {

            linkser = DISK_TO_CPU32( (( LVM_Link_Table_Sector  * )sect_ptr )->Link_Table[i].Partition_Serial_Number );
            if ( ( link_hold = find_link_data( &link_list, linkser )) ) {

                // Add this partition to its parent Volume
                add_os2link( link_hold, region_private_data );
                LOG_DEBUG( "Link Table entry %i, start_RBA == %Ld , sector_count == %Ld\n", i, partition_offset, link_hold->sector_count );
                link_hold->start_sector = partition_offset;
                partition_offset += link_hold->sector_count;
            }
            else {
                LOG_ERROR( "Link Table entry %i, PartitionSN %X metadata missing\n", i, linkser );
                LOGEXIT();
                return 1;
            }
        }
    }

    region->start = 0;
    region->size = partition_offset;
    region_private_data->size_in_sectors = partition_offset;
    region_private_data->flags |= VOLUME_IS_COMPLETE;

    LOGEXIT();
    return 0;
}


/*
 *  Function:  build_os2_compatibility_region
 *
 *  Called during feature discovery to build the storage object that
 *   represents the OS/2 Compatibility Region.  The name and the appropriate
 *   private data fields are filled in.  The region object is then added to
 *   the output_list DLIST we are passed.
 *
 *  Returns: TRUE if object is Ok and can be removed from its original child
 *           object DLIST.
 *
 */
static BOOLEAN build_os2_compatibility_region( void     *Object,
                                               TAG       ObjectTag,
                                               uint      ObjectSize,
                                               void     *ObjectHandle,
                                               void     *Parameters,
                                               BOOLEAN  *FreeMemory,
                                               uint     *Error )
{
    storage_object_t     *segment     = ( storage_object_t * )Object;
    dlist_t               output_list = ( dlist_t )Parameters;
    storage_object_t     *region;
    char                  region_name[EVMS_NAME_SIZE];
    char                  volume_name[VOLUME_NAME_SIZE];
    char                  drive_letter[4];
    int        rc;
    void      *handle;
    orm_private_data_t  * region_private_data;


    LOGENTRY();

    *FreeMemory = FALSE;           /* tells dlist not to free any memory */
    *Error      = DLIST_SUCCESS;   /* tells dlist we were successful     */
    rc          = FALSE;           /* tells dlist not to prune item from list */

    if ( segment ) {

        // build the region name...
        drive_letter[0] = '\0';
        rc = get_DLAT_volume_name( segment, volume_name );
        rc += get_DLAT_drive_letter( segment, drive_letter );
        if ( rc == 0 ) {

            sprintf( region_name, "os2/%s", drive_letter );

            // consume the disk partition, producing an OS/2 region
            region = new_os2_region( region_name );

            // place new region onto output object list
            if ( region ) {

                rc = consume_segment( region, segment );
                if ( !rc ) {

                    region->size += segment->size;
                    region_private_data = ( orm_private_data_t * )region->private_data;
                    region_private_data->flags |= COMPATIBILITY_VOLUME;
                    region_private_data->flags |= VOLUME_IS_COMPLETE;
                    region_private_data->size_in_sectors += segment->size;

                    rc = InsertObject( output_list,
                                       sizeof( storage_object_t ),
                                       region,
                                       REGION_TAG,
                                       NULL,
                                       InsertAtStart,
                                       FALSE,
                                       ( void** )&handle );

                    if ( rc == DLIST_SUCCESS ) {
                        LOGEXIT();
                        return TRUE;   // prune segment from list cuz the OS/2 region consumed it
                    }
                    else {
                        LOG_ERROR( "InsertObject failed, rc == %d\n", rc );
                    }
                }
                else {
                    LOG_ERROR( "consume_segment failed, rc == %d\n", rc );
                }
                // if we got here, we had a problem...
                free_os2_region( region );
            }
        }
    }

    LOGEXIT();
    return FALSE;
}


/*
 * Function:  validate_drivelinksector
 *
 *      This function checks the OS/2 LVM Drivelink Feature Sector
 *
 */
static int validate_drivelinksector( LVM_Link_Table_Sector * link_sector,
                                     int which,
                                     u_int32_t sectorsize )
{
    u_int32_t  link_signature, crc_hold, crc_new;

    if ( !which )  // Sector index is zero-based
        link_signature = LINK_TABLE_MASTER_SIGNATURE;
    else
        link_signature = LINK_TABLE_SIGNATURE;

    if ( DISK_TO_CPU32( link_sector->Link_Table_Signature ) != link_signature ) {
        LOG_ERROR( "Link Table Signature failed.\n" );
        return 1;
    }

    crc_hold = DISK_TO_CPU32( link_sector->Link_Table_CRC );
    link_sector->Link_Table_CRC = 0;
    crc_new = ORM_EngFncs->calculate_CRC( EVMS_INITIAL_CRC, ( char * )link_sector, OS2_BYTES_PER_SECTOR );

    if ( crc_hold != crc_new ) {
        LOG_ERROR( "Link Table crc failed.\n" );
        LOG_DEBUG( "sector_crc == %x , calc_crc == %x \n", crc_hold, crc_new );
        return 1;
    }

    return 0;
}


/*
 *  Function:  new_os2_link_data
 *
 *  Allocate space for the OS/2 Drive Link table.  Read it in and validate
 *   the information contained therein.
 *
 */
static char  * new_os2_link_data( u_int32_t          linksector1,
                                  u_int32_t          linksector2,
                                  u_int32_t          linknumsectors,
                                  storage_object_t  *segment )
{
    char * new_data1, * new_data2, * p1, * p2;
    int  rc, memsize = linknumsectors * OS2_BYTES_PER_SECTOR;
    u_int32_t i, seq1, seq2;

    if ( !( new_data1 = ORM_EngFncs->engine_alloc( memsize )) ) {
        LOG_ERROR( "could not allocate Primary Link metadata area\n" );
        return NULL;
    }
    if ( !( new_data2 = ORM_EngFncs->engine_alloc( memsize )) ) {
        LOG_ERROR( "could not allocate Secondary Link metadata area\n" );
        ORM_EngFncs->engine_free( new_data1 );
        return NULL;
    }

    LOG_DEBUG( "Primary Feature Data starts at RBA %i\n", linksector1 );
    LOG_DEBUG( "Secondary Feature Data starts at RBA %i\n", linksector2 );

    rc = (( struct plugin_functions_s * )( segment->plugin->functions.plugin ))->read( segment, linksector1, linknumsectors, new_data1 );
    if ( rc != 0 ) {
        LOG_ERROR( "I/O error reading Primary Feature Data, rc == %d\n", rc );
    }
    rc = (( struct plugin_functions_s * )( segment->plugin->functions.plugin ))->read( segment, linksector2, linknumsectors, new_data2 );
    if ( rc != 0 ) {
        LOG_ERROR( "I/O error reading Secondary Feature Data, rc == %d\n", rc );
    }

    p1 = new_data1;
    seq1 = DISK_TO_CPU32( (( LVM_Link_Table_Sector * )p1 )->Sequence_Number );
    p2 = new_data2;
    seq2 = DISK_TO_CPU32( (( LVM_Link_Table_Sector * )p2 )->Sequence_Number );

    for ( i = 0; i < linknumsectors; i++, p1 += OS2_BYTES_PER_SECTOR, p2 += OS2_BYTES_PER_SECTOR ) {
        if ( validate_drivelinksector( ( LVM_Link_Table_Sector * )p1, i, OS2_BYTES_PER_SECTOR )) {
            LOG_ERROR( "Drive Link sector 1,%i is not valid\n", i );
            seq1 = 0;
        }
        else  if ( DISK_TO_CPU32( (( LVM_Link_Table_Sector * )p1 )->Sequence_Number ) != seq1 )
            seq1 = 0;
        if ( validate_drivelinksector( ( LVM_Link_Table_Sector * )p2, i, OS2_BYTES_PER_SECTOR )) {
            LOG_ERROR( "Drive Link sector 2,%i is not valid\n", i );
            seq2 = 0;
        }
        else  if ( DISK_TO_CPU32( (( LVM_Link_Table_Sector * )p2 )->Sequence_Number ) != seq2 )
            seq2 = 0;
    }

    LOG_DEBUG( "Primary Feature Data sequence # %i\n", seq1 );
    LOG_DEBUG( "Secondary Feature Data sequence # %i\n", seq2 );

    if ( seq2 > seq1 ) {
        ORM_EngFncs->engine_free( new_data1 );
        return  new_data2;
    }
    else {
        ORM_EngFncs->engine_free( new_data2 );
        if ( !seq1 ) {
            ORM_EngFncs->engine_free( new_data1 );
            new_data1 = NULL;
        }
    }
    return  new_data1;
}


/*
 * Function:  validate_bbrtablesector
 *
 *      This function checks the OS/2 LVM Bad Block Relocation Feature Sector
 *
 */
static int validate_bbrtablesector( LVM_BBR_Table_Sector * bbr_sector,
                                     int which,
                                     u_int32_t sectorsize )
{
    u_int32_t  bbr_signature, crc_hold, crc_new;

    if ( !which )  // Sector index is zero-based
        bbr_signature = BBR_TABLE_MASTER_SIGNATURE;
    else
        bbr_signature = BBR_TABLE_SIGNATURE;

    if ( DISK_TO_CPU32( bbr_sector->Signature ) != bbr_signature ) {
        LOG_ERROR( "BBR Table Signature failed.\n" );
        return 1;
    }

    crc_hold = DISK_TO_CPU32( bbr_sector->CRC );
    bbr_sector->CRC = 0;
    crc_new = ORM_EngFncs->calculate_CRC( EVMS_INITIAL_CRC, ( char * )bbr_sector, OS2_BYTES_PER_SECTOR );

    if ( crc_hold != crc_new ) {
        LOG_ERROR( "BBR Table crc failed.\n" );
        LOG_DEBUG( "sector_crc == %x , calc_crc == %x \n", crc_hold, crc_new );
        return 1;
    }
    return 0;
}


/*
 *  Function:  new_os2_bbr_data
 *
 *  Allocate space for the OS/2 Bad Block Relocation table.  Read it in and validate
 *   the information contained therein.
 *
 */
static char  * new_os2_bbr_data( u_int32_t          bbrsector1,
                                 u_int32_t          bbrsector2,
                                 u_int32_t          bbrnumsectors,
                                 storage_object_t  *segment )
{
    char * new_data1, * new_data2, * p1, * p2;
    int  rc, memsize = bbrnumsectors * OS2_BYTES_PER_SECTOR;
    u_int32_t i, seq1, seq2;

    if ( !( new_data1 = ORM_EngFncs->engine_alloc( memsize )) ) {
        LOG_ERROR( "could not allocate Primary BBR metadata area\n" );
        return NULL;
    }
    if ( !( new_data2 = ORM_EngFncs->engine_alloc( memsize )) ) {
        LOG_ERROR( "could not allocate Secondary BBR metadata area\n" );
        ORM_EngFncs->engine_free( new_data1 );
        return NULL;
    }

    LOG_DEBUG( "Primary Feature Data starts at RBA %i\n", bbrsector1 );
    LOG_DEBUG( "Secondary Feature Data starts at RBA %i\n", bbrsector2 );

    rc = (( struct plugin_functions_s * )( segment->plugin->functions.plugin ))->read( segment, bbrsector1, bbrnumsectors, new_data1 );
    if ( rc != 0 ) {
        LOG_ERROR( "I/O error reading Primary Feature Data, rc == %d\n", rc );
    }
    rc = (( struct plugin_functions_s * )( segment->plugin->functions.plugin ))->read( segment, bbrsector2, bbrnumsectors, new_data2 );
    if ( rc != 0 ) {
        LOG_ERROR( "I/O error reading Secondary Feature Data, rc == %d\n", rc );
    }

    p1 = new_data1;
    seq1 = DISK_TO_CPU32( (( LVM_BBR_Table_Sector * )p1 )->Sequence_Number );
    p2 = new_data2;
    seq2 = DISK_TO_CPU32( (( LVM_BBR_Table_Sector * )p2 )->Sequence_Number );

    for ( i = 0; i < bbrnumsectors; i++, p1 += OS2_BYTES_PER_SECTOR, p2 += OS2_BYTES_PER_SECTOR ) {
        if ( validate_bbrtablesector( ( LVM_BBR_Table_Sector * )p1, i, OS2_BYTES_PER_SECTOR )) {
            LOG_DEBUG( "BBR Table sector 1,%i is not valid\n", i );
            seq1 = 0;
        }
        else  if ( DISK_TO_CPU32( (( LVM_BBR_Table_Sector * )p1 )->Sequence_Number ) != seq1 )
            seq1 = 0;
        if ( validate_bbrtablesector( ( LVM_BBR_Table_Sector * )p2, i, OS2_BYTES_PER_SECTOR )) {
            LOG_DEBUG( "BBR Table sector 2,%i is not valid\n", i );
            seq2 = 0;
        }
        else  if ( DISK_TO_CPU32( (( LVM_BBR_Table_Sector * )p2 )->Sequence_Number ) != seq2 )
                  seq2 = 0;
    }

    LOG_DEBUG( "Primary Feature Data sequence # %i\n", seq1 );
    LOG_DEBUG( "Secondary Feature Data sequence # %i\n", seq2 );

    if ( seq2 > seq1 ) {
        ORM_EngFncs->engine_free( new_data1 );
        return  new_data2;
    }
    else {
        ORM_EngFncs->engine_free( new_data2 );
        if ( !seq1 ) {
            ORM_EngFncs->engine_free( new_data1 );
            new_data1 = NULL;
        }
    }
    return  new_data1;
}


/*
 *  Function:  new_os2_drive_link
 *
 *  Called with a pointer to the current segment and its signature sector.
 *   Use this information to allocate and fill in the Drive Link metadata and
 *   return it to the caller.
 */
static os2_drivelink_t  * new_os2_drive_link( LVM_Signature_Sector  * signature_sector,
                                              storage_object_t      * segment )
{
    int              i;
    u_int32_t        feature, feature_size, sectornum1, sectornum2, sectoroffset;
    os2_drivelink_t  *drive_link_data = ORM_EngFncs->engine_alloc( sizeof( os2_drivelink_t ));

    LOGENTRY();

    if ( drive_link_data ) {
        drive_link_data->start_sector = 0;
        drive_link_data->sector_count = DISK_TO_CPU32( signature_sector->Partition_Size_To_Report_To_User );
        drive_link_data->Partition_Serial_Number = DISK_TO_CPU32( signature_sector->Partition_Serial_Number );
        drive_link_data->bbr_is_active = 0;
        drive_link_data->object = segment;
        drive_link_data->next = NULL;

        sectoroffset = DISK_TO_CPU32( signature_sector->Partition_Start );
        LOG_DEBUG( "Segment start is at LBA %i\n", sectoroffset );

        for ( i = 0; i < OS2LVM_MAX_FEATURES_PER_VOLUME; i++ ) {

            feature = DISK_TO_CPU32( signature_sector->LVM_Feature_Array[i].Feature_ID );

            if ( feature ) {

                feature_size = DISK_TO_CPU32( signature_sector->LVM_Feature_Array[i].Feature_Data_Size );
                LOG_DEBUG( "Entry %d in Feature Table is valid\n", i+1 );
                LOG_DEBUG( "Feature Data size is %i sectors\n", feature_size );

                if ( feature == DRIVE_LINKING_FEATURE_ID ) {

                    LOG_DEBUG( "Feature is:  Drive Linking\n" );
                    sectornum1 = DISK_TO_CPU32( signature_sector->LVM_Feature_Array[i].Location_Of_Primary_Feature_Data ) - sectoroffset;
                    sectornum2 = DISK_TO_CPU32( signature_sector->LVM_Feature_Array[i].Location_Of_Secondary_Feature_Data ) - sectoroffset;
                    drive_link_data->link_data = new_os2_link_data( sectornum1, sectornum2, feature_size, segment );
                }
                else  if ( feature == BBR_FEATURE_ID ) {

                    LOG_DEBUG( "Feature is:  Bad Block Relocation (BBR)\n" );
                    sectornum1 = DISK_TO_CPU32( signature_sector->LVM_Feature_Array[i].Location_Of_Primary_Feature_Data );
                    sectornum2 = DISK_TO_CPU32( signature_sector->LVM_Feature_Array[i].Location_Of_Secondary_Feature_Data );
                    drive_link_data->bbr_data = new_os2_bbr_data( sectornum1, sectornum2, feature_size, segment );
                    // if ( signature_sector->LVM_Feature_Array[i].Feature_Active ) {
                    // need to check for active relocations here.  }
                }
                else {
                    LOG_DEBUG( "Unknown Feature value %d found\n", feature );
                }

                if ( signature_sector->LVM_Feature_Array[i].Feature_Active ) {
                    LOG_DEBUG( "Feature is active\n" );
                }
            }
        }

        LOGEXIT();
        return drive_link_data;
    }
    else {
        LOG_ERROR( "could not allocate drive link metadata area\n" );
    }

    LOGEXIT();
    return NULL;
}


/*
 *  Function:  find_complete_os2_lvm_region
 *
 *  Called with a list of validated OS/2 LVM segments.  Expected to look through
 *   the segments, find an OS/2 LVM region that can be completed, allocate a
 *   parent storage object and return the new ORM region storage object.
 */
static int find_complete_os2_lvm_region( void     *Object,
                                         TAG       ObjectTag,
                                         uint      ObjectSize,
                                         void     *ObjectHandle,
                                         void     *Parameters )
{
    int rc;
    storage_object_t      *segment = ( storage_object_t * )Object;
    storage_object_t      **region  = ( storage_object_t ** )Parameters;
    char                  buffer[OS2_BYTES_PER_SECTOR];
    char                  region_name[EVMS_NAME_SIZE];
    LVM_Signature_Sector  *lvm = ( LVM_Signature_Sector * )buffer;
    os2_drivelink_t       *link_data = NULL;
    orm_private_data_t    * region_private_data;

    LOGENTRY();

    // first grab the signature sector...
    rc = (( struct plugin_functions_s * )( segment->plugin->functions.plugin ))->read( segment, segment->size-1, 1, buffer );
    if ( rc == 0 ) {

        LOG_DEBUG( "Examining LVM segment:  DriveLetter= %c  VolumeSN= %X  PartitionSN= %X\n", lvm->Drive_Letter, lvm->Volume_Serial_Number, lvm->Partition_Serial_Number );

        if (( link_data = new_os2_drive_link( lvm, segment ))) {

            if ( *region == NULL ) {

                // allocate new region and get the needed info.
                if ( lvm->Drive_Letter >= 'C' && lvm->Drive_Letter <= 'Z' )
                    sprintf( region_name, "os2/%c", lvm->Drive_Letter );
                else
                    sprintf( region_name, "os2/%s", lvm->Volume_Name );

                LOG_DEBUG( "Allocating new OS/2 LVM region for %s\n", region_name );

                *region = new_os2_region( region_name );

                if ( *region ) {

                    LOG_DEBUG( "Region was successfully allocated\n" );
                    region_private_data = ( orm_private_data_t * )( *region )->private_data;
                    region_private_data->flags |= LVM_VOLUME;

                    if ( lvm->Drive_Letter < 'C' || lvm->Drive_Letter > 'Z' )
                        region_private_data->flags |= HIDDEN_VOLUME;

                    region_private_data->Volume_Serial_Number = DISK_TO_CPU32( lvm->Volume_Serial_Number );
                    region_private_data->drive_link = link_data;
                }
                else {
                    LOG_ERROR( "could not allocate region\n" );
                    rc = ENOMEM;
                }
            }
            else {  // already have a region... see if this one belongs in it...
                  region_private_data = ( orm_private_data_t * )( *region )->private_data;

                  if ( region_private_data->Volume_Serial_Number == DISK_TO_CPU32( lvm->Volume_Serial_Number )) {

                      LOG_DEBUG( "region already allocated, volume serial numbers match!\n" );
                      add_os2link( link_data, region_private_data );
                  }
                  else {

                      LOG_DEBUG( "region already allocated, volume serial numbers do not match\n" );
                      // need to deallocate the link_data structure...
                      free_os2_drivelinks( link_data );
                  }
            }
        }
        else {
            LOG_ERROR( "could not allocate drivelink metadata\n" );
            rc = ENOMEM;
        }
    }
    else {
        LOG_ERROR( "could not read signature sector, rc == %d\n", rc );
    }

    LOGEXIT();
    return rc;
}


/*
 *  Function:  validate_os2_lvm_objects
 *
 *  Called during feature discovery to validate a storage object. Meaning
 *   that the object is owned by the drive link feature and that the object
 *   appears Ok.  If Ok, the object will be added to the valid_object DLIST
 *   we are passed.  We will use this later to build the volumes.
 *
 *  Returns: TRUE if object is Ok and can be removed from its original child
 *           object DLIST.
 *
 */
static BOOLEAN validate_os2_lvm_objects( void     *Object,
                                         TAG       ObjectTag,
                                         uint      ObjectSize,
                                         void     *ObjectHandle,
                                         void     *Parameters,
                                         BOOLEAN  *FreeMemory,
                                         uint     *Error )
{
    storage_object_t *obj         = ( storage_object_t * )Object;
    dlist_t    valid_object_list  = ( dlist_t )Parameters;
    BOOLEAN    rc;
    void      *handle;

    LOGENTRY();

    *FreeMemory = FALSE;           /* tells dlist not to free any memory */
    *Error      = DLIST_SUCCESS;   /* tells dlist we were successful     */
    rc          = FALSE;           /* tells dlist not to prune item from list */


    if (( obj->object_type == SEGMENT ) &&
        ( obj->data_type   == DATA_TYPE )) {

        if ( isa_lvm_segment( Object ) == TRUE ) {

            *Error = InsertObject( valid_object_list,
                                   sizeof( storage_object_t ),
                                   Object,
                                   SEGMENT_TAG,
                                   NULL,
                                   InsertAtStart ,
                                   0,
                                   ( void** )&handle );

            if ( *Error == DLIST_SUCCESS ) {
                rc = TRUE;   // yes ... dlist can prune object from list
            }
        }
    }

    LOGEXIT();
    return rc;
}


/*
 *  Function:  validate_os2_compatibility_objects
 *
 *  Called during feature discovery to validate a storage object.  This means
 *   that the object is a segment and has proper data type.  The isa function
 *   is then called to validate the private data.  If this is OK, the object
 *   is added to the valid_object DLIST we are passed.
 *
 *  Returns: TRUE if object is Ok and can be removed from its original child
 *           object DLIST.
 *
 */
static BOOLEAN validate_os2_compatibility_objects( void     *Object,
                                                   TAG       ObjectTag,
                                                   uint      ObjectSize,
                                                   void     *ObjectHandle,
                                                   void     *Parameters,
                                                   BOOLEAN  *FreeMemory,
                                                   uint     *Error )
{
    storage_object_t *obj         = ( storage_object_t * )Object;
    dlist_t    valid_object_list  = ( dlist_t )Parameters;
    BOOLEAN    rc;
    void      *handle;

    LOGENTRY();

    *FreeMemory = FALSE;           /* tells dlist not to free any memory */
    *Error      = DLIST_SUCCESS;   /* tells dlist we were successful     */
    rc          = FALSE;           /* tells dlist not to prune item from list */


    if (( obj->object_type == SEGMENT ) &&
        ( obj->data_type   == DATA_TYPE )) {

        if ( isa_os2_compatibility_segment( obj ) == TRUE ) {

            *Error = InsertObject( valid_object_list,
                                   sizeof( storage_object_t ),
                                   Object,
                                   SEGMENT_TAG,
                                   NULL,
                                   InsertAtStart,
                                   0,
                                   ( void** )&handle );

            LOG_DEBUG( "return code from InsertObject == %d \n", *Error );
            if ( *Error == DLIST_SUCCESS ) {
                rc = TRUE;   // yes ... dlist can prune object from list
            }
        }
    }

    LOGEXIT();
    return rc;
}


/*
 *  Function:  Discover_Compatibility_Volumes
 *
 *  Called by the engine with a DLIST of storage objects that it believes
 *   are OS/2 Compatibility Volumes.  Here we examine each object and
 *   determine if it is and create a new storage object ( OS/2 region )
 *   that consumes exactly one object on the input list.
 *
 */
static int Discover_Compatibility_Volumes( dlist_t input_objects,
                                           dlist_t output_objects )
{
    int       rc = 0;
    dlist_t   object_list = CreateList();
    uint      count;

    LOGENTRY();

    if ( object_list ) {

        // validate storage objects found in our discovery DLIST
        // place all valid os2 data segments in the ... valid_object_list
        rc = PruneList( input_objects, validate_os2_compatibility_objects, object_list );

        if ( rc == DLIST_SUCCESS ) {

            GetListSize( object_list, &count );
            LOG_DEFAULT("Found %d OS/2 Compatibility Segment(s)\n", count );

            if ( count ) {

                // consume segments ... turning them into regions
                rc = PruneList( object_list, build_os2_compatibility_region, output_objects );

                GetListSize( output_objects, &count );
                LOG_DEFAULT( "Created %d OS/2 Compatibility Volume(s)\n", count );

                if ( rc == DLIST_SUCCESS ) {
                    rc  = CopyList( output_objects, object_list, InsertAtStart );
                }
            }

            // discard working DLIST but do not free object memory
            DestroyList( &object_list, FALSE );

            if ( rc == DLIST_SUCCESS ) {
                rc = count;
            }
        }
        else {
            LOG_ERROR( "PruneList did not produce a valid object list, rc == %d\n", rc );
        }
    }
    else {
        LOG_ERROR( "failed to create Dlist for new OS/2 Compatibility region objects\n" );
        rc = ENOMEM;
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Function:  Discover_LVM_Volumes
 *
 *  Called by the engine to discover OS/2 LVM regions.  The objects
 *  in the DLIST have all been given a once over by the engine.
 *
 *  Our discovery process requires 4 steps.
 *
 *  Step 1 - validate the storage objects
 *
 *      - run integrity checks on the signature sector - crc, signature, etc.
 *      - prune good objects from original DLIST, placing them in the
 *        valid_object DLIST.
 *
 *  Step 2 - construct OS/2 LVM regions
 *
 *      - find a region we can construct from the objects found in
 *        the valid_object DLIST.
 *      - malloc a new storage object for the region + metadata
 *      - consume child objects belonging to this region from the
 *        valid_object DLIST.
 *      - place new drive link object in the os2_lvm DLIST, removing
 *        consumed child objects from the valid_object DLIST.
 *
 *  Step 3 - merge all DLIST objects back into the output_objects DLIST
 *
 *
 *  Upon Return: the original DLIST may contain 3 different objects.
 *
 *  (1) original objects that failed integrity checks
 *  (2) original objects that we could not use yet because we did not find all
 *      the child objects needed to construct a complete OS/2 LVM region
 *  (3) newly constructed OS/2 LVM regions that consumed some of the
 *      original child objects from the DLIST.
 *
 *  Best Case:  we return only OS/2 LVM region, consuming all the original
 *              child objects.
 *
 *  Worst Case: child objects were invalid due to I/O errors, or bad crc or something
 *              and they are all returned to the engine.
 *
 */
static int Discover_LVM_Volumes( dlist_t input_objects, dlist_t output_objects )
{
    int      rc;
    void    *handle;
    uint     count;
    storage_object_t *region   = NULL;

    dlist_t  valid_object_list = CreateList();
    dlist_t  os2_lvm_list      = CreateList();

    LOGENTRY();

    if (( valid_object_list != NULL ) &&
        ( os2_lvm_list   != NULL )) {

        // validate storage objects found in our discovery DLIST
        // place all valid OS/2 LVM segments in the ... valid_object_list
        rc = PruneList( input_objects, validate_os2_lvm_objects, valid_object_list );

        if ( rc == DLIST_SUCCESS ) {

            GetListSize( valid_object_list, &count );
            LOG_DEFAULT("Found %d OS/2 LVM Segment(s)\n", count );

            if ( count ) {

                // Identify Drive Link Objects that can be completely built from the
                // list of Valid drive link objects. Then build new parent drive link
                // objects for each Drive Link that can be completed.  Leave drive link
                // objects, belonging to incomplete drive link parents, in the valid
                // object list.
                rc = ForEachItem( valid_object_list, find_complete_os2_lvm_region, ( ADDRESS )&region, 1 );

                while ( region ) {

                    rc = validate_os2_lvm_region( region );

                    if ( rc == 0 ) {
                        // insert new ORM Parent Object into temp DLIST
                        rc = InsertObject( os2_lvm_list,
                                           sizeof( storage_object_t ),
                                           region,
                                           EVMS_OBJECT_TAG,
                                           NULL,
                                           InsertAtStart ,
                                           0,
                                           ( void** )&handle );


                        // consume child objects for this ORM LV by removing them from the valid object DLIST
                        if ( rc == DLIST_SUCCESS ) {

                            LOG_DEBUG( "Consuming drive links for region:  VolumeSN= %X ...\n", (( orm_private_data_t * )region->private_data )->Volume_Serial_Number );

                            rc = PruneList( valid_object_list, commit_os2_lvm_region, ( ADDRESS )region );
                        }
                    }
                    else {
                        LOG_ERROR( "Validate Region failed, deleting Region object\n" );
                        PruneList( valid_object_list, purge_os2_lvm_incomplete_region, ( ADDRESS )region );
                        free_os2_region( region );
                    }

                    region = NULL;

                    rc = ForEachItem( valid_object_list, find_complete_os2_lvm_region, ( ADDRESS )&region, 1 );
                }

                // Success, at this point we either have completed all our drive
                // links or we have a combination of:
                //
                // 1 - new drive link storage objects we created
                // 2 - valid child drive link objects whose parent could not be
                //     completed ... YET.
                //
                // RC depends on how many new objects are created
                //

                rc += CopyList( output_objects, os2_lvm_list, InsertAtStart );

                if ( rc == DLIST_SUCCESS ) {

                    if ( GetListSize( os2_lvm_list, &count )) {
                        rc = 0;
                    }
                    else {
                        rc = count;
                        LOG_DEFAULT( "Created %d OS/2 LVM Volume(s)\n", count );
                    }
                }
            }

            // Put objects we did not use on the output list...
            CopyList( output_objects, valid_object_list, InsertAtStart );
        }
        else {
            LOG_ERROR( "PruneList did not produce a valid object list, rc == %d\n", rc );
        }
        DestroyList( &valid_object_list, FALSE );
        DestroyList( &os2_lvm_list, FALSE );
    }
    else {
        LOG_ERROR( "failed to create Dlist or Temp Space for new OS/2 LVM region objects\n" );
        rc = ENOMEM;
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Function:  orm_discover
 *
 *  Called by the engine to discover OS/2 Compatibility Volumes and LVM Volumes.
 *
 *  Our discovery process requires 2 steps.
 *
 *  Step 1 - validate the storage objects and discover Compatibility Volumes
 *
 *      - for every data segment
 *      - ask the segment manager for info and see if the segment is an OS/2
 *        Compatiblilty Volume and consume the segment, producing an ORM
 *        region out of it.
 *
 *  Step 2 - Validate and discover OS/2 LVM Volumes
 *
 *      - for every data segment
 *      - ask the segment manager for info and see if the seg is part of an OS/2
 *        LVM volume.
 *      - find all the segs to make the region and consume them, producing an
 *        ORM region object.
 *
 */
int orm_discover( dlist_t input_objects,
                  dlist_t output_objects,
                  BOOLEAN FinalCall )
{
    int rc = EINVAL, count = 0;

    LOGENTRY();

    // NEGATIVE number means failure
    // ZERO or POSITIVE number means success; the rc then is the number of objects created.
    rc = Discover_Compatibility_Volumes( input_objects, output_objects );

    count = rc;  // save number of objects created

    if ( rc >= 0 ) {

        // NEGATIVE number means failure
        // ZERO or POSITIVE number means success; the rc then is the number of objects created.
        rc = Discover_LVM_Volumes( input_objects, output_objects );

        if ( rc >= 0 ) {
            count += rc;  // total up number of objects created
        }
        else {
            count = rc;   // return errors
        }
    }

    // DLM ... copy whatever remains in the input_object list to the output_object list
    rc  = CopyList( output_objects, input_objects, InsertAtStart );

    LOGEXITRC();
    return count;
}

