/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_pvscan.c
 *
 *	Emulates LVM's 'pvscan' utility using the EVMS Engine. All options
 *	and several status messages are based on the original pvscan command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 *
 *	Not yet implimented:
 *		used size accounting - waiting on GetExtendedInfo functionality
 *		-e,-u - waiting on GetExtendedInfo functionality
 *		-vv - not sure yet
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <frontend.h>

typedef struct cmd_options_s {
	int debug;
	int exported;
	int help;
	int no_vgs;
	int shortdisp;
	int uuid;
	int verbose;
	int version;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

#include "helpers/get_lvmregmgr.c"
#include "helpers/open_engine.c"
#include "helpers/print_size.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	return 0;
}


static int showhelp( void )
{
	showheader();
	printf("\n");
	printf("%s - scan for physical volumes\n\n", cmd);
	printf("%s [-d|--debug] {[-e|--exported] | [-n|--novolumegroup]} [-h|--help]\n", cmd);
	printf("\t[-s|--short] [-u|--uuid] [-v[v]|--verbose [--verbose]] [-V|--version]\n");
	return 0;
}


static int pvscan_end(	handle_array_t		* a,
			handle_object_info_t	* b,
			handle_object_info_t	* c )
{
	if (a) evms_free(a);
	if (b) evms_free(b);
	if (c) evms_free(c);
	evms_close_engine();
	return 0;
}


static int parse_options( int		argc,
			char		** argv )
{
	int		c;
	char		* short_opts = "deh?nsuvV";
	struct option	long_opts[] = {
				{ "debug",		no_argument, NULL, 'd'},
				{ "exported",		no_argument, NULL, 'e'},
				{ "help",		no_argument, NULL, 'h'},
				{ "novolumegroup",	no_argument, NULL, 'n'},
				{ "short",		no_argument, NULL, 's'},
				{ "uuid",		no_argument, NULL, 'u'},
				{ "verbose",		no_argument, NULL, 'v'},
				{ "version",		no_argument, NULL, 'V'},
				{ NULL, 0, NULL, 0} };

	while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'd':
			opts.debug++;
			opts.verbose++;
			break;
		case 'e':
			opts.exported++;
			break;
		case 'h':
		case '?':
			opts.help++;
			break;
		case 'n':
			opts.no_vgs++;
			break;
		case 's':
			opts.shortdisp++;
			break;
		case 'u':
			opts.uuid++;
			break;
		case 'v':
			opts.verbose++;
			break;
		case 'V':
			opts.version++;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	if ( opts.exported && opts.no_vgs ) {
		printf("%s -- option e and n incompatible\n", cmd);
		return EINVAL;
	}

	return 0;
}


int main( int argc, char * argv[] )
{
	object_handle_t		lvmregmgr;
	handle_array_t		* segment_array = NULL;
	handle_object_info_t	* segment_info = NULL;
	handle_object_info_t	* container_info = NULL;
	u_int64_t		current_size = 0;
	u_int64_t		total_size = 0;
	u_int64_t		alloced_size = 0;
	u_int64_t		no_vg_size = 0;
	int			num_pv = 0;
	int			num_alloced = 0;
	int			num_no_vg = 0;
	int			log_level = DEFAULT;
	int			rc, i;
	char			*str1, *str2, *str3;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}

	printf("%s -- reading all physical volumes (this may take a while...)\n", cmd);    

	// Open the engine.
	rc = open_engine(ENGINE_READWRITE, log_level);
	if (rc) {
		return rc;
	}

	// Find the handle for the LVM region manager.
	rc = get_lvmregmgr(&lvmregmgr);
	if (rc) {
		pvscan_end(segment_array, segment_info, container_info);
		return rc;
	}

	// Get the list of all available disks, segments, and regions.
	if ( opts.verbose ) {
		printf("%s -- Getting list of objects from the Engine\n", cmd);
	}
	rc = evms_get_object_list(DISK|SEGMENT|REGION, DATA_TYPE, 0, 0, &segment_array);
	if (rc) {
		printf("%s -- Error getting list of objects from the Engine (%d)\n", cmd, rc);
		pvscan_end(segment_array, segment_info, container_info);
		return rc;
	}
	if ( opts.verbose ) {
		printf("%s -- Got list of objects from the Engine\n", cmd);
	}
	if ( !opts.shortdisp || opts.verbose ) {
		printf("%s -- walking through all physical volumes found\n", cmd);
	}
	if ( opts.no_vgs ) {
		printf("%s -- WARNING! only displaying physical volumes in no volume group\n", cmd);
	}

	// Examine every segment in the list.
	for ( i = 0; i < segment_array->count; i++ ) {

		// Get the info for this segment.
		if ( opts.verbose ) {
			printf("%s -- Getting info for handle %u\n", cmd, segment_array->handle[i]);
		}
		rc = evms_get_info(segment_array->handle[i], &segment_info);
		if (rc) {
			printf("%s -- Error getting info for handle %u (%d)\n", cmd, segment_array->handle[i], rc);
			pvscan_end(segment_array, segment_info, container_info);
			return rc;
		}
		current_size = segment_info->info.segment.size;

		// Is this object in a container?
		if ( ! segment_info->info.segment.consuming_container ) {

			// Does this object have any parent objects?
			if ( segment_info->info.segment.parent_objects->count ) {
				if ( ! opts.shortdisp ) {
					printf("%s -- Unavailable object \"%s\" [%s]\n",
						cmd, segment_info->info.segment.name, str1 = print_size(current_size,0));
					free(str1);
				}
			}
			else {
				total_size += current_size;
				no_vg_size += current_size;
				num_pv++;
				num_no_vg++;
				if ( ! opts.shortdisp ) {
					printf("%s -- Available object \"%s\" [%s]\n",
						cmd, segment_info->info.segment.name, str1 = print_size(current_size,0));
					free(str1);
				}
			}
		}
		else {
			// Get the info for this segment's container.
			if ( opts.verbose ) {
				printf("%s -- Getting info for container handle %u\n", cmd, segment_info->info.segment.consuming_container);
			}
			rc = evms_get_info(segment_info->info.segment.consuming_container, &container_info);
			if (rc) {
				printf("%s -- Error getting info for container handle %u (%d)\n", cmd, segment_info->info.segment.consuming_container, rc);
				pvscan_end(segment_array, segment_info, container_info);
				return rc;
			}

			// Record stats and display info about this PV.
			total_size += current_size;
			alloced_size += current_size;
			num_pv++;
			num_alloced++;
			if ( !opts.shortdisp && !opts.no_vgs ) {
				printf("%s -- ACTIVE PV \"%s\" of VG \"%s\" [%s]\n", cmd, segment_info->info.segment.name,
					container_info->info.container.name, str1 = print_size(current_size,0));
				free(str1);
			}

			evms_free(container_info);
		}

		evms_free(segment_info);
		segment_info = container_info = NULL;
	}

	// Display summary for all PVs.
	printf("%s -- total: %d [%s] / in use: %d [%s] / in no VG: %d [%s]\n", cmd, num_pv, str1 = print_size(total_size,0),
		num_alloced, str2 = print_size(alloced_size,0), num_no_vg, str3 = print_size(no_vg_size,0));
	free(str1);
	free(str2);
	free(str3);

	pvscan_end(segment_array, segment_info, container_info);
	return 0;
}

