/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils/helpers
 * File: revert_volume.c
 */

#ifndef _LVM_HELP_REVERT_VOLUME_
#define _LVM_HELP_REVERT_VOLUME_ 1

/* Function: revert_volume_from_segment
 *
 *	Check the specified segment handle to see if it has any parent
 *	objects. If so, then this segment is being used by another plugin
 *	and cannot be used yet by LVM. If it has no parents, check to see
 *	if it is a compatibility volume. If so, revert the volume.
 */
int revert_volume_from_segment( object_handle_t segment_handle )
{
	handle_object_info_t	* segment_info;
	int			rc;

	// Get the info for this segment.
	rc = evms_get_info(segment_handle, &segment_info);
	if (rc) {
		printf("%s -- Error getting info for segment handle %u (%d)\n", cmd, segment_handle, rc);
		return rc;
	}

	// Check for parent objects.
	if ( segment_info->info.segment.parent_objects->count ) {
		printf("%s -- Segment %s is currently in use.\n", cmd, segment_info->info.segment.name);
		printf("%s -- Please remove it from its current location first.\n", cmd);
		evms_free(segment_info);
		return EINVAL;
	}

	// Revert compatibility volume if necessary.
	if ( segment_info->info.segment.volume ) {
		rc = evms_remove(segment_info->info.segment.volume);
		if (rc) {
			printf("%s -- Error removing compatibility volume from segment %s (%d)\n", cmd, segment_info->info.segment.name, rc);
			evms_free(segment_info);
			return rc;
		}
	}

	evms_free(segment_info);
	return 0;
}


/* Function: revert_volume_from_region
 *
 *	Check to see if there is a compatibility volume on top of this
 *	region. If there is, and the volume points directly at the region,
 *	then the volume must be reverted before the region can be reverted.
 */
int revert_volume_from_region( object_handle_t region_handle )
{
	object_handle_t		volume_handle;
	handle_object_info_t	* region_info;
	handle_object_info_t	* volume_info;
	int			rc;

	// Get the basic info for this region
	rc = evms_get_info(region_handle, &region_info);
	if (rc) {
		printf("%s -- Error getting info for region %d (%d)\n", cmd, region_handle, rc);
		return rc;
	}

	// Is there a volume present?
	volume_handle = region_info->info.region.volume;
	if ( volume_handle ) {
		// Get the basic info for this volume
		rc = evms_get_info(volume_handle, &volume_info);
		if (rc) {
			printf("%s -- Error getting volume info for region %s (%d)\n", cmd, region_info->info.region.name, rc);
			evms_free(region_info);
			return rc;
		}

		// If the volume does not point directly at this region, then
		// it is not a compatibility volume, and should not be deleted.
		// Must use the GUI in this case, since there is other EVMS
		// stuff on top of this region.
		if ( volume_info->info.volume.object != region_handle ) {
			printf("%s -- Region %s is not a top-level object. Cannot remove.\n", cmd, region_info->info.region.name);
			evms_free(region_info);
			evms_free(volume_info);
			return EINVAL;
		}

		// Otherwise, this is a compatibility volume. Use the destroy
		// command to delete both the volume and the region.
		rc = evms_destroy(volume_handle);
		if (rc) {
			printf("%s -- Error destroying volume %s (%d)\n", cmd, volume_info->info.volume.name, rc);
			evms_free(region_info);
			evms_free(volume_info);
			return rc;
		}
		evms_free(volume_info);
	}

	// No volume. Just remove the region.
	else {
		rc = evms_remove(region_handle);
		if (rc) {
			printf("%s -- Error removing region %s (%d)\n", cmd, region_info->info.region.name, rc);
			evms_free(region_info);
			return rc;
		}
	}

	evms_free(region_info);
	return 0;
}

#endif

