/*----------------------------------------------------------------------
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: control.c
 *---------------------------------------------------------------------*/

/*----------------------------------------------------------------------
 * Change History:
 *
 * 11/2001  John Stiles   Initial version.
 *
 *---------------------------------------------------------------------*/


/* Identify this file. */
#define CONTROL_C    1

/*----------------------------------------------------------------------
 * Necessary include files
 *---------------------------------------------------------------------*/
#include <sys/types.h>
#include <string.h>                    /* strcat, strcpy                   */
#include <ctype.h>                     /* toupper                          */
#include <stdio.h>                     /* printf                           */
#include <stdlib.h>                    /* malloc, free                     */

#include "evmsn.h"

/*----------------------------------------------------------------------
 * Global Variables
 *---------------------------------------------------------------------*/

static char work_buffer[] = "                                                                          ";


/*----------------------------------------------------------------------
 * There are No Private Constants
 * There are No Private Type Definitions
 * There are No Private Global Variables.
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * Local Functions
 *---------------------------------------------------------------------*/

int display_viewmenu( void )
{
  int i, j, keyval = 0, start_index = 0, select_index = 0, num_objects = 7,
      screen_cols, screen_lines = CALC_SCREEN3_LINES;
  char *holdbuf;

  draw_box3( _( " Select a View: " ));

  // calculate globals and get storage for buffer
  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;

  holdbuf = malloc( screen_cols + 3 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  do {

    // first blank the lines
    BLANK_BUFFER( holdbuf, screen_cols + 2 )
    for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
      print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

    i = YPOS_BOX3TOP + 2;

    for ( i = 0, j = YPOS_BOX3TOP + 2; i < num_objects; i++, j++ ) {

      BLANK_BUFFER( holdbuf, screen_cols )

      switch ( i ) {
        case 0 :
          memcpy( holdbuf, title_volumes, strlen( title_volumes ));
          break;
        case 1 :
          memcpy( holdbuf, title_availobjs, strlen( title_availobjs ));
          break;
        case 2 :
          memcpy( holdbuf, title_regions, strlen( title_regions ));
          break;
        case 3 :
          memcpy( holdbuf, title_containers, strlen( title_containers ));
          break;
        case 4 :
          memcpy( holdbuf, title_segments, strlen( title_segments ));
          break;
        case 5 :
          memcpy( holdbuf, title_disks, strlen( title_disks ));
          break;
        default:
          memcpy( holdbuf, title_plugins, strlen( title_plugins ));
          break;
      }

      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, j, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, j, COLOR_TITLE, holdbuf );
    }

    draw_status_line( "Select a view and press enter." );

    keyval = key_scroller( &start_index, &select_index, num_objects, screen_lines );

    switch ( keyval ) {

      case KEY_ESC :
        break;

      case KEY_F1  :
      case '1'     :
        draw_help_panel( 5 );
        keyval = KEY_CURSORACT;
        break;

      case KEY_F2  :
      case '2'     :
        keyval = KEY_VOLUMES;
        break;

      case KEY_F5  :
      case '5'     :
        keyval = KEY_COMMIT;
        break;

      case KEY_F10 :
      case '0'     :
        keyval = KEY_EXITPGM;
        break;

      case KEY_ENTER :
        switch ( select_index ) {
          case 1  :
            keyval = KEY_OBJECTS;
            break;
          case 2  :
            keyval = KEY_REGIONS;
            break;
          case 3  :
            keyval = KEY_CONTAINERS;
            break;
          case 4  :
            keyval = KEY_SEGMENTS;
            break;
          case 5  :
            keyval = KEY_DISKS;
            break;
          case 6  :
            keyval = KEY_PLUGINS;
            break;
          default:
            keyval = KEY_VOLUMES;
            break;
        }
        break;

      default:
        keyval = KEY_CURSORACT;
        break;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return keyval;
}


int display_optsmenu( void )
{
  int i, j, keyval = 0, start_index = 0, select_index = 0, num_objects = 1,
      screen_cols, screen_lines = CALC_SCREEN3_LINES;
  char *holdbuf;

  draw_box3( _( " Select an Option: " ));

  // calculate globals and get storage for buffer
  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;

  holdbuf = malloc( screen_cols + 3 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  do {

    // first blank the lines
    BLANK_BUFFER( holdbuf, screen_cols + 2 )
    for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
      print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

    i = YPOS_BOX3TOP + 2;

    for ( i = 0, j = YPOS_BOX3TOP + 2; i < num_objects; i++, j++ ) {

      BLANK_BUFFER( holdbuf, screen_cols )

      switch ( i ) {
        case 0 :
          memcpy( holdbuf, title_nooptions, strlen( title_nooptions ));
          break;
        default:
          break;
      }

      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, j, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, j, COLOR_TITLE, holdbuf );
    }

    draw_status_line( "Select an option to change and press enter." );

    keyval = key_scroller( &start_index, &select_index, num_objects, screen_lines );

    switch ( keyval ) {

      case KEY_ESC :
        break;

      case KEY_F1  :
      case '1'     :
        draw_help_panel( 5 );
        keyval = KEY_CURSORACT;
        break;

      case KEY_F2  :
      case '2'     :
        keyval = KEY_VOLUMES;
        break;

      case KEY_F4  :
      case '4'     :
        keyval = KEY_VIEWMENU;
        break;

      case KEY_F10 :
      case '0'     :
        keyval = KEY_EXITPGM;
        break;

      case KEY_ENTER :
        switch ( select_index ) {
          case 1  :
            break;
          default:
            keyval = KEY_ESC;
            break;
        }
        break;

      default:
        keyval = KEY_CURSORACT;
        break;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return keyval;
}


int commit_changes( int exit_pgm )
{
  int rc = 0;

  if ( evms_changes_pending() ) {
    rc = select_box( _( "Unsaved changes were detected." ),
                     _( "Do not commit changes" ),
                     _( "Commit changes" ),
                     _( "Return to program" ));

    switch ( rc ) {
      case 1 :
        if (exit_pgm) {
            rc = KEY_EXITPGM;
            draw_status_line( _( "Exiting..." ));
        } else {
            rc = 0;
        }    
        break;
      case 2 :
        draw_status_line ( _("Committing changes..."));
      
        rc = evms_commit_changes(NULL);
        if ( rc ) {
          sprintf( work_buffer, "evms_commit_changes() failed, rc == %i, press any key to continue.", rc );
          log_error( "%s: %s\n", __FUNCTION__, work_buffer );
          draw_status_line( work_buffer );
        }
        else
        {
          draw_status_line( _( "All changes were committed successfully. Press any key to continue." )); 
        }
        
        get_inputc();
        
        if (exit_pgm) {
            rc = KEY_EXITPGM;
            draw_status_line( _( "Exiting..." ));            
        }
        break;
      default:
        rc = 0;
        break;
    }
  }
  else
  {
      if (exit_pgm) {
          rc = KEY_EXITPGM;
          draw_status_line( _( "Exiting..." ));
      }
      else
      {
          draw_status_line( _( "Nothing to commit. Press any key to continue." ));
          get_inputc ();
      }
  }

  return rc;
}


/*----------------------------------------------------------------------
 * Public Functions
 *---------------------------------------------------------------------*/

int key_scroller( int *start_index, int *select_index, int object_count, int display_lines )
{
  int start_object = *start_index;
  int selected_object = *select_index;
  int key_value = get_inputc();

  switch ( key_value ) {

    case KEY_CURSORUP :
      if ( selected_object > 0 ) {
        selected_object--;
        if ( selected_object < start_object )
          start_object = selected_object;

        *start_index = start_object;
        *select_index = selected_object;
      }

      return KEY_CURSORACT;
      break;

    case KEY_CURSORDOWN :
    case KEY_TAB :
      if ( object_count && selected_object != object_count - 1 ) {
        selected_object++;
        if ( selected_object - start_object > display_lines )
          start_object++;

        *start_index = start_object;
        *select_index = selected_object;
      }

      return KEY_CURSORACT;
      break;

    case KEY_HOME :
      if ( selected_object > 0 ) {
        selected_object = 0;
        start_object = 0;

        *start_index = start_object;
        *select_index = selected_object;
      }

      return KEY_CURSORACT;
      break;

    case KEY_END :
      if ( object_count && selected_object != object_count - 1 ) {
        selected_object = object_count - 1;
        if ( object_count >= start_object + display_lines )
          start_object = object_count - display_lines - 1;

        *start_index = start_object;
        *select_index = selected_object;
      }

      return KEY_CURSORACT;
      break;

    case KEY_PAGEUP :
      if ( selected_object > 0 ) {
        if ( start_object == 0 )
          selected_object = 0;
        else {
          if ( display_lines > start_object )
            start_object = 0;
          else
            start_object -= display_lines;
        }
        if ( selected_object >= start_object + display_lines )
          selected_object = start_object;

        *start_index = start_object;
        *select_index = selected_object;
      }

      return KEY_CURSORACT;
      break;

    case KEY_PAGEDOWN :
      if ( object_count && selected_object != object_count - 1 ) {
        if ( object_count >= start_object + display_lines )
          start_object += display_lines;
        else
          selected_object = object_count - 1;

        if ( selected_object < start_object )
          selected_object = start_object;

        *start_index = start_object;
        *select_index = selected_object;
      }

      return KEY_CURSORACT;
      break;

    default:
      break;
  }

  return key_value;
}


int screen_scroller( int *start_index, int object_count, int display_lines )
{
  int start_object = *start_index;
  int key_value = get_inputc();

  switch ( key_value ) {

    case KEY_CURSORUP :
      if ( start_object > 0 ) {
        start_object--;

        *start_index = start_object;
      }

      return KEY_CURSORACT;
      break;

    case KEY_CURSORDOWN :
    case KEY_TAB :
      if ( object_count && ( object_count - start_object > display_lines )) {
        start_object++;

        *start_index = start_object;
      }

      return KEY_CURSORACT;
      break;

    case KEY_HOME :
      if ( start_object > 0 ) {
        start_object = 0;

        *start_index = start_object;
      }

      return KEY_CURSORACT;
      break;

    case KEY_END :
      if ( object_count && ( object_count - start_object > display_lines )) {
        start_object = object_count - display_lines;

        *start_index = start_object;
      }

      return KEY_CURSORACT;
      break;

    case KEY_PAGEUP :
      if ( start_object > 0 ) {
        if ( display_lines > start_object )
          start_object = 0;
        else
          start_object -= display_lines - 1;

        *start_index = start_object;
      }

      return KEY_CURSORACT;
      break;

    case KEY_PAGEDOWN :
      if ( object_count && ( object_count - start_object > display_lines )) {
        if ( object_count > start_object + 2 * display_lines )
          start_object += display_lines - 1;
        else
          start_object = object_count - display_lines;

        *start_index = start_object;
      }

      return KEY_CURSORACT;
      break;

    default:
      break;
  }

  return key_value;
}


int select_box( char *message, char *sel1, char *sel2, char *sel3 )
{
  int i, j, rc = 0, keyval = 0, start_index = 0, select_index = 0, item_count,
      screen_cols, screen_lines = CALC_SCREEN4_LINES;
  char *holdbuf;

  screen_cols = GL_screen_cols - XPOS_BOX4TEXT - XPOS_BOX4RIGHT - 1;

  holdbuf = malloc( screen_cols + 3 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  draw_box4( NULL );

  do {

    // first blank the lines
    BLANK_BUFFER( holdbuf, screen_cols + 2 )
    for ( i = YPOS_BOX4TOP + 1; i < YPOS_BOX4TOP + 2 + screen_lines; i++ )
      print_string( XPOS_BOX4TEXT - 1, i, COLOR_DEFAULT, holdbuf );

    BLANK_BUFFER( holdbuf, screen_cols - 1 )
    memcpy( holdbuf, message, strlen( message ));
    print_string( XPOS_BOX4TEXT, YPOS_BOX4TEXT, COLOR_TITLE, holdbuf );

    i = YPOS_BOX4TOP + 2;
    item_count = 0;

    for ( i = 0, j = YPOS_BOX4TEXT + 2; i < 3; i++, j++ ) {

      BLANK_BUFFER( holdbuf, screen_cols - 1 )

      switch ( i ) {
        case 1 :
          if ( sel2 ) {
            memcpy( holdbuf, sel2, strlen( sel2 ));
            item_count++;
          }
          break;
        case 2 :
          if ( sel3 ) {
            memcpy( holdbuf, sel3, strlen( sel3 ));
            item_count++;
          }
          break;
        default:
          memcpy( holdbuf, sel1, strlen( sel1 ));
          item_count++;
          break;
      }

      if ( select_index == i )
        print_string( XPOS_BOX4TEXT, j, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX4TEXT, j, COLOR_TITLE, holdbuf );
    }

    draw_status_line( "Select a choice and press enter." );

    keyval = key_scroller( &start_index, &select_index, item_count, screen_lines );

    switch ( keyval ) {

      case KEY_ESC :
        break;

      case KEY_F1  :
      case '1'     :
        draw_help_panel( 5 );
        keyval = KEY_CURSORACT;
        break;

      case KEY_ENTER :
        switch ( select_index ) {
          case 0  :
            rc = 1;
            break;
          case 1  :
            rc = 2;
            break;
          case 2  :
            rc = 3;
            break;
          case 3  :
            rc = 4;
            break;
          default:
            break;
        }
        break;

      default:
        keyval = KEY_CURSORACT;
        break;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return rc;
}


int naming_box( char *message, int namelen, char *return_name )
{
  int i, rc = 0, screen_cols, screen_lines = CALC_SCREEN4_LINES;
  char *holdbuf;

  screen_cols = GL_screen_cols - XPOS_BOX4TEXT - XPOS_BOX4RIGHT - 1;

  holdbuf = malloc( screen_cols + 3 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  if ( namelen > screen_cols - 2 )
    namelen = screen_cols - 2;

  draw_box4( NULL );

  // first blank the lines
  BLANK_BUFFER( holdbuf, screen_cols + 2 )
  for ( i = YPOS_BOX4TOP + 1; i < YPOS_BOX4TOP + 2 + screen_lines; i++ )
    print_string( XPOS_BOX4TEXT - 1, i, COLOR_DEFAULT, holdbuf );

  BLANK_BUFFER( holdbuf, screen_cols - 1 )
  memcpy( holdbuf, message, strlen( message ));
  print_string( XPOS_BOX4TEXT, YPOS_BOX4TEXT, COLOR_TITLE, holdbuf );

  BLANK_BUFFER( holdbuf, screen_cols - 1 )
  holdbuf[0] = ':';
  holdbuf[1] = ':';
  memcpy( holdbuf + 2, return_name, namelen );
  print_string( XPOS_BOX4TEXT, YPOS_BOX4TEXT + 2, COLOR_TITLE, holdbuf );

  rc = text_entry( XPOS_BOX4TEXT + 2, YPOS_BOX4TEXT + 2, namelen, return_name );

  free( holdbuf );
  return rc;
}


int draw_actions2( object_handle_t object )
{
  int i, rc = 0, screen_cols, screen_lines = CALC_SCREEN3_LINES;
  char *holdbuf;

  if ( GL_screen_context == SC_PLUGINS )
    draw_box3( title_plugin_info );
  else
    draw_box3( title_actions );

  // first blank the lines
  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT;
  holdbuf = malloc( screen_cols + 2 );
  if ( holdbuf ) {
    BLANK_BUFFER( holdbuf, screen_cols )
    for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
      print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );
  }
  else {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return rc;
  }

  switch ( GL_screen_context ) {

    case SC_VOLUMES :

      // volume actions

      rc = volume_actions( object );
      break;

    case SC_OBJECTS :

      // topmost objects actions

      rc = topmost_actions( object );
      break;

    case SC_REGIONS :

      // region actions

      rc = region_actions( object );
      break;

    case SC_CONTAINERS :

      // container actions

      rc = container_actions( object );
      break;

    case SC_SEGMENTS :

      // segment actions

      rc = segment_actions( object );
      break;

    case SC_DISKS :

      // disk actions

      rc = disk_actions( object );
      break;

    case SC_PLUGINS :

      // Just display the plugin information

      rc = plugin_information( object );
      break;

    default:
      break;
  }     /* End switch*/

  free( holdbuf );
  return rc;
}


int draw_screen2( int *select_index, handle_array_t *top_objects )
{
  int i, rc = 0, key_value = 0, start_index = 0, screen_cols,
      screen_lines = CALC_SCREEN2_LINES;
  char *holdbuf;

  if ( top_objects == NULL )                               /* nothing to do*/
    return rc;

  // calculate globals and get storage for buffer
  if ( *select_index >= start_index + screen_lines - 2 )
    start_index = *select_index - ( screen_lines - 2 );

  screen_cols = GL_screen_cols - XPOS_BOX2TEXT - XPOS_BOX2RIGHT;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return rc;
  }

  do {

    // first blank the lines
    BLANK_BUFFER( holdbuf, screen_cols )
    for ( i = YPOS_BOX2TOP + 1; i < YPOS_BOX2TOP + 2 + screen_lines; i++ )
      print_string( XPOS_BOX2TEXT, i, COLOR_DEFAULT, holdbuf );

    switch ( GL_screen_context ) {

      case SC_VOLUMES :

        // volume view

        rc = draw_volumes( start_index, select_index, top_objects );
        break;

      case SC_OBJECTS :
        // topmost objects view

        rc = draw_topmost( start_index, select_index, top_objects );
        break;

      case SC_REGIONS :

        // regions view

        rc = draw_regions( start_index, select_index, top_objects );
        break;

      case SC_CONTAINERS :

        // containers view

        rc = draw_containers( start_index, select_index, top_objects );
        break;

      case SC_SEGMENTS :

        // segments view

        rc = draw_segments( start_index, select_index, top_objects );
        break;

      case SC_DISKS :
        // disks view

        rc = draw_disks( start_index, select_index, top_objects );
        break;

      case SC_PLUGINS :
        // plugin view

        rc = draw_plugins( start_index, select_index, top_objects );
        break;

      default:
          rc = 1;
        break;
    }                                                         /* End switch*/

    if ( top_objects->count && rc == 0 )
      key_value = key_scroller( &start_index, select_index, top_objects->count, screen_lines - 2 );
    else
      key_value = get_inputc();

  }  while ( key_value == KEY_CURSORACT );

  free( holdbuf );
  return key_value;
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  control_loop
 *
 *   Descriptive Name:  main control loop
 *
 *   Input:   int key_value == the value of the beginning keystroke
 *
 *   Output:  int == the keystroke to return ( usually exit or select / options ).
 *
 *   Error Handling:  If an error occurs, all memory allocated by this
 *                     program is freed, and an error message is output
 *                     to the user.
 *
 *   Side Effects:  The state of the disks, partitions, and/or volumes
 *                   in the system may be altered.  Nodes may be
 *                   created or removed from the /dev tree.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
int control_loop( int key_value )
{
  int  rc = 0, selected_object = 0, keyval = key_value,
       screen_lines = CALC_SCREEN2_LINES - 2;

  handle_array_t *top_objects = NULL;

  // Do forever until quit key pressed
  do {

    rc = 0;

    switch ( keyval ) {

      case KEY_RESIZE :
        update_screen_info();                    /* pick up new dimensions.*/
        draw_frame();
        screen_lines = CALC_SCREEN2_LINES - 2;
        draw_fnkeys( KEYS_DEFAULT );
        // draw_status_line( _( "Resize event." ));
        // rc = get_inputc();
        rc = 0;
        break;

      case KEY_F1  :
      case '1'     :
        draw_help_panel( 5 );
        // clear the screen to get around Gnome terminal bug
        draw_frame();
        draw_fnkeys( KEYS_DEFAULT );
        break;

      case KEY_F2  :
      case KEY_VOLUMES  :
      case '2'     :
        if ( GL_screen_context != SC_VOLUMES ) {

          if ( top_objects != NULL ) {
            evms_free( top_objects );
            top_objects = NULL;
          }

          GL_screen_context = SC_VOLUMES;
          selected_object = 0;
          // This has the effect of clearing the screen
          draw_frame();
          draw_fnkeys( KEYS_DEFAULT );

          rc = evms_get_volume_list( 0, &top_objects );
          if ( rc == 0 ) {
            log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_vollist, top_objects->count, info_msg_entries );
          }
          else {
            sprintf( work_buffer, _( "%s %i." ), err_msg_get_vollist, rc );
            log_error ( "%s: %s\n", __FUNCTION__, work_buffer );
            draw_status_line( work_buffer );
            rc = 0;
          }
        }
        break;

      case KEY_OBJECTS  :
        if ( GL_screen_context != SC_OBJECTS ) {

          if ( top_objects != NULL ) {
            evms_free( top_objects );
            top_objects = NULL;
          }

          GL_screen_context = SC_OBJECTS;
          selected_object = 0;
          // This has the effect of clearing the screen
          draw_frame();
          draw_fnkeys( KEYS_DEFAULT );

          rc = evms_get_object_list( 0, DATA_TYPE, 0, TOPMOST, &top_objects );
          if ( rc == 0 ) {
            log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_objlist, top_objects->count, info_msg_entries );
          }
          else {
            sprintf( work_buffer, _( "%s %i." ), err_msg_get_objlist, rc );
            log_error ( "%s: %s\n", __FUNCTION__, work_buffer );
            draw_status_line( work_buffer );
            rc = 0;
          }
        }
        break;

      case KEY_REGIONS  :
        if ( GL_screen_context != SC_REGIONS ) {

          if ( top_objects != NULL ) {
            evms_free( top_objects );
            top_objects = NULL;
          }

          GL_screen_context = SC_REGIONS;
          selected_object = 0;
          // This has the effect of clearing the screen
          draw_frame();
          draw_fnkeys( KEYS_DEFAULT );

          rc = evms_get_object_list( REGION, 0, 0, 0, &top_objects );
          if ( rc == 0 ) {
            log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_objlist, top_objects->count, info_msg_entries );
          }
          else {
            sprintf( work_buffer, _( "%s %i." ), err_msg_get_objlist, rc );
            log_error ( "%s: %s\n", __FUNCTION__, work_buffer );
            draw_status_line( work_buffer );
            rc = 0;
          }
        }
        break;

      case KEY_CONTAINERS  :
        if ( GL_screen_context != SC_CONTAINERS ) {

          if ( top_objects != NULL ) {
            evms_free( top_objects );
            top_objects = NULL;
          }

          GL_screen_context = SC_CONTAINERS;
          selected_object = 0;
          // This has the effect of clearing the screen
          draw_frame();
          draw_fnkeys( KEYS_DEFAULT );

          rc = evms_get_container_list( 0, &top_objects );
          if ( rc == 0 ) {
            log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_contlist, top_objects->count, info_msg_entries );
          }
          else {
            sprintf( work_buffer, _( "%s %i." ), err_msg_get_contlist, rc );
            log_error ( "%s: %s\n", __FUNCTION__, work_buffer );
            draw_status_line( work_buffer );
            rc = 0;
          }
        }
        break;

      case KEY_SEGMENTS  :
        if ( GL_screen_context != SC_SEGMENTS ) {

          if ( top_objects != NULL ) {
            evms_free( top_objects );
            top_objects = NULL;
          }

          GL_screen_context = SC_SEGMENTS;
          selected_object = 0;
          // This has the effect of clearing the screen
          draw_frame();
          draw_fnkeys( KEYS_DEFAULT );

          rc = evms_get_object_list( SEGMENT, 0, 0, 0, &top_objects );
          if ( rc == 0 ) {
            log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_objlist, top_objects->count, info_msg_entries );
          }
          else {
            sprintf( work_buffer, _( "%s %i." ), err_msg_get_objlist, rc );
            log_error ( "%s: %s\n", __FUNCTION__, work_buffer );
            draw_status_line( work_buffer );
            rc = 0;
          }
        }
        break;


      case KEY_DISKS  :
        if ( GL_screen_context != SC_DISKS ) {

          if ( top_objects != NULL ) {
            evms_free( top_objects );
            top_objects = NULL;
          }

          GL_screen_context = SC_DISKS;
          selected_object = 0;
          // This has the effect of clearing the screen
          draw_frame();
          draw_fnkeys( KEYS_DEFAULT );

          rc = evms_get_object_list( DISK, 0, 0, 0, &top_objects );
          if ( rc == 0 ) {
            log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_objlist, top_objects->count, info_msg_entries );
          }
          else {
            sprintf( work_buffer, _( "%s %i." ), err_msg_get_objlist, rc );
            log_error ( "%s: %s\n", __FUNCTION__, work_buffer );
            draw_status_line( work_buffer );
            rc = 0;
          }
        }
        break;

      case KEY_PLUGINS  :
        if ( GL_screen_context != SC_PLUGINS ) {

          if ( top_objects != NULL ) {
            evms_free( top_objects );
            top_objects = NULL;
          }

          GL_screen_context = SC_PLUGINS;
          selected_object = 0;
          // This has the effect of clearing the screen
          draw_frame();
          draw_fnkeys( KEYS_DEFAULT );

          rc = evms_get_plugin_list( EVMS_NO_PLUGIN, 0, &top_objects );
          if ( rc == 0 ) {
            log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_pluglist, top_objects->count, info_msg_entries );
          }
          else {
            sprintf( work_buffer, _( "%s %i." ), err_msg_get_pluglist, rc );
            log_error ( "%s: %s\n", __FUNCTION__, work_buffer );
            draw_status_line( work_buffer );
            rc = 0;
          }
        }
        break;

      case KEY_F4  :
      case KEY_VIEWMENU  :
      case '4'     :
        rc = display_viewmenu();
        // clear the screen to get around Gnome terminal bug
        draw_frame();
        draw_fnkeys( KEYS_DEFAULT );
        break;

      case KEY_F5  :
      case KEY_COMMIT  :
      case '5'     :
        rc = commit_changes(FALSE);
        // clear the screen to get around Gnome terminal bug
        draw_frame();
        draw_fnkeys( KEYS_DEFAULT );
        break;
      
      case KEY_F10 :
      case KEY_EXITPGM  :
      case '0'     :
        rc = commit_changes(TRUE);
        // clear the screen to get around Gnome terminal bug
        draw_frame();
        draw_fnkeys( KEYS_DEFAULT );
        break;

      case KEY_ENTER :
        if ( top_objects->count || GL_screen_context == SC_CONTAINERS ||
             GL_screen_context == SC_REGIONS || GL_screen_context == SC_OBJECTS ) {
          if ( top_objects->count )
            rc = draw_actions2( top_objects->handle[selected_object] );
          else
            rc = draw_actions2( 0 );
          // someone returning a selection key wants a refresh
          if ( rc >= KEY_VOLUMES && rc <= KEY_PLUGINS )
            GL_screen_context = SC_NONE;
          // clear the screen to get around Gnome terminal bug
          draw_frame();
          draw_fnkeys( KEYS_DEFAULT );
        }
        break;

      default:
        sprintf( work_buffer, " key 0x%x  ", keyval );
        draw_fnkeys( KEYS_DEFAULT );
        draw_status_line( work_buffer );
        break;
    }                                                       /* End switch*/

    if ( rc == 0 )
      keyval = draw_screen2( &selected_object, top_objects );
    else  if ( rc != KEY_EXITPGM ) {
      keyval = rc;
      rc = 0;
    }

  } while ( rc == 0 );

  if ( top_objects != NULL )
    evms_free( top_objects );

  GL_screen_context = SC_NONE;

  return rc;
}

