/*----------------------------------------------------------------------
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: volumes.c
 *---------------------------------------------------------------------*/

/*----------------------------------------------------------------------
 * Change History:
 *
 * 12/2001  John Stiles   Initial version.
 *
 *---------------------------------------------------------------------*/


/* Identify this file. */
#define VOLUMES_C    1

/*----------------------------------------------------------------------
 * Necessary include files
 *---------------------------------------------------------------------*/
#include <sys/types.h>
#include <string.h>                    /* strcat, strcpy                   */
#include <ctype.h>                     /* toupper                          */
#include <stdio.h>                     /* printf                           */
#include <stdlib.h>                    /* malloc, free                     */

#include "evmsn.h"

/*----------------------------------------------------------------------
 * Global Variables
 *---------------------------------------------------------------------*/

static char work_buffer[] = "                                                ";


/*----------------------------------------------------------------------
 * There are No Private Constants
 * There are No Private Type Definitions
 * There are No Private Global Variables.
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * Local Functions
 *---------------------------------------------------------------------*/

int set_volume_info( object_handle_t object )
{
  int rc = 0, screen_cols;
  char *holdbuf;

  screen_cols = GL_screen_cols - XPOS_BOX4TEXT - XPOS_BOX4RIGHT - 6;
  holdbuf = malloc( screen_cols + 1 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  BLANK_BUFFER( holdbuf, screen_cols )
  rc = naming_box( _( "Enter the New Volume Name: " ),
                   screen_cols, holdbuf );
  if ( rc == 0 ) {
    rc = evms_set_volume_name( object, holdbuf );
    if ( rc == 0 ) {
      draw_status_line( info_msg_set_name_ok );
    }
    else {
      sprintf( work_buffer, _( "Modify Volume Properties failed, rc == %i" ), rc );        
      draw_status_line( work_buffer );
    }
    get_inputc();
  }

  free( holdbuf );
  return rc;
}

/*
 * This routine returns a handle array containing FSIMs that indicate
 * that they can make a file system on the supplied volume.
 */
int get_fsims_for_volume (object_handle_t volume, handle_array_t **fsim_array)
{
    int             rc = 0;
    handle_array_t *plugins;
    
    *fsim_array = NULL;
    
    rc = evms_get_plugin_list (EVMS_FILESYSTEM_INTERFACE_MODULE, 0, &plugins);
    
    if (rc != 0)
    {
        log_error ("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        *fsim_array = calloc (0, (sizeof (handle_array_t) + (sizeof (object_handle_t) * plugins->count)));
        
        if (*fsim_array) 
        {
            int i;        
        
            (*fsim_array)->count = 0;
            
            for (i=0; i < plugins->count; i++)
            {
                if (evms_can_mkfs (volume, plugins->handle[i]) == 0)
                {
                    (*fsim_array)->handle[(*fsim_array)->count] = plugins->handle[i];
                    (*fsim_array)->count++;
                }
            }
        }
        else
        {
            rc = ENOMEM;
            log_error ("%s: Unable to allocate memory for array of FSIMs.\n", __FUNCTION__);
        }
        
        evms_free (plugins);
    }
    
    return rc;
}

int mkfs_on_volume (object_handle_t volume)
{    
  int rc = 0, menu_state = 1;
  plugin_handle_t plugin_object;
  task_handle_t mkfs_context = 0;
  handle_array_t *fsim_array = NULL;    
  handle_array_t *resultant_objects;
    
  while ( menu_state ) {

    switch ( menu_state ) {

      case 1 :
        menu_state = 0;  /* initialize in case of error */

        rc = get_fsims_for_volume( volume, &fsim_array );

        if ( rc == 0 ) {
          rc = get_plugin_selection( fsim_array, &plugin_object, _("Select File System Interface Module") );
        }
        else {
          log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_get_pluglist, rc );
        }
      
        if ( rc ) {
          if ( rc == ESC_PRESSED )
            return rc;

          draw_status_line( err_msg_no_plugins );

          get_inputc();
        }
        else  if ( plugin_object )
                menu_state = 2;
        break;

      case 2 :
        menu_state = 0;  /* initialize in case of error */

        if ( !mkfs_context ) {

          log_debug( "%s: Creating mkfs task using object handle %i\n", __FUNCTION__, plugin_object );
          rc = evms_create_task( plugin_object, EVMS_Task_mkfs, &mkfs_context );

          if ( rc == 0 )
            menu_state = 3;
          else
            log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_create_task, rc );
        }
        break;

      case 3 : {
          handle_array_t ha;
          menu_state = 0;  /* initialize in case of error */
                
          ha.count = 1;
          ha.handle[0] = volume;
          
          /*
           * Make our volume the selected object for the task.
           */
          rc = evms_set_selected_objects (mkfs_context, &ha, NULL, NULL);

          if ( rc == 0 ) {
            menu_state = 4;
          }
          else {
            sprintf( work_buffer, "%s %i", _("FSIM did not accept the volume selection. rc =="), rc );
            draw_status_line( work_buffer );

            get_inputc();
          }
        }
        break;

      case 4 :
        menu_state = 0;  /* initialize in case of error */
        rc = set_options_menu( mkfs_context, _( " Make File System Configuration Options " ));

        if ( rc == 0 ) {
          rc = evms_invoke_task( mkfs_context, &resultant_objects );
          if ( rc == 0 ) {
            if ( resultant_objects ) {
              log_debug( "%s: Make File System task successful, %i resultant objects\n", __FUNCTION__, resultant_objects->count );
              evms_free( resultant_objects );
            }
          }
        }
        else  if ( rc == ESC_PRESSED )
          menu_state = 3;
        break;

      default:
        menu_state = 0;  /* unknown state - drop out  */
        break;
    }
  }
    
  if ( rc == 0 )
    draw_status_line( _( "Make will be complete when changes are committed. Press any key to continue" ));
  else  {
    sprintf( work_buffer, _( "Make File System failed, rc == %i" ), rc );
    draw_status_line( work_buffer );
  }
  get_inputc();

  if ( mkfs_context )
    rc = evms_destroy_task( mkfs_context );
  
  return rc;    
}

int fsck_volume (object_handle_t volume)
{
  int rc = 0, menu_state = 1;
  task_handle_t fsck_context = 0;
  handle_array_t *resultant_objects;

  rc = select_box( _( "Check/Repair File System?" ),
                   _( "Do not check/repair file system" ),
                   _( "Check/Repair" ),
                   NULL  );

  if ( rc == 2 ) {

  while ( menu_state ) {
    
        switch ( menu_state ) {
    
          case 1 :
            menu_state = 0;  /* initialize in case of error */
    
            if ( !fsck_context ) {
    
              log_debug( "%s: Creating fsck task using object handle %i\n", __FUNCTION__, volume );
              rc = evms_create_task( volume, EVMS_Task_fsck, &fsck_context );
    
              if ( rc == 0 )
                menu_state = 2;
              else
                log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_create_task, rc );
            }
            break;
    
          case 2 :
            menu_state = 0;  /* initialize in case of error */
            rc = set_options_menu( fsck_context, _( " Check/Repair File System Configuration Options " ));
    
            if ( rc == 0 ) {
              rc = evms_invoke_task( fsck_context, &resultant_objects );
              if ( rc == 0 ) {
                if ( resultant_objects ) {
                  log_debug( "%s: Check File System task successful, %i resultant objects\n", __FUNCTION__, resultant_objects->count );
                  evms_free( resultant_objects );
                }
              }
            }
            break;
    
          default:
            menu_state = 0;  /* unknown state - drop out  */
            break;
        }
      }
    
      if (rc != ESC_PRESSED ) {
          if ( rc == 0 )
            draw_status_line( _( "Operation will be complete when changes are committed. Press any key to continue" ));
          else  {
            sprintf( work_buffer, _( "Check/Repair File System failed, rc == %i" ), rc );
            draw_status_line( work_buffer );
          }
          get_inputc();
      }
    
      if ( fsck_context )
        rc = evms_destroy_task( fsck_context );
  }
  
  return rc;
}

int defrag_volume (object_handle_t volume)
{
  int rc = 0, menu_state = 1;
  task_handle_t defrag_context = 0;
  handle_array_t *resultant_objects;

  rc = select_box( _( "Defragment File System?" ),
                   _( "Do not defragment file system" ),
                   _( "Defragment" ),
                   NULL  );

  if ( rc == 2 ) {

  while ( menu_state ) {
    
        switch ( menu_state ) {
    
          case 1 :
            menu_state = 0;  /* initialize in case of error */
    
            if ( !defrag_context ) {
    
              log_debug( "%s: Creating defrag task using object handle %i\n", __FUNCTION__, volume );
              rc = evms_create_task( volume, EVMS_Task_defrag, &defrag_context );
    
              if ( rc == 0 )
                menu_state = 2;
              else
                log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_create_task, rc );
            }
            break;
    
          case 2 :
            menu_state = 0;  /* initialize in case of error */
            rc = set_options_menu( defrag_context, _( " Defragment File System Configuration Options " ));
    
            if ( rc == 0 ) {
              rc = evms_invoke_task( defrag_context, &resultant_objects );
              if ( rc == 0 ) {
                if ( resultant_objects ) {
                  log_debug( "%s: Defragment File System task successful, %i resultant objects\n", __FUNCTION__, resultant_objects->count );
                  evms_free( resultant_objects );
                }
              }
            }
            break;
    
          default:
            menu_state = 0;  /* unknown state - drop out  */
            break;
        }
      }
    
      if (rc != ESC_PRESSED ) {
          if ( rc == 0 )
            draw_status_line( _( "Operation will be complete when changes are committed. Press any key to continue" ));
          else  {
            sprintf( work_buffer, _( "Defragment File System failed, rc == %i" ), rc );
            draw_status_line( work_buffer );
          }
          get_inputc();
      }
    
      if ( defrag_context )
        rc = evms_destroy_task( defrag_context );
  }
  
  return rc;
}

int unmkfs_on_volume (object_handle_t volume)
{
  int rc = 0;

  rc = select_box( _( "Remove File System from Volume?" ),
                   _( "Do not remove file system" ),
                   _( "Remove" ),
                   NULL  );

  if ( rc == 2 ) {
    rc = evms_unmkfs( volume );
    if ( rc == 0 ) {
      draw_status_line( _( "Removal will be complete when changes are committed. Press any key to continue" ));
    }
    else {
      sprintf( work_buffer, _( "Remove failed, rc == %i" ), rc );
      draw_status_line( work_buffer );
    }
    get_inputc();
  }

  return rc;
}

/*
 *
 *   int can_volume_be_formatted (object_handle_t)
 *
 *   Description:
 *      This routine checks to see if any FSIM can mkfs it
 *      and if so returns TRUE.
 * 
 *   Entry:
 *      volume - volume's object handle
 *
 *   Exit:
 *      See Description.
 *
 */
int can_volume_be_formatted (object_handle_t volume)
{
    int              rc;
    int              result = FALSE;
    handle_array_t  *plugins;
    
    rc = evms_get_plugin_list (EVMS_FILESYSTEM_INTERFACE_MODULE, 0, &plugins);

    if (rc != 0)
    {
        log_error ("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        int i;

        for (i=0; i < plugins->count && result == FALSE; i++)
        {
            result = evms_can_mkfs (volume, plugins->handle[i]) == 0;
        }
            
        evms_free (plugins);
    }
    
    return result;
}


/*----------------------------------------------------------------------
 * Public Functions
 *---------------------------------------------------------------------*/

int volume_actions( object_handle_t object )
{
  int i, rc = 0, keyval = 0, start_index = 0, select_index = 0,
      screen_cols, screen_lines = CALC_SCREEN3_LINES, removx = 99,
      destrx = 99, xpandx = 99, shrnkx = 99, setinfx = 99, mkfsx = 99,
      unmkfsx = 99, defragx = 99, fsckx = 99;
  handle_object_info_t *object_info;
  char *holdbuf;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  BLANK_BUFFER( holdbuf, screen_cols )
  rc = evms_get_info( object, &object_info );
  if ( rc == 0 ) {
    sprintf( holdbuf, "%s:  %s", vol_name_col, object_info->info.volume.name );
    print_string( XPOS_BOX3TEXT - 1, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );
    evms_free( object_info );
  }
  else {
    log_error( "%s: %s VOLUME %s %i.\n", __FUNCTION__, err_msg_get_info, err_msg_rc, rc );
  }

  drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, screen_cols );

  do {

    i = 0;
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, vol_displaydetsel, strlen( vol_displaydetsel ));
    if ( select_index == i )
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_BAR, holdbuf );
    else
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_TITLE, holdbuf );

    if ( evms_can_remove( object ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, vol_removesel, strlen( vol_removesel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      removx = i;
    }
    if ( evms_can_destroy( object ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, vol_destroysel, strlen( vol_destroysel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      destrx = i;
    }
//    if ( evms_can_move( object ) == 0 ) {
//      i++;
//      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, "Move Volume" );
//    }
    if ( evms_can_expand( object ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, vol_expandsel, strlen( vol_expandsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      xpandx = i;
    }
    if ( evms_can_shrink( object ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, vol_shrinksel, strlen( vol_shrinksel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      shrnkx = i;
    }
    if ( evms_can_set_volume_name( object ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, vol_setinfsel, strlen( vol_setinfsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      setinfx = i;
    }
    if ( can_volume_be_formatted( object ) ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, vol_mkfssel, strlen( vol_mkfssel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      mkfsx = i;
    }
    if ( evms_can_fsck( object ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, vol_fscksel, strlen( vol_fscksel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      fsckx = i;
    }
    if ( evms_can_defrag( object ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, vol_defragsel, strlen( vol_defragsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      defragx = i;
    }    
    if ( evms_can_unmkfs( object ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, vol_unmkfssel, strlen( vol_unmkfssel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      unmkfsx = i;
    }    
    
    draw_status_line( action_selectmsg );

    keyval = key_scroller( &start_index, &select_index, i + 1, screen_lines );

    switch ( keyval ) {

      case KEY_ESC :
        break;

      case KEY_F1  :
      case '1'     :
        draw_help_panel( 5 );
        keyval = KEY_CURSORACT;
        break;

      case KEY_ENTER :
        if ( select_index == removx ) {
          rc = remove_object( object, _( "Remove the Volume?" ));
          if ( rc == 0 )
            rc = KEY_VOLUMES;
        }
        else  if ( select_index == destrx ) {
          rc = destroy_object( object, _( "Destroy the Volume?" ));
          if ( rc == 0 )
            rc = KEY_VOLUMES;
        }
        else  if ( select_index == xpandx ) {
          rc = resize_object( object, EVMS_Task_Expand );
          if ( rc == 0 )
            rc = KEY_VOLUMES;
        }
        else  if ( select_index == shrnkx ) {
          rc = resize_object( object, EVMS_Task_Shrink );
          if ( rc == 0 )
            rc = KEY_VOLUMES;
        }
        else  if ( select_index == setinfx ) {
          rc = set_volume_info( object );
          if ( rc == 0 )
            rc = KEY_VOLUMES;
        }                
        else  if ( select_index == mkfsx ) {
          rc = mkfs_on_volume( object );
          if ( rc == 0 )
            rc = KEY_VOLUMES;
        }
        else  if ( select_index == fsckx ) {
          rc = fsck_volume( object );
          if ( rc == 0 )
            rc = KEY_VOLUMES;
        }        
        else  if ( select_index == defragx ) {
          rc = defrag_volume( object );
          if ( rc == 0 )
            rc = KEY_VOLUMES;
        }        
        else  if ( select_index == unmkfsx ) {
          rc = unmkfs_on_volume( object );
          if ( rc == 0 )
            rc = KEY_VOLUMES;
        }                
        else {
          rc = object_extended_info( object );
        }
        break;

      default:
        break;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return rc;
}


int draw_volumes( int start_index, int *select_index, handle_array_t *volumes )
{
  int i, j, rc = 0, item_count, screen_cols,
      screen_lines = CALC_SCREEN2_LINES;
  handle_object_info_t *object;
  char *holdbuf, *col1, *col2, *col3, *col4;

  if ( volumes == NULL )                               /* nothing to do*/
    return 1;

  screen_cols = GL_screen_cols - XPOS_BOX2TEXT - XPOS_BOX2RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  // volume view

  draw_box2( title_volumes );

  drawhline( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 1, screen_cols );

  if ( volumes->count ) {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, vol_name_col, strlen( vol_name_col ));
    col1 = holdbuf + X_VOL_SIZE;
    memcpy( col1, vol_size_col, strlen( vol_size_col ));
    col2 = holdbuf + X_VOL_MINOR;
    memcpy( col2, vol_minor_col, strlen( vol_minor_col ));
    col3 = holdbuf + X_VOL_DIRTY;
    memcpy( col3, vol_dirty_col, strlen( vol_dirty_col ));
    col4 = holdbuf + X_VOL_NEW;
    memcpy( col4, vol_new_col, strlen( vol_new_col ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 2, COLOR_TITLE, holdbuf );

    if ( start_index + screen_lines > volumes->count )
      item_count = volumes->count - start_index + YPOS_BOX2TEXT;
    else
      item_count = screen_lines - 1 + YPOS_BOX2TEXT;

    for ( i = YPOS_BOX2TEXT, j = start_index; i < item_count; i++, j++ ) {

      rc = evms_get_info( volumes->handle[j], &object );
      if ( rc == 0 ) {
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object->info.volume.name, strlen( object->info.volume.name ));
        draw_sector_size( object->info.volume.vol_size, work_buffer );
        memcpy( col1, work_buffer, strlen( work_buffer ));
        sprintf( work_buffer, "%d", object->info.volume.minor_number );
        memcpy( col2, work_buffer, strlen( work_buffer ));
        if( object->info.volume.flags & VOLFLAG_DIRTY )
          *col3 = ITEM_SELECTED_CHAR;
        if( object->info.volume.flags & VOLFLAG_NEW )
          *col4 = ITEM_SELECTED_CHAR;
        if ( j == *select_index )
          print_string( XPOS_BOX2TEXT, i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX2TEXT, i, COLOR_TITLE, holdbuf );

        log_debug( "%s: entry %i is: %s \n", __FUNCTION__, i, object->info.volume.name );
        evms_free( object );
      }
      else {
        log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, i, err_msg_rc, rc );
      }
    }
    draw_status_line( vol_selectmsg );
  }
  // else no volumes
  else {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, info_no_volumes, strlen( info_no_volumes ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT, COLOR_BAR, holdbuf );
    draw_status_line( NULL );                        /* redraw status line.*/
    free( holdbuf );
    return 1;
  }

  free( holdbuf );
  return 0;
}

