/*
 * mpdev.c - Multipath Device - RAM disk driver 
 * 
 * This code is based on lots of ramdisk code like ... 
 * drivers/block/rd.c
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/blkdev.h>
#include <linux/bio.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/hdreg.h>
#include <asm/uaccess.h>

#include "mpdev.h"

#define MP_NUM_DISKS		16
#define MP_MAJOR		252
#define MP_SECTORS		2048
#define MP_SECTOR_SIZE		512

#define MP_NAME			"mp"
#define MP_HEADS		16
#define MP_CYLINDERS		2

static struct gendisk *rd_disks[MP_NUM_DISKS];
static struct request_queue *rd_queue[MP_NUM_DISKS];
static int mpdev_indexes[MP_NUM_DISKS];
static int mpdev_device_status[MP_NUM_DISKS]; 

static void *mpdev_data;

static unsigned int major = MP_MAJOR;
static unsigned int paths = MP_NUM_DISKS;
static unsigned int size = MP_SECTORS / 2;
static unsigned int sectors;


static int request_should_fail(int dev)
{       
	if (dev >= 0 && dev < paths) {
		return mpdev_device_status[dev];
	}
	return 1;
}


static int mpdev_make_request(request_queue_t * q, struct bio *bio)
{
	sector_t lsn = bio->bi_sector;
	long   len = bio->bi_size >> 9;
	struct bio_vec *bvec;
	int rw = bio_data_dir(bio);
	int i;          
	int dev=0;
	char *src,*dst;
	int vsectors;
	int bytes;


	if (lsn + len > sectors)
		goto fail;

	if (rw==READA)
		rw=READ;

	dev = bio->bi_bdev->bd_dev & 0xff;

	if ( request_should_fail(dev) )
		goto fail;

	if (rw==READ)
		printk(KERN_INFO "mpdev: read(%d) lsn = %ld count = %ld (SUCCESS)\n",
		       dev, (unsigned long)lsn, len);
	else
		printk(KERN_INFO "mpdev: write(%d) lsn = %ld  count = %ld (SUCCESS)\n",
		       dev, (unsigned long)lsn, len);     

	bio_for_each_segment(bvec, bio, i) {            

		bytes    = bvec->bv_len;
		vsectors = bvec->bv_len>>9;                             

		if (rw == READ) {
			src = mpdev_data + (lsn<<9);
			dst = bio_data(bio); 
		} else {
			src = bio_data(bio); 
			dst = mpdev_data + (lsn<<9);
		}

		memcpy(dst, src, bytes);

		lsn += vsectors;
		++bio->bi_idx;
	}

	bio_endio(bio, bio->bi_size, 0);
	return 0;

	fail:
	if (rw==READ)
		printk(KERN_INFO "mpdev: read(%d) lsn = %ld  count = %ld (FAILED)\n",
		       dev, (unsigned long)lsn, len);
	else
		printk(KERN_INFO "mpdev: write(%d) lsn = %ld  count = %ld (FAILED)\n",
		       dev, (unsigned long)lsn, len);

	bio_io_error(bio, bio->bi_size);
	return 0;
} 


static int mpdev_open(struct inode * inode, struct file * filp)
{
	return 0;
}


static int mpdev_ioctl(struct inode * inode, struct file *file,
		       unsigned int cmd, unsigned long arg)
{
	int error;
	int unit_number;

	switch (cmd) {
	case HDIO_GETGEO:
		{
			struct hd_geometry g;
			struct hd_geometry *geometry = (struct hd_geometry *)arg;

			printk(KERN_INFO "HDIO GETGEO cmd\n");  

			g.heads = MP_HEADS;
			g.cylinders = MP_CYLINDERS;
			g.sectors = sectors / (MP_HEADS * MP_CYLINDERS);
			g.start = 0;
			error = copy_to_user(geometry, &g, sizeof g) ? -EFAULT : 0;
			break;
		}

	case MPDEV_CMD:  
		{    
			mpdev_cmd_t mpdev_cmd;

			unit_number = *((int*)inode->i_bdev->bd_disk->private_data);

			if (copy_from_user(&mpdev_cmd, (mpdev_cmd_t *)arg, sizeof(mpdev_cmd_t)))
				return -EFAULT;

			if (unit_number >= 0  && unit_number < paths) {
				mpdev_device_status[unit_number] = mpdev_cmd.cmd;
				if (mpdev_cmd.cmd == 1) {
					printk(KERN_INFO "mpdev: failing unit device %d\n", unit_number);
				} else {
					printk(KERN_INFO "mpdev: recovering unit device %d\n", unit_number);
				}
				error = 0;
			} else {
				error = -EINVAL;
			}                    

			mpdev_cmd.status = error;
			copy_to_user((mpdev_cmd_t *)arg, &mpdev_cmd, sizeof(mpdev_cmd));
			return 0;
		}


	default:
		error = -EINVAL;
	}

	return error;
}

static struct block_device_operations rd_bd_op = {
	.owner =        THIS_MODULE,
	.open =         mpdev_open,
	.ioctl = mpdev_ioctl,
};

static void __exit mpdev_cleanup(void)
{
	int i;

	for (i = 0 ; i < paths; i++) {
		del_gendisk(rd_disks[i]);
		put_disk(rd_disks[i]);
		blk_put_queue(rd_queue[i]);
	}
	vfree(mpdev_data);
	devfs_remove(MP_NAME);
	unregister_blkdev(major, MP_NAME);

	printk("mpdev driver unloaded\n");
}

static int __init mpdev_init(void)
{
	int i, rc;

	if (paths > MP_NUM_DISKS) {
		rc = -EINVAL;
		goto err1;
	}

	sectors = size * 2;

	for (i = 0; i < paths; i++) {
		mpdev_device_status[i] = 0; 
	}

	rc = register_blkdev(major, MP_NAME);
	if (rc)
		goto err1;

	devfs_mk_dir(MP_NAME);

	mpdev_data = vmalloc(sectors * MP_SECTOR_SIZE);
	if (!mpdev_data) {
		printk(KERN_ERR "mpdev: failed to malloc ramdisk area\n");
		rc = -ENOMEM;
		goto err2;
	}
	memset(mpdev_data, 0, sectors * MP_SECTOR_SIZE);

	for (i = 0; i < paths; i++) {
		rd_disks[i] = alloc_disk(1);
		if (!rd_disks[i])
			goto err3;

		rd_queue[i] = blk_alloc_queue(GFP_KERNEL);
		if (!rd_queue[i])
			goto err3;
	}

	for (i = 0; i < paths; i++) {
		struct gendisk *disk = rd_disks[i];

		mpdev_indexes[i] = i;

		blk_queue_make_request(rd_queue[i], &mpdev_make_request);
		rd_queue[i]->queuedata = &mpdev_indexes[i];

		disk->major = major;
		disk->first_minor = i;
		disk->fops = &rd_bd_op;
		disk->queue = rd_queue[i];
		disk->private_data = &mpdev_indexes[i];
		sprintf(disk->disk_name, "%s%d", MP_NAME, i);
		sprintf(disk->devfs_name, "%s/%d", MP_NAME, i);
		set_capacity(disk, sectors);
		add_disk(disk);
	}

	printk("mpdev driver initialized: major %d. "
	       "%d multipath disks of %d kB\n", major, paths, size);

	return 0;

err3:
	for (i = 0; i < paths; i++) {
		if (rd_disks[i])
			put_disk(rd_disks[i]);
		if (rd_queue[i])
			blk_put_queue(rd_queue[i]);
	}
	vfree(mpdev_data);
err2:
	devfs_remove(MP_NAME);
	unregister_blkdev(major, MP_NAME);
err1:
	return rc;
}

module_param(major, uint, 0);
MODULE_PARM_DESC(major, "The major number of the mp driver");

module_param(paths, uint, 0);
MODULE_PARM_DESC(major, "The number of paths to create to the ram-disk");

module_param(size, uint, 0);
MODULE_PARM_DESC(major, "The size of the ram-disk in kB");

module_init(mpdev_init);
module_exit(mpdev_cleanup);
MODULE_LICENSE("GPL");

