/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: dm.c
 *
 * Implementation of the Device-Mapper common-services. The code in this file
 * is independent of the DM interface. Code that is dependent on the interface
 * version is in dm-ioctl*.[ch].
 *
 * Each common-service API is routed into this file first, where common handling
 * is performed. Then the current interface version is checked, and the
 * appropriate version-specific routine is called to complete the work for the
 * service.
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "fullengine.h"
#include "engine.h"
#include "memman.h"
#include "dm.h"
#include "dm-ioctl.h"

/* The major number of the Device-Mapper ioctl interface. */
static int dm_version_major = 0;

/* Flag indicating whether a Device-Mapper device is suspended. */
int dm_device_suspended = FALSE;

/**
 * dm_allocate_target
 *
 * @type:	Type of target to allocate. DM_TARGET_*
 * @start:	Logical start of this target within the DM device.
 * @length:	Length of this target.
 * @num_devs:	Total number of devices when allocating a STRIPE, MIRROR,
 *		or MULTIPATH. Otherwise set to 0.
 * @num_groups:	Number of groups when allocating a MULTIPATH. Otherwise
 *		set to 0.
 *
 * Allocate a DM target and initialize the common fields.
 **/
dm_target_t *dm_allocate_target(dm_target_type type,
				u_int64_t start,
				u_int64_t length,
				u_int32_t num_devs,
				u_int32_t num_groups)
{
	dm_target_t *target = NULL;
	unsigned long size;
	int rc;

	LOG_PROC_ENTRY();

	if (type > DM_TARGET_MAX) {
		/* Bad parameters. */
		goto out;
	}

	LOG_DEBUG("Request to allocate a %s target.\n",
		  dm_target_type_info[type].name);

	target = engine_alloc(sizeof(dm_target_t));
	if (!target) {
		goto out;
	}

	size = dm_target_type_info[type].struct_size;
	if (size) {
		target->data.linear = engine_alloc(size);
		if (!target->data.linear) {
			engine_free(target);
			target = NULL;
			goto out;
		}

		rc = dm_target_type_info[type].allocate_target(target,
							       num_devs,
							       num_groups);
		if (rc) {
			engine_free(target->data.linear);
			engine_free(target);
			target = NULL;
			goto out;
		}
	}

	target->start = start;
	target->length = length;
	target->type = type;

out:
	if (!target) {
		LOG_ERROR("Error allocating memory for target.\n");
		LOG_ERROR("   Type: %d, Start %"PRIu64", Length %"PRIu64"\n",
			  type, start, length);
	}
	LOG_PROC_EXIT_PTR(target);
	return target;
}

/**
 * dm_add_target
 *
 * @target:		New target to insert in the list.
 * @target_list:	Pointer to the head of the target list.
 *
 * Add a DM target to a list of targets in preparation for an activate or
 * reactivate. The target is inserted in order based on the "start" field.
 **/
void dm_add_target(dm_target_t *target, dm_target_t **target_list)
{
	dm_target_t **t;

	LOG_PROC_ENTRY();

	for (t = target_list; *t; t = &(*t)->next) {
		if ((*t)->start > target->start) {
			break;
		}
	}

	target->next = (*t) ? (*t)->next : NULL;
	*t = target;

	LOG_PROC_EXIT_VOID();
}

/**
 * dm_deallocate_targets
 *
 * Free the memory for a list of DM targets. This assumes the targets were
 * allocated with dm_allocate_target() and linked together with dm_add_target().
 **/
void dm_deallocate_targets(dm_target_t *target_list)
{
	dm_target_t *target;

	LOG_PROC_ENTRY();

	for (target = target_list; target; target = target_list) {
		target_list = target->next;
		if (target->data.linear) {
			dm_target_type_info[target->type].deallocate_target(target);
			engine_free(target->data.linear);
		}
		engine_free(target);
	}

	LOG_PROC_EXIT_VOID();
}

/**
 * dm_deallocate_device_list
 *
 * Free the memory for a list of dm_device_list's.
 **/
void dm_deallocate_device_list(dm_device_list_t *device_list)
{
	dm_device_list_t *device;

	LOG_PROC_ENTRY();

	for (device = device_list; device; device = device_list) {
		device_list = device->next;
		engine_free(device);
	}

	LOG_PROC_EXIT_VOID();
}

/**
 * dm_deallocate_module_list
 *
 * Free the memory for a list of dm_module_list's.
 **/
void dm_deallocate_module_list(dm_module_list_t *module_list)
{
	dm_module_list_t *module;

	LOG_PROC_ENTRY();

	for (module = module_list; module; module = module_list) {
		module_list = module->next;
		engine_free(module);
	}

	LOG_PROC_EXIT_VOID();
}

/**
 * dm_suspend
 *
 * @object:	EVMS storage object to suspend/resume.
 * @suspend:	TRUE for suspend. FALSE for resume.
 *
 * Suspend or resume the Device-Mapper device representing an active EVMS
 * object.
 **/
int dm_suspend(storage_object_t *object, int suspend)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!object) {
		/* Bad parameter. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to %s object %s\n",
		  suspend ? "suspend" : "resume", object->name);

	if (!(object->flags & SOFLAG_ACTIVE)) {
		/* Can't suspend inactive objects. */
		rc = EINVAL;
		goto out;
	}

	if (suspend && (object->flags & SOFLAG_SUSPENDED)) {
		/* Can't suspend an object that's already suspended. On the
		 * other hand, we *can* resume an object that isn't suspended.
		 */
		rc = 0;
		goto out;
	}

	switch (dm_version_major) {
	case 3:
		rc = dm_suspend_v3(object->name, suspend);
		break;
	case 4:
		rc = dm_suspend_v4(object->name, suspend);
		break;
	default:
		rc = EINVAL;
	}
	if (rc) {
		goto out;
	}

	/* Update the object's state. */
	if (suspend) {
		object->flags |= SOFLAG_SUSPENDED;
	} else {
		object->flags &= ~SOFLAG_SUSPENDED;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_suspend_volume
 *
 * @volume:	EVMS logical volume to suspend/resume.
 * @suspend:	TRUE for suspend. FALSE for resume.
 *
 * Suspend or resume the Device-Mapper device representing an active EVMS
 * volume.
 **/
int dm_suspend_volume(logical_volume_t *volume, int suspend)
{
	char *base_name;
	int rc;

	LOG_PROC_ENTRY();

	if (!volume) {
		/* Bad parameter. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to %s volume %s\n",
		  suspend ? "suspend" : "resume", volume->name);

	if (memcmp(volume->name, EVMS_DEV_NODE_PATH, EVMS_DEV_NODE_PATH_LEN)) {
		LOG_ERROR("Volume %s does not have the \"%s\" prefix.\n",
			  volume->name, EVMS_DEV_NODE_PATH);
		rc = EINVAL;
		goto out;
	}
	base_name = volume->name + EVMS_DEV_NODE_PATH_LEN;

	if (!(volume->flags & VOLFLAG_ACTIVE)) {
		/* Can't suspend inactive volumes. */
		rc = EINVAL;
		goto out;
	}

	if (suspend && (volume->flags & VOLFLAG_SUSPENDED)) {
		/* Can't suspend a volume that's already suspended. On the other
		 * hand, we *can* resume a volume that isn't suspended.
		 */
		rc = 0;
		goto out;
	}

	switch (dm_version_major) {
	case 3:
		rc = dm_suspend_v3(base_name, suspend);
		break;
	case 4:
		rc = dm_suspend_v4(base_name, suspend);
		break;
	default:
		rc = EINVAL;
	}
	if (rc) {
		goto out;
	}

	/* Update the volume's state. */
	if (suspend) {
		volume->flags |= VOLFLAG_SUSPENDED;
	} else {
		volume->flags &= ~VOLFLAG_SUSPENDED;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_activate
 *
 * @object:	EVMS storage object to activate.
 * @target_list:List of targets that will make up the mapped device.
 *
 * Activate (or re-activate) an EVMS object as a Device-Mapper device. After
 * activating, update the object's device major:minor and mark object as active.
 **/
int dm_activate(storage_object_t *object, dm_target_t *target_list)
{
	int reactivate, read_only, rc;

	LOG_PROC_ENTRY();

	if (!object || !target_list) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	reactivate = object->flags & SOFLAG_ACTIVE;
	read_only = object->flags & SOFLAG_READ_ONLY;

	LOG_DEBUG("Request to %sactivate object %s\n",
		  reactivate ? "re" : "", object->name);

	/* Create the ASCII parameter strings for all targets. */
	rc = build_target_type_params(target_list);
	if (rc) {
		goto out;
	}

	switch (dm_version_major) {
	case 3:
		rc = dm_activate_v3(object->name,
				    target_list, reactivate, read_only,
				    &object->dev_major, &object->dev_minor);
		break;
	case 4:
		rc = dm_activate_v4(object->name,
				    target_list, reactivate, read_only,
				    &object->dev_major, &object->dev_minor);
		break;
	default:
		rc = EINVAL;
	}
	if (rc) {
		goto out;
	}

	object->flags |= SOFLAG_ACTIVE;

out:
	/* Free the target parameter strings. */
	deallocate_target_type_params(target_list);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_activate_volume
 *
 * @volume:	EVMS volume to activate.
 * @target_list:List of targets that will make up the mapped device.
 *
 * Activate (or re-activate) an EVMS volume as a Device-Mapper device. After
 * activating, update the volume's device major:minor and mark volume as active.
 **/
int dm_activate_volume(logical_volume_t *volume, dm_target_t *target_list)
{
	char *base_name;
	int reactivate, read_only, rc;

	LOG_PROC_ENTRY();

	if (!volume || !target_list) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	if (!(volume->flags & VOLFLAG_HAS_OWN_DEVICE)) {
		/* This volume doesn't need its own DM device. */
		rc = 0;
		goto out;
	}

	base_name = volume->name + EVMS_DEV_NODE_PATH_LEN;
	reactivate = volume->flags & VOLFLAG_ACTIVE;
	read_only = volume->flags & VOLFLAG_READ_ONLY;

	LOG_DEBUG("Request to %sactivate volume %s\n",
		  reactivate ? "re" : "", volume->name);

	/* Create the ASCII parameter strings for all targets. */
	rc = build_target_type_params(target_list);
	if (rc) {
		goto out;
	}

	if (memcmp(volume->name, EVMS_DEV_NODE_PATH,
		   EVMS_DEV_NODE_PATH_LEN) != 0) {
		LOG_ERROR("Volume %s does not have the \"%s\" prefix.\n",
			  volume->name, EVMS_DEV_NODE_PATH);
		rc = EINVAL;
		goto out;
	}

	switch (dm_version_major) {
	case 3:
		rc = dm_activate_v3(base_name,
				    target_list, reactivate, read_only,
				    &volume->dev_major, &volume->dev_minor);
		break;
	case 4:
		rc = dm_activate_v4(base_name,
				    target_list, reactivate, read_only,
				    &volume->dev_major, &volume->dev_minor);
		break;
	default:
		rc = EINVAL;
	}
	if (rc) {
		goto out;
	}

	volume->flags |= VOLFLAG_ACTIVE;

out:
	/* Free the target parameter strings. */
	deallocate_target_type_params(target_list);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_deactivate
 *
 * @object: EVMS storage object to deactivate.
 *
 * Deactivate the Device-Mapper device representing the EVMS storage object.
 * Update the object's information.
 **/
int dm_deactivate(storage_object_t *object)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!object) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to deactivate object %s\n", object->name);

	if (!(object->flags & SOFLAG_ACTIVE)) {
		/* Object is not active. */
		rc = 0;
		goto out;
	}

	switch (dm_version_major) {
	case 3:
		rc = dm_deactivate_v3(object->name);
		break;
	case 4:
		rc = dm_deactivate_v4(object->name);
		break;
	default:
		rc = EINVAL;
	}
	if (rc) {
		goto out;
	}

	/* Update the object's information. */
	object->dev_major = 0;
	object->dev_minor = 0;
	object->flags &= ~SOFLAG_ACTIVE;

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_deactivate_volume
 *
 * @volume: EVMS logical volume to deactivate.
 *
 * Deactivate the Device-Mapper device representing the EVMS logical volume.
 * Update the volume's information.
 **/
int dm_deactivate_volume(logical_volume_t *volume)
{
	char *base_name;
	int rc;

	LOG_PROC_ENTRY();

	if (!volume) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to deactivate volume %s\n", volume->name);

	if (!(volume->flags & VOLFLAG_ACTIVE) ||
	    !(volume->flags & VOLFLAG_HAS_OWN_DEVICE)) {
		/* Volume is not active, or doesn't have it's own DM device. */
		rc = 0;
		goto out;
	}

	if (memcmp(volume->name, EVMS_DEV_NODE_PATH,
		   EVMS_DEV_NODE_PATH_LEN) != 0) {
		LOG_ERROR("Volume %s does not have the \"%s\" prefix.\n",
			  volume->name, EVMS_DEV_NODE_PATH);
		rc = EINVAL;
		goto out;
	}

	base_name = volume->name + EVMS_DEV_NODE_PATH_LEN;

	switch (dm_version_major) {
	case 3:
		rc = dm_deactivate_v3(base_name);
		break;
	case 4:
		rc = dm_deactivate_v4(base_name);
		break;
	default:
		rc = EINVAL;
	}
	if (rc) {
		goto out;
	}

	/* Update the volume's information. */
	volume->dev_major = 0;
	volume->dev_minor = 0;
	volume->flags &= ~VOLFLAG_ACTIVE;

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_rename
 *
 * @object:	The object whose name is changing
 * @old_name:	The object's old name.
 * @new_name:	The object's new name.
 *
 * Tell Device-Mapper to change the name for the specified active object. Both
 * the old name and the new name are parameters to this function, so as to not
 * force a particular ordering on when the object's actual name is changed.
 * This function does not change the name in the object, and the caller can
 * change that name before or after calling this API.
 **/
int dm_rename(storage_object_t * object,
	      char * old_name,
	      char * new_name)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!object || !old_name || !new_name) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to rename object %s from %s to %s\n",
		  object->name, old_name, new_name);

	if (!(object->flags & SOFLAG_ACTIVE)) {
		/* Object is not active. */
		rc = EINVAL;
		goto out;
	}

	switch (dm_version_major) {
	case 3:
		rc = dm_rename_v3(old_name, new_name);
		break;
	case 4:
		rc = dm_rename_v4(old_name, new_name);
		break;
	default:
		rc = EINVAL;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_rename_volume
 *
 * @object:	The volume whose name is changing
 * @old_name:	The volume's old name.
 * @new_name:	The volume's new name.
 *
 * Tell Device-Mapper to change the name for the specified active volume. Both
 * the old name and the new name are parameters to this function, so as to not
 * force a particular ordering on when the volume's actual name is changed.
 * This function does not change the name in the volume, and the caller can
 * change that name before or after calling this API.
 **/
int dm_rename_volume(logical_volume_t * volume,
		     char * old_name,
		     char * new_name)
{
	char *old_base_name, *new_base_name;
	int rc;

	LOG_PROC_ENTRY();

	if (!volume || !old_name || !new_name) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to rename volume %s from %s to %s\n",
		  volume->name, old_name, new_name);

	if (!(volume->flags & VOLFLAG_ACTIVE)) {
		/* Volume is not active. */
		rc = EINVAL;
		goto out;
	}

	if (memcmp(old_name, EVMS_DEV_NODE_PATH, EVMS_DEV_NODE_PATH_LEN) != 0) {
		LOG_ERROR("Name %s does not have the \"%s\" prefix.\n",
			  old_name, EVMS_DEV_NODE_PATH);
		rc = EINVAL;
		goto out;
	}
        old_base_name = old_name + EVMS_DEV_NODE_PATH_LEN;
	
	if (memcmp(new_name, EVMS_DEV_NODE_PATH, EVMS_DEV_NODE_PATH_LEN) != 0) {
		LOG_ERROR("Name %s does not have the \"%s\" prefix.\n",
			  new_name, EVMS_DEV_NODE_PATH);
		rc = EINVAL;
		goto out;
	}
        new_base_name = new_name + EVMS_DEV_NODE_PATH_LEN;

	switch (dm_version_major) {
	case 3:
		rc = dm_rename_v3(old_base_name, new_base_name);
		break;
	case 4:
		rc = dm_rename_v4(old_base_name, new_base_name);
		break;
	default:
		rc = EINVAL;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_create
 *
 * @object: EVMS object to create a new DM device for.
 *
 * Create a new Device-Mapper device for the specified EVMS object. This
 * device will have no initial mapping, and will be created in a suspended
 * state. The device must have a mapping added using dm_load_targets and
 * then be activated by a call to dm_suspend.
 *
 * This API is only available with DM interface version 4.
 **/
int dm_create(storage_object_t * object)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!object) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	if (object->flags & SOFLAG_ACTIVE) {
		LOG_WARNING("Object %s already has an active DM device.\n",
			    object->name);
		rc = EEXIST;
		goto out;
	}

	LOG_DEBUG("Request to create inactive DM device for object %s\n",
		  object->name);

	switch (dm_version_major) {
	case 4:
		rc = dm_create_v4(object->name,
				  &object->dev_major, &object->dev_minor);
		break;
	default:
		rc = EINVAL;
	}
	if (rc) {
		goto out;
	}

	object->flags |= SOFLAG_ACTIVE;

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_update_status
 *
 * @object: EVMS object to retrieve status for.
 *
 * Query device-mapper for the status of this EVMS object. If the device
 * exists in the kernel, it is marked active, and the remaining object
 * information is updated. If the device does not exist in the kernel, the
 * object is marked inactive.
 **/
int dm_update_status(storage_object_t *object)
{
	int active = FALSE;
	int read_only = FALSE;
	int rc;

	LOG_PROC_ENTRY();

	if (!object) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to update the status of object %s\n", object->name);

	switch (dm_version_major) {
	case 3:
		rc = dm_update_status_v3(object->name,
					 &active, &read_only,
					 &object->dev_major, &object->dev_minor);
		break;
	case 4:
		rc = dm_update_status_v4(object->name,
					 &active, &read_only,
					 &object->dev_major, &object->dev_minor);
		break;
	default:
		rc = EINVAL;
	}
	if (rc) {
		goto out;
	}

	object->flags |= (active    ? SOFLAG_ACTIVE    : 0) |
			 (read_only ? SOFLAG_READ_ONLY : 0);

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_update_volume_status
 *
 * @volume: EVMS volume to retrieve status for.
 *
 * Query device-mapper for the status of this EVMS volume. If the device
 * exists in the kernel, it is marked active, and the remaining volume
 * information is updated. If the device does not exist in the kernel, the
 * volume is marked inactive.
 **/
int dm_update_volume_status(logical_volume_t *volume)
{
	char *base_name;
	int active = FALSE;
	int read_only = FALSE;
	int rc;

	LOG_PROC_ENTRY();

	if (!volume) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to update the status of volume %s\n", volume->name);

	if (memcmp(volume->name, EVMS_DEV_NODE_PATH,
		   EVMS_DEV_NODE_PATH_LEN) != 0) {
		LOG_ERROR("Volume %s does not have the \"%s\" prefix.\n",
			  volume->name, EVMS_DEV_NODE_PATH);
		rc = EINVAL;
		goto out;
	}

	base_name = volume->name + EVMS_DEV_NODE_PATH_LEN;

	switch (dm_version_major) {
	case 3:
		rc = dm_update_status_v3(base_name,
					 &active, &read_only,
					 &volume->dev_major, &volume->dev_minor);
		break;
	case 4:
		rc = dm_update_status_v4(base_name,
					 &active, &read_only,
					 &volume->dev_major, &volume->dev_minor);
		break;
	default:
		rc = EINVAL;
	}
	if (rc) {
		goto out;
	}

	volume->flags |= (active    ? VOLFLAG_ACTIVE    : 0) |
			 (read_only ? VOLFLAG_READ_ONLY : 0);

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_targets
 *
 * @object:	Object to retrieve target information for.
 * @target_list:Return pointer to list of targets.
 *
 * Query device-mapper for the set of targets that comprise the specified
 * object. This function sets target_list to point at a list of targets. When
 * the caller is done using this list, it must be freed with a call to
 * dm_deallocate_targets().
 **/
int dm_get_targets(storage_object_t *object, dm_target_t **target_list)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!object || !target_list) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to get the targets for object %s\n", object->name);

	switch (dm_version_major) {
	case 3:
		rc = dm_get_targets_v3(object->name, target_list);
		break;
	case 4:
		rc = dm_get_targets_v4(object->name, target_list);
		break;
	default:
		rc = EINVAL;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_volume_targets
 *
 * @volume:	Volume to retrieve target information for.
 * @target_list:Return pointer to list of targets.
 *
 * Query device-mapper for the set of targets that comprise the specified
 * volume. This function sets target_list to point at a list of targets. When
 * the caller is done using this list, it must be freed with a call to
 * dm_deallocate_targets().
 **/
int dm_get_volume_targets(logical_volume_t *volume, dm_target_t **target_list)
{
	int rc;
	char *base_name;

	LOG_PROC_ENTRY();

	if (!volume || !target_list) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to get the targets for volume %s\n", volume->name);

	if (memcmp(volume->name, EVMS_DEV_NODE_PATH,
		   EVMS_DEV_NODE_PATH_LEN) != 0) {
		LOG_ERROR("Volume %s does not have the \"%s\" prefix.\n",
			  volume->name, EVMS_DEV_NODE_PATH);
		rc = EINVAL;
		goto out;
	}

	base_name = volume->name + EVMS_DEV_NODE_PATH_LEN;

	switch (dm_version_major) {
	case 3:
		rc = dm_get_targets_v3(base_name, target_list);
		break;
	case 4:
		rc = dm_get_targets_v4(base_name, target_list);
		break;
	default:
		rc = EINVAL;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_load_targets
 *
 * @object:		Object to load new mappings for.
 * @target_list:	List of targets for the new mapping.
 *
 * Load new mappings into the Device-Mapper device for the specified object.
 * The mappings will not be activated until a subsequent call to dm_resume.
 *
 * This API is only available with DM interface version 4.
 **/
int dm_load_targets(storage_object_t * object,
		    dm_target_t * target_list)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!object || !target_list) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	if (!(object->flags & SOFLAG_ACTIVE)) {
		/* Can't load mappings into an inactive device. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to load new targets for object %s\n", object->name);

	/* Create the ASCII parameter strings for all targets. */
	rc = build_target_type_params(target_list);
	if (rc) {
		goto out;
	}

	switch (dm_version_major) {
	case 4:
		rc = dm_load_targets_v4(object->name, target_list,
					(object->flags & SOFLAG_READ_ONLY));
		break;
	default:
		rc = EINVAL;
	}

out:
	/* Free the target parameter strings. */
	deallocate_target_type_params(target_list);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_clear_targets
 *
 * @object: Object to clear an inactive mapping for.
 *
 * Clear the mappings that were added with dm_load_targets. This can only be
 * called before the device has been resumed.
 *
 * This API is only available with DM interface version 4.
 **/
int dm_clear_targets(storage_object_t * object)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!object) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	if (!(object->flags & SOFLAG_ACTIVE)) {
		/* Can't clear mappings for an inactive device. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to clear the targets for object %s\n", object->name);

	switch (dm_version_major) {
	case 4:
		rc = dm_clear_targets_v4(object->name);
		break;
	default:
		rc = EINVAL;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_info
 *
 * @object:	Object to retrieve target information for.
 * @info:	Return pointer to info string.
 *
 * Query device-mapper for the latest status info for the specified object.
 * This function sets info to point at a newly-allocated string. When the
 * caller is done using that string, it must be freed with a call to the
 * engine_free service. Status info is target-dependent. The caller should
 * know how to interpret the returned string.

 * Currently, this function assumes the desired device has only one target.
 * Thus, only one string will be returned. May change this later to return
 * an array of strings, one for each target in the device. However, since
 * currently only snapshot provides any meaningful info for this call, there
 * is no need for multi-target info.
 **/
int dm_get_info(storage_object_t *object, char ** info)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!object || !info) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to get info for object %s\n", object->name);

	switch (dm_version_major) {
	case 3:
		rc = dm_get_info_v3(object->name, info);
		break;
	case 4:
		rc = dm_get_info_v4(object->name, info);
		break;
	default:
		rc = EINVAL;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_devices
 *
 * @device_list: List of devices returned to caller.
 *
 * Get a list of all devices current active in Device-Mapper.
 *
 * This API is only available with DM interface version 4.
 **/
int dm_get_devices(dm_device_list_t ** device_list)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!device_list) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	switch (dm_version_major) {
	case 4:
		rc = dm_get_devices_v4(device_list);
		break;
	default:
		rc = EINVAL;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_modules
 *
 * @modules_list: List of modules returned to caller.
 *
 * Get a list of all target-modules current loaded in Device-Mapper.
 *
 * This API is only available with DM interface version 4.
 **/
int dm_get_modules(dm_module_list_t ** module_list)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!module_list) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	switch (dm_version_major) {
	case 4:
		rc = dm_get_modules_v4(module_list);
		break;
	default:
		rc = EINVAL;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_wait
 *
 * @object:	Object to wait for.
 * @event_nr:	Number of event to wait for.
 * @info:	Return pointer to info string.
 *
 * Wait for an event from a Device-Mapper device. This call will wait in the
 * kernel until the event occurs. The caller must specify the event-number they
 * wish to wait for. After the wait completes, the event-number will be filled
 * in with the next event-number that will occur for this device. Also, the
 * "info" string will be filled in with the status-info for the device (in the
 * same format used for dm_get_info).
 *
 * This API is only available with DM interface version 4.
 **/
int dm_wait(storage_object_t * object, unsigned int * event_nr, char ** info)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!object || !event_nr || !info) {
		/* Bad parameters. */
		rc = EINVAL;
		goto out;
	}

	LOG_DEBUG("Request to wait on object %s\n", object->name);

	switch (dm_version_major) {
	case 4:
		rc = dm_wait_v4(object->name, event_nr, info);
		break;
	default:
		rc = EINVAL;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_run_command
 *
 * Send an ioctl to the Device-Mapper driver.
 **/
int dm_run_command(void *dmi, unsigned long command)
{
	int rc;

	LOG_PROC_ENTRY();

	switch (dm_version_major) {
	case 3:
		rc = run_command_v3(dmi, command);
		break;
	case 4:
		rc = run_command_v4(dmi, command);
		break;
	default:
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_check_version
 *
 * Query device-mapper for the current version of the ioctl interface. If the
 * kernel's version of the interface does not match what EVMS requires, return
 * an error. This function should only be called by the engine core, so it is
 * not exported as a common-service API.
 *
 * This now checks for both version 3 and 4, and sets the dm_version_major
 * global variable appropriately.
 **/
int dm_check_version(void)
{
	int major = 0, minor, patch;
	int rc;

	LOG_PROC_ENTRY();

	/* Use version 4 to get the version number. If that returns an error,
	 * try the same command with version 3. If version 3 is running, it
	 * will print a harmless warning message in syslog. Hopefully users
	 * will see this warning and start migrating to the new kernel
	 * interface. :)
	 */
	rc = dm_get_version_v4(&major, &minor, &patch);
	if (rc) {
		rc = dm_get_version_v3(&major, &minor, &patch);
	}

	LOG_DEFAULT("Device-Mapper interface version: %d.%d.%d\n",
		    major, minor, patch);

	/* EVMS works with version 3 or 4. */
	if (!(major == 3 || major == 4)) {
		LOG_WARNING("Device-Mapper interface version mismatch.\n");
		LOG_WARNING("  EVMS requires: 3.x.x or 4.x.x\n");
		dm_version_major = 0;
		rc = EINVAL;
	} else {
		dm_version_major = major;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_version
 *
 * Simple API to get the current DM version. The plugins will be able to use
 * this in case they have things they can do that require version 4. A return
 * value of zero means DM isn't open.
 **/
int dm_get_version(void)
{
	LOG_PROC_ENTRY();
	LOG_PROC_EXIT_INT(dm_version_major);
	return dm_version_major;
}

/**
 * dm_set_suspended_flag
 *
 * @suspended: TRUE if any device is suspended, FALSE if otherwise.
 *
 * Set a global flag indicating there is a DM device in a suspended state. The
 * engine will use this flag to determine if it is allowed to do things like
 * print log entries or user messages.
 *
 * This function should be called *before* the device is suspended, and
 * *after* the device is resumed.
 **/
void dm_set_suspended_flag(int suspended)
{
	LOG_PROC_ENTRY();
	LOG_DETAILS("Setting dm_device_suspended to TRUE.\n");
	dm_device_suspended = suspended;
	LOG_DETAILS("Setting dm_device_suspended to FALSE.\n");
	LOG_PROC_EXIT_VOID();
}

