/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/*************************************************************
*
*     file:        lexinit.c
*
*     Purpose:    Reads in ASCII initial data files.
*
*      lexical analyzer version, using lex defs in datafile.lex
*/

#include "include.h"
#include "lex.h"
#include "ytab.h"
#define yylex kb_yylex

/***********************************************************
*
*  Function: reset_web()
*
*  Purpose: Zero out and initialize web storage structures
*
*/

void reset_web()
{
  int j;
  int cflag = web.torus_clip_flag;
  int bflag = web.torus_body_flag;

  
  unload_libraries(); /* unload dynamic libraries */
  clear_symtable();
  clear_globals();
  web.global_count = 0; /* because all other global stuff wiped when web reset */
  reset_skeleton();  /* cleans out web and resets to 0 */
  dymemsize = 0;
  if ( dymem ) { myfree(dymem); dymem = NULL; }

  random_seed = 1;  srand(random_seed); srand48(random_seed);

  /* set total transformation matrix */
  matcopy(view,identmat,HOMDIM,HOMDIM);

  /* reset nonzero flags and stuff */
  transform_gen_count = 0;
  if ( view_transform_gens ) 
     { free_matrix3(view_transform_gens); view_transform_gens = NULL;}

  if ( option_q == 2 ) option_q = 0;
  uminus_flag = 0;
  hessian_slant_cutoff = 0.0;
  last_error = 0;
  check_increase_flag = 0;
  brightness = DEFAULT_BRIGHTNESS;
  volgrads_every_flag = 0;
  zener_drag_flag = 0 ;
  pickvnum = pickenum = pickfnum = 0;
  keep_macros_flag = 0;
  needed_version[0] = 0;
  new_vertex_id = new_edge_id = new_facet_id = new_body_id = 0;
  sqcurve_ignore_constr = 0;
  ackerman_flag = 0;
  ps_colorflag = -1;
  boundary_expr_flag = 0;
  gridflag = -1;
  crossingflag = -1;
  labelflag = -1;
  web.target_tolerance = DEFAULT_TARGET_TOLERANCE;
  autorecalc_flag = 0;
  interp_bdry_param = 0;
  web.headvnum = 1;
  thickness = .001;
  user_thickness_flag = 0;
  last_eigenvalue = 0.0;
  last_hessian_scale = 0.0;
  optparamcount = 0;
  thickenflag = 0;
  rotorder_var = -1;
  quantities_only_flag = everything_quantities_flag = 0;
  linear_metric_mix = .50;
  min_square_grad_flag = 0;
  hessian_linear_metric_flag = 0;
  hess_move_con_flag = 1;
  eigen_neg = eigen_pos = eigen_zero = 0;
  innerflag = outerflag = 1;
  normal_sq_mean_curvature_mi = -1;
  eff_area_sq_mean_curvature_mi = -1;
  sq_mean_curvature_mi = -1;
  mix_sq_mean_curvature_mi = -1;
  star_normal_sq_mean_curvature_mi = -1;
  star_eff_area_sq_mean_curvature_mi = -1;
  star_sq_mean_curvature_mi = -1;
  gravity_quantity_num = -1;
  sq_mean_curv_quantity_num = -1;
  mean_curv_int_quantity_num = -1;
  view_4D_flag = 0;
  hessian_normal_flag = 1;
  hessian_special_normal_flag = 0;
  hessian_normal_perp_flag = 0;
  hessian_double_normal_flag = 0;
  hessian_normal_one_flag = 0;
  hessian_quiet_flag = 1;
  hessian_by_diff_flag = 0;
  hess_debug = 0;
  quiet_go_flag = 0;
  quiet_flag = 0;
  pressure_set_flag = 1;
  self_similar_flag = 0;
  make_pos_def_flag = 0;
  datafile_view_flag = 0;
  post_project_flag = 0;
  read_command_flag = 0;
  ribiere_flag = 1;
  assume_oriented_flag = 0;
  conj_grad_flag = 0;
  mobility_flag = mobility_tensor_flag = 0;
  old_area_flag = 0;
  area_fixed_flag = 0;
  sqgauss_flag = 0;
  square_curvature_flag = 0;
  mean_curv_int_flag = 0;
  boundary_curvature_flag = 0;
  normal_curvature_flag = 0;
  approx_curve_flag = 0;
  kusner_flag = 0;
  klein_metric_flag = 0;
  conf_edge_curv_flag = 0;
  effective_area_flag = 0;
  autochop_flag = 0; autopop_flag = 0;
  runge_kutta_flag = 0;
  web.dimension = 2;
  web.representation = SOAPFILM;
  web.skel[VERTEX].ctrlpts = 1;
  web.skel[EDGE].ctrlpts = 2;
  web.skel[FACET].ctrlpts = 3;
  star_fraction = web.dimension + 1.0;
  areaname = "area";
  web.sdim = DEFAULT_SDIM;
  total_time = 0.0;
  homothety_target = 1.0;
  sym_flags = 0;
  symmetry_name = NULL;
  sym_wrap = NULL;
  sym_form_pullback = NULL;
  sym_inverse = NULL;
  sym_compose = NULL;
  web.modeltype = LINEAR;
  web.lagrange_order = 1;
  web.hide_flag = 1;
  web.torus_clip_flag = cflag;
  web.torus_body_flag = bflag;
  web.torus_flag = 0;
  web.full_flag = 0;
  web.meritfactor = 0.0;
  web.grav_const = 1.0;
  web.symmetric_content = 0;
  web.pressure_flag = 0;
  web.projection_flag = 0;
  web.area_norm_flag = 0;
  web.vol_flag = 0;
  web.jiggle_flag = 0;
  web.temperature = 0.05;
  web.total_area = 0.0;
  web.total_facets = 0;
  web.scale = 0.1;
  web.scale_scale = 1.0;
  web.maxscale = 1.0;
  web.pressure = 0.0;
  web.bodycount = 0;
  web.wulff_count = web.wulff_flag = 0;
  web.min_area = 0.1;
  web.min_length = 0.25;
  web.max_len = 1.4;
  web.max_angle = 0.1;
  web.spring_constant = 1.0;
  web.gauss1D_order = 3;  /* can do 3  degree exactly (need default 7 in quadratic)*/
  set_by_user_gauss_1D = 0;
  web.gauss2D_order = 6; /* can do 6th degree poly */ 
  set_by_user_gauss_2D = 0;
  web.tolerance = DEFAULT_TOLERANCE;
  reflevel = 0;
  no_refine = 0;
  zoom_number = 1;
  web.zoom_v = NULLVERTEX;
  web.zoom_radius = 99999.0;
  for ( j = 0 ; j < SDIM ; j++ )
     web.torus_period[j][j] = 1.0;
  transform_count = 0;
  transforms_flag = 0;
  transform_expr[0] = '\0';
  if ( view_transform_det )
     { myfree((char*)view_transform_det); view_transform_det = NULL; }
  if ( transform_colors )
     { myfree((char*)transform_colors); transform_colors = NULL; }
  transform_colors_flag = 0;
  phase_flag = 0;
  metric_convert_flag = 0;
  end_geomview_object();
  end_normal_motion();
  for ( j = 0 ; j < NUMELEMENTS ; j++ )
     { tree_copy(show_expr[j],NULL);
         show_expr[j] = NULL;
     }
  for (j = 0 ; j < 127; j++ ) free_expr(single_redefine+j);
  if ( gauss1poly ) { free_matrix(gauss1poly); gauss1poly = NULL; }
  if ( gauss1polyd ) { free_matrix(gauss1polyd); gauss1polyd = NULL; }
  if ( gpoly ) { free_matrix(gpoly); gpoly = NULL; }
  if ( gpolypartial ) { free_matrix3(gpolypartial); gpolypartial = NULL; }

}

/******************************************************************
*
*  Function: initialize()
*
*  Purpose:  read in data file and create initial triangulation.
*
*/


#define MAXLIST 100
#define LINESIZE 1000


/* temporary lists of elements */
  vertex_id *vlist = NULL;
  int vmaxlist;
  edge_id *elist = NULL;
  int emaxlist;
  facet_id *flist = NULL;
  int fmaxlist;
  int facecount; 
  body_id *blist = NULL;
  int bmaxlist;

void initialize()  /* initialize from data file */
{
  int f;
  facetedge_id fe;
  int i,j,k;
  int topflag;
  int dataflag;
  int esize;
  REAL modulus;

  yylex_init();  /* reset lex */
  macro_init();  /* reset macros in parser */
  line_no = 1;
  parse_error_flag = 0;
  parse_errors = 0;
  recovery_flag = 0;
  facecount = 0;
  bare_edge_count = 0;
  verb_flag = 0;
  lists_flag = LISTS_OFF;
  vlist = NULL;
  elist = NULL;
  flist = NULL;
  blist = NULL;
  quantity_init();

  reset_timestamp = top_timestamp = ++global_timestamp;
  graph_timestamp = ++global_timestamp;
  /* read in any header information */
  topflag = 1;
  tok = yylex();
  while ( (tok != 0) && topflag )
    switch ( tok )
     { 
        case KEEP_MACROS_ : keep_macros_flag = 1; tok = yylex();  break;
        case VERSION_:
             tok = yylex();
             strcpy(needed_version,yytext);
             tok = yylex();
             if ( strcmp(needed_version,evolver_version) > 0 )
             { sprintf(errmsg,"\nDatafile %s needs Evolver version at least %s.  This is version %s.\n\n",datafilename,needed_version,evolver_version);
               kb_error(2066,errmsg,RECOVERABLE);
             }
             break;
        case LOAD_LIBRARY_: 
             tok = yylex();
             if ( tok == QUOTATION_ )
                {  load_library(yytext);
                   tok = yylex();
                }
             else
                  kb_error(1976,"LOAD_LIBRARY file name missing.\n",DATAFILE_ERROR);
             recovery_flag = 0;
             break;

        case INTERP_BDRY_PARAM_:
             interp_bdry_param = 1; tok = yylex(); break;

        case EVERYTHING_QUANTITIES_:
             /*everything_quantities_flag = quantities_only_flag = 1;*/
             if ( !option_q) option_q = 2;
             tok = yylex();
             /* should warn if anything else done yet */
             break;

        case SPACE_DIMENSION_:
             if ( (tok = gettok(INTEGER_)) != INTEGER_ )
             { kb_error(1060,"Dimension of space missing.\n",DATAFILE_ERROR);
               break;
             }
             if ( int_val > MAXCOORD )
             { sprintf(msg,"Space dimension too high.  Recompile with -DMAXCOORD=%d as compiler option.\n",int_val);
                kb_error(1061,msg,RECOVERABLE);
             }
             web.sdim = int_val;
             web.skel[BODY].dimension = int_val;
             if ( web.sdim != SDIM )
             { sprintf(errmsg,"This Evolver compiled strictly for space dimension %d.\n",
                  SDIM);
               kb_error(1062,errmsg,RECOVERABLE);
             }
             tok = yylex();
             break;

        case SURFACE_DIMENSION_:
             if ( (tok = gettok(INTEGER_)) != INTEGER_ )
             { kb_error(1063,"Dimension of surface missing.\n",DATAFILE_ERROR);
               break;
             }
             web.dimension = int_val;
             if ( int_val > web.sdim )
                kb_error(1064,"Surface dimension higher than space dimension.\n",RECOVERABLE);

             star_fraction = web.dimension + 1.0;
             if ( web.representation == SIMPLEX )
             { web.skel[EDGE].ctrlpts = web.dimension;
                web.skel[EDGE].dimension = web.dimension - 1;
                web.skel[FACET].ctrlpts = web.dimension+1;
                web.skel[FACET].dimension = web.dimension;
             }
             else if ( web.dimension == 1 ) web.representation = STRING;
             tok = yylex();
             break;

        case SIMPLEX_REP_:
             web.representation = SIMPLEX;
             web.skel[EDGE].ctrlpts = web.dimension;
             web.skel[EDGE].dimension = web.dimension - 1;
             web.skel[FACET].ctrlpts = web.dimension+1;
             web.skel[FACET].dimension = web.dimension;
             tok = yylex();
             break;

        case DEFINE_:  /* extra attribute definition */
          { int dim,e_type=0,attr_type=0;
             char name[ATTR_NAME_SIZE+1];
             tok = yylex();
             switch ( tok )
             { case VERTICES_: e_type = VERTEX; break;
               case EDGES_:     e_type = EDGE; break;
               case FACES_:     e_type = FACET; break;
               case BODIES_:    e_type = BODY;  break;
               case FACETEDGES_: e_type = FACETEDGE; break;
               default: kb_error(1065,"Bad element type in 'define'.\n",DATAFILE_ERROR);
             };
             tok = yylex();
             if ( tok != ATTRIBUTE_ )
                kb_error(1066,"Need ATTRIBUTE keyword.\n",DATAFILE_ERROR);

             tok = yylex();
             if ( tok != NEWIDENT_)
                kb_error(1067,"Need new identifier. \n",DATAFILE_ERROR);

             strncpy(name,yytext,ATTR_NAME_SIZE);
             tok = yylex();
             switch ( tok )
             { case REAL_TYPE_ : attr_type = REAL_ATTR; break;
                case INTEGER_TYPE_: attr_type = INTEGER_ATTR; break;
                case ULONG_TYPE_: attr_type = ULONG_ATTR; break;
                default: kb_error(1068,"Need attribute datatype.\n",DATAFILE_ERROR);
             };
             tok = yylex();
             if ( tok == '[' )
             { 
                tok = yylex();
                if ( tok != INTEGER_ )
                  kb_error(1069,"Need dimension number.\n",DATAFILE_ERROR);
                dim = int_val;
                if ( dim < 1 )
                  kb_error(1070,"Attribute dimension must be at least 1.\n",DATAFILE_ERROR);
                tok = yylex();  
                tok = yylex(); /* eat ] */
             }
             else dim = 1;

             if ( tok == FUNCTION_ )
             { /* have calculable attribute */
               kb_error(4354,"Sorry, attribute function definitions cannot be in top of datafile.\nOmit function code in top, and do full definition in READ section.\n",
                     DATAFILE_ERROR);
             }
             else add_attribute(e_type,name,attr_type,dim,DUMP_ATTR,NULL);
            
            } /* end DEFINE */
            break;
          
        case CONFORMAL_:  /* have background metric on domain */
             web.metric_flag = 1;
             web.conformal_flag = 1;
             recovery_flag = 0;
             uminus_flag = 0;
             if (exparse(SDIM,&web.metric[0][0],USERCOPY) <= 0 )
                { sprintf(errmsg,"Bad conformal metric definition.\n");
                  kb_error(1071,errmsg,DATAFILE_ERROR);
                }
             metric = dmatrix(0,SDIM-1,0,SDIM-1);
             metric_partial = dmatrix3(SDIM,SDIM,SDIM);
             det_array = dmatrix(0,web.dimension-1,0,web.dimension-1);
             tok = yylex();
             break;
             
        case KLEIN_METRIC_:
             klein_metric_flag = 1;
             web.metric_flag = 1;
             tok = yylex();
             break;
          
        case METRIC_:  /* have background metric on domain */
             web.metric_flag = 1;
             recovery_flag = 0;
             lists_flag = LISTS_SOME;
             uminus_flag = 1;
             for ( i = 0 ; i < SDIM ; i++ )
                for ( j = 0 ; j < SDIM ; j++ )
                { 
                  esize = exparse(SDIM,&web.metric[i][j],USERCOPY);
                  if ( esize <= 0 )
                  { sprintf(errmsg,
                        "Bad metric g[%d][%d] definition.\n",i,j);
                    kb_error(1072,errmsg,DATAFILE_ERROR);
                  }
                }
             metric = dmatrix(0,SDIM-1,0,SDIM-1);
             metric_partial = dmatrix3(SDIM,SDIM,SDIM);
             det_array = dmatrix(0,web.dimension-1,0,web.dimension-1);
             lists_flag = LISTS_OFF;
             tok = yylex();
             break;
             
        case SYMMETRY_GROUP_:
             web.symmetry_flag = 1;
             tok = yylex();
             if ( tok == QUOTATION_ )
                { struct sym_registry *reg;
                  for ( reg = sym_register ; reg->name != NULL ; reg++ )
                    if ( stricmp(yytext,reg->name) == 0 )
                      {
                         symmetry_name = reg->name;
                         sym_wrap = reg->wrapper;
                         sym_form_pullback = reg->pullback;
                         sym_inverse = reg->inverse;
                         sym_compose = reg->compose;
                         sym_flags = reg->flags;
                         break;
                      }
                  if ( reg->name == NULL ) /* search failed */
                      { sprintf(errmsg,
                          "Symmetry name '%s' not found in registry.c \n", yytext);
                          kb_error(1073,errmsg,DATAFILE_ERROR);

                      }
                    tok = yylex();
                }
             else
                  kb_error(1074,"Missing symmetry group name.\n",DATAFILE_ERROR);
             recovery_flag = 0;
             break;


        case TORUS_:
             web.torus_flag = 1;
             web.symmetry_flag = 1;
             sym_wrap = torus_wrap;
             sym_form_pullback = torus_form_pullback;
             sym_inverse = torus_inverse;
             sym_compose = torus_compose;
             sym_flags = 0;
             tok = yylex();
             recovery_flag = 0;
             break;

        case TORUS_FILLED_:
             web.torus_flag = 1;
             web.symmetry_flag = 1;
             sym_wrap = torus_wrap;
             sym_form_pullback = torus_form_pullback;
             sym_inverse = torus_inverse;
             sym_compose = torus_compose;
             sym_flags = 0;
             web.full_flag = 1;
             tok = yylex();
             recovery_flag = 0;
             break;

        case STRING_:
             web.dimension = 1;
             web.representation = STRING;
             web.skel[EDGE].dimension = 1;
             web.skel[FACET].dimension = 2;
             star_fraction = web.dimension + 1.0;
             tok = yylex();
             recovery_flag = 0;
             break;

        case SOAPFILM_:
             web.dimension = 2;
             web.representation = SOAPFILM;
             star_fraction = web.dimension + 1.0;
             tok = yylex();
             recovery_flag = 0;
             break;

        case LINEAR_:
             web.modeltype = LINEAR;
             tok = yylex();
             recovery_flag = 0;
             break;
              
        case QUADRATIC_:
             linear_to_quad();
             tok = yylex();
             recovery_flag = 0;
             break;

        case LAGRANGE_:
             switch ( web.modeltype )
             { case LINEAR: linear_to_lagrange(1); break;
               case QUADRATIC: quad_to_lagrange(1); break;
               case LAGRANGE:  lagrange_to_lagrange(1); break;
             }
             tok = yylex();
             recovery_flag = 0;
             break;

        case LAGRANGE_ORDER_:
             if ( (tok = gettok(INTEGER_)) != INTEGER_ )
             { kb_error(2407,"Lagrange order missing.\n",DATAFILE_ERROR);
               break;
             }
             switch ( web.modeltype )
             { case LINEAR: linear_to_lagrange(int_val); break;
               case QUADRATIC: quad_to_lagrange(int_val); break;
               case LAGRANGE:  lagrange_to_lagrange(int_val); break;
             }
             tok = yylex();
             recovery_flag = 0;
             break;
              
        case SYMMETRIC_CONTENT_:
             web.symmetric_content = 1;
             tok = yylex();
             recovery_flag = 0;
             break;
              
        case MEAN_CURV_:
             web.area_norm_flag = 1;
             tok = yylex();
             recovery_flag = 0;
             break;

        case BOUNDARY_CURVATURE_:
             boundary_curvature_flag = 1;
             tok = yylex();
             recovery_flag = 0;
             break;

        case EFFECTIVE_AREA_:
             effective_area_flag = 1;
             tok = yylex();
             recovery_flag = 0;
             break;
              
        case JIGGLE_:
             web.jiggle_flag = 1;
             tok = yylex();
             recovery_flag = 0;
             break;
              

        case WULFF_: 
             tok = yylex();
             if ( tok == QUOTATION_ )
                {  wulff_initialize(yytext);
                   tok = yylex();
                }
             else
                  kb_error(1076,"Wulff file name missing.\n",DATAFILE_ERROR);
             recovery_flag = 0;
             break;

        case PHASEFILE_: 
             tok = yylex();
             if ( tok == QUOTATION_ )
                {  phase_initialize(yytext);
                   tok = yylex();
                }
             else
                  kb_error(1077,"Cannot find phasefile name.\n",DATAFILE_ERROR);

             recovery_flag = 0;
             break;

        case PERIODS_:  recovery_flag = 0; 
                lists_flag = LISTS_SOME;
                uminus_flag = 1;
                read_periods();
                lists_flag = LISTS_OFF;
                tok = yylex(); /* lookahead */ 
                break;

        case VIEW_MATRIX_: recovery_flag = 0; 
             lists_flag = LISTS_SOME;
             uminus_flag = 1;
             for ( i = 0 ; i <= SDIM ; i++ )
              { for ( j = 0 ; j <= SDIM ; j++ ) 
                 { 
                    if ( read_const(&view[i][j]) <= 0 )
                     { kb_error(1861,"Not enough values for view matrix.\n",DATAFILE_ERROR);
                        break;
                     }
                 }     
              }
             datafile_view_flag = 1;
             lists_flag = LISTS_OFF;
             if ( i > SDIM ) tok = yylex();  /* lookahead */
             break;

        case VIEW_TRANSFORMS_: recovery_flag = 0; uminus_flag = 1; read_transforms(0); break;
        case VIEW_TRANSFORM_GENS_: recovery_flag = 0; 
             uminus_flag = 1;
             read_transform_generators(0); break;

        case PARAMETERS_:  recovery_flag = 0; uminus_flag = 0; read_parameter(); break;

        case OPTIMIZING_PARAMETER_:  recovery_flag = 0; uminus_flag = 0; read_parameter(); break;

        case BOUNDARY_:  recovery_flag = 0; uminus_flag = 0; read_boundary(); break;

        case CONSTRAINT_:  recovery_flag = 0; uminus_flag = 0; read_constraint(); break;

        case SURFACE_ENERGY_: recovery_flag = 0; uminus_flag = 0; read_surface_energy(); break;

        case QUANTITY_: recovery_flag = 0; uminus_flag = 0; read_quantity(); break;

        case METHOD_INSTANCE_: recovery_flag = 0; uminus_flag = 0; read_method_instance(); break;

        case AREA_FIXED_:
             kb_error(3860,"Area_fixed is obsolete.  Replace with named quantity.\n",DATAFILE_ERROR);
             area_fixed_flag = 1;
             uminus_flag = 0;
             if ( read_const(&area_fixed_target) <= 0 )
             { kb_error(1078,"Missing fixed area value.\n",DATAFILE_ERROR);
                if ( tok == AREA_FIXED_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             recovery_flag = 0;
             break;

        case DIFFUSION_:
             web.diffusion_flag = 1;
             web.diffusion_const = 0.0;
             uminus_flag = 0;
             if ( read_const(&web.diffusion_const) <= 0 )
             { kb_error(1079,"Missing DIFFUSION value.\n",WARNING);
                if ( tok == DIFFUSION_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             recovery_flag = 0;
             break;

        case HOMOTHETY_:
             web.homothety = 1;
             uminus_flag = 0;
             if ( read_const(&homothety_target) <= 0 )
                homothety_target = 1.0; 
             else tok = yylex();
             break;

        case AUTOPOP_:
             autopop_flag = 1;
             tok = yylex();
             recovery_flag = 0;
             break;
              
        case AUTOCHOP_:
             autochop_flag = 1;
             autochop_size = 1.0;
      uminus_flag = 0;
             if ( read_const(&autochop_size) <= 0 )
             { kb_error(1080,"Missing AUTOCHOP length.\n",WARNING);
               if ( tok == AUTOCHOP_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             recovery_flag = 0;
             break;
             
        case APPROX_CURV_:
             approx_curve_flag = 1;
             tok = yylex();
             recovery_flag = 0;
             break;

        case CONDUCTING_KNOT_ENERGY_:
             { tok = yylex();
               if ( tok != MODULUS_ ) unput_tok();
               uminus_flag = 0;
               if (read_const(&modulus) <= 0) modulus = 1.0;
               else tok = yylex();
               add_standard_quantity("knot_energy",modulus);
               recovery_flag = 0;
               break;
             }
             
        case INSULATING_KNOT_ENERGY_:
              { tok = yylex();
                if ( tok != MODULUS_ ) unput_tok();
                uminus_flag = 0;
                if (read_const(&modulus) <= 0) modulus = 1.0;
                else tok = yylex();
                add_standard_quantity("uniform_knot_energy",modulus);
                recovery_flag = 0;
                break;
              }
             
        case MEAN_CURV_INT_:
             mean_curv_int_flag = 1;
             mean_curvature_param = add_global("mean_curvature_modulus");
             globals[mean_curvature_param].flags |= ORDINARY_PARAM | SURFACE_PARAMETER;
             uminus_flag = 0;
             if ( read_const(&globals[mean_curvature_param].value.real) <= 0 )
             { kb_error(1081,"Missing integral mean curvature modulus value.\n",WARNING);
                if ( tok == MEAN_CURV_INT_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             if ( web.representation != SOAPFILM )
                  kb_error(1082,"Can do integral of mean curvature only in SOAPFILM model.\n",
                     DATAFILE_ERROR);
             recovery_flag = 0;
             break;

        case GAUSS_CURVATURE_:
              { tok = yylex();
                if ( tok != MODULUS_ ) unput_tok();
                uminus_flag = 0;
                if (read_const(&modulus) <= 0) modulus = 1.0;
                else tok = yylex();
                if ( (web.representation != SOAPFILM) )
                  kb_error(1083,"Can do gauss curvature only in SOAPFILM model.\n",
                     DATAFILE_ERROR);
                else add_standard_quantity("gauss_curvature_integral",modulus);
                recovery_flag = 0;
                break;
              }
             
        case SQGAUSS_:
             sqgauss_flag = 1;
             sqgauss_param = add_global("square_gauss_modulus");
             globals[sqgauss_param].flags |= ORDINARY_PARAM | SURFACE_PARAMETER;
             uminus_flag = 0;
             if ( read_const(&globals[sqgauss_param].value.real) <= 0 )
             { kb_error(1084,"Missing square gaussian modulus value.\n",WARNING);
                if ( tok == SQGAUSS_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             if ( web.representation != SOAPFILM )
                 kb_error(1085,"Can do square gauss curvature only in SOAPFILM model.\n",
                     DATAFILE_ERROR);
             recovery_flag = 0;
             break;

        case SQUARE_CURVATURE_:
             square_curvature_flag = 1;
             square_curvature_param = add_global("sq_curvature_modulus");
             globals[square_curvature_param].flags |= ORDINARY_PARAM | SURFACE_PARAMETER;
             uminus_flag = 0;
             if ( read_const(&globals[square_curvature_param].value.real) <= 0 )
             { kb_error(1086,"Missing square curvature modulus value.\n",WARNING);
               if ( tok == SQUARE_CURVATURE_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             recovery_flag = 0;
             break;

        case NORMAL_CURVATURE_:
             normal_curvature_flag = 1;
             tok = yylex();
             break;

        case MOBILITY_:
             recovery_flag = 0;
             uminus_flag = 0;
             esize = exparse(SDIM,&mobility_formula,USERCOPY);
             tok = yylex();
             if ( esize <= 0 )
                kb_error(1087,"Bad mobility definition.\n",DATAFILE_ERROR);
             else mobility_flag = 1; 
             break;
             
        case MOBILITY_TENSOR_:
             mobility_flag = 1;
             mobility_tensor_flag = 1;
             recovery_flag = 0;
             lists_flag = LISTS_SOME;
             uminus_flag = 1;
             for ( i = 0 ; i < SDIM ; i++ )
                for ( j = 0 ; j < SDIM ; j++ )
                { esize = exparse(SDIM,&mobility_tensor[i][j],USERCOPY);
                  if ( esize <= 0 )
                  { sprintf(errmsg,
                       "Bad mobility_tensor[%d][%d] definition.\n",i,j);
                    kb_error(1088,errmsg,DATAFILE_ERROR);
                  }
                }
             lists_flag = LISTS_OFF;
             tok = yylex();
             break;
             
        case RUNGE_KUTTA_: runge_kutta_flag = 1; tok = yylex(); break;

        case SCALE_LIMIT_:
             uminus_flag = 0;
             if ( read_const(&web.maxscale) <= 0 )
             { kb_error(1089,"Missing SCALE_LIMIT value.\n",WARNING);
               if ( tok == SCALE_LIMIT_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             recovery_flag = 0;
             break;

        case TOTAL_TIME_:
             uminus_flag = 0;
             if ( read_const(&total_time) <= 0 )
             { kb_error(1090,"Missing TOTAL_TIME value.\n",WARNING);
                if ( tok == TOTAL_TIME_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             recovery_flag = 0;
             break;

        case ZOOM_RADIUS_:
             uminus_flag = 0;
             if ( read_const(&web.zoom_radius) <= 0 )
             { kb_error(1091,"Missing ZOOM RADIUS value.\n",WARNING);
               if ( tok == ZOOM_RADIUS_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             recovery_flag = 0;
             break;

        case ZOOM_VERTEX_:
             tok = yylex();
             if ( tok != INTEGER_ )
                kb_error(1092,"Missing ZOOM VERTEX number.\n",WARNING);
             else { zoom_number = int_val; tok = yylex(); }
             recovery_flag = 0;
             break;

        case V_INTEGRAL_ORDER:
             tok = yylex();
             if ( tok != INTEGER_ )
                { kb_error(1093,"Missing INTEGRAL_ORDER value.\n",WARNING);
                  break;
                }
             if ( int_val < 1 )
              { sprintf(errmsg,"Invalid INTEGRAL_ORDER value %d.\n",int_val);
                kb_error(1094,errmsg,WARNING);
              }
             else
             { web.gauss1D_order = 2*int_val-1; 
               set_by_user_gauss_1D = web.gauss1D_order;
               web.gauss2D_order = int_val;
               set_by_user_gauss_2D = web.gauss2D_order;
             }
             tok = yylex();
             recovery_flag = 0;
             break;

        case V_INTEGRAL_ORDER_1D:
             tok = yylex();
             if ( tok != INTEGER_ )
                { kb_error(1095,"Missing INTEGRAL_ORDER_1D value.\n",WARNING);
                  break;
                }
             if ( int_val < 1 )
              { sprintf(errmsg,"Invalid INTEGRAL_ORDER_1D value %d.\n",int_val);
                 kb_error(1096,errmsg,WARNING);
              }
             else {  web.gauss1D_order = int_val; 
                     set_by_user_gauss_1D = web.gauss1D_order;
                  }
             tok = yylex();
             recovery_flag = 0;
             break;

        case V_INTEGRAL_ORDER_2D:
             tok = yylex();
             if ( tok != INTEGER_ )
                { kb_error(1097,"Missing INTEGRAL_ORDER_2D value.\n",WARNING);
                  break;
                }
             if ( int_val < 1 )
              { sprintf(errmsg,"Invalid INTEGRAL_ORDER_2D value %d.\n",int_val);
                kb_error(1098,errmsg,WARNING);
              }
             else {  web.gauss2D_order = int_val; 
                     set_by_user_gauss_2D = web.gauss2D_order;
                  }
             tok = yylex();
             recovery_flag = 0;
             break;

        case CONSTRAINT_TOLERANCE_:
             uminus_flag = 0;
             if ( read_const(&web.tolerance) <= 0 )
             { kb_error(1099,"Missing CONSTRAINT_TOLERANCE value.\n",WARNING);
               if ( tok == CONSTRAINT_TOLERANCE_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             if ( web.tolerance <= 0.0 )
             { kb_error(2333,"Tolerance must be positive.\n",WARNING);
               web.tolerance = DEFAULT_TOLERANCE;
             }
             recovery_flag = 0;
             break;

        case MERITFACTOR_:
             uminus_flag = 0;
             if ( read_const(&web.meritfactor) <= 0 )
                kb_error(1100,"Missing MERIT FACTOR value.\n",WARNING);
             else tok = yylex();
             recovery_flag = 0;
             break;

        case GRAV_CONST_:
             uminus_flag = 0;
             if ( read_const(&web.grav_const) <= 0 )
             { kb_error(1101,"Missing GRAVITY_CONSTANT value.\n",WARNING);
               if ( tok == GRAV_CONST_ ) tok = yylex(); /* ensure porgress */
             }
             else 
             { if ( web.grav_const != 0.0 )  web.gravflag = 1;
               tok = yylex();
             }
             recovery_flag = 0;
             break;

        case SPRING_CONSTANT_:
        case GAP_CONSTANT_:
             uminus_flag = 0;
             if ( read_const(&web.spring_constant) <= 0 )
             { kb_error(1102,"Missing GAP_CONSTANT value.\n",WARNING);
               if ( tok == SPRING_CONSTANT_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             recovery_flag = 0;
             break;

        case SCALE_:
             uminus_flag = 0;
             if ( read_const(&web.scale) <= 0 )
             { kb_error(1103,"Missing SCALE value.\n",WARNING);
               if ( tok == SCALE_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             if ( tok == FIXED_ )
             { web.motion_flag = 1;
               tok = yylex();
             }
             recovery_flag = 0;
             break;

        case TEMPERATURE_:
             uminus_flag = 0;
             if ( read_const(&web.temperature) <= 0 )
             { kb_error(1104,"Missing TEMPERATURE value.\n",WARNING);
               if ( tok == TEMPERATURE_ ) tok = yylex(); /* ensure porgress */
             }
             else tok = yylex();
             recovery_flag = 0;
             break;

        case PRESSURE_:
             uminus_flag = 0;
             if ( read_const(&web.pressure) <= 0 )
             { kb_error(1105,"Missing PRESSURE value.\n",WARNING);
               if ( tok == PRESSURE_ ) tok = yylex(); /* ensure porgress */
             }
             else 
             { tok = yylex(); web.pressure_flag = 1; }
             recovery_flag = 0;
             break;

        case VERTICES_: lists_flag = LISTS_FULL; topflag = 0; break;  
            /* done with top stuff */

        default: 
             if ( !recovery_flag )
                { sprintf(errmsg,"Illegal token '%s'.\n",yytext);
                  kb_error(1106,errmsg,PARSE_ERROR);
                }
             tok = yylex();
             break;
     }
  dataflag = 1;
  uminus_flag = 1;
  while ( (tok != 0) && dataflag )
    switch ( tok )
     { 
        case VERTICES_: 
             if ( (web.dimension > 2) && (web.representation != SIMPLEX) )
                kb_error(1107,"Must have simplex representation for surface dimension over 2.\n",UNRECOVERABLE);
             recovery_flag = 0; 
             read_vertices();
             break;

        case EDGES_: 
             recovery_flag = 0; 
             read_edges();
             break;

        case FACES_: 
             recovery_flag = 0; 
             read_faces();
             break;

        case BODIES_: 
             recovery_flag = 0; 
             read_bodies();
             break;

        case READ_: 
             /* read commands from datafile */
             lists_flag = LISTS_OFF;
             read_command_flag = 1; 
             dataflag = 0;  /* end of element data */
             line_no++; /* kludge */
             break;

        default: 
             if ( !recovery_flag )
                { sprintf(errmsg,"Illegal token '%s'.\n",yytext);
                  kb_error(1108,errmsg,PARSE_ERROR);
                }
             brace_depth = parens = in_quote = 0; /* just in case */
             tok = yylex();
             break;
     }
             
  if ( parse_errors ) return;
  
  datafile_flag = 1;  /* got zeroed by pop_commandfd; need to read x1 */
  if ( option_q )
     convert_to_quantities();
  if ( (web.modeltype == LAGRANGE) && !everything_quantities_flag )
  { outstring("Converting to named quantities for Lagrange model.\n");
    option_q = 1;  /* so convert_to_quantities doesn't try to calc */
    convert_to_quantities();
    option_q = 0;
  }
  for ( i = 0 ; i < BDRYMAX ; i++ )
     if ( (web.boundaries[i].attr & (CON_ENERGY|CON_CONTENT)) && !everything_quantities_flag  )
      { outstring("Converting to named quantities for boundary integrals.\n");
        option_q = 1;  /* so convert_to_quantities doesn't try to calc */
        convert_to_quantities();
        option_q = 0;
        break;
      }

  /* set up gaussian quadrature */
  gauss_setup();

  if ( web.dimension == 1 ) areaname = "length";
  else areaname = "area";

  /* create initial triangulation of each face */
  if ( web.representation == SOAPFILM )
    for ( f = 1 ; f <= facecount ; f++ )
     { 
        int edgecount = 0;

        if ( !valid_id(flist[f]) ) continue;

        /* see how many edges, and if we want to refine */
        generate_facet_fe_init();
        while ( generate_facet_fe(flist[f],&fe) )
            edgecount++;
        if ( edgecount > 3 ) face_triangulate(flist[f],edgecount);
     }

  if ( web.torus_flag )
     calc_periods(ADJUST_VOLUMES);  /* adjust torus volume constants */

  /* straighten out facet order around edges */
  if ( web.representation == SOAPFILM )
  { edge_id e_id;
     FOR_ALL_EDGES(e_id)
        fe_reorder(e_id);
  }

  /* put facet-edges on string network if no facets defined */

  if ( web.representation == STRING )
     string_fixup();


  /* phase boundary energies */
  if ( phase_flag && (web.representation == STRING) )
     { edge_id e_id;
        FOR_ALL_EDGES(e_id)
          set_e_phase_density(e_id);
     }
  if ( phase_flag && (web.representation != STRING) )
     { facet_id f_id;
        FOR_ALL_FACETS(f_id)
          set_f_phase_density(f_id);
     }

  /* run preliminary checks */
  if ( web.representation != SIMPLEX  )
     if ( facetedge_check(PRELIMCHECK) || facet_body_check() )
        kb_error(1109,"Bad data file.\n",DATAFILE_ERROR);
  if ( vlist == NULL )
     kb_error(1110,"No vertices found in datafile.\n",RECOVERABLE);
  if ( (elist == NULL) && (web.representation != SIMPLEX) )
     kb_error(1111,"No edges found in datafile.\n",RECOVERABLE);


  if ( vlist ) myfree((char *)vlist);
  if ( elist ) myfree((char *)elist);
  if ( flist ) myfree((char *)flist);
  if ( blist ) myfree((char *)blist);
         
  for ( k = 2, web.simplex_factorial = 1.0 ; k <= web.dimension ; k++ )
     web.simplex_factorial *= k;
  volume_factorial = web.simplex_factorial*SDIM;
  if ( web.representation == SIMPLEX )
    {
      refine_simplex_init();
    }
  else  if ( web.modeltype == LINEAR )
  { calc_facet_energy =  facet_energy_l;
     calc_facet_forces =  facet_force_l;
     calc_facet_volume = facet_volume_l;
     film_grad = film_grad_l;
     calc_edge_energy = edge_energy_l;
     calc_edge_forces  = edge_force_l;
     calc_edge_area = edge_area_l;
     string_grad = string_grad_l;
  }

  if ( autopop_flag )
  { int n;
    if ( web.representation == STRING )
      sprintf(msg,"Number of vertices popped: %d\n", n = verpop_str());
    else
      sprintf(msg,"Number of vertices popped: %d\n", n = edgepop_film());
    outstring(msg);
    if ( n > 0 ) update_display();
  }

  if ( web.homothety && (web.skel[BODY].count == 0) )
     { web.homothety = 0;
       kb_error(1112,"Cannot do homothety without bodies. Homothety OFF.\n",RECOVERABLE);
     }
  datafile_flag = 0;

  if ( sym_flags & NEED_FORM_UNWRAPPING )
     if ( auto_convert_flag ) convert_to_quantities();

}  /* end initialize */


/************************************************************************
*
*  Function: read_extra()
*
*  purpose: read extra attribute of element
*/
void read_extra(el_id)
element_id el_id;
{ char *spot;
  int i,j;
  int type = id_type(el_id);
  REAL value;
  struct extra *ex;

  for ( i = 0, ex = web.skel[type].extras ;
                  i < web.skel[type].extra_count ; i++,ex++ )
     if ( strcmp(ex->name,yytext) == 0 ) break;

  if ( i >= web.skel[type].extra_count )
     kb_error(1113,"Invalid extra attribute.\n",DATAFILE_ERROR);

  spot = get_extra(el_id,i);
  for ( j = 0 ; j < ex->dim ; j++ )
     { if ( read_const(&value) < 0 ) 
        {  kb_error(1114,"Missing attribute value.\n",DATAFILE_ERROR);
            return;
        }
        switch ( ex->type )
        { case REAL_ATTR : ((REAL*)spot)[j] = value; break;
          case INTEGER_ATTR : ((int *)spot)[j] = (int)value; break;
          case ULONG_ATTR : ((unsigned long *)spot)[j] = (unsigned long)value; break;
        }
     }
  tok = yylex(); /* lookahead */
}

/************************************************************************
*
*  Function: read_vertices()
*
*/

void read_vertices()
{
  int k;
  REAL c[MAXCOORD];  /* temporary number buffer */
  int cnum,bnum,pcount;
  struct boundary *bdry;
  struct constraint *constr;
  int more_attr;

  /* allocate space in structures */
  expand_attribute(VERTEX,V_COORD_ATTR,SDIM);
  expand_attribute(VERTEX,V_FORCE_ATTR,SDIM);
  expand_attribute(VERTEX,V_VELOCITY_ATTR,SDIM);
  if ( web.maxparam > 0 )
  { expand_attribute(VERTEX,V_PARAM_ATTR,web.maxparam);
     expand_attribute(VERTEX,V_BOUNDARY_ATTR,1);
  }
  /* read in vertex coordinates */
  vmaxlist = MAXLIST;
  vlist = (vertex_id *)mycalloc(sizeof(vertex_id),vmaxlist);
  if ( tok != VERTICES_  ) 
            kb_error(1115,"Cannot find VERTICES section of the datafile.\n",UNRECOVERABLE);

  tok = yylex();
  while ( tok == LEAD_INTEGER_ )
    { 
      k = int_val;
      if ( k < 1 ) 
          kb_error(2371,"Vertex number must be positive.\n",DATAFILE_ERROR);
      for ( pcount = 0 ; pcount < SDIM ; pcount++ )
         { if ( read_const(&c[pcount]) <= 0 ) break;
            if ( tok == LEAD_INTEGER_ ) { pcount++; break; }
         }
      tok = yylex(); /* get lookahead */

      while ( k >= vmaxlist )
         { int spot = vmaxlist; 
            vlist = (vertex_id *)kb_realloc((char *)vlist,
              (k+MAXLIST)*sizeof(vertex_id),vmaxlist*sizeof(vertex_id));
            vmaxlist = k + MAXLIST;
            for ( ; spot < vmaxlist ; spot ++ ) vlist[spot] = NULLID;
         }
      if ( valid_id(vlist[k]) )
         { sprintf(errmsg,"Duplicate vertex number %d\n",k);
           kb_error(1117,errmsg,DATAFILE_ERROR);
         }
      move_to_free_front(VERTEX,k); /* so id will be k */
      vlist[k] = new_vertex(c,NULLID);
      set_original(vlist[k],k);

      /* attributes */
      if ( web.con_global_map )
         { set_v_global(vlist[k]);
         }
      for ( more_attr = 1 ; more_attr ; )
         switch ( tok )
             {
                 case EXTRA_ATTRIBUTE_:
                    read_extra(vlist[k]);
                    break;

                 case BARE_:
                    set_attr(vlist[k],BARE_NAKED);
                    tok = yylex();
                    break;

                 case AXIAL_POINT_:
                    set_attr(vlist[k],AXIAL_POINT);
                    tok = yylex();
                    break;

                 case ORIGINAL_:
                    if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                    { kb_error(2400,"ORIGINAL number missing.\n",DATAFILE_ERROR);
                      break;
                    }
                    set_original(vlist[k],int_val);
                    tok = yylex();
                    break;

                 case FIXED_:
                    set_attr(vlist[k],FIXED);
                    tok = yylex();
                    break;

                 case METHOD_:  /* apply method instance to edge */
                    tok = yylex();  /* get name */
                 case METHOD_NAME_:
                    apply_method(vlist[k],yytext);
                    tok = yylex();
                    break;

                 case QUANTITY_NAME_: /* name of quantity */
                    apply_quantity(vlist[k],globals[yylval].value.quant);
                    tok = yylex();
                    break;

                 case IDENT_:  /* maybe method or quantity */
                    if ( globals[yylval].flags & METHOD_NAME )
                      apply_method(vlist[k],yytext);
                    else if ( globals[yylval].flags & QUANTITY_NAME )
                      apply_quantity(vlist[k],globals[yylval].value.quant);
                    else 
                    { sprintf(errmsg,"Illegal use of identifier '%s'.\n",yytext);
                      kb_error(1118,errmsg,DATAFILE_ERROR);
                    }
                    tok = yylex();
                    break;


                 case QUANTITY_:
                    tok = yylex();
                     if ( tok == IDENT_ )
                        { /* have named quantity/method pair */
                          char qname[32];
                          strncpy(qname,yytext,sizeof(qname));
                          if ( globals[yylval].flags & QUANTITY_NAME )
                             apply_quantity(vlist[k],globals[yylval].value.quant);
                          else
                          { sprintf(errmsg,"Undefined quantity: %s.\n",yytext);
                            kb_error(1119,errmsg,DATAFILE_ERROR);
                          }
                          tok = yylex();
                          if ( stricmp(yytext,"method")==0 )
                             { tok = yylex();
                                kb_error(1120,"Obsolete quantity syntax.\n  Methods must be listed in quantity definition.",DATAFILE_ERROR);

                                tok = yylex();
                             }
                          continue;
                        }
                     else kb_error(1121,"Need quantity name and method.\n",DATAFILE_ERROR);
                     break;

                 case BOUNDARY_:
                    {
                      REAL *x,*param;
                      int n;

                      tok = gettok(INTEGER_);
                      if ( tok != INTEGER_ )  
                         { kb_error(1122,"Need boundary number.\n",DATAFILE_ERROR);
                            break;
                         }
                      bnum = abs(int_val);
                      if ( (bnum >= BDRYMAX) 
                                || (web.boundaries[bnum].coordf[0] == NULL) )
                         {
                            sprintf(errmsg,
                                "Bad boundary number %d for vertex %d.\n",bnum,k);
                            kb_error(1123,errmsg,DATAFILE_ERROR);
                            yylex();
                            break;
                         }          
                      set_attr(vlist[k],BOUNDARY);
                      if ( int_val < 0 )  set_attr(vlist[k],NEGBOUNDARY); 
                      set_boundary_num(vlist[k],bnum);
                      bdry = get_boundary(vlist[k]); 
                      if ( pcount != bdry->pcount )
                        { sprintf(errmsg, "Wrong number of parameters for vertex %d.\n",k);
                          kb_error(1124,errmsg,DATAFILE_ERROR);
                        }          
                      if ( (bdry->attr & CON_ENERGY) && (yytext[0] != '0') )
                         set_attr(vlist[k], BDRY_ENERGY);
                      if ( bdry->attr & CON_CONTENT )
                         set_attr(vlist[k], BDRY_CONTENT);
                      param = get_param(vlist[k]);
                      x = get_coord(vlist[k]);
                      for ( n = 0 ; n < web.maxparam ; n++ )
                            param[n] = x[n];
                      /* initial coordinate calculation later, after all info */
                      tok = yylex();
                    }
                break;

      /* see if constraint point */
      case CONSTRAINT_:
            while ( (tok = gettok(INTEGER_)) == INTEGER_ )
              {
                 cnum = abs(int_val);
                 if ( (cnum >= MAXCON) || !web.constraint_addr[cnum] )
                    {
                      sprintf(errmsg,
                          "Bad constraint number %d for vertex %d.\n",cnum,k);
                      kb_error(1125,errmsg,DATAFILE_ERROR);
                      tok = yylex();
                      break;
                    }
                 set_attr(vlist[k],CONSTRAINT);
                 if ( int_val < 0 )  set_attr(vlist[k],NEGBOUNDARY); 
                 set_v_constraint_map(vlist[k],cnum);
                 constr = get_constraint(cnum); 
                 if ( (constr->attr & CON_ENERGY) && (yytext[0] != '0') )
                    set_attr(vlist[k], BDRY_ENERGY);
                 if ( constr->attr & CON_CONTENT )
                    set_attr(vlist[k], BDRY_CONTENT);
              }
            /* projection later, after all vertex info read in */
            break;

            case EDGES_: case FACES_: case BODIES_: case READ_: 
            case LEAD_INTEGER_: case NO_TOKEN:
                more_attr = 0 ; break;  /* error recovery */
            default: 
                sprintf(errmsg,"Unexpected token: %s\n",yytext);
                kb_error(2265,errmsg,WARNING);
                tok = yylex();
                break;

          }

      if ((get_vattr(vlist[k])&(BOUNDARY|CONSTRAINT)) == (BOUNDARY|CONSTRAINT))
          kb_error(1126,"Cannot have constraint and boundary.",DATAFILE_ERROR);

      if ( !(get_vattr(vlist[k])&BOUNDARY) && (pcount != SDIM) )
         { sprintf(errmsg,"Wrong number of coordinates for vertex %d.\n",k);
            kb_error(1127,errmsg,WARNING);

          }
      if ( get_vattr(vlist[k]) & BOUNDARY )
      { REAL * x = get_coord(vlist[k]);
         int n;
         bdry = get_boundary(vlist[k]);
          for ( n = 0 ; n < SDIM ; n++ )
             if ( bdry->coordf[n]->root != NULL )
                 x[n] = eval(bdry->coordf[n],get_param(vlist[k]),vlist[k]);
      }
      if ( get_vattr(vlist[k]) & CONSTRAINT )
         project_v_constr(vlist[k],ACTUAL_MOVE);
    }
  web.zoom_v = vlist[zoom_number];  /* vertex for zooming in on */
}  /* end read_vertices() */


/************************************************************************
*
*  Function: read_edges()
*
*/

void read_edges()
{
  int i,k;
  int head,tail;
  int cnum;
  struct boundary *bdry;
  struct constraint *constr;
  int more_attr;
  REAL value;  /* for constant expression values */
  WRAPTYPE wrap;
  int compcount;  /* proper number of components for integrands */
  int numv; /* vertices to read in association with a facet */
  int edim = (web.representation==STRING) ? 1 : web.dimension - 1;

  if ( web.representation == SIMPLEX )
     compcount = binom_coeff(SDIM,edim);
  else compcount = SDIM; 

  /* optional attributes */
  expand_attribute(EDGE,E_VERTICES_ATTR,web.skel[EDGE].ctrlpts);
  if ( web.surfen_count )
     expand_attribute(EDGE,E_SURFEN_MAP_ATTR,1);
  if ( web.quantity_count ) /* numbered */
     expand_attribute(EDGE,E_SURFEN_MAP_ATTR,1);
  if ( web.maxparam )
     expand_attribute(EDGE,E_BOUNDARY_ATTR,1);
  if ( web.symmetry_flag )
     expand_attribute(EDGE,E_WRAP_ATTR,1);

  if ( web.representation == SIMPLEX )
  { if ( web.modeltype == LAGRANGE )
        numv = binom_coeff(web.lagrange_order+edim,edim); 
     else numv = web.dimension;
  }
  else if ( web.modeltype == LAGRANGE )
  { numv = binom_coeff(web.lagrange_order+edim,edim);
  }
  else if ( web.modeltype == QUADRATIC ) numv = 3;
  else numv = 2;

  /* read in edges */
  emaxlist = MAXLIST;
  elist = (edge_id *)mycalloc(sizeof(edge_id),emaxlist);
  while ( (tok != EDGES_) && (tok != 0 ) ) 
     tok = yylex();
  if ( tok != EDGES_ ) return;
  tok = yylex();
  while ( tok == LEAD_INTEGER_ )
    { int have_mid = 0;
    
      wrap = 0;

      /* check edge number */
      k = int_val;
      if ( k < 1 ) 
          kb_error(2373,"Edge number must be positive.\n",DATAFILE_ERROR);
      while ( k >= emaxlist )
         { int spot = emaxlist; 
            elist = (edge_id *)kb_realloc((char *)elist,
              (k+MAXLIST)*sizeof(edge_id),emaxlist*sizeof(edge_id));
            emaxlist= k + MAXLIST;
            for ( ; spot < emaxlist ; spot ++ ) elist[spot] = NULLID;
         }
      if ( valid_id(elist[k]) )
         { sprintf(errmsg,"Duplicate edge number %d\n",k);
            kb_error(1130,errmsg,DATAFILE_ERROR);
         }

      { /* read vertex list */
			int vercount;
         vertex_id *v,*vv;

         move_to_free_front(EDGE,k); /* so id will be k */
         elist[k] = new_edge(NULLID,NULLID,NULLID);
         vv = v = get_edge_vertices(elist[k]);
         for ( vercount = 0 ; vercount < numv ; vercount++ )
         { if ( (tok = gettok(INTEGER_)) != INTEGER_ )
            { kb_error(1131,"Too few vertices for edge.\n",DATAFILE_ERROR); 
              return; 
            }
            if ( !valid_id(vlist[int_val]) )
             { sprintf(errmsg,"Edge %d: vertex %d is not defined.\n",k,int_val);
                kb_error(1132,errmsg,DATAFILE_ERROR);
                return;
             }
             *(v++) = vlist[int_val];
          }
         if ( web.modeltype == QUADRATIC )
         { head = vv[1]; tail = vv[0]; set_edge_midv(elist[k],vv[2]); }
         else { head = vv[numv-1]; tail = vv[0]; }
         set_edge_headv(elist[k],head);
         set_edge_tailv(elist[k],tail);
         if ( web.modeltype == QUADRATIC )
         { have_mid =1; /* for later */
         }
         else if ( web.modeltype == LAGRANGE )
         { for ( i = 1 ; i < numv-1 ; i++ )
            { set_attr(vv[i],Q_MIDEDGE); 
              set_vertex_edge(vv[i],elist[k]);
            }
         }
      } 
      tok = yylex();
      if ( web.torus_flag )
         { read_wrap_flag = 1;
            for ( i = 0 ; i < SDIM  ; i++, tok = yylex() )
              switch ( tok )
                 {
                    case '+':
                        wrap += POSWRAP << (i*TWRAPBITS);
                        break;

                    case '*':  break;

                    case '-':
                    case UMINUS_:
                        wrap += NEGWRAP << (i*TWRAPBITS);
                        break;

                    case POW: /* ** */
                                 i++; break;
                    default :
                      kb_error(1133,"Edge wraps must immediately follow endpoints.\n",
                          WARNING);
                      break;
                }
         }
      read_wrap_flag = 0;

      set_original(elist[k],k);

      if ( web.representation == STRING )
         set_edge_density(elist[k],1.0);
      else
         set_edge_density(elist[k],0.0);
      /* check attributes */
      for ( more_attr = 1; more_attr ; )
         switch ( tok )
            {
                 case EXTRA_ATTRIBUTE_:
                    read_extra(elist[k]);
                    break;

                 case ORIENTATION_:
                    tok = gettok(INTEGER_);
                    if ( tok != INTEGER_ )
                    { kb_error(2401,"ORIENTATION value missing.\n",DATAFILE_ERROR);
                      break;
                    }
                    if ( int_val < 0 ) set_attr(elist[k],NEGBOUNDARY);
                    tok = yylex();
                    break;

                 case NO_REFINE_:
                    set_attr(elist[k],NO_REFINE);
                    tok = yylex();
                    break;

                 case BARE_:
                    set_attr(elist[k],BARE_NAKED);
                    tok = yylex();
                    break;

                 case ORIGINAL_:
                    if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                    { kb_error(2402,"ORIGINAL number missing.\n",DATAFILE_ERROR);
                      break;
                    }
                    set_original(elist[k],int_val);
                    tok = yylex();
                    break;

              case WRAP_:
                     if ( read_const(&value) < 0 ) 
                        kb_error(1135,"Missing wrap value.\n",DATAFILE_ERROR);
                     else tok = yylex();
                     wrap = (WRAPTYPE)value;
                     if ( !web.symmetry_flag )
                         kb_error(1134,"Cannot do wraps without torus or symmetry group.\n",
                            DATAFILE_ERROR);
                     else set_edge_wrap(elist[k],wrap);
                     break;

              case FIXED_:
                     set_attr(elist[k],FIXED);  
                     tok = yylex();
                     break;

              case EFIXED_: /* edge only, not vertices */ 
                     set_attr(elist[k],FIXED);  
                     tok = yylex();
                     break;

              case COLOR_:
                     if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                     { kb_error(1136,"Color missing.\n",DATAFILE_ERROR);
                       break;
                     }
                     set_edge_color(elist[k],(short)int_val);
                     tok = yylex();
                     break;

              case BOUNDARY_:
                     tok = gettok(INTEGER_);
                     if ( tok != INTEGER_ )
                      { kb_error(2403,"BOUNDARY number missing.\n",DATAFILE_ERROR);
                         break;
                      }
                     cnum = abs(int_val);
                     if ( (cnum >= BDRYMAX) || 
                                   (web.boundaries[cnum].coordf[0] == NULL) )
                        { sprintf(errmsg,
                             "Bad boundary number %d for edge %d.\n",cnum,k);
                          kb_error(1137,errmsg,DATAFILE_ERROR);
                        }
                     set_attr(elist[k],BOUNDARY);
                     if ( int_val < 0 )  set_attr(vlist[k],NEGBOUNDARY); 
                     set_edge_boundary_num(elist[k],cnum);
                     bdry = get_edge_boundary(elist[k]); 
                     if ( (bdry->attr & CON_ENERGY) && (yytext[0] == '0') )
                        set_attr(elist[k], BDRY_ENERGY);
                     if ( bdry->attr & CON_CONTENT )
                        set_attr(elist[k], BDRY_CONTENT);
                     tok = yylex();
                     break;

            case CONSTRAINT_:
            while ( (tok = gettok(INTEGER_)) == INTEGER_ )
              { struct constraint *con;
                 cnum = abs(int_val);
                 if ( (cnum >= MAXCON) || !web.constraint_addr[cnum] )
                    { sprintf(errmsg,
                          "Bad constraint number %d for edge %d.\n",cnum,k);
                      kb_error(1138,errmsg,DATAFILE_ERROR);
                    }
                 con = get_constraint(cnum);
                 /* check consistency of number of components */
                 if ( con->attr & (CON_ENERGY | CON_CONTENT) )
                    if ( con->compcount != compcount )
                      kb_error(1139,"Inconsistent number of components in constraint edge content integrands.\n",
                        WARNING);
                 set_attr(elist[k],CONSTRAINT);
                 if ( int_val < 0 )  set_attr(elist[k],NEGBOUNDARY); 
                 set_e_constraint_map(elist[k],cnum);
                 constr = get_constraint(cnum); 
                 if ( (constr->attr & CON_ENERGY) && (yytext[0] != '0') )
                    set_attr(elist[k], BDRY_ENERGY);
                 if ( constr->quantity_map ) /* quantities done with energies */
                    set_attr(elist[k], BDRY_ENERGY);
                 if ( constr->attr & CON_CONTENT )
                    set_attr(elist[k], BDRY_CONTENT);
              }
            break;

            case DENSITY_:
                 if ( read_const(&value) <= 0 )
                    kb_error(1140,"Missing DENSITY value.\n",WARNING);
                 else tok = yylex();
                 set_attr(elist[k],DENSITY);
                 set_edge_density(elist[k],value);
                 break;

          case QUANTITY_NAME_: /* name of quantity */
             apply_quantity(elist[k],globals[yylval].value.quant);
             tok = yylex();
             break;

          case IDENT_:  /* maybe method or quantity */
             if ( globals[yylval].flags & METHOD_NAME )
                apply_method(elist[k],yytext);
             else if ( globals[yylval].flags & QUANTITY_NAME )
                apply_quantity(elist[k],globals[yylval].value.quant);
             else
              { sprintf(errmsg,"Illegal use of identifier '%s'.\n",yytext);
                kb_error(1141,yytext,DATAFILE_ERROR);
              }
             tok = yylex();
             break;

          case METHOD_:  /* apply method instance to edge */
             tok = yylex();
          case METHOD_NAME_:
          { char name[100];
             strcpy(name,yytext);
             tok = yylex();
             if ( tok == UMINUS_ )
             { apply_method(inverse_id(elist[k]),name);
                tok = yylex();
             }
             else apply_method(elist[k],name);
             break;
          }

          case QUANTITY_:
          /* see if quantity */
                tok = gettok(INTEGER_);
                for ( i = 0 ; i < QUANTMAX ; i++, tok = gettok(INTEGER_) ) 
                  {
                     if ( tok == IDENT_ )
                        { /* have named quantity/method pair */
                          char qname[32];
                          strncpy(qname,yytext,sizeof(qname));
                          if ( globals[yylval].flags & QUANTITY_NAME )
                             apply_quantity(elist[k],globals[yylval].value.quant);
                          else 
                          { sprintf(errmsg,"Undefined quantity '%s'.\n",yytext);
                            kb_error(1142,errmsg,DATAFILE_ERROR);
                          }
                          tok = yylex();
                          if ( stricmp(yytext,"method")==0 )
                            { tok = yylex();
                              kb_error(1143,"Obsolete quantity syntax.\n  Methods must be listed in quantity definition.",DATAFILE_ERROR);
                              tok = yylex();
                            }
                          continue;
                        }
                     else if ( tok == NEWIDENT_ )
                        { sprintf(errmsg,"Undefined quantity: %s\n",yytext);
                          kb_error(1144,errmsg,DATAFILE_ERROR);
                        }
                     else if ( tok != INTEGER_ ) break;
                     if ( tok != INTEGER_ ) break;
                     cnum = int_val;
                     if ( (cnum < 0 ) || (cnum >= QUANTMAX)
                                   || (web.quants[cnum].quanvect[0] == NULL) )
                        { sprintf(errmsg,
                              "Bad quantity number %d for edge %d.\n",cnum,k);
                          kb_error(1145,errmsg,DATAFILE_ERROR);
                        }
                     set_attr(elist[k],SURF_QUANTITY);
                     set_e_quant_map(elist[k],cnum);  
                  }
                break;


          case ENERGY_:
          /* see if surface energy */
                tok = gettok(INTEGER_);
                for ( i = 0 ; i < SURFENMAX ; i++, tok = gettok(INTEGER_) ) 
                  { if ( tok != INTEGER_ ) break;
                    kb_error(1147,"Obsolete. Implement edge integral energy with constraint.\n",WARNING);
                  }
                break;

            case FACES_: case BODIES_: case READ_: case LEAD_INTEGER_: 
            case NO_TOKEN:    more_attr = 0 ; break;  /* error recovery */
            default: 
                sprintf(errmsg,"Unexpected token: %s\n",yytext);
                kb_error(2266,errmsg,WARNING);
                tok = yylex();
                break;

         }

      if ( web.symmetry_flag )
         {  set_edge_wrap(elist[k],wrap);
             if ( (web.modeltype == QUADRATIC) && !have_mid )
              { /* have to adjust midpoints */
                 REAL *x = get_coord(get_edge_midv(elist[k]));
                 REAL *t = get_coord(get_edge_tailv(elist[k]));
                 REAL h[MAXCOORD];
                 (*sym_wrap)(get_coord(get_edge_headv(elist[k])),h,wrap);
                 for ( i = 0 ; i < SDIM ; i++ )
                         x[i] = 0.5*(t[i] + h[i]);
              }
          }
      if ((get_eattr(elist[k])&(BOUNDARY|CONSTRAINT)) == (BOUNDARY|CONSTRAINT))
          kb_error(1148,"Cannot have constraint and boundary.",DATAFILE_ERROR);

     }
} /* end read_edges() */

/************************************************************************
*
*  Function: read_faces()
*
*/

void read_faces()
{
  int i,k;
  int e;
  facetedge_id old_fe;
  facet_id ff_id;
  int cnum,bnum;
  struct constraint *constr;
  REAL value;  /* for constant expression values */
  int numv; /* vertices to read with facet */

  /* optional attributes */
  expand_attribute(FACET,F_VERTICES_ATTR,web.skel[FACET].ctrlpts);

  if ( web.maxparam > 1)
     expand_attribute(FACET,F_BOUNDARY_ATTR,1);
  if ( web.surfen_count )
     expand_attribute(FACET,F_SURFEN_MAP_ATTR,1);
  if ( web.quantity_count ) /* numbered */
     expand_attribute(FACET,F_SURFEN_MAP_ATTR,1);
  if ( phase_flag )
     expand_attribute(FACET,F_PHASE_ATTR,1);

  if ( web.representation == SIMPLEX )
  { if ( web.modeltype == LAGRANGE )
        numv = binom_coeff(web.lagrange_order+web.dimension,web.dimension); 
    else numv = web.dimension;
  }
  else if ( web.modeltype == LAGRANGE )
  { numv = binom_coeff(web.lagrange_order+web.dimension,web.dimension);
  }
  else if ( web.modeltype == QUADRATIC ) numv = 0;
  else numv = 0;

  /* read in faces */
  fmaxlist = MAXLIST;
  flist = (facet_id *)mycalloc(sizeof(facet_id),fmaxlist);

  tok = yylex();
  while ( tok == LEAD_INTEGER_ )
    {
      int more_attr;
      vertex_id hh,tt;
      int edge_count = 0;

      k = int_val;
      if ( k < 1 ) 
          kb_error(2374,"Face number must be positive.\n",DATAFILE_ERROR);
      old_fe = NULLFACETEDGE;
      while ( k >= fmaxlist )
         { int spot = fmaxlist;

            flist = (facet_id *)kb_realloc((char *)flist,
              (k+MAXLIST)*sizeof(facet_id),fmaxlist*sizeof(facet_id));
            fmaxlist = k + MAXLIST;
            for ( ; spot < fmaxlist ; spot++ ) flist[spot] = NULLID;
         }
      if ( valid_id(flist[k]) )
         { sprintf(errmsg,"Duplicate face number %d\n",k);
            kb_error(1152,errmsg,DATAFILE_ERROR);

         }
      move_to_free_front(FACET,k); /* so id will be k */
      flist[k] = new_facet();
      set_original(flist[k],k);
      if ( web.representation == SIMPLEX )
      { /* read vertex list */
         int vercount = 0;
         vertex_id *v = get_facet_vertices(flist[k]);
         while ( (tok = gettok(INTEGER_)) == INTEGER_ )
          { if ( vercount++ > numv )
                kb_error(1153,"Too many vertices for facet.\n",DATAFILE_ERROR);

             if ( !valid_id(vlist[int_val]) )
             { sprintf(errmsg,"Facet %d: vertex %d is not defined.\n",k,int_val);
                kb_error(1154,errmsg,DATAFILE_ERROR);

             }
             *(v++) = vlist[int_val];
          }
         if ( vercount < numv )
            kb_error(1155,"Too few vertices for facet.\n",DATAFILE_ERROR);

      }
      else  /* facet_edge representation */
      { facetedge_id fe = NULLID;
         while ( (tok = gettok(INTEGER_)) == INTEGER_ )
         { 
            edge_id e_id;
            facetedge_id edge_fe;
            
            edge_count++;
            e = int_val;
            if ( abs(e) >= emaxlist )
             { sprintf(errmsg,"Facet %d: edge %d is not defined.\n",k,abs(e));
					 e = 0;
                kb_error(1156,errmsg,DATAFILE_ERROR);
                continue;
             }
            e_id =  e > 0 ? elist[e] : edge_inverse(elist[-e]);

            if ( !valid_id(e_id) )
             { sprintf(errmsg,"Facet %d: edge %d is not defined.\n",k,e);
                kb_error(1157,errmsg,DATAFILE_ERROR);
                continue;
             }

            fe = new_facetedge(flist[k],e_id);
            if ( !valid_id(get_facet_fe(flist[k])) )
              { set_facet_fe(flist[k],fe);}
            else
              {
                 hh = get_fe_headv(old_fe);
                 tt = get_fe_tailv(fe);
                 if ( !equal_id(hh,tt) ) 
                    {sprintf(msg,"Inconsistency in face %d, edge %d tail vertex disagrees with previous head.\n",
                         k,e);
                     kb_error(1976,msg,DATAFILE_ERROR);
                    }
                 set_next_edge(old_fe,fe);
              }
            set_prev_edge(fe,old_fe);
            old_fe = fe;

            /* add to edge facet list, not in geometric order */
            edge_fe = get_edge_fe(e_id);
            if ( valid_id(edge_fe) )
              { /* insert in chain */
                 set_next_facet(fe,get_next_facet(edge_fe));
                 set_prev_facet(fe,edge_fe);
                 set_prev_facet(get_next_facet(fe),fe);
                 set_next_facet(edge_fe,fe);
              }
            else
              { set_next_facet(fe,fe);
                 set_prev_facet(fe,fe);
                 set_edge_fe(e_id,fe);      /* link edge to rest of world */
              }

         }
      if ( ((web.representation == STRING) && (edge_count < 1))
              || ((web.representation == SOAPFILM) && (edge_count < 3)) )
         { sprintf(errmsg,"Face %d has too few edges.\n",k);
            kb_error(1158,errmsg,DATAFILE_ERROR);
         }
      ff_id = get_facet_fe(flist[k]);
      if ( !valid_id(ff_id) ) continue; /* something went wrong */
      tt = get_fe_tailv(ff_id);
      hh = get_fe_headv(fe);
      if ( equal_id(tt,hh) ) 
         {
            set_next_edge(fe,get_facet_fe(flist[k]));  /* close up ring */
            set_prev_edge(get_facet_fe(flist[k]),fe);
         }
      else 
         { if ( web.representation != STRING )
             { sprintf(errmsg,
                  "Inconsistency in face %d first and last edges.\n",k);
                kb_error(1159,errmsg,DATAFILE_ERROR);
             }
         }

      if ( (web.modeltype == LAGRANGE) && (web.representation == SOAPFILM) )
      { /* read vertex list */
         int vercount = 0;
         vertex_id *v;
         if ( tok != VERTICES_ )
            kb_error(1160,"Need facet vertices in Lagrange model.\n",DATAFILE_ERROR);
         v = get_facet_vertices(flist[k]);
         while ( (tok = gettok(INTEGER_)) == INTEGER_ )
          { if ( vercount++ >= numv )
                kb_error(1161,"Too many vertices for edge.\n",DATAFILE_ERROR);
             if ( !valid_id(vlist[int_val]) )
             { sprintf(errmsg,"Facet %d: vertex %d is not defined.\n",k,int_val);
                kb_error(1162,errmsg,DATAFILE_ERROR);
             }
             *(v++) = vlist[int_val];
          }
         if ( vercount < numv )
            kb_error(1163,"Too few vertices for facet.\n",DATAFILE_ERROR);
        }
      } 
      if ( web.modeltype == LAGRANGE )
        { vertex_id *v = get_facet_vertices(flist[k]);
          for ( i = 0 ; i < numv ; i++ ) 
             if ( !(get_vattr(v[i]) & Q_MIDEDGE ) ) set_attr(v[i],Q_MIDFACET);
          for ( i = 0 ; i <= web.dimension ; i++ )
             unset_attr(v[web.skel[FACET].extreme[i]],Q_MIDFACET);
        }

      set_facet_density(flist[k],1.0);
      /* have attributes, maybe */
      for ( more_attr = 1 ; more_attr ; )
        switch ( tok )
         { 
            case EXTRA_ATTRIBUTE_:
                    read_extra(flist[k]);
                    break;

            case ORIENTATION_:
                    if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                    { kb_error(2404,"ORIENTATION value missing.\n",DATAFILE_ERROR);
                      break;
                    }
                    if ( int_val < 0 ) set_attr(flist[k],NEGBOUNDARY);
                    tok = yylex();
                    break;


            case ORIGINAL_:
                    if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                    { kb_error(2405,"ORIGINAL value missing.\n",DATAFILE_ERROR);
                      break;
                    }
                    else 
                    { set_original(flist[k],int_val);
                      tok = yylex();
                    }
                    break;

            case DENSITY_:
                 if ( read_const(&value) <= 0 )
                    kb_error(1164,"Missing DENSITY value.\n",WARNING);
                 else tok = yylex();
                 set_attr(flist[k],DENSITY);
                 set_facet_density(flist[k],value);
                 break;

            case NODISPLAY_:
          /* see if want not to be displayed */
                 set_attr(flist[k],NODISPLAY);
                 tok = yylex();
                 break;

             case NO_REFINE_:
                    set_attr(flist[k],NO_REFINE);
                    tok = yylex();
                    break;

          case FIXED_:
          /* see if fixed in place */
                 set_attr(flist[k],FIXED);
                 tok = yylex();
                 break;

          case PHASE_: 
                if ( !phase_flag )
                  kb_error(1165,"Phases not in effect.\n",DATAFILE_ERROR);

                if ( web.representation != STRING )
                  kb_error(1166,"Phases on facets only in STRING model.\n",DATAFILE_ERROR);

                if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                { kb_error(1167,"Phase missing.\n",DATAFILE_ERROR);
                  break;
                }
                if ( (int_val < 0) || (int_val > phasemax) )
                  kb_error(1168,"Illegal phase value.\n",DATAFILE_ERROR);

                set_f_phase(flist[k],int_val);
                tok = yylex();
                break;

          case COLOR_:
                if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                { kb_error(1169,"Color missing.\n",DATAFILE_ERROR);
                  break;
                }
                set_facet_color(flist[k],(short)int_val);
                tok = yylex();
                break;

          case FRONTCOLOR_:
                if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                { kb_error(1170,"Frontcolor missing.\n",DATAFILE_ERROR);
                  break;
                }
                set_facet_frontcolor(flist[k],(short)int_val);
                tok = yylex();
                break;

          case BACKCOLOR_:
                if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                { kb_error(1171,"Backcolor missing.\n",DATAFILE_ERROR);
                  break;
                }
                set_facet_backcolor(flist[k],(short)int_val);
                tok = yylex();
                break;

          case TAG_:
          /* optional inheritable tag */
                expand_attribute(FACET,F_TAG_ATTR,1);
                if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                { kb_error(1172,"Tag missing.\n",DATAFILE_ERROR);
                  break;
                }
                set_tag(flist[k],(tagtype)int_val);
                tok = yylex();
                break;

          case BOUNDARY_:
          /* see if boundary facet */
                if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                { kb_error(1173,"Boundary number missing.\n",DATAFILE_ERROR);
                  break;
                }
                bnum = int_val;
                if ( (bnum < 0 ) || (bnum >= BDRYMAX) 
                                      || (web.boundaries[bnum].coordf[0] == NULL))
                 {
                    sprintf(errmsg,"Bad boundary number %d for facet %d.\n",bnum,k);
                    kb_error(1174,errmsg,DATAFILE_ERROR);

                 }
                set_attr(flist[k],BOUNDARY);
                set_facet_boundary_num(flist[k],bnum);
                tok = yylex();
                break;

          case CONSTRAINT_:
          /* see if constraint */
                while ( (tok = gettok(INTEGER_)) == INTEGER_ )
                  {
                     cnum = abs(int_val);
                     if ( (cnum >= MAXCON) || !web.constraint_addr[cnum] )
                        { sprintf(errmsg,
                              "Bad constraint number %d for face %d.\n",cnum,k);
                          kb_error(1175,errmsg,DATAFILE_ERROR);
                        }
                     set_attr(flist[k],CONSTRAINT);
/* ?? */          if ( int_val < 0 ) set_attr(flist[k],NEGBOUNDARY);
                     set_f_constraint_map(flist[k],cnum);
                     constr = get_constraint(cnum); 
                     if ( constr->attr & CON_ENERGY )
                        set_attr(flist[k], BDRY_ENERGY);
                  }
                break;

          case ENERGY_:
          /* see if surface energy */
                tok = gettok(INTEGER_);
                for ( i = 0 ; i < SURFENMAX ; i++, tok = gettok(INTEGER_) ) 
                  {
                     if ( tok != INTEGER_ ) break;
                     cnum = int_val;
                     if ( (cnum < 0 ) || (cnum >= SURFENMAX)
                                     || (web.surfen[cnum].envect[0] == NULL) )
                        { sprintf(errmsg,
                              "Bad energy number %d for face %d.\n",cnum,k);
                          kb_error(1177,errmsg,DATAFILE_ERROR);
                        }
                     set_attr(flist[k],SURF_ENERGY);
                     set_f_surfen_map(flist[k],cnum);
                  }
                break;

          case QUANTITY_NAME_: /* name of quantity */
             apply_quantity(flist[k],globals[yylval].value.quant);
             tok = yylex();
             break;

          case IDENT_:  /* maybe method or quantity */
             if ( globals[yylval].flags & METHOD_NAME )
                apply_method(flist[k],yytext);
             else if ( globals[yylval].flags & QUANTITY_NAME )
                apply_quantity(flist[k],globals[yylval].value.quant);
             else 
             { sprintf(errmsg,"Illegal use of identifier '%s'.\n",yytext);
               kb_error(1178,errmsg,DATAFILE_ERROR);
             }
             tok = yylex();
             break;

          case METHOD_:  /* apply method instance to edge */
             tok = yylex(); /* fall through */
          case METHOD_NAME_:
          { char name[100];
             strcpy(name,yytext);
             tok = yylex();
             if ( tok == UMINUS_ )
             { apply_method(inverse_id(flist[k]),name);
                tok = yylex();
             }
             else apply_method(flist[k],name);
             break;
          }
          case QUANTITY_:
          /* see if quantity */
                tok = gettok(INTEGER_);
                for ( i = 0 ; i < QUANTMAX ; i++, tok = gettok(INTEGER_) ) 
                  {
                     if ( tok == IDENT_ )
                        { /* have named quantity/method pair */
                          char qname[32];
                          strncpy(qname,yytext,sizeof(qname));
                          if ( globals[yylval].flags & QUANTITY_NAME )
                             apply_quantity(flist[k],globals[yylval].value.quant);
                          else 
                          { sprintf(errmsg,"Undefined quantity '%s'.\n",yytext);
                            kb_error(1179,errmsg,DATAFILE_ERROR);
                          }
                          tok = yylex();
                          if ( stricmp(yytext,"method")==0 )
                            {tok = yylex();
                                kb_error(1180,"Obsolete quantity syntax.\n  Methods must be listed in quantity definition.",DATAFILE_ERROR);
                                tok = yylex();
                            }
                          continue;
                        }
                     else if ( tok != INTEGER_ ) break;
                     cnum = int_val;
                     if ( (cnum < 0 ) || (cnum >= QUANTMAX)
                                            || (web.quants[cnum].quanvect[0] == NULL) )
                        { sprintf(errmsg,
                              "Bad quantity number %d for face %d.\n",cnum,k);
                          kb_error(1181,errmsg,DATAFILE_ERROR);
                        }
                     set_attr(flist[k],SURF_QUANTITY);
                     set_f_quant_map(flist[k],cnum);
                  }
                break;

            case BODIES_: case READ_: case LEAD_INTEGER_: case NO_TOKEN:
                more_attr = 0 ; break;  /* error recovery */
            default: 
                sprintf(errmsg,"Unexpected token: %s\n",yytext);
                kb_error(2267,errmsg,WARNING);
                tok = yylex();
                break;
         }

      if ((get_fattr(flist[k])&(BOUNDARY|CONSTRAINT)) == (BOUNDARY|CONSTRAINT))
          kb_error(1182,"Cannot have constraint and boundary.",DATAFILE_ERROR);


      if ( k > facecount ) facecount = k;
    }
} /* end read_faces() */

/************************************************************************
*
*  Function: read_facet_edges()
*
*  Obsolete.
*/

void read_facet_edges()
{
  int k;
  int f;
  facetedge_id fe=NULLID,old_fe,first_fe=NULLID;
  facet_id f_id;

  /* read in facet-edge facet chains (obsolete) */
    { tok = yylex();
      while ( tok == LEAD_INTEGER_ )
      { 
      k = int_val;
      if ( !valid_id(elist[k]) )
         { sprintf(errmsg,"Edge %d is not defined.\n",k);
            kb_error(1183,errmsg,DATAFILE_ERROR);
         }
      old_fe = NULLFACETEDGE;
      while ( (tok = gettok(INTEGER_)) == INTEGER_ )
         { int loopcount = 0;  /* for detecting errors in edge list */ 
            f = int_val;
            f_id = f > 0 ? flist[f] : facet_inverse(flist[-f]);
            if ( !valid_id(f_id) )
              { sprintf(errmsg,"Edge %d: face %d is not defined.\n",k,f);
                 kb_error(1184,errmsg,DATAFILE_ERROR);
              }
            fe = get_facet_fe(f_id);
            while ( !equal_id(get_fe_edge(fe),elist[k]) ) 
              { fe = get_next_edge(fe);
                 if ( loopcount++ > 1000 )
                    { sprintf(errmsg,
                        "Facet-edge inconsistency for edge %d and facet %d.\n",k,f);
                      kb_error(1185,errmsg,DATAFILE_ERROR);
                    }
              }
            if ( valid_id(old_fe) )
              { set_next_facet(old_fe,fe);
                 set_prev_facet(fe,old_fe);
              }
            else first_fe = fe;
            old_fe = fe;
         }
      set_next_facet(fe,first_fe);  /* close up chain */
      set_prev_facet(first_fe,fe);
      }
    }
} /* end read_facet_edges() */

/************************************************************************
*
*  Function: read_bodies()
*
*/

void read_bodies()
{
  int k;
  int f=0;
  facetedge_id fe;
  facet_id f_id=NULLID;
  int more_attr;
  REAL value;  /* for constant expression values */
  int lagmulflag = 0;

  /* read in bodies */
  bmaxlist = MAXLIST;
  blist = (body_id *)mycalloc(sizeof(body_id),bmaxlist);

  tok = yylex();
  while ( tok == LEAD_INTEGER_ )  /* body loop */
  { 
    REAL den,vol;
    int face_count = 0;

    k = int_val;
    if ( k < 1 ) 
       kb_error(2375,"Body number must be positive.\n",DATAFILE_ERROR);
    while ( k >= bmaxlist )
       { int spot = bmaxlist; 
          blist = (body_id *)kb_realloc((char *)blist,
            (k+MAXLIST)*sizeof(body_id),bmaxlist*sizeof(body_id));
          bmaxlist = k + MAXLIST;
          for ( ; spot < bmaxlist ; spot ++ ) blist[spot] = NULLID;
       }
    if ( valid_id(blist[k]) )
      { sprintf(errmsg,"Duplicate body number %d\n",k);
        kb_error(1187,errmsg,DATAFILE_ERROR);
      }
    move_to_free_front(BODY,k); /* so id will be k */
    blist[k] = new_body();
    set_original(blist[k],k);
    f_id = NULLID;  /* in case no facets */
    while ( (tok = gettok(INTEGER_)) == INTEGER_ )
       { 
          face_count++;
          f = int_val;
          if ( abs(f) >= fmaxlist )
           { sprintf(errmsg,"Body %d: face %d is not defined.\n",k,f);
              kb_error(1188,errmsg,DATAFILE_ERROR);
           }
          f_id = f > 0 ? flist[f] : facet_inverse(flist[-f]);
          if ( !valid_id(f_id) )
           { sprintf(errmsg,"Body %d: face %d is not defined.\n",k,f);
              kb_error(1189,errmsg,DATAFILE_ERROR);
           }
          set_facet_body(f_id, blist[k]);
       }
    if ( (web.representation != STRING) && (face_count < 1) )
       { sprintf(errmsg,"Body %d has no faces.\n",k);
          kb_error(1190,errmsg,WARNING);
       }
    if ( valid_id(f_id) )  /* use last one */
      { fe = get_facet_fe(f_id); 
        set_body_fe(blist[k],fe);
      }

    more_attr = 1;
    while ( more_attr )
      switch ( tok )
      {
        case EXTRA_ATTRIBUTE_:
              read_extra(blist[k]);
              break;

        case ORIGINAL_:
              if ( (tok = gettok(INTEGER_)) != INTEGER_ )
              { kb_error(2406,"ORIGINAL value missing.\n",DATAFILE_ERROR);
                break;
              }
              set_original(vlist[k],int_val);
              tok = yylex();
              break;

        case VOLUME_:
              /* have a fixed volume constraint */
              if ( read_const(&vol) <= 0 )
                kb_error(1191,"Missing VOLUME value.\n",DATAFILE_ERROR);
              else 
              { set_attr(blist[k],FIXEDVOL);
                set_body_fixvol(blist[k],vol);
                tok = yylex();
              }
              break;

        case ACTUAL_VOLUME_:
              /* have a declared volume */
              if ( read_const(&vol) <= 0 )
                kb_error(1191,"Missing ACTUAL_VOLUME value.\n",DATAFILE_ERROR);
              else 
              { set_attr(blist[k],ACTUALVOL);
                set_body_actualvolume(blist[k],vol);
                tok = yylex();
              }
              break;


        case VOLCONST_:
              /* have a body volume adjustment */
              if ( read_const(&vol) <= 0 )
                kb_error(1192,"Missing VOLCONST value.\n",WARNING);
              else tok = yylex();
              set_body_volconst(blist[k],vol); 
              break;

        case DENSITY_:
              /* have density for gravity */
              if ( read_const(&den) <= 0 )
                kb_error(1193,"Missing DENSITY value.\n",WARNING);
              else tok = yylex();
              web.gravflag = 1;
              set_body_density(blist[k],den);
              set_attr(blist[k],DENSITY);
              break;

        case PRESSURE_:
              /* have prescribed pressure */
              web.pressflag = 1;
              set_attr(blist[k],PRESSURE);
              if ( read_const(&value) <= 0 )
                kb_error(1194,"Missing PRESSURE value.\n",WARNING);
              else 
              { set_body_pressure(blist[k],value);
                tok = yylex();
              }
              break;

        case LAGRANGE_MULTIPLIER_:
              if ( read_const(&value) <= 0 )
                kb_error(2194,"Missing lagrange_multiplier value.\n",DATAFILE_ERROR);
              else 
              { set_body_pressure(blist[k],value); lagmulflag = 1; 
                tok = yylex();
              }
              break;
              
        case PHASE_: 
              if ( !phase_flag )
                kb_error(1195,"Phases not in effect.\n",DATAFILE_ERROR);

              if ( web.representation == STRING )
                kb_error(1196,"Phases must be on facets in STRING model.\n",

                   DATAFILE_ERROR);
              if ( (tok = gettok(INTEGER_)) != INTEGER_ )
               { kb_error(1197,"Phase missing.\n",DATAFILE_ERROR);
                  break;
               }
              if ( (int_val < 0) || (int_val > phasemax) )
                kb_error(1198,"Illegal phase value.\n",DATAFILE_ERROR);

              else set_b_phase(blist[k],int_val);
              tok = yylex();
              break;

           case METHOD_:  /* apply method instance to edge */
              tok = yylex();
           case METHOD_NAME_:
              apply_method(blist[k],yytext);
              tok = yylex();
              break;

           case IDENT_:  /* maybe method or quantity */
              if ( globals[yylval].flags & METHOD_NAME )
                apply_method(blist[k],yytext);
              else if ( globals[yylval].flags & QUANTITY_NAME )
                apply_quantity(blist[k],globals[yylval].value.quant);
              else 
              { sprintf(errmsg,"Illegal use of identifier: %s.\n",yytext);
                kb_error(1199,errmsg,DATAFILE_ERROR);
              }
              tok = yylex();
              break;

      case READ_: case LEAD_INTEGER_: case NO_TOKEN:
          more_attr = 0 ; break;  /* error recovery */
      default: 
          sprintf(errmsg,"Unexpected token: %s\n",yytext);
          kb_error(2266,errmsg,WARNING);
          tok = yylex();
          break;

      }

    /* can't have both pressure and volume */
    if ((get_battr(blist[k]) & (FIXEDVOL|PRESSURE)) == (FIXEDVOL|PRESSURE))
        kb_error(1203,"Body can't have fixed volume and fixed pressure.\n",DATAFILE_ERROR);
  } /* end body loop */

  if ( web.bodycount > 0 )
    {      
      web.projection_flag = 1;
    }
    if ( !lagmulflag ) pressure_set_flag = 0;

    if ( web.pressure_flag)
    {
        if ( !web.full_flag && !valid_id(web.outside_body) )
                add_outside();
    }

} /* end read_bodies() */

