/*
    ext2.c -- generic ext2 stuff
    Copyright (C) 1998, 1999, 2000 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_c[] = "$Id: ext2.c,v 1.29 2004/09/30 14:01:40 sct Exp $";

#define USE_EXT2_IS_DATA_BLOCK
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/ioctl.h>
#include "ext2.h"
#include "ext2_fs.h"

/* ext2 stuff ****************************************************************/

unsigned char _bitmap[8] = {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80};


void ext2_print_version(FILE *outfile, char *progname)
{
	fprintf(outfile, "%s v%s - 2001/03/18 for EXT2FS %s\n",
		progname, VERSION, EXT2FS_VERSION);
}

void ext2_copy_block(struct ext2_fs *fs, blk_t from, blk_t to)
{
	unsigned char buf[fs->blocksize];

	ext2_bcache_flush(fs, from);
	ext2_bcache_flush(fs, to);

	ext2_read_blocks(fs, buf, from, 1);
	ext2_write_blocks(fs, buf, to, 1);
}

int ext2_get_block_state(struct ext2_fs *fs, blk_t block)
{
	struct ext2_buffer_head *bh;
	int group;
	int offset;
	int state;

	block -= fs->sb.s_first_data_block;
	group = block / fs->sb.s_blocks_per_group;
	offset = block % fs->sb.s_blocks_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_block_bitmap);
	state = check_bit(bh->data, offset);
	ext2_brelse(bh, 0);

	return state;
}

blk_t ext2_find_free_block(struct ext2_fs *fs)
{
	int i;

	for (i = 0; i < fs->numgroups; i++)
		if (fs->gd[i].bg_free_blocks_count) {
			blk_t j;
			blk_t offset;

			offset = i * fs->sb.s_blocks_per_group + fs->sb.s_first_data_block;
			for (j = fs->itoffset + fs->inodeblocks;
			     j < fs->sb.s_blocks_per_group; j++)
				if (ext2_is_data_block(fs, offset + j) &&
				    !ext2_get_block_state(fs, offset + j))
					return offset + j;

			fprintf(stderr, "inconsistent group descriptors!\n");
		}

	fprintf(stderr, "filesystem full!\n");
	return 0;
}

ino_t ext2_find_free_inode(struct ext2_fs *fs)
{
	int i;

	for (i = 0; i < fs->numgroups; i++)
		if (fs->gd[i].bg_free_inodes_count) {
			ino_t j;
			ino_t offset;

			offset = i * fs->sb.s_inodes_per_group + 1;
			for (j = 0;j < fs->sb.s_inodes_per_group; j++)
				if (!ext2_get_inode_state(fs, offset + j))
					return offset + j;

			fprintf(stderr, "inconsistent group descriptors!\n");
		}

	fprintf(stderr, "filesystem full!\n");
	return 0;
}

void ext2_move_blocks(struct ext2_fs *fs, blk_t src, blk_t num, blk_t dest)
{
	unsigned char *buf;
	blk_t i;

	if ((buf = malloc(num << fs->logsize)) != NULL) {
		ext2_bcache_flush_range(fs, src, num);
		ext2_bcache_flush_range(fs, dest, num);

		ext2_read_blocks(fs, buf, src, num);
		ext2_write_blocks(fs, buf, dest, num);

		free(buf);
	} else if (src > dest)
		for (i = 0; i < num; i++)
			ext2_copy_block(fs, src + i, dest + i);
	else
		for (i = num - 1; i >= 0; i--)
			ext2_copy_block(fs, src + i, dest + i);
}

void ext2_read_blocks(struct ext2_fs *fs, void *ptr, blk_t block, blk_t num)
{
	fs->devhandle->ops->read(fs->devhandle->cookie, ptr, block, num);
}

void ext2_set_block_state(struct ext2_fs *fs, blk_t block, int state,
			  int updatemetadata)
{
	struct ext2_buffer_head *bh;
	int			 group;
	int			 offset;

	/* printf("marking block %d %s\n", block, state ? "used" : "free"); */
	block -= fs->sb.s_first_data_block;
	group = block/fs->sb.s_blocks_per_group;
	offset = block%fs->sb.s_blocks_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_block_bitmap);
	bh->dirty = 1;
	if (state)
		set_bit(bh->data, offset);
	else
		clear_bit(bh->data, offset);
	ext2_brelse(bh, 0);

	if (updatemetadata) {
		int diff;

		diff = state ? -1 : 1;

		fs->gd[group].bg_free_blocks_count += diff;
		fs->sb.s_free_blocks_count += diff;
		fs->metadirty |= EXT2_META_SB | EXT2_META_GD;
	}
}

void ext2_write_blocks(struct ext2_fs *fs, void *ptr, blk_t block, blk_t num)
{
	fs->devhandle->ops->write(fs->devhandle->cookie, ptr, block, num);
}

void ext2_zero_blocks(struct ext2_fs *fs, blk_t block, blk_t num)
{
	unsigned char *buf;
	blk_t i;

	if ((buf = malloc(num << fs->logsize)) != NULL) {
		memset(buf, 0, num << fs->logsize);
		ext2_bcache_flush_range(fs, block, num);
		ext2_write_blocks(fs, buf, block, num);
		free(buf);
		return;
	}

	if ((buf = malloc(fs->blocksize)) != NULL) {
		memset(buf, 0, fs->blocksize);

		for (i = 0; i < num; i++) {
			ext2_bcache_flush(fs, block+i);
			ext2_write_blocks(fs, buf, block+i, 1);
		}

		free(buf);
		return;
	}

	for (i = 0; i < num; i++) {
		struct ext2_buffer_head *bh;

		bh = ext2_bcreate(fs, block+i);
		bh->dirty = 1;
		ext2_brelse(bh, 1);
	}
}

off_t ext2_get_inode_offset(struct ext2_fs *fs, ino_t inode, blk_t *block)
{
	int group;
	int offset;

	inode--;

	group = inode / fs->sb.s_inodes_per_group;
	offset = (inode % fs->sb.s_inodes_per_group) * sizeof(struct ext2_inode);

	*block = fs->gd[group].bg_inode_table + (offset >> fs->logsize);

	return offset & (fs->blocksize - 1);
}

int ext2_get_inode_state(struct ext2_fs *fs, ino_t inode)
{
	struct ext2_buffer_head *bh;
	int			 group;
	int			 offset;
	int			 ret;

	inode--;
	group = inode / fs->sb.s_inodes_per_group;
	offset = inode % fs->sb.s_inodes_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_inode_bitmap);
	ret = check_bit(bh->data, offset);
	ext2_brelse(bh, 0);

	return ret;
}

void ext2_read_inode(struct ext2_fs *fs, ino_t ino, struct ext2_inode *inode)
{
	struct ext2_buffer_head *bh;
	blk_t			 blk;
	off_t			 off;

	off = ext2_get_inode_offset(fs, ino, &blk);

	bh = ext2_bread(fs, blk);
	memcpy(inode, bh->data + off, sizeof(struct ext2_inode));
	ext2_brelse(bh, 0);
}

void ext2_set_inode_state(struct ext2_fs *fs, ino_t ino, int state,
			  int updatemetadata)
{
	struct ext2_buffer_head *bh;
	int			 group;
	int			 offset;

	ino--;
	group = ino / fs->sb.s_inodes_per_group;
	offset = ino % fs->sb.s_inodes_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_inode_bitmap);
	bh->dirty = 1;
	if (state)
		set_bit(bh->data, offset);
	else
		clear_bit(bh->data, offset);
	ext2_brelse(bh, 0);

	if (updatemetadata) {
		int diff;

		diff = state ? -1 : 1;

		fs->gd[group].bg_free_inodes_count += diff;
		fs->sb.s_free_inodes_count += diff;
		fs->metadirty = EXT2_META_SB | EXT2_META_GD;
	}
}

int ext2_block_iterate(struct ext2_fs *fs, struct ext2_inode *inode,
		       blk_t block, int action)
{
	struct ext2_buffer_head *bh;
	__u32			*udata;
	blk_t			 curblock;
	int			 count = 0;
	int			 i;
	int			 i512perblock = 1 << (fs->logsize - 9);

	if (block == 0 || inode->i_mode == 0)
		return -1;

	if (fs->flags & FL_DEBUG) {
		switch (action) {
			case EXT2_ACTION_ADD:
				printf("adding 0x%04x to inode\n", block);
				break;
			case EXT2_ACTION_DELETE:
				printf("deleting 0x%04x from inode\n", block);
				break;
			case EXT2_ACTION_FIND:
				printf("finding 0x%04x in inode\n", block);
				break;
		}
	}

	/* Direct blocks for first 12 blocks */
	for (i = 0, curblock = 0; i < EXT2_NDIR_BLOCKS; i++, curblock++) {
		if (action == EXT2_ACTION_ADD && !inode->i_block[i]) {
			size_t new_size = (curblock + 1) * fs->blocksize;

			if (fs->flags & FL_DEBUG)
				printf("add %d as direct block\n", curblock);
			inode->i_block[i] = block;
			/* i_blocks is in 512 byte blocks */
			inode->i_blocks += i512perblock;
			if (new_size > inode->i_size)
				inode->i_size = new_size;

			inode->i_mtime = time(NULL);
			ext2_set_block_state(fs, block, 1,
					     !(fs->flags & FL_ONLINE));
			return curblock;
		}
		if (inode->i_block[i] == block) {
			if (action == EXT2_ACTION_DELETE) {
				if (fs->flags & FL_DEBUG)
					printf("del %d as direct block\n",
					      curblock);
				inode->i_block[i] = 0;
				inode->i_blocks -= i512perblock;
				inode->i_mtime = time(NULL);
				if (!(fs->flags & FL_ONLINE))
					ext2_set_block_state(fs, block, 0, 1);
			}
			return i;
		}
		if (inode->i_block[i])
			count += i512perblock;
	}

	count += inode->i_block[EXT2_IND_BLOCK] ? i512perblock : 0;
	count += inode->i_block[EXT2_DIND_BLOCK] ? i512perblock : 0;
	count += inode->i_block[EXT2_TIND_BLOCK] ? i512perblock : 0;

	if (!inode->i_block[EXT2_IND_BLOCK] ||
	    (count >= inode->i_blocks && action != EXT2_ACTION_ADD))
		return -1;

	bh = ext2_bread(fs, inode->i_block[EXT2_IND_BLOCK]);
	udata = (__u32 *)bh->data;

	/* Indirect blocks for next 256/512/1024 blocks (for 1k/2k/4k blocks) */
	for (i = 0; i < fs->u32perblock; i++, curblock++) {
		if (action == EXT2_ACTION_ADD && !udata[i]) {
			size_t new_size = (curblock + 1) * fs->blocksize;

			if (fs->flags & FL_DEBUG)
				printf("add %d to ind block %d\n", curblock,
				       inode->i_block[EXT2_IND_BLOCK]);
			bh->dirty = 1;
			udata[i] = block;
			inode->i_blocks += i512perblock;
			if (new_size > inode->i_size)
				inode->i_size = new_size;
			inode->i_mtime = time(NULL);
			ext2_set_block_state(fs, block, 1,
					     !(fs->flags & FL_ONLINE));
			ext2_brelse(bh, 0);
			return curblock;
		}
		if (udata[i] == block) {
			if (action == EXT2_ACTION_DELETE) {
				if (fs->flags & FL_DEBUG)
					printf("del %d from ind block %d\n",
					      curblock,
					      inode->i_block[EXT2_IND_BLOCK]);
				bh->dirty = 1;
				udata[i] = 0;
				inode->i_blocks -= i512perblock;
				inode->i_mtime = time(NULL);
				if (!(fs->flags & FL_ONLINE))
					ext2_set_block_state(fs, block, 0, 1);
			}
			ext2_brelse(bh, 0);
			return curblock;
		}
		if (udata[i]) {
			count += i512perblock;
			if (count >= inode->i_blocks &&
			    action != EXT2_ACTION_ADD)
				return -1;
		}
	}

	ext2_brelse(bh, 0);

	if (!inode->i_block[EXT2_DIND_BLOCK] ||
	    (count >= inode->i_blocks && action != EXT2_ACTION_ADD))
		return -1;
	bh = ext2_bread(fs, inode->i_block[EXT2_DIND_BLOCK]);
	udata = (__u32 *)bh->data;

	/* Double indirect blocks for next 2^16/2^18/2^20 1k/2k/4k blocks */
	for (i = 0; i < fs->u32perblock; i++) {
		struct ext2_buffer_head	*bh2;
		__u32			*udata2;
		int			 j;

		if (!udata[i]) {
			ext2_brelse(bh, 0);
			ext2_brelse(bh2, 0);
			return -1;
		}
		bh2 = ext2_bread(fs, udata[i]);
		udata2 = (__u32 *)bh2->data;
		count += i512perblock;

		for (j = 0; j < fs->u32perblock; j++, curblock++) {
			if (action == EXT2_ACTION_ADD && !udata2[j]) {
				size_t new_size = (curblock+1) * fs->blocksize;

				if (fs->flags & FL_DEBUG)
					printf("add %d to dind block %d\n",
					       curblock, udata[i]);
				bh2->dirty = 1;
				udata2[j] = block;
				inode->i_blocks += i512perblock;
				if (new_size > inode->i_size)
					inode->i_size = new_size;
				inode->i_mtime = time(NULL);
				ext2_set_block_state(fs, block, 1,
						     !(fs->flags & FL_ONLINE));
				ext2_brelse(bh, 0);
				ext2_brelse(bh2, 0);
				return curblock;
			}
			if (udata2[j] == block) {
				if (action == EXT2_ACTION_DELETE) {
					if (fs->flags & FL_DEBUG)
						printf("del %d from dind %d\n",
						       curblock, udata[i]);
					bh2->dirty = 1;
					udata2[j] = 0;
					inode->i_blocks -= i512perblock;
					inode->i_mtime = time(NULL);
					if (!(fs->flags & FL_ONLINE))
						ext2_set_block_state(fs, block,
						     0, 1);
				}
				ext2_brelse(bh, 0);
				ext2_brelse(bh2, 0);
				return curblock;
			}
			if (udata2[j]) {
				count += i512perblock;
				if (count >= inode->i_blocks &&
				    action != EXT2_ACTION_ADD)
					return -1;
			}
		}
		ext2_brelse(bh2, 0);
	}
	ext2_brelse(bh, 0);

	/* Note: we don't try to handle triple-indirect blocks for resizing
	 * because this is not really useful in a normal filesystem.
	 */

	return -1;
}

void ext2_write_inode(struct ext2_fs *fs, ino_t ino,
		      const struct ext2_inode *inode)
{
	struct ext2_buffer_head *bh;
	blk_t			 blk;
	off_t			 off;

	off = ext2_get_inode_offset(fs, ino, &blk);

	/* If the filesystem is currently mounted, write only this inode
	 * out, so we don't corrupt other inodes in this block.
	 */
	if (fs->flags & FL_ONLINE) {
		fs->devhandle->ops->direct_write(fs->devhandle->cookie,
			(void *)inode, (loff_t)blk * fs->blocksize + off,
			sizeof(struct ext2_inode));
	} else {
		bh = ext2_bread(fs, blk);
		bh->dirty = 1;
		memcpy(bh->data + off, inode, sizeof(struct ext2_inode));
		ext2_brelse(bh, 0);
	}
}

void ext2_zero_inode(struct ext2_fs *fs, ino_t ino)
{
	struct ext2_inode inode;

	memset(&inode, 0, sizeof(struct ext2_inode));
	ext2_write_inode(fs, ino, &inode);
}

unsigned int ext2_list_backups(struct ext2_fs *fs, unsigned int *three,
			       unsigned int *five, unsigned int *seven)
{
	unsigned int *min = three;
	int mult = 3;
	unsigned int ret;

	if (!(fs->sb.s_feature_ro_compat&EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER)) {
		ret = *min;
		*min += 1;
		return ret;
	}

	if (*five < *min) {
		min = five;
		mult = 5;
	}
	if (*seven < *min) {
		min = seven;
		mult = 7;
	}

	ret = *min;
	*min *= mult;

	return ret;
}

int ext2_bg_has_super(struct ext2_fs *fs, int group)
{
	unsigned int three = 1, five = 5, seven = 7;
	unsigned int grp;

	if (!(fs->flags & FL_SPARSE))
		return 1;

	if (group == 0)
		return 1;

	while ((grp = ext2_list_backups(fs, &three, &five, &seven)) < group)
	       /* do nothing */;


	return (grp == group);
}


void ext2_close(struct ext2_fs *fs)
{
#if 0
	ext2_journal_deinit(fs);
#endif
	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	ext2_commit_metadata(fs, EXT2_META_BACKUP);
	ext2_sync(fs);

	ext2_bcache_deinit(fs);

	fs->devhandle->ops->close(fs->devhandle->cookie);

	free(fs->gd);
	free(fs);
}

void ext2_commit_metadata(struct ext2_fs *fs, int copies)
{
	int		 i;
	int		 num;
	int		 wmeta = fs->metadirty & copies;
	unsigned char	 sbp[fs->blocksize]; /* primary */
	unsigned char	 sbb[fs->blocksize]; /* backup(s) */
	__u16		*block_group_p;
	__u16		*block_group_b;
	int		 sb_block;

	/* See if there is even anything to write... */
	if (wmeta == EXT2_META_CLEAN)
		return;

	if (fs->flags & FL_IOCTL && wmeta & EXT2_META_PRIMARY) {
		fprintf(stderr, "%s: writing metadata on live filesystem!\n",
			__FUNCTION__);
		exit(5);
	}

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	fs->sb.s_r_blocks_count = (fs->r_frac * (loff_t)fs->sb.s_blocks_count)/
				  100;

	memset(sbp, 0, fs->blocksize);
	memset(sbb, 0, fs->blocksize);

	/* The Primary superblock is always 1024 bytes from the start
	 * of the filesystem, regardless of the block size.
	 */
	if (fs->sb.s_first_data_block) {
		memcpy(sbp, &fs->sb, 1024);
		block_group_p = (__u16 *)
			(&(((struct ext2_super_block *)sbp)->s_block_group_nr));
	} else {
		memcpy(sbp + 1024, &fs->sb, 1024);
		block_group_p = (__u16 *)(&(((struct ext2_super_block *)
			    (sbp + 1024))->s_block_group_nr));
	}
	/* The Backup superblock is always at the start of the block, even
	 * if we have 2k or 4k blocks, so we handle it separately.
	 */
	memcpy(sbp, &fs->sb, 1024);
	block_group_b = (__u16 *)
		(&(((struct ext2_super_block *)sbp)->s_block_group_nr));

	num = copies & EXT2_META_BACKUP ? fs->numgroups : 1;

	for (i = 0, sb_block = fs->sb.s_first_data_block; i < num;
	     i++, sb_block += fs->sb.s_blocks_per_group) {
		if (!ext2_bg_has_super(fs, i))
			continue;

		if (i == 0 && wmeta & EXT2_META_PRIMARY_SB) {
			/* write primary superblock to disk */
			ext2_bcache_flush_range(fs, sb_block, 1);
			ext2_write_blocks(fs, sbp, sb_block, 1);
		} else if (i != 0 && wmeta & EXT2_META_SB) {
			/* write backup superblock to disk */
			if (fs->sb.s_rev_level > EXT2_GOOD_OLD_REV)
				*block_group_b = i;
			ext2_bcache_flush_range(fs, sb_block, 1);
			ext2_write_blocks(fs, sbb, sb_block, 1);
		}
		if ((i == 0 && wmeta & EXT2_META_PRIMARY_GD) ||
		    (i != 0 && wmeta & EXT2_META_GD)) {
			ext2_bcache_flush_range(fs, sb_block + 1, fs->gdblocks);
			ext2_write_blocks(fs, fs->gd, sb_block + 1,
					  fs->gdblocks);
		}
	}

	/* Clear the flags of the components we just finished writing. */
	fs->metadirty &= ~copies;
}

static int ext2_determine_itoffset(struct ext2_fs *fs)
{
	int	group;
	int	bpg = fs->sb.s_blocks_per_group;
	blk_t	start, itend;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	fs->itoffset = fs->gd[0].bg_inode_table - fs->sb.s_first_data_block;
	fs->stride = 0;

	if (fs->flags & FL_DEBUG)
		printf("setting itoffset to +%d\n", fs->itoffset);

	/* Try to keep RAID spacings.  We don't check the first few groups as
	 * an old ext2prepare might have moved the block and inode bitmaps,
	 * and the last group may also be strange because it is smaller.
	 */
	if (fs->gd[0].bg_block_bitmap > fs->gd[0].bg_inode_table) {
		if (fs->numgroups == 1)
			fs->stride = 2;
		else if (fs->numgroups == 2)
			fs->stride = fs->gd[1].bg_block_bitmap -
				     fs->gd[0].bg_block_bitmap - bpg;
		else {
			int last;

			if (fs->numgroups == 3)
				last = fs->numgroups;
			else
				last = min(fs->numgroups - 1, 10);

			fs->stride = fs->gd[last - 2].bg_block_bitmap -
				     fs->gd[last - 3].bg_block_bitmap - bpg;
			if (fs->stride != fs->gd[last - 1].bg_block_bitmap-
					  fs->gd[last - 2].bg_block_bitmap-bpg)
				fprintf(stderr, "This is odd, the RAID stride "
						"is not constant at %d!\n",
						fs->stride);
		}
		if (fs->stride < 0) {
			fprintf(stderr, "This is odd, the RAID stride "
				"is negative (%d)!\n",
				fs->stride);
			fs->stride = 0;
		}
		printf("Using a RAID stride value of %d\n", fs->stride);
	}

	/* We no longer need to check the bitmap and table offsets */
	if (!(fs->flags & FL_VERBOSE))
		return 1;

	itend = fs->itoffset + fs->inodeblocks;
	for (group = 0, start = fs->sb.s_first_data_block;
	     group < fs->numgroups;
	     group++, start += fs->sb.s_blocks_per_group) {
		blk_t bb;
		blk_t ib;
		blk_t it;
		int has_sb;

		it = start + fs->itoffset;
		if (start + bpg > fs->sb.s_blocks_count)
			bpg = fs->sb.s_blocks_count - start;
		has_sb = ext2_bg_has_super(fs, group);

		if (fs->stride) {
			blk_t bmap = start + itend +
				(fs->stride * group % (bpg - itend));
			bb = bmap >= start + bpg ? itend : bmap;
		} else
			bb = has_sb ? it - 2 : start;
		ib = (bb + 1 >= start + bpg) ? start + itend : bb + 1;

		if (fs->gd[group].bg_block_bitmap != bb && fs->flags & FL_DEBUG)
			fprintf(stderr,
				"group %d block bitmap has offset %d, not %d\n",
				group, fs->gd[group].bg_block_bitmap, bb);
		if (fs->gd[group].bg_inode_bitmap != ib && fs->flags & FL_DEBUG)
			fprintf(stderr,
				"group %d inode bitmap has offset %d, not %d\n",
				group, fs->gd[group].bg_inode_bitmap, ib);
		if (fs->gd[group].bg_inode_table != it) {
			fprintf(stderr,
				"group %d inode table has offset %d, not %d\n",
				group, fs->gd[group].bg_inode_table-start,
				it-start);
			/*
			fprintf(stderr,
				"This ext2 filesystem has a strange layout!\n"
				"Please use dumpe2fs and send output to: "
				"<ext2resize-devel@lists.sourceforge.net>.\n"
				"I won't resize it, sorry.\n");
			return 0;
			 */
		}
	}

	return 1;
}

void ext2_sync(struct ext2_fs *fs)
{
	ext2_commit_metadata(fs, EXT2_META_PRIMARY);
	ext2_bcache_sync(fs);
	fs->devhandle->ops->sync(fs->devhandle->cookie);
}

int ext2_ioctl(struct ext2_fs *fs, int ioc, void *arg)
{
	int ret = 1;

	if (fs->flags & FL_DEBUG) {
		if (ioc == EXT2_IOC_GROUP_EXTEND)
			printf("%s: EXTEND group to %u blocks\n",
			       __FUNCTION__, *(__u32 *)arg);
		else if (ioc == EXT2_IOC_GROUP_ADD)
			printf("%s: ADD group %u\n", 
			       __FUNCTION__, *(__u32 *)arg);
		else
			printf("%s: unknown IOC %x\n", __FUNCTION__, ioc);
	}
	if (fs->devhandle->ops->ioctl(fs->devhandle->cookie, ioc, arg) < 0) {
		char msg[4097];

		snprintf(msg, 4096, "%s: %s", fs->prog,  __FUNCTION__);
		msg[4096] = '\0';
		perror(msg);

		ret = 0;
	}

	return ret;
}

static blk_t ext2_get_reserved(struct ext2_fs *fs, struct ext2_inode *inode)
{
	int	resgdblocks;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	/* How many blocks to reserve for each group future GDT expansion.
	 * This assumes Bond has the right number of blocks assigned.  It
	 * may not be reliable unless e2fsck starts fixing Bond for us.
	 */
	if (inode->i_blocks == 0)
		resgdblocks = 0;

	/* This is the official method of storing reserved blocks. */
	else if (fs->sb.s_feature_compat & EXT2_FEATURE_COMPAT_RESIZE_INODE) {
		resgdblocks = fs->sb.s_reserved_gdt_blocks;
		if (fs->flags & FL_DEBUG)
			printf("Found %d blocks in s_reserved_gdt_blocks\n",
			       resgdblocks);
	}

	/* This is the new/preferred method of storing reserved blocks
	 * We want to also allow the old method in case someone
	 * prepared a filesystem and they now want to resize with
	 * the new code.  This can be removed in a future release.
	 */
	else if (inode->i_generation && inode->i_generation < 256) {
		if (fs->flags & FL_DEBUG)
			printf("Found %d reserved blocks in i_generation\n",
			       inode->i_generation);
		resgdblocks = inode->i_generation;
	} else {
		int sb_groups = 0;
		int i;

		for (i = 0; i < fs->numgroups; i++)
			sb_groups += ext2_bg_has_super(fs, i);
		resgdblocks = ((inode->i_blocks >> (fs->logsize - 9)) -
			       (inode->i_block[EXT2_IND_BLOCK] ? 1 : 0) -
			       (inode->i_block[EXT2_DIND_BLOCK] ? 1 : 0)) /
			      sb_groups;
		if (fs->flags & FL_DEBUG)
			printf("Calculated %d reserved blocks from i_blocks\n",
			       resgdblocks);
	}

	if (resgdblocks > fs->blocksize / 4 ||
	    resgdblocks < 0) { /* Bogosity test */
		fprintf(stderr,
			"%s: wrong resgdblocks found (%d), using 0.\n"
			"\tyou may need to run ext2prepare before online "
			"resizing.\n",
			fs->devhandle->prog, resgdblocks);
		resgdblocks = 0;
	}

	if (fs->flags & FL_VERBOSE)
		printf("using %u reserved group descriptor blocks\n",
		       resgdblocks);

	return resgdblocks;
}

#define EXT2_OPEN_COMPAT_UNSUPP ~(EXT2_FEATURE_COMPAT_DIR_PREALLOC | \
				  EXT3_FEATURE_COMPAT_HAS_JOURNAL | \
				  EXT2_FEATURE_COMPAT_DIR_INDEX | \
				  EXT2_FEATURE_COMPAT_EXT_ATTR | \
				  EXT2_FEATURE_COMPAT_RESIZE_INODE)
#define EXT2_OPEN_RO_COMPAT_UNSUPP ~(EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER | \
				     EXT2_FEATURE_RO_COMPAT_LARGE_FILE)
#define EXT2_OPEN_INCOMPAT_UNSUPP ~EXT2_FEATURE_INCOMPAT_FILETYPE

struct ext2_fs *ext2_open(struct ext2_dev_handle *handle, blk_t newblocks,
			  int flags)
{
	struct ext2_fs *fs;
	struct ext2_inode inode;
	int maxgroups;
	blk_t devsize;
	blk_t residue;

	if (flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	if ((fs = (struct ext2_fs *)malloc(sizeof(struct ext2_fs))) == NULL) {
		fprintf(stderr, "%s: error allocating fs struct\n",
			handle->prog);
		goto error;
	}

	fs->flags = flags;
	fs->prog = handle->prog;
	fs->devhandle = handle;
	handle->ops->set_blocksize(handle->cookie, 10);

	ext2_read_blocks(fs, &fs->sb, 1, 1);
	if (fs->sb.s_magic != EXT2_SUPER_MAGIC) {
		fprintf(stderr, "%s: ext2_open: invalid superblock\n",fs->prog);
		goto error_free_fs;
	}

	if (!(fs->flags & FL_FORCE) && fs->sb.s_state & EXT2_ERROR_FS) {
		fprintf(stderr, "%s: ext2_open: filesystem has errors! "
			"e2fsck first!\n", fs->prog);
		goto error_free_fs;
	}

	if (!(fs->flags & (FL_FORCE | FL_ONLINE)) &&
	    !(fs->sb.s_state & EXT2_VALID_FS)) {
		fprintf(stderr,
			"%s: ext2_open: filesystem was not cleanly unmounted! "
			"e2fsck first!\n", fs->prog);
		goto error_free_fs;
	}

	if (!(fs->flags & FL_FORCE) &&
	    (fs->sb.s_feature_compat & EXT2_OPEN_COMPAT_UNSUPP ||
	     (fs->sb.s_feature_incompat & EXT2_OPEN_INCOMPAT_UNSUPP &
	      ~(fs->flags & FL_ONLINE ? EXT3_FEATURE_INCOMPAT_RECOVER : 0)) ||
	     (fs->sb.s_feature_ro_compat & EXT2_OPEN_RO_COMPAT_UNSUPP)))
	{
		fprintf(stderr,
			"ext2_open: fs has unsupported feature(s) enabled:");
		if(fs->sb.s_feature_compat & EXT2_OPEN_COMPAT_UNSUPP)
			fprintf(stderr, " compat %x", fs->sb.s_feature_compat &
						      EXT2_OPEN_COMPAT_UNSUPP);
		if(fs->sb.s_feature_ro_compat & EXT2_OPEN_RO_COMPAT_UNSUPP)
			fprintf(stderr, " ro_compat %x",
						fs->sb.s_feature_ro_compat &
						EXT2_OPEN_RO_COMPAT_UNSUPP);
		if (!(fs->flags & FL_ONLINE) &&
		    (fs->sb.s_feature_incompat &EXT3_FEATURE_INCOMPAT_RECOVER))
			fprintf(stderr, " recover");
		if (fs->sb.s_feature_incompat & EXT2_OPEN_INCOMPAT_UNSUPP &
		    ~EXT3_FEATURE_INCOMPAT_RECOVER)
			fprintf(stderr, " incompat %x",
				fs->sb.s_feature_incompat &
				EXT2_OPEN_INCOMPAT_UNSUPP);
		fprintf(stderr, "\n");
		goto error_free_fs;
	}

	if (fs->sb.s_feature_incompat & EXT3_FEATURE_INCOMPAT_RECOVER)
	      fs->flags |= FL_IOCTL;

	fs->logsize = fs->sb.s_log_block_size + 10;
	handle->ops->set_blocksize(handle->cookie, fs->logsize);

	if (!ext2_bcache_init(fs))
		goto error_free_fs;

	fs->blocksize = 1 << fs->logsize;
	fs->u32perblock = fs->blocksize >> 2;

	devsize = handle->ops->get_size(fs->devhandle->cookie);
	if (newblocks == 0)
		newblocks = devsize;
	else if (fs->flags & FL_KB_BLOCKS)
		newblocks >>= fs->logsize - 10;

	if (fs->flags & FL_VERBOSE)
		printf("new filesystem size %d\n", newblocks);

	fs->numgroups = howmany(fs->sb.s_blocks_count -
				fs->sb.s_first_data_block,
				fs->sb.s_blocks_per_group);
	fs->gdblocks = howmany(fs->numgroups * sizeof(struct ext2_group_desc),
			       fs->blocksize);
	fs->inodeblocks = howmany(fs->sb.s_inodes_per_group *
				  sizeof(struct ext2_inode), fs->blocksize);
	fs->r_frac = howmany(100 * (loff_t)fs->sb.s_r_blocks_count,
			     fs->sb.s_blocks_count);

	if (devsize < newblocks && !(fs->flags & FL_PREPARE)) {
		char *junk[fs->blocksize];

		fprintf(stderr, "%s: warning - device size %u, specified %u\n",
			fs->prog, devsize, newblocks);

		ext2_read_blocks(fs, junk, newblocks - 1, 1);
	} else if (fs->sb.s_blocks_count > devsize) {
		char *junk[fs->blocksize];
		fprintf(stderr, "%s: warning - device size %u, filesystem %u\n",
			fs->prog, devsize, fs->sb.s_blocks_count);

		ext2_read_blocks(fs, junk, fs->sb.s_blocks_count - 1, 1);
	}

	residue = (newblocks - fs->sb.s_first_data_block) %
		fs->sb.s_blocks_per_group;
	if (residue && residue <= fs->inodeblocks + fs->gdblocks + 50) {
		fprintf(stderr, "%s: %i is a bad size for an ext2 fs! "
			"rounding down to %i\n",
			fs->prog, newblocks, newblocks - residue);
		newblocks -= residue;
	}

	fs->newblocks = newblocks;
	fs->newgroups = howmany(newblocks - fs->sb.s_first_data_block,
				fs->sb.s_blocks_per_group);

	/* round up to the next block boundary, since we read full blocks */
	maxgroups = max(fs->newgroups, fs->numgroups) +
		fs->blocksize / sizeof(struct ext2_group_desc);

	fs->newgdblocks = howmany(fs->newgroups*sizeof(struct ext2_group_desc),
				  fs->blocksize);

	if (fs->sb.s_feature_ro_compat & EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER)
		fs->flags |= FL_SPARSE;

	if ((fs->gd = (struct ext2_group_desc *)
	     malloc(maxgroups * sizeof(struct ext2_group_desc))) == NULL) {
		fprintf(stderr, "%s: ext2_open: descriptor alloc error!\n",
			fs->prog);
		goto error_free_bcache;
	}
	ext2_read_blocks(fs, fs->gd, fs->sb.s_first_data_block+1, fs->gdblocks);

	if (!ext2_determine_itoffset(fs))
		goto error_free_gd;

	ext2_read_inode(fs, EXT2_BAD_INO, &inode);
	if (inode.i_blocks > 0) {
		fprintf(stderr,
			"%s: ext2_open: Can't handle fs with bad blocks!!\n"
			"Contact <ext2resize-devel@lists.sourceforge.net> to "
			"request this feature.\n", fs->prog);
		goto error_free_gd;
	}

	/* Bring in Mr. Bond */
	ext2_read_inode(fs, EXT2_RESIZE_INO, &inode);
	fs->resgdblocks = ext2_get_reserved(fs, &inode);

#if 0
	ext2_journal_init(fs);
#endif

	return fs;

error_free_gd:
	free(fs->gd);
error_free_bcache:
	ext2_bcache_deinit(fs);
error_free_fs:
	free(fs);
error:
	return NULL;
}
