/*
    ext2_resize.c -- ext2 resizer
    Copyright (C) 1998, 1999, 2000 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_resize_c[] = "$Id: ext2_resize.c,v 1.16 2004/09/30 14:01:41 sct Exp $";

#define USE_EXT2_IS_DATA_BLOCK
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include "ext2.h"

static int ext2_add_group(struct ext2_fs *fs, blk_t bpg)
{
	struct ext2_inode tmp;
	struct ext2_inode *inode = &tmp;
	blk_t admin;
	int   group = fs->numgroups;
	blk_t start;
	blk_t new_bb, new_ib;
	int   new_gdblocks;
	int   new_itend;
	int   has_sb;
	struct ext2_buffer_head *bh;
	int   i;

	if (fs->flags & FL_DEBUG)
		printf("%s %d\n", __FUNCTION__, group);

	if (fs->sb.s_blocks_count !=
	    fs->sb.s_first_data_block + group * fs->sb.s_blocks_per_group) {
		fprintf(stderr, "ext2_add_group: last (existing) group "
			"isn't complete!\n");

		return 0;
	}

	has_sb = ext2_bg_has_super(fs, group);

	if (bpg < fs->gdblocks + fs->inodeblocks + 50 ||
	    bpg > fs->sb.s_blocks_per_group) {
		fprintf(stderr,
			"ext2_add_group: groups of %i blocks are impossible!\n",
			bpg);

		return 0;
	}

	ext2_read_inode(fs, EXT2_RESIZE_INO, inode);
	inode->i_mtime = 0;

	new_gdblocks = howmany((group + 1) *
			      sizeof(struct ext2_group_desc), fs->blocksize);
	if (new_gdblocks > fs->gdblocks) {
		if (fs->flags & FL_DEBUG)
			printf("add group descriptor block %d\n", new_gdblocks);
		for (group = 0, start = fs->sb.s_first_data_block;
		     group < fs->numgroups;
		     group++, start += fs->sb.s_blocks_per_group) {
			if (ext2_bg_has_super(fs, group)) {
				if (fs->resgdblocks)
					ext2_block_iterate(fs, inode,
							   start+fs->gdblocks+1,
							   EXT2_ACTION_DELETE);
				ext2_set_block_state(fs, start +fs->gdblocks +1,
						     1, 1);
			}
		}
		if (inode->i_mtime)
			inode->i_generation = --fs->resgdblocks;
		fs->gdblocks++;
	}
	fs->numgroups++;
	start = fs->sb.s_first_data_block + group * fs->sb.s_blocks_per_group;
	admin = fs->inodeblocks + (has_sb ? fs->gdblocks + 3 : 2);

	fs->sb.s_inodes_count += fs->sb.s_inodes_per_group;
	fs->sb.s_blocks_count += bpg;
	fs->sb.s_free_blocks_count += bpg - admin;
	fs->sb.s_free_inodes_count += fs->sb.s_inodes_per_group;
	fs->metadirty |= EXT2_META_SB;

	new_itend = fs->itoffset + fs->inodeblocks;
	if (fs->stride) {
		blk_t new_bmap = new_itend +
			(fs->stride * group % (bpg - new_itend));
		new_bb = new_bmap >= bpg ? new_itend : new_bmap;
	} else
		new_bb = has_sb ? fs->itoffset - 2 : 0;
	new_ib = (new_bb + 1 >= bpg) ? new_itend : new_bb +1;

	fs->gd[group].bg_block_bitmap = start + new_bb;
	fs->gd[group].bg_inode_bitmap = start + new_ib;
	fs->gd[group].bg_inode_table = start + fs->itoffset;
	fs->gd[group].bg_free_blocks_count = bpg - admin;
	fs->gd[group].bg_free_inodes_count = fs->sb.s_inodes_per_group;
	fs->gd[group].bg_used_dirs_count = 0;
	fs->metadirty |= EXT2_META_GD;

	if (fs->flags & FL_DEBUG)
		printf("group %d %s: bb %d (+%d), ib %d (+%d), it %d (+%d)\n",
		       group, has_sb ? "has sb" : "no sb",
		       fs->gd[group].bg_block_bitmap, new_bb,
		       fs->gd[group].bg_inode_bitmap, new_ib,
		       fs->gd[group].bg_inode_table, fs->itoffset);

	bh = ext2_bcreate(fs, fs->gd[group].bg_block_bitmap);

	if (has_sb) {
		if (fs->flags & FL_DEBUG)
			printf ("mark backup superblock and %d gdblocks used\n",
				fs->gdblocks);
		for (i = 0; i <= fs->gdblocks; i++)
			set_bit(bh->data, i);
		for (i = 1; i <= fs->resgdblocks; i++)
			ext2_block_iterate(fs, inode, start + fs->gdblocks + i,
					   EXT2_ACTION_ADD);
	}

	set_bit(bh->data, new_bb);
	set_bit(bh->data, new_ib);

	for (i = fs->itoffset; i < new_itend; i++)
		set_bit(bh->data, i);

	for (i = bpg; i < fs->sb.s_blocks_per_group; i++)
		set_bit(bh->data, i);

	ext2_brelse(bh, 0);	/* this is a block bitmap */

	ext2_zero_blocks(fs, fs->gd[group].bg_inode_bitmap, 1);
	ext2_zero_blocks(fs, fs->gd[group].bg_inode_table, fs->inodeblocks);

	if (inode->i_mtime)
		ext2_write_inode(fs, EXT2_RESIZE_INO, inode);

	if (fs->flags & FL_SAFE)
		ext2_sync(fs);

	return 1;
}

static int ext2_del_group(struct ext2_fs *fs)
{
	blk_t admin;
	int   group = fs->numgroups - 1;
	int   bpg;
	blk_t new_gdblocks;
	int   has_sb;

	if (fs->flags & FL_DEBUG)
		printf("%s %d\n", __FUNCTION__, group);

	has_sb = ext2_bg_has_super(fs, group);

	admin = fs->inodeblocks + (has_sb ? fs->gdblocks + 3 : 2);

	bpg = fs->sb.s_blocks_count - fs->sb.s_first_data_block -
		group * fs->sb.s_blocks_per_group;

	if (fs->sb.s_free_blocks_count < bpg - admin) {
		fprintf(stderr, "%s: filesystem is too full to remove a group "
			"(%d used blocks)!\n",
			fs->prog,
			fs->sb.s_blocks_count - fs->sb.s_free_blocks_count);

		return 0;
	}

	if (fs->sb.s_free_inodes_count < fs->sb.s_inodes_per_group) {
		fprintf(stderr, "%s: filesystem has too many used inodes "
			"to remove a group (%d used inodes)!\n",
			fs->prog,
			fs->sb.s_inodes_count - fs->sb.s_free_inodes_count);
		return 0;
	}

	if (fs->gd[group].bg_free_inodes_count != fs->sb.s_inodes_per_group) {
		/* This should not happen anymore */
		fprintf(stderr,
			"%s: free inodes != inodes_per_group (%d != %d)!\n",
			fs->prog, fs->gd[group].bg_free_inodes_count,
			fs->sb.s_inodes_per_group);

		return 0;
	}

	/* If this isn't true, we have an error - see what is wrong */
	if (fs->gd[group].bg_free_blocks_count != bpg - admin) {
		blk_t j;
		blk_t start;

		start = fs->sb.s_first_data_block +
			group * fs->sb.s_blocks_per_group;

		for (j = 0; j < bpg; j++)
			if (ext2_is_data_block(fs, start + j) &&
			    ext2_get_block_state(fs, start + j)) {
				fprintf(stderr, "error: block relocator "
					"should have relocated %i\n",
					start + j);

				return 0;
			}
	}

	new_gdblocks = howmany(group *
			       sizeof(struct ext2_group_desc), fs->blocksize);

	if (new_gdblocks != fs->gdblocks) {
		struct ext2_inode tmp;
		struct ext2_inode *inode = &tmp;
		blk_t start;

		ext2_read_inode(fs, EXT2_RESIZE_INO, inode);
		inode->i_mtime = 0;

		for (group = 0, start = fs->sb.s_first_data_block;
		     group < fs->numgroups - 1;
		     group++, start += fs->sb.s_blocks_per_group)
			if (ext2_bg_has_super(fs, group)) {
				ext2_set_block_state(fs, start + fs->gdblocks,
						     0, 1);
				if (inode->i_blocks)
					ext2_block_iterate(fs, inode,
							   start+fs->gdblocks,
							   EXT2_ACTION_ADD);
			}

		if (inode->i_mtime) {
			inode->i_generation = ++fs->resgdblocks;
			ext2_write_inode(fs, EXT2_RESIZE_INO, inode);
		}
		fs->gdblocks--;
	}

	fs->numgroups--;

	fs->sb.s_inodes_count -= fs->sb.s_inodes_per_group;
	fs->sb.s_blocks_count -= bpg;
	fs->sb.s_free_blocks_count -= bpg - admin;
	fs->sb.s_free_inodes_count -= fs->sb.s_inodes_per_group;
	fs->metadirty |= EXT2_META_SB;

	if (fs->flags & FL_SAFE)
		ext2_sync(fs);

	return 1;
} /* ext2_del_group */

static int ext2_grow_group(struct ext2_fs *fs, blk_t newsize)
{
	int   group;
	blk_t start;
	blk_t bpg;
	blk_t i;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	group = fs->numgroups - 1;
	start = fs->sb.s_first_data_block + group * fs->sb.s_blocks_per_group;
	bpg = fs->sb.s_blocks_count - start;

	if (newsize < bpg) {
		fprintf(stderr, "ext2_grow_group: called to shrink group!\n");
		return 0;
	}

	if (bpg == newsize) {
		if (fs->flags & FL_DEBUG)
			fprintf(stderr, "ext2_grow_group: nothing to do!\n");
		return 0;
	}

	for (i = bpg; i < newsize; i++)
		ext2_set_block_state(fs, start + i, 0, 1);

	fs->sb.s_blocks_count += newsize - bpg;
	fs->metadirty |= EXT2_META_SB;

	if (fs->flags & FL_SAFE)
		ext2_sync(fs);

	return 1;
}

static int ext2_shrink_group(struct ext2_fs *fs, blk_t newsize)
{
	int   group;
	blk_t start;
	blk_t bpg;
	int   itend = fs->itoffset + fs->inodeblocks;
	int   i;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	group = fs->numgroups - 1;

	start = fs->sb.s_first_data_block + group * fs->sb.s_blocks_per_group;
	bpg = fs->sb.s_blocks_count - start;

	if (newsize < itend + (fs->stride ? 2 : 0)) {
		fprintf(stderr, 
			"%s: can't shrink group %d to %d blocks\n",
			__FUNCTION__, group, newsize);
		return 0;
	}

	if (newsize > bpg) {
		fprintf(stderr, "ext2_shrink_group: called to grow group!\n");
			return 0;
	}

	if (bpg == newsize) {
		if (fs->flags & FL_DEBUG)
			fprintf(stderr, "ext2_shrink_group: nothing to do!\n");
		return 0;
	}

	for (i = newsize; i < bpg; i++) {
		if (start + i == fs->gd[group].bg_block_bitmap)
			ext2_block_bitmap_push(fs, group, itend - i);
		else if (start + i == fs->gd[group].bg_inode_bitmap)
			ext2_inode_bitmap_push(fs, group, itend + 1 - i);
		else if (ext2_get_block_state(fs, start + i)) {
			fprintf(stderr,
				"error: block relocator should have moved %i\n",
				start + i);
			return 0;
		}
		ext2_set_block_state(fs, start + i, 1, 0);
	}

	i = bpg - newsize;
	fs->sb.s_blocks_count -= i;
	fs->sb.s_free_blocks_count -= i;
	fs->gd[group].bg_free_blocks_count -= i;
	fs->metadirty |= EXT2_META_SB | EXT2_META_GD;

	if (fs->flags & FL_SAFE)
		ext2_sync(fs);

	return 1;
}


static int ext2_grow_fs(struct ext2_fs *fs)
{
	blk_t diff;
	blk_t sizelast;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	if (!ext2_block_relocate(fs))
		return 0;

	if (!ext2_metadata_push(fs))
		return 0;

	diff = fs->newblocks - fs->sb.s_blocks_count;
	sizelast = fs->sb.s_blocks_count - fs->sb.s_first_data_block -
		(fs->numgroups - 1) * fs->sb.s_blocks_per_group;

	if (sizelast != fs->sb.s_blocks_per_group) {
		blk_t growto;

		growto = sizelast + diff;
		if (growto > fs->sb.s_blocks_per_group)
			growto = fs->sb.s_blocks_per_group;

		if (!ext2_grow_group(fs, growto))
			return 0;

		diff -= growto - sizelast;
	}

	while (diff) {
		sizelast = min(diff, fs->sb.s_blocks_per_group);
		if (!ext2_add_group(fs, sizelast))
			return 0;

		diff -= sizelast;
	}

	return 1;
}

static int ext2_shrink_fs(struct ext2_fs *fs)
{
	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	if (fs->sb.s_blocks_count - fs->sb.s_free_blocks_count + fs->itoffset >
	    fs->newblocks) {
		fprintf(stderr,
			"%s: filesystem is too full to resize to %d blocks.\n"
			"Try resizing it to a larger size first, like %d\n",
			fs->prog, fs->newblocks,
			fs->sb.s_blocks_count - fs->sb.s_free_blocks_count +
			fs->itoffset);
		return 0;
	}

	if (fs->sb.s_inodes_count - fs->sb.s_free_inodes_count >
	    fs->newgroups * fs->sb.s_inodes_per_group) {
		fprintf(stderr,
			"%s: filesystem has too many used inodes to resize"
			"to %i blocks.\n", fs->prog, fs->newblocks);
		return 0;
	}

	if (!ext2_inode_relocate(fs))
		return 0;

	if (!ext2_block_relocate(fs))
		return 0;

	while (fs->sb.s_blocks_count > fs->newblocks) {
		blk_t sizelast = (fs->sb.s_blocks_count -
				  fs->sb.s_first_data_block) -
				 (fs->numgroups - 1) *fs->sb.s_blocks_per_group;

		if (fs->sb.s_blocks_count - sizelast < fs->newblocks) {
			if (!ext2_shrink_group(fs, sizelast + fs->newblocks -
					       fs->sb.s_blocks_count))
				return 0;
		} else {
			if (!ext2_del_group(fs))
				return 0;
		}
	}

	return 1;
}


int ext2_resize_fs(struct ext2_fs *fs)
{
	int status;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	if (fs->newblocks == fs->sb.s_blocks_count) {
		fprintf(stderr, "%s: new size is same as current (%u)\n",
			fs->prog, fs->newblocks);
		return 1;
	}

	if ((fs->relocator_pool =
	     (unsigned char *)malloc(ext2_relocator_pool_size << 10)) == NULL) {
		fprintf(stderr,
			"ext2_resize_fs: error allocating relocator pool!\n");
		return 0;
	}
	fs->relocator_pool_end = fs->relocator_pool +
		(ext2_relocator_pool_size << 10);

	if (fs->newblocks < fs->sb.s_blocks_count)
		status = ext2_shrink_fs(fs);
	else
		status = ext2_grow_fs(fs);

	free(fs->relocator_pool);
	fs->relocator_pool = NULL;
	fs->relocator_pool_end = NULL;

	return status;
}
