/*
 * server.cpp
 *
 * (C) 1998-2000 Murat Deligonul
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * ---
 * does: server code; listening for connections, accepting, logging
 * ---
 *
 */

#include "autoconf.h"

#include <unistd.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifdef HAVE_SYS_TIME_H
    #include <sys/time.h>
#endif
#ifdef HAVE_STDARG_H
    #include <stdarg.h>
#endif
#include <errno.h>
#include <time.h>

#include "server.h"
#include "ruleset.h"
#include "conn.h"
#include "linkedlist.h"
#include "socket.h"

#define MARK fprintf(stderr, "%s: %s: %d: marked\n", __FILE__, __PRETTY_FUNCTION__, __LINE__)


class listen_psock : public pollsocket
{
public:
    listen_psock(int f, u_short port);
protected:
    u_short port;
    virtual int event_handler(const struct pollfd * pfd);

};



time_t time_ctr;

static void kill_conns(void);
int  check_timers(void);

static bool terminate_request = 0;
static bool rehash_request = 0;             
static int  fd_log = -1;
    
static linkedlist<listen_psock> sock_list;

 /*
 * The destructor for linkedlist will kill the all nodes it has
 * allocated but is it up to us to free the data they point to.
 */
int stop_proxy(void)
{
    list_iterator<listen_psock> i(&sock_list);
    do {
        listen_psock * p = i.get();
        delete p;
    } while (++i);

    kill_conns();
    return 1;
}

int ircproxy_listen(u_short port, const struct in_addr *iface)
{
    struct sockaddr_in sin;
    int sock, parm;
    memset(&sin, 0, sizeof sin);
    sin.sin_family = AF_INET;
    sin.sin_port = htons(port);
    if (iface)
        sin.sin_addr = *iface;
    else
        sin.sin_addr.s_addr = INADDR_ANY;
    sock = socket(AF_INET, SOCK_STREAM, 0);
    parm = 1; 
    setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (const char * ) &parm, sizeof(int));
    if (bind(sock, (struct sockaddr *) &sin, sizeof(sin)) == -1)
        return 0;
    if (::listen(sock, 20) == -1)
        return 0;

    /* every thing seems to work */
    DEBUG("Adding a socket fd %d\n", sock);
    listen_psock * p = new listen_psock(sock, port);
    sock_list.add(p);
    return 1;      
} 

int start_proxy(void)
{
    time_t wait_time = 30, old_time;

    old_time = update_time_ctr();
    printlog("Server ready\n");
    
    while (1)
    {   
        if (pollsocket::poll_all(wait_time * 1000) <= 0)
            update_time_ctr();
        
        if (terminate_request)
        {
            terminate_request = 0;
            conn::broadcast("... proxy shutting down now.");
            kill_conns();
            printlog("SIGTERM received: terminating...\n");
            return 1;
        }
        if (rehash_request)
        {
            rehash_request = 0;
            printlog("Reloading config file (signal)\n");
            switch (reload_config_file(&config, config.configfile))
            {
            case 1:
                printlog("Reload successful\n");
                break;
            case 0:
                printlog("Reload failed: %s\n", strerror(errno));
                break;
            case -1:
                printlog("Reload failed\n");
            default:
                break;
            }
        }
        
        /* This does the periodic checks */
        wait_time = time_ctr - old_time;
        if (wait_time < 30)
            wait_time = 30 - wait_time;
        else 
        {
            check_timers();
            wait_time = 30;
            old_time = time_ctr;
        }
    }
    return 1;
}  


/* 
 *  Requests server termination
 *  if now == 1, all conn objects are freed and server exit()s.
 *  if it's 0, a flag is set which the server can read and terminate
 *  when ready.
 */
int ircproxy_die(bool now, const char *reason)
{
    char buff[256];
    if (now)
    {
        strcpy(buff, "Server terminating: ");
        strncat(buff, reason, (sizeof buff) - strlen(buff) + 1);
        conn::broadcast(buff);
        kill_conns();
        exit(0);
        return 1;       
    }
    terminate_request = 1;
    strcpy(buff, "Terminate request: ");
    strncat(buff, reason, (sizeof buff) - strlen(buff) + 1);
    conn::broadcast(buff);   
    return 1;
}   


static void kill_conns(void)
{
    while (conn *c = conn::first())
        delete c;
}

int ircproxy_startlog(const char *logfile)
{
    fd_log = open(logfile, O_CREAT | O_APPEND | O_WRONLY, 0600);
    return (fd_log < 0) ? 0 : 1;
}

int ircproxy_closelog()
{
    printlog("Server log ended\n");
    int ret = close(fd_log);
    fd_log = -1;
    return (ret >= 0);
}

/* we could also have called fdprintf() with fd_log as first argument, but
  since i wanted a nice little timestamp automatically put in front of every 
  log entry, i made a function just for it.*/
int printlog(const char *format, ...)
{
    static char buffer[1024];
    buffer[0] = 0;
    const char *z = timestamp();
    va_list ap;
    
    write(fd_log, z, strlen(z));
    va_start(ap,format);
    vsprintf(buffer, format, ap);
    va_end(ap);
    write(fd_log, buffer, strlen(buffer));
    return 1;   
}


void ircproxy_rehash(void)
{
    rehash_request = 1;
}

void ircproxy_redir_stdxxx(void)
{
    dup2(fd_log, STDOUT_FILENO);
    dup2(fd_log, STDERR_FILENO);   
}


listen_psock::listen_psock(int f, u_short p)
 : pollsocket::pollsocket(f, POLLIN)
{
    port = p;
}

int listen_psock::event_handler(const struct pollfd *)
{
    conn * c = new conn(fd);
    if (c->dead())
        delete c;
    return 1;
}


/* 
 * This function will be called every 30 seconds at least.
 */
int check_timers(void)
{
    
    static int half_min;
    half_min++;
    
    if (!(half_min % 10))
    {
        rule_set *r = rule_set::first(), *r2;
        DEBUG("TIMER: Deleting stale rulesets\n");
        if (r)
        {
            do {
                if (r->dead())
                {
                    r2 = r->next();
                    DEBUG("deleting dead ruleset 0x%lx\n", r);
                    delete r;
                    r = r2;
                }
                else
                    r = r->next();
            } while (r);
        }

        pollsocket::compress();
    }
    
    /* Get rid of idle dccs every minute */
    if (!(half_min % 2))
    {
        DEBUG("TIMER: Killing idle dccs\n");
        if (dcc::num_waiting)
        {
            list_iterator<dcc_list_t> i(&dcc_list_t::list);
            dcc_list_t * dt;
            while (1)
            {
                dt = i++;
                if (!dt)
                    break;
                if ((dt->ptr->lsock) &&
                    (time_ctr - dt->ptr->start_time >= 90))
                {
                    dt->ptr->stat |= 0x800; /* TIMES_UP */
                    dt->ptr->lsock->event_handler(NULL);
                }
            }
        }
    }
        
        
    DEBUG("TIMER: Killing the idlers/deleting stale conns\n");
    if (conn::first())
          conn::do_timers(time_ctr);

    if (half_min == 10)
        half_min = 0;

    return 1;
}

 
