/*
 * socket.cpp
 *
 * (C) 2000 Murat Deligonul
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 */

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>

#ifdef HAVE_POLL_H
#   include <poll.h>
#endif

#ifdef HAVE_SYS_POLL_H
#   include <sys/poll.h>
#endif


#include "autoconf.h"                     
#include "socket.h"
#include "linkedlist.h"
#include "server.h"

int             pollsocket::num_socks = 0;

linkedlist<pollsocket> pollsocket::list;

 
 
#ifdef USE_SELECT
fd_set pollsocket::fds_read  = { };
fd_set pollsocket::fds_write = { };
fd_set pollsocket::res_read = { };
fd_set pollsocket::res_write = { };
int pollsocket::highest_fd = -1;

#warning -----------------------------------------------
#warning Your system does not appear to support poll():
#warning Using select(). 
#warning -----------------------------------------------

#else
struct pollfd * pollsocket::pfds = 0;
int             pollsocket::num_pfds = 0;

/* static */ struct pollfd * pollsocket::find_empty_slot(void)
{
    for (int i = 0; i < num_pfds; i++)
        if (pfds[i].fd == -1)
            return &pfds[i];
        
    return 0;
}

#endif 




pollsocket::pollsocket(int f, int events)
{
    fd = f;
#ifdef USE_SELECT
    if (!num_socks)
    {
        FD_ZERO(&fds_read);
        FD_ZERO(&fds_write);
        FD_ZERO(&res_read);
        FD_ZERO(&res_write);
    }
    if (events & POLLIN)
        FD_SET(fd, &fds_read);
    if (events & POLLOUT)
        FD_SET(fd, &fds_write);
    if (fd > highest_fd)
        highest_fd = fd;
#else
    /* Register this file descriptor */
    struct pollfd * pfd = find_empty_slot();
    DEBUG("Constructing 0x%x: empty slot? 0x%x\n", this, pfd);
    if (!pfd)
    {
        if (!pfds)
        {
            pfds = pfd = (struct pollfd *) malloc(sizeof(struct pollfd));
            num_pfds = 1;
            idx = 0;
        }
        else
        {
            pfds = (struct pollfd *) realloc(pfds, sizeof(struct pollfd) * ++num_pfds);
            pfd = &pfds[num_pfds - 1];
            idx = num_pfds - 1;
        }
    } else
    {
        DEBUG("Base is: %p diff is %d\n",pfds, pfd - pfds);
        idx = (pfd - pfds);
    }
    DEBUG("Constructing %x using pollfd structure @ %x idx %d for fd %d\n", this, pfd, idx, f);
    pfd->fd = f;
    pfd->events = events;
    pfd->revents = 0;
#endif
    list.add(this);
    num_socks++;
}


int pollsocket::close()
{
    DEBUG("pollsocket::close() for %x\n", this);
    if (fd > -1)
    {
        ::close(fd);
#ifdef USE_SELECT
        FD_CLR(fd, &fds_write);
        FD_CLR(fd, &fds_read);
#else
        struct pollfd * pfd = &pfds[idx];
        pfd->fd = -1;
        pfd->events = pfd->revents = 0;
        idx = -1;
#endif
        fd = -1;
        return 1;
    }
    return 0;
}

pollsocket::~pollsocket()
{
    /* If we're the last one on the list... decrease the
     * pollfd count */
#ifdef USE_SELECT
    close();
    num_socks--;
#else
    if (idx > -1 && idx + 1 == num_pfds)
        num_pfds--;
    close();
    num_socks--;
    if (!num_socks)
    {
        free(pfds);
        pfds = 0;
        num_pfds = 0;
    }
    DEBUG("Destructing %x num_socks: %d num_pfds: %d\n", this, num_socks, num_pfds);
#endif
    list.remove(this);
}


int pollsocket::set_revents(int revents)
{
#ifdef USE_SELECT
    if (fd < 0)
        return 0;
    FD_CLR(fd, &res_read);
    FD_CLR(fd, &res_write);
    if (revents & POLLIN)
        FD_SET(fd, &res_read);
    if (revents & POLLOUT)
        FD_SET(fd, &res_write);        
#else
    if (fd > -1)
        pfds[idx].revents = revents;
    else 
        return 0;
#endif
    return revents;
}
    

int pollsocket::set_events(int events)
{
#ifdef USE_SELECT
    if (fd < 0)
        return 0;
    FD_CLR(fd, &fds_read);
    FD_CLR(fd, &fds_write);
    if (events & POLLIN)
        FD_SET(fd, &fds_read);
    if (events & POLLOUT)
        FD_SET(fd, &fds_write);
#else
    if (fd > -1)
        pfds[idx].events = events;
    else 
        return 0;
#endif
    return events;
}


/* FIXME: 
 * We have to keep re-starting the list walk-thru because
 * the list can always change from under us. One call to
 * the event_handler can mean not only the object will be
 * deleted, but others as well.
 */
/* static */ int pollsocket::poll_all(int wait)
{
#ifdef USE_SELECT
    /* FIXME: does not set timeval::tv_usec */
    struct timeval tv;
    int secs = (wait / 1000);
    tv.tv_sec = secs;
    tv.tv_usec = 0;
    memcpy(&res_read, &fds_read, sizeof(fd_set));
    memcpy(&res_write, &fds_write, sizeof(fd_set));

    int r = select(highest_fd + 1, &res_read, &res_write, (fd_set *) 0, &tv);

    if (r > 0)
    {
        pollsocket * p;

        update_time_ctr();
        list_iterator<pollsocket> iter(&list);
        do
        { 
            p = iter++;
            if (!p)
                break;
            
            if (p->fd > -1)
            {
                /* Since the event handler expects a valid pollfd structure,
                 * we must create one */
                struct pollfd pfd;
                pfd.revents = 0;
                pfd.events = 0;
                pfd.fd = p->fd;
                if (FD_ISSET(p->fd, &res_read))
                    pfd.revents |= POLLIN;
                else if (FD_ISSET(p->fd, &res_write))
                    pfd.revents |= POLLOUT;
                if (pfd.revents)
                {
                    int fd = p->fd;
                    p->event_handler(&pfd);
                    FD_CLR(fd, &res_read);
                    FD_CLR(fd, &res_write);
                    iter.set(0);
                }
            }   
        } while (p);
    }
    
#else
    int r = poll(pfds, num_pfds, wait);
    pollsocket * p;
    if (r > 0)
    {
        update_time_ctr();
        for (int i = 0; i < num_pfds; i++)
        {
            if (pfds[i].fd > -1 && pfds[i].revents)
            {
                list_iterator<pollsocket> iter(&list);
                for (; (p = iter.get()); ++iter)
                {
                    if (p->fd == pfds[i].fd)
                    {
                        p->event_handler(&pfds[i]);
                        break;
                    }
                }
            }
        }
    }
#endif
    return r;
}

#ifdef USE_SELECT
int pollsocket::revents() const
{
    int r = 0;
    if (fd < 0)
        return 0;
    if (FD_ISSET(fd, &res_read))
        r |= POLLIN;
    if (FD_ISSET(fd, &res_write))
        r |= POLLOUT;
    return r;
}

int pollsocket::revents(int flags) const
{
    return (revents() & flags);
}
#endif

/*
 * 'Compresses' the pollfd array to get rid of gaps and
 * such:
 */
/* static */ int pollsocket::compress(void)
{
#ifdef USE_SELECT
    /* FIXME: We don't need to do anything. However, we could loop 
     * through and find a new highest_fd */
    return 0;
#else
    int num_confirmed = 0;
    if (num_pfds == num_socks)
        return 0;
    
    DEBUG("pollsocket::compress(): entering with array size of %d\n", num_pfds);
    for (int i = 0; i < num_pfds; i++)
    {
        /* Found an empty slot */
        if (pfds[i].fd == -1)
        {
            DEBUG("Found a hole at location %d\n", i);
            for (int n = i + 1; n < num_pfds; n++)
            {
                if (pfds[n].fd > -1)
                {
                    /* 'Slide' it down */
                    DEBUG("Sliding pfd @ location %d (fd: %d) to %d\n", n, pfds[n].fd, i);
                    memcpy(&pfds[i], &pfds[n], sizeof(struct pollfd));
                    pfds[n].fd = -1;
                    num_confirmed++;
                    break;
                }
            }
        } else
            num_confirmed++;
        DEBUG("Confirmed %d/%d...\n", num_confirmed, num_pfds);
    }

    if (num_confirmed != num_pfds)
    {
        pollsocket * p;
        pfds = (struct pollfd *) realloc(pfds, sizeof (struct pollfd) * num_confirmed);
        
        /* pfds[] array is compressed now:
         * now walk the list of socks and for each
         * run through the pfds[] array and finding the matching fd;
         * update the idx accordingly -- couldn't think of a better way
         * of doing this! */

        list_iterator<pollsocket> iter(&list);
        for (; (p = iter.get()); ++iter)
        {
            if (p->idx > -1 && p->idx < num_pfds && pfds[p->idx].fd != p->fd)
            {
                DEBUG("Trying to match idx for %p (fd: %d)\n", p, p->fd);
                for (int x = 0; x < num_confirmed; x++)
                {
                    if (pfds[x].fd == p->fd)
                    {
                        /* Update idx */
                        DEBUG("Updating idx for %p to %d (fd: %d)\n", p, x, pfds[x].fd);
                        p->idx = x;
                        break;
                    }
                }
            }
        }
        num_pfds = num_confirmed;
    }
    DEBUG("pollsocket::compress(): Exiting with new array size of %d\n", num_pfds);
#endif
    return 1;
}
