/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "ferite.h"

/* this is so that each registered class gets a unique id. this allows for native
 * code to check whether they are being passed one of their objects or someone elses. */
long class_counter = 1000;

/*!
 *  \fn FeriteClass *ferite_register_inherited_class( FeriteScript *script, FeriteNamespace *ns, char *name, char *parent )
 *  \brief Registers, creates and returns a class object
 *  \param script The current script being run
 *  \param ns     The namespace in which the class should be registered
 *  \param name   The name of the class eg. Socket
 *  \param parent The name of the parent class, this is the class that will be inherited from, if NULL no inheirtance is performed.
 * 
 *  This is the only way to create a class natively.
 */
FeriteClass *ferite_register_inherited_class( FeriteScript *script, FeriteNamespace *ns, char *name, char *parent )
{
   FeriteClass *ptr = NULL, *newc = NULL;

   FE_ENTER_FUNCTION;

   if( (__ferite_namespace_element_exists( script, ns, name )) != NULL )
   {
      ferite_warning( NULL, "Class %s already exists can't register\n", name );
      FE_LEAVE_FUNCTION(NULL);
   }
   if( parent != NULL )
   {
      ptr = __ferite_find_class( script, ns, parent );
      if( ptr == NULL )
      {
		 ferite_warning( NULL, "Parent class %s does not exist. Not inheriting from it for %s.\n", parent, name );
      }
   }
   newc = fmalloc( sizeof( FeriteClass ) );
   newc->name = fstrdup( name );
   /* a cunning hack to automatically inhherit varaibles */
   newc->variables = ( ptr == NULL ? 
					   __ferite_variable_hash_alloc( script, FE_CLASS_VARIABLE_HASH_SIZE ) :
					   __ferite_duplicate_variable_hash( script, ptr->variables ) );
   newc->functions = __ferite_create_hash( script, FE_CLASS_FUNCTION_HASH_SIZE );
   newc->id = ++class_counter;
   newc->parent = ptr;
   newc->next = NULL;
   __ferite_register_ns_class( script, ns, newc );

   FE_LEAVE_FUNCTION( newc );
}

/*!
 * \fn int ferite_register_class_function( FeriteScript *script, FeriteClass *classp, FeriteFunction *f )
 * \brief Register a function within a class
 * \param script The current script
 * \param classp The class to place the function in
 * \param f      The function structure
 * 
 * It must be noted that the function being passed must not have the super, and self variables within their signitue, this
 * function will handle that automatically 
 */
int ferite_register_class_function( FeriteScript *script, FeriteClass *classp, FeriteFunction *f )
{
   FE_ENTER_FUNCTION;
   if( classp != NULL )
   {
      if( __ferite_hash_get( script, classp->functions, f->name ) != NULL )
      {
		 ferite_warning( script, "Function %s() already exists in class %s\n", f->name, classp->name );
		 FE_LEAVE_FUNCTION(0);
      }
      __ferite_hash_add( script, classp->functions, f->name, f );
      if( f->type == FNC_IS_EXTRL && !(f->is_static) )
      {
		 /* these are sekret vars :) */
		 f->signature[f->arg_count] = fmalloc( sizeof( FeriteParameterRecord* ) );
		 f->signature[f->arg_count]->variable = __ferite_create_object_variable( "super" );
		 f->signature[f->arg_count++]->has_default_value = 0;
		 f->signature[f->arg_count] = fmalloc( sizeof( FeriteParameterRecord* ) );
		 f->signature[f->arg_count]->variable = __ferite_create_object_variable( "self" );
		 f->signature[f->arg_count++]->has_default_value = 0;
      }
      FE_LEAVE_FUNCTION( 1 );
   }
   FE_LEAVE_FUNCTION( 0 );
}

/*!
 * \fn int ferite_register_class_variable( FeriteScript *script, FeriteClass *classptr, FeriteVariable *variable, int is_static )
 * \brief Register a variable within a class
 * \param script    The current script
 * \param classptr  The class to attach the variable to
 * \param variable  The variable to register
 * \param is_static Boolean, 0 = not static, 1 = static. Allows the specifcation of whether or not the variable has static access within the class
 */
int ferite_register_class_variable( FeriteScript *script, FeriteClass *classptr, FeriteVariable *variable, int is_static )
{
   FE_ENTER_FUNCTION;

   if( classptr != NULL )
   {
      if( variable != NULL )
	  {
		 __ferite_add_variable_to_hash( script, classptr->variables, variable );
		 variable->flags.is_static = is_static;
	  }
	  else
		ferite_error( script, "Can't register a NULL variable in class %s", classptr->name );
   }
   else
     ferite_error( script, "Can't register a variable in a non existant class" );

   FE_LEAVE_FUNCTION( 1 );
}

/*!
 * \fn void __ferite_delete_class( FeriteScript *script, FeriteClass *classp )
 * \breif Clean up and free the memory a class takes up
 * \param script The current script
 * \param classp The class to be deleted
 */
void __ferite_delete_class( FeriteScript *script, FeriteClass *classp )
{
   FE_ENTER_FUNCTION;

   if( classp != NULL )
   {
      FUD(("Deleting Class: %s\n", classp->name ));
      ffree( classp->name );
      __ferite_delete_variable_hash( script, classp->variables );
      __ferite_delete_function_hash( script, classp->functions );
      ffree( classp );
   }

   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn FeriteClass *__ferite_find_class( FeriteScript *script, FeriteNamespace *ns, char *name )
 * \brief Find a class within a namespace
 * \param script The current script
 * \param ns     The top level namespace to look in
 * \param name   The name of the class to find.
 */
FeriteClass *__ferite_find_class( FeriteScript *script, FeriteNamespace *ns, char *name )
{
   FeriteClass *ptr = NULL;
   FeriteNamespaceBucket *nsb = NULL;

   FE_ENTER_FUNCTION;
   FUD(("Trying to find class %s\n", name));
   nsb = __ferite_find_namespace(script,ns,name,FENS_CLS);
   if( nsb )
     ptr = nsb->data;
   FE_LEAVE_FUNCTION( ptr );
}

/*!
 * \fn long __ferite_find_class_id( FeriteScript *script, FeriteNamespace *ns, char *name )
 * \brief Find a class within a namespace, and return it's unique id
 * \param script The current script
 * \param ns     The top level namespace to look in
 * \param name   The name of the class to find.
 */
long __ferite_find_class_id( FeriteScript *script, FeriteNamespace *ns, char *name )
{
   FeriteClass *ptr = NULL;

   FE_ENTER_FUNCTION;

   FUD(("Trying to find class %s\n", name));
   ptr = __ferite_find_class( script, ns, name );
   if( ptr )
   {
      FE_LEAVE_FUNCTION( ptr->id );
   }
   FE_LEAVE_FUNCTION( 0 );
}

/*!
 * \fn void __ferite_delete_class_object( FeriteObject *object )
 * \brief Dispose of an object, the only part of ferite that should be calling this is the garbage collector.
 * \param object The object to be nuked.
 * 
 * If you want to get rid of an object the accepted method is as follows:
 * 
 *    object->refcount = 0;
 *    __ferite_check_gc();
 * 
 * This will dispose of the object properly. The object destructor will get called if the objects' class is native.
 */
void __ferite_delete_class_object( FeriteObject *object )
{
   FeriteScript *script = NULL;
   FeriteFunction *func = NULL;
   FeriteVariable *retv;
   void *(*fptr)( FeriteScript *s, FeriteVariable **plist );
   FeriteVariable **params;
   FeriteClass *clas;

   FE_ENTER_FUNCTION;
#warning __ferite_delete_class_object - you know the score

   if( object != NULL )
   {
      if( object->tmpl != NULL && object->tmpl->functions != NULL )
      {
		 for( clas = object->tmpl; func == NULL && clas != NULL; clas = clas->parent )
		 {
			func = __ferite_hash_get( script, clas->functions, "Destructor" );
		 }
		 if( func != NULL )
		 {
	    /*  we have the destructor */
			if( func->type == FNC_IS_EXTRL )
			{
			   FUD(( "OPS: Calling destructor (%s) in class %s\n", func->name, object->tmpl->name));
			   params = fmalloc( sizeof( FeriteVariable * ) * 3 );
			   params[0] = __ferite_variable_alloc();
			   params[0]->name = fstrdup("DestrcutorVariable");
			   params[0]->type = F_VAR_OBJ;
			   VAO(params[0]) = object;
			   params[1] = params[0];
			   params[2] = NULL;
			   fptr = func->fncPtr;
			   retv = (fptr)( script, params );
			   ffree( params[0]->name );
			   ffree( params[0] );
			   ffree( params );
			   __ferite_variable_destroy( script, retv );
			}
		 }
      }

      FUD(( "Deleting class %s's variable hash %p\n", object->name, object->variables ));
      if( object->variables != NULL )
		__ferite_delete_variable_hash( script, object->variables );

      if( object->name != NULL )
		ffree( object->name );

      ffree( object );
   }
   else
     printf( "Error: trying to delete null object\n" );
   FE_LEAVE_FUNCTION( NOWT );
}

FeriteFunction *__ferite_class_get_function(FeriteScript *script, FeriteClass *cls, char *name) {
   FeriteFunction *ptr = NULL;
   int offset = 0;

   FE_ENTER_FUNCTION;

   if(cls == NULL) {
      FE_LEAVE_FUNCTION( NOWT );
   }

   ptr = __ferite_hash_get(script, cls->functions, name);
   if(ptr == NULL) {
      cls = cls->parent;
      while(cls != NULL) {
         ptr = __ferite_hash_get(script, cls->functions, name);
         if(ptr != NULL) {
            FE_LEAVE_FUNCTION( ptr );
         }
      }
      FE_LEAVE_FUNCTION( NULL );
   }
   FE_LEAVE_FUNCTION( ptr );
}

int __ferite_class_has_function(FeriteScript *script, FeriteClass *cls, char *name) {
   FeriteFunction *ptr = __ferite_class_get_function(script,cls,name);
   if(ptr == NULL) {
      return 0;
   }
   return 1;
}

/*! 
 * \fn FeriteFunction *__ferite_find_function_in_object( FeriteScript *script, FeriteObject *obj, char *function )
 * \brief Find a function in an object, go up the inheirtance tree if necessary
 * \param script   The current script
 * \param obj      The object to search
 * \param function The name of the function to find
 */
FeriteFunction *__ferite_find_function_in_object( FeriteScript *script, FeriteObject *obj, char *function )
{
   FeriteFunction *ptr = NULL;
   FeriteClass *cp = NULL;
   int offset = 0;

   FE_ENTER_FUNCTION;

   FUD(("Looking for function %s\n", function));
   for( offset = strlen( function ) - 1; offset >= 0; offset-- )
   {
      if( function[offset] == '.' )
      {
		 offset++;
		 break;
      }
   }
   if( offset < 0 )
	 offset = 0;
   if( function[offset] == '.' )
     offset++;

   FUD(("Searching for function %s in %s, offset: %d\n", function+offset, obj->name, offset ));
   ptr = __ferite_hash_get( script, obj->tmpl->functions, function+offset );
   if( ptr == NULL )
   {
      /* we need to go up the class tree */
      if( obj->tmpl != NULL )
      {
		 for( cp = obj->tmpl->parent; cp != NULL; cp = cp->parent )
		 {
			ptr = __ferite_hash_get( script, cp->functions, function+offset );
			if( ptr != NULL )
			  break;
		 }
      }
   }
   FE_LEAVE_FUNCTION( ptr );
}

int __ferite_object_has_var(FeriteScript* script,FeriteObject* obj, char *id) {
   FeriteIterator *iter = __ferite_create_iterator(script);
   FeriteHashBucket *buk;
   while((buk = (FeriteHashBucket*)__ferite_hash_walk(script,obj->variables,iter)) != NULL) {
      if(!strcmp(buk->id,id)) {
         return 1;
      }
   }
   return 0;
   ffree(iter);
}

void __ferite_object_set_var(FeriteScript* script,FeriteObject* obj, char *id, void *data) {
   FeriteIterator *iter = __ferite_create_iterator(script);
   FeriteHashBucket *buk;
   while((buk = (FeriteHashBucket*)__ferite_hash_walk(script,obj->variables,iter)) != NULL) {
      if(!strcmp(buk->id,id)) {
         buk->data = __ferite_duplicate_variable(script,(FeriteVariable*)data);
         return;
      }
   }
   ffree(iter);
}

void __ferite_object_call_super( FeriteScript *script, FeriteObject *object, FeriteVariable **params )
{
   FeriteFunction *function = NULL;
   
   FE_ENTER_FUNCTION;   
   if( object->tmpl->parent == NULL )
   {
	  /* return no parental constructor to call */
	  FE_LEAVE_FUNCTION( NOWT );
   }
   
   function = __ferite_find_function_in_object( script, object, object->tmpl->parent->name );
   if( function != NULL )
   {
	  __ferite_variable_destroy( script, __ferite_call_function( script, function, params ) );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

int __ferite_object_is_sublass( FeriteObject *obj, char *name )
{
   FeriteClass *ptr;
   
   FE_ENTER_FUNCTION;
   if( obj != NULL )
   {
	  for( ptr = obj->tmpl; ptr != NULL; ptr = ptr->parent )
	  {
		 if( strcmp( ptr->name, name ) == 0 )
		 {
			FE_LEAVE_FUNCTION(1);
		 }
	  }
   }
   FE_LEAVE_FUNCTION(0);
}
