/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *   
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity 
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *    
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "ferite.h"
#include <math.h>

FeriteIterator *__ferite_create_iterator( FeriteScript *script )
{
   FeriteIterator *i;

   FE_ENTER_FUNCTION;
   i = fmalloc(sizeof(FeriteIterator));
   i->curbucket = NULL;
   i->curindex = 0;
   FE_LEAVE_FUNCTION( i );
}

FeriteArrayItem *__ferite_create_array_item( FeriteScript *script, FeriteHashBucket *ptr, int index)
{
   FeriteArrayItem *i;

   FE_ENTER_FUNCTION;

   i = fmalloc(sizeof(FeriteArrayItem));
   i->var = ptr;
   i->index = index;
   i->in_hash = 0;
   i->next = NULL;
   i->prev = NULL;

   FE_LEAVE_FUNCTION(i);
}

void __ferite_uarray_destroy_item( FeriteScript *script, FeriteUnifiedArray *array, FeriteArrayItem *item )
{
   FE_ENTER_FUNCTION;
/*   printf( "DELETE THE REFERENCE TO __ferite_uarray_destroy_item\n" );*/
   ffree( item );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_uarray_destroy( FeriteScript *script, FeriteUnifiedArray *array )
{
   FeriteArrayItem *next, *cur;

   FE_ENTER_FUNCTION;

   cur = array->linear;
   while( cur != NULL )
   {
      next = cur->next;
      FUD(("deleting index %d...(size %d) next: %p\n", cur->index, array->size, cur->next));
      FUD(("(%s)\n", PTR2VAR(cur->var->data)->name ));
      if( !cur->in_hash )
	__ferite_delete_hash_bucket_list( script, cur->var, (void (*)(FeriteScript*,void *))__ferite_variable_destroy );
      ffree( cur );
      cur = next;
   }
   __ferite_delete_hash( script, array->hash, (void (*)(FeriteScript*,void *))__ferite_variable_destroy );

   if(array->iter != NULL)
   {
      ffree(array->iter);
   }

   /* printf("uarray dead.\n"); */
   ffree( array );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_item_link_after( FeriteScript *script, FeriteArrayItem *cur, FeriteArrayItem *next )
{
   FeriteArrayItem *tmp;

   FE_ENTER_FUNCTION;
   tmp = cur->next;
   cur->next = next;
   next->prev = cur;
   if(tmp != NULL)
   {
      ffree(tmp);
   }
   FE_LEAVE_FUNCTION( NOWT );
}

int __ferite_item_find( FeriteScript *script, FeriteArrayItem *dest, FeriteUnifiedArray *array, int index )
{
   FeriteArrayItem *tmp, *cur;

   FE_ENTER_FUNCTION;
   tmp = array->linear;
   cur = NULL;

   while(tmp->index > index && tmp != NULL)
   {
      cur = tmp;
      tmp = tmp->next;
   }

   /* so, tmp->index is either equal to or less then index */
   if(tmp != NULL)
   {
      dest = tmp;
      FE_LEAVE_FUNCTION( 1 );
   }

   dest = cur;
   FE_LEAVE_FUNCTION( 0 );

}

void __ferite_uarray_add_var( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var, FeriteVariable *index )
{
   FE_ENTER_FUNCTION;

   if(index->type == F_VAR_LONG)
   {
      __ferite_uarray_add( script, array,var,NULL,VAI(index) );
   }
   else if(index->type == F_VAR_STR)
   {
      __ferite_uarray_add( script, array,var,VAS(index),-1 );
   }
   else if(index->type == F_VAR_DOUBLE)
   {
      __ferite_uarray_add( script, array,var,NULL,floor(VAF(index)) );
   };
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_uarray_add( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var, char *id, int index )
{
   FeriteHashBucket *ptr;
   FeriteArrayItem *item, *cur, *prev;
   int in_hash = 0;

   FE_ENTER_FUNCTION;

   /* the array is now bigger  if there is an id, generate the ptr from a hash insert */
   if(id != NULL)
   {
      ptr = __ferite_hash_add_get( script, array->hash, id, var);
      in_hash = 1;
   }
   else
   {
      /* otherwise, generate it by hand, with no id */
      ptr = __ferite_create_hash_bucket( script, NULL, var );
   }

   if(index == -1)
   {
      index = array->size;
   }

   array->size++;
   /* printf("array size %d - adding at %d\n",array->size,index); */

   ptr->index = index;
   item = __ferite_create_array_item( script, ptr, index );

   if(in_hash == 1)
   {
      item->in_hash = 1;
   }

   /* err, easiest way to do this is to treat array->linear == NULL as a special case */
   if(array->linear == NULL)
   {
      array->linear = item;
      {
	 FE_LEAVE_FUNCTION( NOWT );
      }
   }

   if(index == 0)
   {
      cur = array->linear;
      array->linear = item;
      if(cur->index > 0)
      {
	 item->next = cur;
	 cur->prev = item;
      }
      else
      {
	 if(cur->next != NULL)
	 {
	    item->next = cur->next;
	    cur->next->prev = item;
	 }
	 __ferite_uarray_destroy_item( script, array, cur );
      }
      FE_LEAVE_FUNCTION( NOWT );
   }

   cur = array->linear;
   prev = NULL;

   /*  get to the right spot */
   while(cur != NULL && cur->index < index)
   {
		/* printf("at %d...\n",cur->index); */
      prev = cur;
      cur = cur->next;
   }

   /*  ok, prev and cur and now pointing to the right places. */
   /*  NOTE: this function overwrites values, it's not a push */
   /*  first, repoint prev's pointer */
   /*  */
   if(cur != NULL)
   {
      item->next = cur->next;
      __ferite_uarray_destroy_item( script, array, cur );
   }

   if(prev != NULL)
   {
      prev->next = item;
   }

   item->prev = prev;

   FE_LEAVE_FUNCTION( NOWT );

}

FeriteArrayItem *__ferite_uarray_get_index( FeriteScript *script, FeriteUnifiedArray *array, int index )
{
   FeriteArrayItem *item;

   FE_ENTER_FUNCTION;

   if(array->size == 0)
   {
      ferite_error( script,"Invalid index: array size is 0");
      FE_LEAVE_FUNCTION( NULL );
   }
   else
   {
      if(index > array->size)
      {
	 ferite_error( script,"index outside bounds");
	 FE_LEAVE_FUNCTION( NULL );
      }
   }

   item = array->linear;

   while(item != NULL && item->index != index)
   {
      item = item->next;
   }

   if(item == NULL)
   {
      ferite_error( script,"Invalid index (item NULL)");
      FE_LEAVE_FUNCTION( NULL );
   }

   if(item->index == index)
   {
      FE_LEAVE_FUNCTION(item);
   }

   ferite_error( script,"Invalid index (ran out of checks)");
   FE_LEAVE_FUNCTION( NULL );
}

FeriteVariable *__ferite_uarray_get_id( FeriteScript *script, FeriteUnifiedArray *array, char *key )
{
   FeriteHashBucket *buk;

   FE_ENTER_FUNCTION;

   buk = __ferite_hash_get_bucket( script, array->hash, key );

   if(buk == NULL)
   {
      FE_LEAVE_FUNCTION( NULL );
   }

   FE_LEAVE_FUNCTION(buk->data);
}

FeriteVariable *__ferite_uarray_get( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var )
{
   FE_ENTER_FUNCTION;
   switch(var->type)
   {
    case F_VAR_LONG:
      FE_LEAVE_FUNCTION( AIV(__ferite_uarray_get_index( script, array,VAI(var))) );
      break;
    case F_VAR_DOUBLE:
      FE_LEAVE_FUNCTION ( AIV(__ferite_uarray_get_index( script, array,floor(VAF(var)))) );
      break;
    case F_VAR_STR:
      FE_LEAVE_FUNCTION(  __ferite_uarray_get_id( script, array,VAS(var)) );
      break;
   }
   FE_LEAVE_FUNCTION( NULL );
}

FeriteVariable *__ferite_uarray_op(FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *index, void *rhs)
{
   FeriteVariable *ptr = NULL;

   FE_ENTER_FUNCTION;
   if(index->type == F_VAR_LONG && VAI(index) == -1)
   {
      ptr = __ferite_create_void_variable( "UARRAY VARIABLE-1" );
      __ferite_uarray_add( script, array, ptr, NULL, -1);
   }
   else
   {
      ptr = __ferite_uarray_get( script, array,index );
      if(ptr == NULL)
      {
		 ptr = __ferite_create_void_variable( "UARRAY VARIABLE" );
		 __ferite_uarray_add_var( script, array, ptr, index );
      }
   }
   FE_LEAVE_FUNCTION(  ptr );
}

int __ferite_uarray_find_index( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *index)
{
   FeriteHashBucket *buk;

   FE_ENTER_FUNCTION;
   if(index->type == F_VAR_LONG)
   {
      FE_LEAVE_FUNCTION( VAI(index) );
   }
   if(index->type == F_VAR_STR)
   {
      buk = __ferite_hash_get_bucket( script, array->hash,VAS(index));
      if(buk == NULL)
      {
	 FE_LEAVE_FUNCTION( -1 );
      }
      FE_LEAVE_FUNCTION( buk->index );
   }
   FE_LEAVE_FUNCTION( -1 );
}

void __ferite_uarray_del_var( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *index)
{
   int i;

   FE_ENTER_FUNCTION;
   i = __ferite_uarray_find_index( script, array, index );
   __ferite_uarray_del_index( script, array, i );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_uarray_del_index( FeriteScript *script, FeriteUnifiedArray *array, int i)
{
   FeriteArrayItem *item;
   FeriteVariable *var;

   FE_ENTER_FUNCTION;
   item = __ferite_uarray_get_index( script, array, i );

   if(item == NULL)
   {
      ferite_error( script,"Invalid index");
   }

   /*  reindex the linear array first */
   /*  */
   if(item->prev != NULL)
   {
      if(item->next != NULL)
      {
	 item->prev->next = item->next;
	 item->next->prev = item->prev;
      }
      else
      {
	 item->prev->next = NULL;
      }
   }
   else
   {
      if(item->next != NULL)
      {
	 item->next->prev = NULL;
	 array->linear = item->next;
      }
      else
      {
	 array->linear = NULL;
      }
   }

   /*  remove from the hash */
   if(item->in_hash)
   {
      var = __ferite_hash_get( script, array->hash, item->var->id );
      if( var != NULL )
	__ferite_variable_destroy( script, var ); /* ferite_hash_delete will only drop the bucket */
      __ferite_hash_delete( script, array->hash, item->var->id );

      ffree(item);
   }
   else
   {
      __ferite_delete_hash_bucket_list( script, item->var, (void (*)(FeriteScript*,void *))__ferite_variable_destroy );
      ffree( item );
   }
   array->size--;
   FE_LEAVE_FUNCTION( NOWT );
}

/*  this will reindex until it hits a hole */
void __ferite_uarray_reindex_up( FeriteScript *script, FeriteUnifiedArray *array)
{
   FeriteArrayItem *item;
   int cur = 0;

   FE_ENTER_FUNCTION;
   item = array->linear;
   if(item == NULL)
   {
      FE_LEAVE_FUNCTION( NOWT );
   }

   while(item->index == cur && item != NULL)
   {
      item->index++;
      cur++;
      item = item->next;
      if(item == NULL)
      {
	 FE_LEAVE_FUNCTION( NOWT );
      }
   }
   FE_LEAVE_FUNCTION( NOWT );
}

/*  this will renindex down until it hits a hole (starts at 1) */
void __ferite_uarray_reindex_down( FeriteScript *script, FeriteUnifiedArray *array)
{
   FeriteArrayItem *item;
   int cur = 1;

   FE_ENTER_FUNCTION;
   item = array->linear;
   if(item == NULL)
   {
      FE_LEAVE_FUNCTION( NOWT );
   }

   while(item->index == cur && item != NULL)
   {
      item->index--;
      cur++;
      item = item->next;
      if(item == NULL)
      {
	 FE_LEAVE_FUNCTION( NOWT );
      }
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_uarray_clean( FeriteScript *script, FeriteUnifiedArray *array)
{
   FeriteArrayItem *cur, *next;

   FE_ENTER_FUNCTION;
   __ferite_delete_hash( script, array->hash, NULL );
   cur = array->linear;
   while(cur != NULL)
   {
      next = cur->next;
      ffree(cur);
      cur = next;
   }
   FE_LEAVE_FUNCTION( NOWT );
}

FeriteUnifiedArray *__ferite_uarray_dup( FeriteScript *script,FeriteUnifiedArray *array, void *(*ddup)(FeriteScript*,FeriteVariable *data) )
{
   FeriteUnifiedArray *out;
   FeriteArrayItem *item, *cur;
   FeriteHashBucket *buk;

   FE_ENTER_FUNCTION;
   out = fmalloc(sizeof(FeriteUnifiedArray));
   out->hash = __ferite_create_hash( script, array->hash->size );
   out->linear = NULL;
   out->iter = NULL;
   out->size = array->size;

   item = array->linear;
   cur = NULL;
   while(item != NULL)
   {
      if(item->var->id != NULL && strcmp(item->var->id,"") != 0)
      {
	 buk = __ferite_hash_add_get( script, out->hash, item->var->id, (ddup)(script,item->var->data));
      }
      else
      {
	 buk = __ferite_create_hash_bucket( script, item->var->id, (ddup)(script,item->var->data));
      }
      if(cur == NULL)
      {
	 out->linear = __ferite_create_array_item( script, buk, item->index);
	 out->linear->in_hash = item->in_hash;
	 cur = out->linear;
      }
      else
      {
	 cur->next = __ferite_create_array_item( script, buk, item->index);
	 cur->next->in_hash = item->in_hash;
	 cur = cur->next;
      }
      item = item->next;
   }
   FE_LEAVE_FUNCTION( out );
}

void __ferite_uarray_push( FeriteScript *script, FeriteUnifiedArray *array, void *var )
{
   FeriteVariable *v;

   FE_ENTER_FUNCTION;
   v = __ferite_duplicate_variable( script, var );
   __ferite_uarray_add( script, array, v, NULL, -1 );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_uarray_unshift( FeriteScript *script, FeriteUnifiedArray *array, void *var )
{
   FeriteVariable *v;

   FE_ENTER_FUNCTION;
   v = __ferite_duplicate_variable( script, var );
   __ferite_uarray_reindex_up( script, array );
   __ferite_uarray_add( script, array, v, NULL, 0);
   FE_LEAVE_FUNCTION( NOWT );
}

FeriteVariable *__ferite_uarray_pop( FeriteScript *script, FeriteUnifiedArray *array )
{
   FeriteVariable *out;
   FeriteArrayItem *i;

   FE_ENTER_FUNCTION;
   i = __ferite_uarray_get_index( script, array,(array->size)-1);
   out = __ferite_duplicate_variable( script, AIV(i));
   __ferite_uarray_del_index( script, array,(array->size)-1);
   FE_LEAVE_FUNCTION( out );
}

FeriteVariable *__ferite_uarray_shift( FeriteScript *script, FeriteUnifiedArray *array )
{
   FeriteVariable *out;
   FeriteArrayItem *i;

   FE_ENTER_FUNCTION;
   i = __ferite_uarray_get_index( script, array,0);
   out = __ferite_duplicate_variable( script, AIV(i));
   __ferite_uarray_del_index( script, array, 0 );
   __ferite_uarray_reindex_down( script, array );
   MARK_VARIABLE_AS_DISPOSABLE( out );
   FE_LEAVE_FUNCTION(  out );
}

FeriteVariable *__ferite_uarray_item_to_var( FeriteScript *script, FeriteArrayItem *item )
{
   FE_ENTER_FUNCTION;
   if(item != NULL )
   {
      FE_LEAVE_FUNCTION((FeriteVariable *)(item->var->data));
   }
   FE_LEAVE_FUNCTION( NULL );
}

FeriteHashBucket *__ferite_uarray_walk( FeriteScript *script, FeriteUnifiedArray *array, FeriteIterator *iter )
{
   FeriteHashBucket *buk;

   FE_ENTER_FUNCTION;
   if(iter == NULL) {
      if(array->iter == NULL) {
         iter = array->iter = __ferite_create_iterator(script);
      } else {
         iter = array->iter;
      }
   }
   buk = (FeriteHashBucket *)__ferite_hash_walk( script, array->hash,iter );
   FE_LEAVE_FUNCTION( buk );
}

FeriteHashBucket *__ferite_uarray_walk_linear( FeriteScript *script, FeriteUnifiedArray *array, FeriteIterator *iter )
{
   FeriteArrayItem *item;

   FE_ENTER_FUNCTION;
   if(iter == NULL)
   {
      iter = __ferite_create_iterator( script );
   }
   if(iter->curindex == array->size)
   {
      ffree(iter);
      FE_LEAVE_FUNCTION( NULL );
   }
   item = __ferite_uarray_get_index( script, array,iter->curindex );
   iter->curindex++;
   FE_LEAVE_FUNCTION( item->var );
}
