/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *   
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity 
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *    
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "ferite.h"

int variable_count = 0;

FeriteVariable *__ferite_variable_alloc(void)
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   variable_count++;
   ptr = fmalloc( sizeof( FeriteVariable ) );
   ptr->name = NULL;
   ptr->type = 0;
   ptr->data.lval = 0;
   ptr->flags.disposable = 0;
   ptr->flags.compiled = 0;
   ptr->flags.constant = 0;
   ptr->flags.is_static = 0;
   FE_LEAVE_FUNCTION( ptr );
}

void __ferite_variable_destroy( FeriteScript *script, FeriteVariable *var )
{
   FE_ENTER_FUNCTION;
   if( var != NULL )
   {
      FUD(("Freeing %s [%p](%s)\n", var->name, var, __ferite_variable_id_to_str( script, var->type ) ));
      if( var->name )
		ffree( var->name );
      FUD(("Variable Name Free'd\n"));
      if( var->type == F_VAR_STR )
      {
		 FUD(("It's a string. Freeing\n"));
		 FUD(("Trying to free data member\n"));
		 ffree( var->data.sval );
		 FUD(("Variable String Data Free'd\n"));
      }
      if( var->type == F_VAR_OBJ && VAO(var) != NULL )
      {
		 FUD(("Reducing reference count for object variable"));
		 VAO(var)->refcount--;
		 FUD((" :: %d\n", VAO(var)->refcount ));
      }
      if( var->type == F_VAR_UARRAY )
      {
		 __ferite_uarray_destroy( script, VAUA(var));
      }
      ffree( var );
      FUD(("Variable Free'd\n"));
      var = NULL;
   }
   FE_LEAVE_FUNCTION( NOWT );
}

FeriteVariableHash *__ferite_variable_hash_alloc( FeriteScript *script, int size )
{
   FeriteHash *ptr;

   FE_ENTER_FUNCTION;
   ptr = __ferite_create_hash( script, size );
   FE_LEAVE_FUNCTION( ptr );
}

FeriteVariable *__ferite_duplicate_variable( FeriteScript *script, FeriteVariable *var )
{
   FeriteVariable *ptr = NULL;

   FE_ENTER_FUNCTION;
   if( var != NULL )
   {
      switch( var->type )
      {
       case F_VAR_VOID:
		 ptr = __ferite_create_void_variable( var->name );
		 break;
       case F_VAR_LONG:
		 ptr = __ferite_create_number_long_variable( var->name, VAI(var) );
		 break;
       case F_VAR_DOUBLE:
		 ptr = __ferite_create_number_double_variable( var->name, VAF(var) );
		 break;
       case F_VAR_STR:
		 ptr = __ferite_create_string_variable( var->name, VAS(var) );
		 break;
       case F_VAR_OBJ:
		 ptr = __ferite_create_object_variable( var->name );
		 __ferite_variable_destroy( script, __ferite_op_assign( script, ptr, var ) );
		 break;
       case F_VAR_UARRAY:
		 ptr = __ferite_create_uarray_variable( var->name, VAUA(var)->size );
		 __ferite_uarray_destroy( script, VAUA(ptr) );
		 VAUA(ptr) = __ferite_uarray_dup( script, VAUA(var), (void *(*)(FeriteScript*,FeriteVariable*))__ferite_duplicate_variable );
		 break;
       default:
		 ferite_error( script, "Can not duplicate variable of type %d",  var->type );
		 FE_LEAVE_FUNCTION( ptr );
      }
      ptr->flags.constant = var->flags.constant;
	  ptr->flags.is_static = var->flags.is_static;
   }
   FE_LEAVE_FUNCTION( ptr );
}

FeriteVariableHash *__ferite_duplicate_variable_hash( FeriteScript *script, FeriteVariableHash *hash )
{
   FeriteVariableHash *ptr = NULL;

   FE_ENTER_FUNCTION;
   ptr = __ferite_hash_dup( script, hash, (void *(*)(FeriteScript *,void *))__ferite_duplicate_variable );
   FE_LEAVE_FUNCTION( ptr );
}

void __ferite_delete_variable_hash( FeriteScript *script, FeriteVariableHash *hash )
{
   FE_ENTER_FUNCTION;
   if( hash != NULL )
   {
      __ferite_delete_hash( script, hash, (void (*)(FeriteScript *,void *))__ferite_variable_destroy );
   }
   else
     FUD(("ERROR: Trying to free no existant variable hash\n"));
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_add_variable_to_hash( FeriteScript *script, FeriteVariableHash *hash, FeriteVariable *var )
{
   FE_ENTER_FUNCTION;
   __ferite_hash_add( script, hash, var->name, var );
   FE_LEAVE_FUNCTION( NOWT );
}

FeriteVariable *__ferite_get_variable_from_hash( FeriteScript *script, FeriteVariableHash *hash, char *name )
{
   FeriteVariable *ptr;
   FE_ENTER_FUNCTION;
   ptr = __ferite_hash_get( script, hash, name );
   FE_LEAVE_FUNCTION( ptr );
}

FeriteVariable     *__ferite_create_string_variable( char *name, char *data )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   ptr = __ferite_variable_alloc();
   ptr->type = F_VAR_STR;
   ptr->name = fstrdup( name );
   ptr->data.sval = ( data == NULL ) ? fstrdup("") : fstrdup( data );
   FE_LEAVE_FUNCTION( ptr );
}

FeriteVariable     *__ferite_create_number_long_variable( char *name, long data )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   ptr = __ferite_variable_alloc();
   ptr->type = F_VAR_LONG;
   ptr->name = fstrdup( name );
   VAI( ptr ) = data;
   FE_LEAVE_FUNCTION( ptr );
}

FeriteVariable *__ferite_create_number_double_variable( char *name, double data )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   ptr = __ferite_variable_alloc();
   ptr->type = F_VAR_DOUBLE;
   ptr->name = fstrdup( name );
   VAF( ptr ) = data;
   FE_LEAVE_FUNCTION( ptr );
}

FeriteVariable *__ferite_create_object_variable( char *name )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   ptr = __ferite_variable_alloc();
   ptr->type = F_VAR_OBJ;
   ptr->name = fstrdup( name );
   VAO(ptr) = NULL;
   FE_LEAVE_FUNCTION( ptr );
}

FeriteVariable *__ferite_create_uarray_variable( char *name, int size)
{
   FeriteVariable *ptr;
   int rsize = 0;

   FE_ENTER_FUNCTION;
   rsize = (size == 0 ? FE_ARRAY_DEFAULT_SIZE  : size );
   ptr = __ferite_variable_alloc();
   ptr->type = F_VAR_UARRAY;
   ptr->name = fstrdup( name );
   VAUA(ptr) = fmalloc(sizeof(FeriteUnifiedArray));
   VAUA(ptr)->size = 0;
   VAUA(ptr)->hash = __ferite_create_hash( NULL, rsize );
   VAUA(ptr)->linear = NULL;
   VAUA(ptr)->iter = NULL;
   FE_LEAVE_FUNCTION( ptr );
}

FeriteVariable *__ferite_create_void_variable( char *name )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   ptr = __ferite_variable_alloc();
   ptr->type = F_VAR_VOID;
   ptr->name = fstrdup( name );
   FE_LEAVE_FUNCTION( ptr );
}

void __ferite_variables_dump( FeriteVariableHash *hash )
{
   FE_ENTER_FUNCTION;
   FE_LEAVE_FUNCTION( NOWT );
}

char *__ferite_variable_id_to_str( FeriteScript *script, int variable )
{
   FE_ENTER_FUNCTION;
   switch( variable )
   {
    case F_VAR_VOID:
      FE_LEAVE_FUNCTION( "\"void\"" );
    case F_VAR_STR:
      FE_LEAVE_FUNCTION( "\"string\"" );
    case F_VAR_LONG:
      FE_LEAVE_FUNCTION( "\"number\"" );
    case F_VAR_DOUBLE:
      FE_LEAVE_FUNCTION( "\"number\"" );
    case F_VAR_OBJ:
      FE_LEAVE_FUNCTION( "\"object\"" );
    case F_VAR_UARRAY:
      FE_LEAVE_FUNCTION( "\"uarray\"" );
    case F_VAR_NS:
      FE_LEAVE_FUNCTION( "\"namespace\"" );
    case F_VAR_CLASS:
      FE_LEAVE_FUNCTION( "\"class\"" );
   }
   ferite_error( script,  "Trying to id variable of type %d - UNKNOWN", variable );
   FE_LEAVE_FUNCTION( NULL );
}

int __ferite_variable_is_false( FeriteScript *script, FeriteVariable *var )
{
   int retval = 0;

   FE_ENTER_FUNCTION;
   if( var != NULL )
   {
      switch( var->type )
      {
       case F_VAR_VOID:
		 retval = 1;
		 break;
       case F_VAR_LONG:
		 if( VAI(var) == 0 )
		   retval = 1;
		 break;
       case F_VAR_DOUBLE:
		 if( VAF(var) < 0.000001 && VAF(var) > -0.000001 )
		   retval = 1;
		 break;
       case F_VAR_STR:
		 if( strlen( VAS(var) ) == 0 )
		   retval = 1;
		 break;
       case F_VAR_OBJ:
		 if( VAO(var) == NULL )
		   retval = 1;
		 break;
       case F_VAR_UARRAY:
		 if( VAUA(var)->size == 0 )
		   retval = 1;
		 break;
       default:
		 ferite_error( script,  "Can't tell whether type %d has a truth value", var->type );
      }
   }
   else
     retval = 1;
   FE_LEAVE_FUNCTION( retval );
}
